/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*------------------------------------------------------------------------

                    STRUCTURED DISPLAY FILE MANAGER

This module contains the basic routines for managing a structured
display file.  The module was designed to support a network
implementation of the SILT VLSI design-aid system, but has fairly
general applicability to any needs for a structured graphical data
base.

The display file is a structured representation of a list of graphical
objects such as rectangles, points, lines, and symbols, the basic
structured units.  This list is traversed by a window manager to
display the graphics objects appropriately.  This package only builds
the structure.  Other routines are necessary to display it.

The basic unit in the display file is the display record.  This record
is composed of fields indicating the type of the object represented and
other data describing the particular attributes of this realization of
the object. Also included in the display records are link fields to
facilitate the structured representation.

The symbol definition record type is the header record for a circular
list of records which collectively represent a "symbol".  These records
are linked together by the rightSib field in the display records.  The
list may include calls to other symbols, which will be expanded by the
window manager when asked to display the symbol being defined.  A
symbol call is represented by a record type of "symbol call" which has
a link to the appropriate symbol definition record placed in its
leftChild field.

This structure is interfaced via the routines given here.  Procedures
are given for defining new symbols, calling already existent symbols,
inserting primitive graphical objects into a symbol's definition list,
deleting symbol definitions, and editing symbol definitions by
inserting or deleting records from its defining list.

A set of lower level routines including those routines necessary for
maintaining a hash table for locating named symbols is presented
elsewhere (hash.c).  These routines are used by the
procedures presented here, and should be of no interest to the user.
The one exception is a Find(item) procedure which returns a
DISPLAY_RECORD_PTR which gives the location of the display record with
the given name.

                                  Rocky Rhodes
                                  February 18, 1982
------------------------------------------------------------------------
*/

/* Modifications:  Tom Davis added code to keep track of the minimum
 * and maximum dimensions of the symbols. March 26, 1982.
 *
 * Bill Nowicki August 1982
 *	- Moved SdfTrace here from yaleinssym.c
 *	- AddItem now takes optional char * parameter
 *	- Added InquireItem() and InquireCall
 *	- return zero on error instead of printf-ing
 *	- Keep track of redrawing coordinates
 *	- Added CreateSDF and DeleteSDF calls
 *
 * David Kaelbling March 1983
 *	- Updated to include SDF_SPLINE type.
 */

# include "Vgts.h"
# include "sdf.h"
# include "bitmaps.h"
# include "splines.h"

extern short Debug;


SdfTableType SdfTable[NumberSDF];


short InitSDF()
  {
    /*
     * Initializes the SDF Table
     */
     register SdfTableType *sdf = SdfTable;
     register int i;
     
     for (i=0; i<NumberSDF; i++, sdf++)
       {
         sdf->type=UNUSED;
      }
  }


short CreateSDF()
  {
    /*
     * Returns the number of an available SDF. by finding an unused
     * entry in the SdfTable.
     */
     register SdfTableType *sdf = SdfTable;
     register short i;
     
     for (i=0; i<NumberSDF; i++, sdf++)
       {
         if (sdf->type!=UNUSED) continue;
	 sdf->type = GRAPHICS;
         return(i);
      }
    return(-1);
  }



DeleteSDF(i,andStrings)
  short i;
  int andStrings;
    {
      /*
       * Remove all symbol definitions from the indicated SDF.
       * if andStrings is true, all the strings will be freed as well.
       */
       register SdfTableType *sdf = SdfTable + i;
       
       if (i<0 || i >= NumberSDF ) return(-1);
       if (sdf->type==UNUSED) return(i);
       FreeSdfAll(i,andStrings);
       sdf->type = UNUSED;
       return(i);
    }

/* DefineSymbol(item,text_ptr) is called to enter item
in the structured display file.  A display record is created containing
this name and a pointer to some text information describing the symbol.
This entry is also added to the hash tables to enable future calls to
find the proper items.  A list of items should follow this definition
item, and all will be linked together using the rightSib field of the
display records.  The list is terminated by a call to EndSymbol which
also fills in the other fields of the symbol definition entry.  */

DefineSymbol (sdf, item, name)
  short sdf;
  short   item;
  DISPLAY_RECORD_PTR name;	/* name is actually a char pointer */
{
    DISPLAY_RECORD_PTR Insert();
    register SdfTableType *s = SdfTable + sdf;

    s->definingSymbol = s->currentItem = Insert (sdf, item);
    if (s->currentItem==NULL) return(0);
/*  if (Debug) printf("Defining Symbol %d", item ); */
 /* Create a display record with this name and add to hash table. */
    s->currentItem -> type = SDF_SYMBOL_DEF;
    s->currentItem -> rightSib = s->definingSymbol;
    s->currentItem -> leftChild = name;
    s->currentItem -> item = item;
    s->openXmin = s->openYmin = MaxCoord;
    s->openXmax = s->openYmax = -MaxCoord;
    s->redrawXmin = s->redrawYmin = MaxCoord;
    s->redrawXmax = s->redrawYmax = -MaxCoord;
    return (item);
}

/*
 * EndSymbol is called at the end of a list of items defining a
 * symbol.  It closes the named symbol to further additions and fills in
 * some needed information about the symbol that might not have been
 * available at the initial call to DefineSymbol. 
 */

EndSymbol(sdf, item, vgt)
  short sdf;
  short   item;
  unsigned char   vgt;
  {
    register SdfTableType *s = SdfTable + sdf;

    if (s->definingSymbol==NULL || s->definingSymbol -> item != item)
	return(0);

    s->definingSymbol -> xmin = s->openXmin;
    s->definingSymbol -> xmax = s->openXmax;
    s->definingSymbol -> ymin = s->openYmin;
    s->definingSymbol -> ymax = s->openYmax;
    s->definingSymbol -> typedata = vgt;
    s->currentItem = NIL;
    s->definingSymbol = NIL;

    if (vgt) 
      {
        if (s->needToClear)
    	    RedrawVGT( vgt, s->redrawXmin, s->redrawXmax, 
	    		    s->redrawYmin, s->redrawYmax);
	 else
    	    RedrawPartialVGT( vgt, s->redrawXmin, s->redrawXmax, 
	    		    s->redrawYmin, s->redrawYmax, s->addList);
      }
    return(item);
  }

/*
 *  AddItem is the main routine for adding a item to the list of items
 * defining a symbol.  The routine is called with all the necessary
 * information for the display record, and a display record is created
 * with these characteristics.  The global name is used to enter this
 * location in the hash table for future reference.  The global variable
 * "s->currentItem" is used to remember the last item added to the current
 * symbol definition list, and is updated by this routine. 
 */

AddItem(sdf, item, xmin, xmax, ymin, ymax, typedata, type, string)
  short sdf;
  short   item,
        xmin,
        xmax,
        ymin,
        ymax;
  unsigned char   type;
  unsigned char   typedata;
  char	*string;
{
	/*
	 * Create new display record, insert entry in hash table, and link it
	 *  to the chain of items defined for the currently open symbol. 
	 * if the item number is zero, we don't bother putting it into
	 * the hash table.
	 */
    register DISPLAY_RECORD_PTR ptr;
    DISPLAY_RECORD_PTR Insert ();
    register SdfTableType *s = SdfTable + sdf;

    if (s->definingSymbol==NULL) return(0);
    if (s->currentItem == NIL) return(0);
    if ( type<1 || type>SDF_MaxCode) return(0);
    /* Verify some of the user's data for SDF_SPLINE */
    if (type == SDF_SPLINE)
      {
	register SPLINE *item = (SPLINE *) string;
	
        if ((item->order <= 0) || (item->order >= 6))
          {
	    return( 0 );
          }
        if (item->numvert < 1)
          {
	    return( 0 );
          }
	else if (item->numvert == 1)
	  {
	    item->order = 1;
	  }
        if (((int) item->nib < 0) || ((int) item->nib >= (int) NibEnd))
          {
	    if (Debug) printf("AddItem:  Invalid nib %d.\n\r", item->nib );
	    return( 0 );
          }
        if (((int) item->pat < 0) || ((int) item->pat >= (int) PatEnd))
          {
	    if (Debug) printf("AddItem:  Invalid fill pattern %d.\n\r", item->pat );
	    return( 0 );
          }
        if (item->filled && !(item->closed))
          {
	    return( 0 );
          }
      }
    
    ptr = Insert(sdf, item);
    if (ptr==NULL) return(0);
    s->currentItem -> rightSib = ptr;
    switch (type)
      {
	  /*
	   * set up some of the fields to known values, to prevent
	   * users from messing us up
	   */
	case SDF_TEXT:
		TextSize(string,typedata,&xmin,&xmax,&ymin,&ymax);
		break;

	case SDF_VERTICAL_LINE:
    		xmax = xmin + 1;
		break;

	case SDF_HORIZONTAL_LINE:
    		ymax = ymin;
		ymin--;
		break;

	case SDF_POINT:
		ymax = ymin + 1;
		xmax = xmin + 1;
		break;

	case SDF_GENERAL_LINE:
		if (xmin==xmax)
		  {
		    type = SDF_VERTICAL_LINE;
		    xmax = xmin+1;
		    if (ymin > ymax)
		      {
		        register temp;
		        temp = ymin; ymin = ymax; ymax = temp;
		      }
		    break;
		  }
		if (ymin==ymax)
		  {
		    type = SDF_HORIZONTAL_LINE;
		    ymin = ymax-1;
		    if (xmin > xmax)
		      {
		        register temp;
		        temp = xmin; xmin = xmax; xmax = temp;
		      }
		    break;
		  }
    		if (xmin > xmax)
		  {
		    /*
		     * Swap points so (xmin,ymin) is on the left
		     */
		    register temp;
		    temp = xmin; xmin = xmax; xmax = temp;
		    temp = ymin; ymin = ymax; ymax = temp;
		  }
		if (ymin > ymax)
		  {
		    register temp;
		    type = SDF_DownLine;
		    temp = ymin; ymin = ymax; ymax = temp;
		  }
		 else
		  {
		    type = SDF_UpLine;
		  }
		break;
		
	case SDF_RASTER:
		break;

	case SDF_SPLINE:
	    {
		register temp;
		register SPLINE *sptr;
		register POINT *pptr;
		register short xm, ym;
		
		sptr = (SPLINE *) string;
		xm = xmin;
		ym = ymin;
		xmin = xmax = sptr->head.x += xm;
		ymin = ymax = sptr->head.y += ym;
		pptr = &(sptr->head);
		for (temp = 1;  temp < sptr->numvert;  temp++)
		  {	
		    pptr[temp].x += xm;
		    pptr[temp].y += ym;
		    xmax = max( pptr[temp].x, xmax );
		    xmin = min( pptr[temp].x, xmin );
		    ymax = max( pptr[temp].y, ymax );
		    ymin = min( pptr[temp].y, ymin );
		  }
		ymax += 10;  /* Random fudging to take into account	*/
		xmax += 10;  /*  the width of the pens.  (These are	*/
		ymin -= 10;  /*  in World coordinates, and nibs have	*/
		xmin -= 10;  /*  constant size in screen coords.)	*/
		
		/* Convert to relative coordinates */
		for (temp = 0;  temp < sptr->numvert;  temp++)
		  {
		    pptr[temp].x -= xmin;
		    pptr[temp].y -= ymin;
		  }
		break;
	    }
      }
    s->currentItem = ptr;
    if (s->addList==NULL) s->addList = ptr;
    ptr -> item = item;
    ptr -> xmin = xmin;
    ptr -> xmax = xmax;
    ptr -> ymin = ymin;
    ptr -> ymax = ymax;
    ptr -> typedata = typedata;
    ptr -> type = type;
    ptr -> rightSib = s->definingSymbol;

    if (type != SDF_SYMBOL_CALL)
      {
        s->openXmin = min (s->openXmin, xmin);
        s->openYmin = min (s->openYmin, ymin);
        s->openXmax = max (s->openXmax, xmax);
        s->openYmax = max (s->openYmax, ymax);
        s->redrawXmin = min (s->redrawXmin, xmin);
        s->redrawYmin = min (s->redrawYmin, ymin);
        s->redrawXmax = max (s->redrawXmax, xmax);
        s->redrawYmax = max (s->redrawYmax, ymax);
      }
    switch (type)
      {
        case SDF_SIMPLE_TEXT:
        case SDF_TEXT:
	case SDF_HORIZONTAL_REF:
	case SDF_VERTICAL_REF:
	case SDF_SEL_HORIZ_REF:
	case SDF_SEL_VERT_REF:
	case SDF_InternalText:
	case SDF_SPLINE:
    		ptr -> leftChild = (DISPLAY_RECORD_PTR) string;
		break;

	case SDF_RASTER:
	    {   
	      register MemRaster *raster = 
	      		(MemRaster *)(malloc(sizeof(MemRaster)));

		if (string==NULL) 
		  {
		    ptr->leftChild = NULL;
		    return(0);
		  }
		ptr -> leftChild = (DISPLAY_RECORD_PTR) raster;
		if (raster==NULL) return(0);
		raster->next = raster;
		raster->start = (short *)string;
		raster->stride = raster->height = ymax - ymin + 1;
		raster->width = xmax - xmin + 1;
		raster->bitOffset = 0;
		raster->where = (typedata & InverseRaster)
		    |HeadRaster|MemoryRaster|LinkedRaster;
		ptr->xmax++; ptr->ymax++;
		ptr->xmin--; ptr->ymin--;
		break;
	    }
	default:
		ptr -> leftChild = NIL;
		break;
      }
    return(item);
}


AddCall(sdf, item, xoffset, yoffset, symbolName)
  short sdf;
  short   item,
        xoffset,
        yoffset,
        symbolName;
{
/*
 * AddCall is called to insert an instance of one symbol in the
 * definition list of another symbol.  The hashed directory is consulted
 * and a dummy entry is made for this symbol if the definition is not
 * already there. Any future attempts to define this symbol will use this
 * dummy item as the start of the symbol definition list, maintaining
 * consistency with any already existing calls to the symbol.  
 */
    DISPLAY_RECORD_PTR Find (), symbolDef;
    unsigned char   typedata = 0;
    register SdfTableType *s = SdfTable + sdf;

    AddItem(sdf, item, xoffset, 0, yoffset, 0,
	    typedata, SDF_SYMBOL_CALL, NULL);
    if (s->currentItem == NIL)
	return;
    s->currentItem -> leftChild = symbolDef = Find (sdf, symbolName);
    if (symbolDef==NULL) return(0);

	 /*
	  * A symbol call is the same as a AddItem, except the leftChild 
	  * of the created display record is linked to the symbol definition
	  * item of the called symbol type. 
	  */

    s->openXmin = min (xoffset + symbolDef -> xmin, s->openXmin);
    s->openYmin = min (yoffset + symbolDef -> ymin, s->openYmin);
    s->openXmax = max (xoffset + symbolDef -> xmax, s->openXmax);
    s->openYmax = max (yoffset + symbolDef -> ymax, s->openYmax);
    s->redrawXmin = min (xoffset + symbolDef -> xmin, s->redrawXmin);
    s->redrawYmin = min (yoffset + symbolDef -> ymin, s->redrawYmin);
    s->redrawXmax = max (xoffset + symbolDef -> xmax, s->redrawXmax);
    s->redrawYmax = max (yoffset + symbolDef -> ymax, s->redrawYmax);
    return(item);
}


DeleteItem(sdf, item)
  short sdf;
  short   item;
{
	/*
	 * DeleteItem: will delete item from the currently open symbol
	 * definition, if possible.  The display record will be freed,
	 * and item will be stricken from the hash table. Symbol
	 * calls may be deleted just like any other item, but an entire symbol
	 * definition must be deleted by using the DeleteSymbol routine.
	 */

    DISPLAY_RECORD_PTR disprec, saveRec;
    register SdfTableType *s = SdfTable + sdf;

    if (!s->definingSymbol)
	return(0);			/* No symbol is open. Can't delete */

    if (disprec=Find(sdf, item))
      {
        s->needToClear = TRUE;
	if ((disprec -> type == SDF_SYMBOL_CALL) && (disprec->leftChild))
	  {
	    s->redrawXmin = min (s->redrawXmin,
		    disprec -> xmin + disprec -> leftChild -> xmin);
	    s->redrawXmax = max (s->redrawXmax,
		    disprec -> xmin + disprec -> leftChild -> xmax);
	    s->redrawYmin = min (s->redrawYmin,
		    disprec -> ymin + disprec -> leftChild -> ymin);
	    s->redrawYmax = max (s->redrawYmax,
		    disprec -> ymin + disprec -> leftChild -> ymax);
	  }
	else
	  {
	    s->redrawXmin = min (s->redrawXmin, disprec -> xmin);
	    s->redrawXmax = max (s->redrawXmax, disprec -> xmax);
	    s->redrawYmin = min (s->redrawYmin, disprec -> ymin);
	    s->redrawYmax = max (s->redrawYmax, disprec -> ymax);
	  }
      }

    disprec = s->definingSymbol;	/* Start the search for item at the
				   symbol definition item. */

    s->openXmin = s->openYmin = MaxCoord;/* need to re-compute these */
    s->openXmax = s->openYmax = -MaxCoord;

    while ((disprec->rightSib) && (disprec->rightSib->type != SDF_SYMBOL_DEF))
    /* Search until back to the beginning of the circular list. */
    {
	if (disprec -> rightSib -> item == item)
	  {
	    saveRec = disprec -> rightSib;
	    if (saveRec == NIL)
		return;
	    DeleteHash(sdf, item);/* Delete hash table entry for item. */
	    disprec -> rightSib = disprec -> rightSib -> rightSib;
	    ReturnDisplayRecord (saveRec);
  	  }
	if ((disprec->rightSib) && (disprec->rightSib->type!=SDF_SYMBOL_DEF))
	    disprec = disprec -> rightSib;
	if ((disprec -> type == SDF_SYMBOL_CALL) && (disprec->leftChild))
	  {
	    s->openXmin = min (s->openXmin,
		    disprec -> xmin + disprec -> leftChild -> xmin);
	    s->openXmax = max (s->openXmax,
		    disprec -> xmin + disprec -> leftChild -> xmax);
	    s->openYmin = min (s->openYmin,
		    disprec -> ymin + disprec -> leftChild -> ymin);
	    s->openYmax = max (s->openYmax,
		    disprec -> ymin + disprec -> leftChild -> ymax);
	  }
	else
	  {
	    s->openXmin = min (s->openXmin, disprec -> xmin);
	    s->openXmax = max (s->openXmax, disprec -> xmax);
	    s->openYmin = min (s->openYmin, disprec -> ymin);
	    s->openYmax = max (s->openYmax, disprec -> ymax);
	  }
      }
    s->currentItem = disprec;	/* s->currentItem should point to the last
				   item in the circular list. */
    return(item);
}


InquireItem(sdf, item, xmin, xmax, ymin, ymax, typedata, type, string)
    short sdf;
    short item, *xmin, *xmax, *ymin, *ymax;
    unsigned char   *type;
    unsigned char   *typedata;
    char	*string;
  {
	/*
	 * InquireItem -- return the values in the SDF record indicated.
	 * returns non-zero if the item was found.
	 */
    register DISPLAY_RECORD_PTR disp;
    register SdfTableType *s = SdfTable + sdf;

    disp = Find (sdf, item);
    if (disp)
      {
       if (xmin) *xmin = disp->xmin;
       if (xmax) *xmax = disp->xmax;
       if (ymin) *ymin = disp->ymin;
       if (ymax) *ymax = disp->ymax;
       if (typedata) *typedata = disp->typedata;
       if (type) *type = disp->type;
       if (string) strcpy( string, disp->leftChild );
       return(item);
      }
    return(0);
  }


short InquireCall(sdf, item)
    short sdf;
    short item;
  {
	/*
	 * InquireCall -- return the name of the symbol called by the item.
	 * returns zero if item is not found or not a symbol call.
	 */
    register DISPLAY_RECORD_PTR disp;

    disp = Find (sdf, item);
    if (disp && disp->type == SDF_SYMBOL_CALL)
      {
        disp = disp->leftChild;
        if (disp) return(disp->item);
      }
    return(0);
  }


/*
 * ChangeItem(item,xmin,xmax,ymin,ymax,typedata,type,string) is the 
 * only procedure that (possibly) alters the database outside the 
 * currently open symbol.  It merely changes the parameters of an already
 * existent item.  
 */

ChangeItem(sdf,item, xmin, xmax, ymin, ymax, typedata, type, string)
  short sdf;
  short item,
        xmin,
        xmax,
        ymin,
        ymax;
  unsigned char   type;
  unsigned char   typedata;
  char	*string;
{
    register    DISPLAY_RECORD_PTR disprec;
    register SdfTableType *s = SdfTable + sdf;

    disprec = Find (sdf, item);
    if (disprec)
     {		/* If its already there, change it. */
	s->redrawXmin = min( disprec -> xmin, s->redrawXmin);
	s->redrawYmin = min( disprec -> ymin, s->redrawYmin);
	s->redrawXmax = max( disprec -> xmax, s->redrawXmax);
	s->redrawYmax = max( disprec -> ymax, s->redrawYmax);
        s->needToClear = TRUE;

        if (disprec->type != SDF_SYMBOL_CALL)
        {
        switch (type)
          {
	  /*
	   * set up some of the fields to known values, to prevent
	   * users from messing us up.  Note that we do not let
	   * users change symbol calls to other items types.
	   */
	case SDF_TEXT:
		if (string==NULL) string = (char *)disprec->leftChild;
		TextSize(string,typedata,&xmin,&xmax,&ymin,&ymax);
		break;
 
	case SDF_VERTICAL_LINE:
    		xmax = xmin + 1;
		break;

	case SDF_HORIZONTAL_LINE:
    		ymax = ymin;
		ymin--;
		break;

	case SDF_GENERAL_LINE:
		if (xmin==xmax)
		  {
		    type = SDF_VERTICAL_LINE;
		    xmax = xmin+1;
		    if (ymin > ymax)
		      {
		        register temp;
		        temp = ymin; ymin = ymax; ymax = temp;
		      }
		    break;
		  }
		if (ymin==ymax)
		  {
		    type = SDF_HORIZONTAL_LINE;
		    ymax = ymin+1;
		    if (xmin > xmax)
		      {
		        register temp;
		        temp = xmin; xmin = xmax; xmax = temp;
		      }
		    break;
		  }
    		if (xmin > xmax)
		  {
		    /*
		     * Swap points so (xmin,ymin) is on the left
		     */
		    register temp;
		    temp = xmin; xmin = xmax; xmax = temp;
		    temp = ymin; ymin = ymax; ymax = temp;
		  }
		if (ymin > ymax)
		  {
		    register temp;
		    type = SDF_DownLine;
		    temp = ymin; ymin = ymax; ymax = temp;
		  }
		 else
		  {
		    type = SDF_UpLine;
		  }
		break;

	   case SDF_SPLINE:
      {
	register SPLINE *item = (SPLINE *) string;
	
        if ((item->order <= 0) || (item->order >= 6))
          {
	    return( 0 );
          }
        if (item->numvert < 1)
          {
	    if (Debug) printf("ChangeItem:  Can't draw %d vertices.\n\r", item->numvert );
	    return( 0 );
          }
	else if (item->numvert == 1)
	  {
	    item->order = 1;
	  }
        if (((int) item->nib < 0) || ((int) item->nib >= (int) NibEnd))
          {
	    if (Debug) printf("ChangeItem:  Invalid nib %d.\n\r", item->nib );
	    return( 0 );
          }
        if (((int) item->pat < 0) || ((int) item->pat >= (int) PatEnd))
          {
	    if (Debug) printf("ChangeItem:  Invalid fill pattern %d.\n\r", item->pat );
	    return( 0 );
          }
        if (item->filled && !(item->closed))
          {
	    return( 0 );
          }
      }
	    {
		register temp;
		register SPLINE *sptr;
		register POINT *pptr;
		register short xm, ym;
		
		sptr = (SPLINE *) string;
		xm = xmin;
		ym = ymin;
		xmin = xmax = sptr->head.x += xm;
		ymin = ymax = sptr->head.y += ym;
		pptr = &(sptr->head);
		for (temp = 1;  temp < sptr->numvert;  temp++)
		  {
		    pptr[temp].x += xm;
		    pptr[temp].y += ym;
		    xmax = max( pptr[temp].x, xmax );
		    xmin = min( pptr[temp].x, xmin );
		    ymax = max( pptr[temp].y, ymax );
		    ymin = min( pptr[temp].y, ymin );
		  }
		ymax += 10;  /* Random fudging to take into account	*/
		xmax += 10;  /*  the width of the pens.  (These are	*/
		ymin -= 10;  /*  in World coordinates, and nibs have	*/
		xmin -= 10;  /*  constant size in screen coords.)	*/
		
		/* Convert to relative coordinates */
		for (temp = 0;  temp < sptr->numvert;  temp++)
		  {
		    pptr[temp].x -= xmin;
		    pptr[temp].y -= ymin;
		  }
		break;
	    }
		break;
          }
    	  disprec -> type = type;
 	 }
	disprec -> xmin = xmin;
	disprec -> xmax = xmax;
	disprec -> ymin = ymin;
	disprec -> ymax = ymax;
	disprec -> typedata = typedata;
	if (string)
	  disprec -> leftChild = (DISPLAY_RECORD_PTR)string;
		/*
		 * should really change this to recalculate the
		 * whole bounding box as in delete item.
		 */
	s->openXmin = min( xmin, s->openXmin);
	s->openYmin = min( ymin, s->openYmin);
	s->openXmax = max( xmax, s->openXmax);
	s->openYmax = max( ymax, s->openYmax);
	s->redrawXmin = min( xmin, s->redrawXmin);
	s->redrawYmin = min( ymin, s->redrawYmin);
	s->redrawXmax = max( xmax, s->redrawXmax);
	s->redrawYmax = max( ymax, s->redrawYmax);
	return (item);
      }
    else
	return (0);
}

/* 
 * EditSymbol(item) opens item for editing by
 * initializing the fields s->definingSymbol and s->currentItem to
 * the symbol definition item and the last item in the circular definition
 * list before the symbol definition, respectively.  This command has the
 * effect of calling DefineSymbol, inserting all the already existing
 * entries to the definition list, and waiting for additional insertions
 * or deletions. The editing process is ended the same way the initial
 * definition process was ended - a call to EndSymbol. 
 */

EditSymbol(sdf, item)
  short sdf;
  short   item;
{
    register SdfTableType *s = SdfTable + sdf;

    s->definingSymbol = Find (sdf, item);

    if ((s->definingSymbol == NIL) ||
	(s->definingSymbol -> type != SDF_SYMBOL_DEF))
	    return(0);
    s->currentItem = s->definingSymbol;
    s->needToClear = FALSE;

    while (s->currentItem != NIL &&
		s->currentItem -> rightSib != NIL &&
		s->currentItem -> rightSib -> type != SDF_SYMBOL_DEF)
	s->currentItem = s->currentItem -> rightSib;
	 /* Set s->currentItem to the last item in the symbol. */

    s->openXmin = s->definingSymbol -> xmin;
    s->openXmax = s->definingSymbol -> xmax;
    s->openYmin = s->definingSymbol -> ymin;
    s->openYmax = s->definingSymbol -> ymax;
    s->redrawXmin = s->redrawYmin = MaxCoord;
    s->redrawXmax = s->redrawYmax = -MaxCoord;
    s->addList = NULL;
    return(item);
}


/* 
 * DeleteSymbol(item) is used to "unexpand" the definition of a
 * symbol.  The hash table entry will remain and the symbol definition
 * item will remain, in order to prevent dangling references to this
 * symbol by other unknown referees, but all the items within the
 * definition list will be deleted. 
 */

DeleteSymbol(sdf, item)
  short sdf;
  short   item;
{
    register SdfTableType *s = SdfTable + sdf;

    if (!EditSymbol(sdf, item))
	return(0);

    while ((s->definingSymbol->rightSib)
    	&& (s->definingSymbol->rightSib->type!=SDF_SYMBOL_DEF))
	/*
	 * Traverse the circular symbol definition list, deleting everything
         * along the way. 
	 */
	DeleteItem(sdf, s->definingSymbol -> rightSib -> item);
    EndSymbol(sdf, item, 0);
    return(item);
 }


SdfTrace(sdfPtr, indent)
  DISPLAY_RECORD_PTR sdfPtr;
  short indent;
{
	/*
	 * debug trace of a structured display file.
	 * moved from yaleinssym.c by WIN
	 */
  short i;
  char string[32];
  char c;				/* MMT */
  char GetInputChar();

  while (1)
   {
    if (sdfPtr == 0)
        return;
    printf("nxt?");
    c = GetInputChar();			/* MMT */
    if (c == 'n')
        return;

    for (i = 0; i < indent; i++)	/* tab in */
        putchar(' ');

    switch (sdfPtr->type)
      {
        case SDF_SYMBOL_CALL:
    	    strcpy( string,  "Call");
    	    break;

        case SDF_SYMBOL_DEF:
    	    strcpy( string,  "Defn");
    	    break;

        case SDF_FILLED_RECTANGLE:
    	    strcpy( string,  "Rect");
    	    break;

        case SDF_SIMPLE_TEXT:
        case SDF_TEXT:
    	    strcpy( string,  "Text");
    	    break;

        case SDF_OUTLINE:
    	    strcpy( string,  "Outl");
    	    break;

        case SDF_HORIZONTAL_REF:
        case SDF_SEL_HORIZ_REF:
    	    strcpy( string,  "Href");
    	    break;

        case SDF_VERTICAL_REF:
        case SDF_SEL_VERT_REF:
    	    strcpy( string,  "Vref");
    	    break;

        case SDF_FREE:
    	    strcpy( string,  "Free!?!?");
     	    break;

        default:
    	    sprintf( string, "Undefined (%d)", sdfPtr->type );
    	    break;
      }
    
    printf("%s: Num: %d Data: %d, (%d, %d)-(%d, %d)\n",
    	string,
    	sdfPtr->item,
    	sdfPtr->typedata,
    	sdfPtr->xmin,
    	sdfPtr->ymin,
    	sdfPtr->xmax,
    	sdfPtr->ymax);

    if (sdfPtr->type == SDF_SYMBOL_CALL)
        SdfTrace(sdfPtr->leftChild, indent + 4);
    else
        printf("\"%s\"\n", sdfPtr->leftChild ?
	      (char *) sdfPtr->leftChild : "NULL" );
    if (sdfPtr->rightSib==NULL)
       {
         printf( "Right Sibling pointer is NULL!\n");
	 return;
       }
    sdfPtr = sdfPtr->rightSib;
    if (sdfPtr->type == SDF_SYMBOL_DEF) return;
  }
}
