/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * manager.c - the view manager
 *
 * Based on a package for the Yale layout editor by Thomas R. Davis
 *
 *
 * Added dynamic user feedback
 *	Bill Nowicki April 1983
 *
 * Changed Main command menus to optimize text operations
 *	Bill Nowicki July 1983
 *
 */

# include "Vgts.h"
# include "sdf.h"

#define C_CreateView 0
#define C_ExecControl 1
#define C_Expansion 3
#define C_DeleteView 4
#define C_CenterPoint 5
#define C_MoveEdges 6
#define C_MoveView 7
#define C_Redraw 8
#define C_ToggleGrid 9
#define C_Return 10
#define C_Debug 11
#define C_StretchEdges 12
#define C_MakeTop 13
#define C_Zoom 14
#define C_Graphics 15

extern short MouseX, MouseY, MouseButtons;
extern short ScreenLimitX, ScreenLimitY;	/* framebuffer limits */
DISPLAY_RECORD_PTR windowList;

extern short Debug;

static PopUpEntry viewMenu[] =
    {
      "Create View", C_CreateView,
      "Delete View", C_DeleteView,
      "Move Viewport", C_MoveView,
      "Make Top", C_MakeTop,
      "Exec Control", C_ExecControl,
      "Graphics Commands", C_Graphics,
      0, 0
    };

static PopUpEntry graphicsMenu[] =
    {
      "Center Window", C_CenterPoint,
      "Move Edges", C_MoveEdges,
      "Move Edges + Object", C_StretchEdges,
      "Move Viewport", C_MoveView,
      "Expansion Depth", C_Expansion,
      "Redraw", C_Redraw,
      "Toggle Grid", C_ToggleGrid,
      "Debug", C_Debug,
      "Zoom", C_Zoom,
      0, 0
    };

#define SWAP(X, Y) {temp = X; X = Y; Y = temp;}


short FindTopVGT();
short FindTopView();
short FindMouseClick();

/*******************************************************************/


/*
 * manager:
 * Top-level window manager routine.  
 *
 */

short manager(pad)
  short pad;		/* current input pad */
{
  short popCommand, popup(), newpad;

  popCommand = popup(viewMenu);
  switch (popCommand)
	 {
	    case C_CreateView:
		wCreateView();
		break;

	    case C_DeleteView:
		wDeleteView();
		break;

	    case C_ExecControl:
		newpad = ExecControl(pad);
		if (newpad>0) return(newpad);
		break;

	    case C_MoveView:
		wMoveView();
		break;

	    case C_MakeTop:
		wMakeTop();
		break;

	    case C_Graphics:
	    	return( graphics() );
	  }
  return(pad);
}

static graphics(pad)
  short pad;		/* current input pad */
{
  short popCommand, popup(), newpad;

  popCommand = popup(graphicsMenu);
  switch (popCommand)
	 {
	    case C_CreateView:
		wCreateView();
		break;

	    case C_DeleteView:
		wDeleteView();
		break;

	    case C_Debug:
	    	Debug = !Debug;
		break;

	    case C_Expansion:
		wExpansionDepth();
		break;

	    case C_CenterPoint:
    		wCenterPoint();
		break;

	    case C_MoveEdges:
		wMoveEdges(FALSE);
		break;
			
	    case C_StretchEdges:
	        wMoveEdges(TRUE);
		break;

	    case C_Redraw:
	 	SdfExpand ();
		break;

	    case C_ToggleGrid:
		ToggleGrid();
		break;

	    case C_MoveView:
		wMoveView();
		break;

	    case C_MakeTop:
		wMakeTop();
		break;

	    case C_Zoom:
	    	TtyMessage( "Zoom Mode: Left out, Middle in B:", 1);
		return(ZoomMode(pad));
    }
  return(pad);
}


static ZoomMode(pad)
    short pad;
  {
	/*
	 * Zoom in or out in this mode
	 */
      short newpad;

    SetZoomCursor();
    while (1)
      switch (GetMouseInput())
        {
          case LeftButton:			/* ZOOM out */
    		newpad = wZoomOut(MouseX, MouseY);
		if (newpad>0) 
		  {
		    MakeTopXY(MouseX,MouseY);
		    SetMainCursor();
		    return(newpad);
		  }
		break;

          case MiddleButton:			/* Zoom in */
    		newpad = wZoomIn(MouseX, MouseY);
		if (newpad>0) 
		  {
		    MakeTopXY(MouseX,MouseY);
		    SetMainCursor();
		    return(newpad);
		  }
		break;

          case RightButton:			/* return */
		SetMainCursor();
	        newpad = FindTopVGT(MouseX, MouseY);
	        return(newpad>0 ? newpad : pad);
	}

  }

/*******************************************************************/


/*
 * pixelNear:
 * Checks whether x and y are within MaxPixelMiss distance of each other.
 */

static BOOLEAN pixelNear(x, y)
short x, y;
{
    short temp;

    if (x < y)
        SWAP(x, y);
    if ((x - y) < MaxPixelMiss)
        return(TRUE);
    return(FALSE);
}


/*******************************************************************/


/*
 * wMoveView:
 * Rigidly moves the view to the new position.
 */

static wMoveView()
{
    short xNew, yNew, sXmin, sYmin, sXmax, sYmax, sXdisp, x, y,
      sYdisp, oldXmin, oldXmax, oldYmin, oldYmax, window;
    register View_PTR w;  

    TtyPutString("Move viewport rigidly from B: ");

    WaitForDownButton();
    x = MouseX;
    y = MouseY;
    if ((window = FindTopView(x, y, TRUE)) == -1)
      {
        TtyBlinkError("View not found");
        return;
      }
    w = ViewTable + window;

    TtyPutString("to B: ");

    sXmin = oldXmin = w->ScreenXmin;
    sXmax = oldXmax = w->ScreenXmax;
    sYmin = oldYmin = w->ScreenYmin;
    sYmax = oldYmax = w->ScreenYmax;

    FlashOutline(sXmin,sYmin,sXmax,sYmax,AllEdges);
    while (MouseButtons!=0)
      {
        ReadMouseInput();

	FlashOutline(sXmin,sYmin,sXmax,sYmax,AllEdges);
	if (MouseButtons==7)
    	    return(TtyPutString(" Aborted\r\n"));
        xNew = MouseX;
        yNew = MouseY;
        sXdisp = xNew - x;
        sYdisp = yNew - y;
        sXmin = oldXmin + sXdisp;
        sYmin = oldYmin + sYdisp;
	sXmax = sXmin + oldXmax - oldXmin;
	sYmax = sYmin + oldYmax - oldYmin;
	CursorUpdate();
	FlashOutline(sXmin,sYmin,sXmax,sYmax,AllEdges);
      }
    FlashOutline(sXmin,sYmin,sXmax,sYmax,AllEdges);

    TtyPutString("\r\n");

    w->ScreenXmin = sXmin;
    w->ScreenXmax = sXmax;
    w->ScreenYmin = sYmin;
    w->ScreenYmax = sYmax;

    DisplayBackground(w);
    Recalculate(w, w->zoom, sXmin, w->ScreenXmax, sYmin, w->ScreenYmax );
    RedrawScreen(oldXmin,oldXmax,oldYmin,oldYmax,w->next);
    Cursor();
    RedrawEntireView(window);
}

static wMakeTop()
{
    short xNew, yNew, sXmin, sYmin, sXdisp, window, i,
      sYdisp, oldXmin, oldXmax, oldYmin, oldYmax;
    register View_PTR w;
    	/*
	 * Make a selected window thetop one
	 */

    TtyPutString("Bring view to top B: ");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
    if ((window = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("View not found");
        return;
      }
    w = ViewTable + window;

    TtyPutString("\r\n");
    MakeTop(w);
    SmashViews();
    RedrawEntireView(window);
}


/*******************************************************************/


/*
 * wCreateView:
 * Gets a pair of mouse clicks and builds a window
 * using those two spots as corners.
 */

static wCreateView()
{
    short xCorner, yCorner, xStart, yStart;
    short xCorner1, yCorner1, temp, vgt;

    TtyPutString("Creating a view, corners at B: ");
    SetViewCursor();
    WaitForDownButton();
 
    xCorner1 = xCorner = xStart = MouseX;
    yCorner1 = yCorner = yStart = MouseY;

    TtyPutString("and B: ");
    while (MouseButtons!=0)
      {
        ReadMouseInput();
	
	FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
	if (MouseButtons==7)
	  {
            SetMainCursor();
    	    TtyPutString(" Aborted\r\n");
    	    return;
	  }

	CursorUpdate();

	xCorner  = xStart;
	yCorner  = yStart;
        xCorner1 = MouseX;
        yCorner1 = MouseY;

/* Guarantee that (xCorner, yCorner) is upper-left. */

        if (xCorner > xCorner1) SWAP(xCorner, xCorner1);
        if (yCorner > yCorner1) SWAP(yCorner, yCorner1);

	FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
      }

    TtyPutString("\r\n");

    TtyPutString("Select VGT to use B: ");
    temp = GetMouseInput();
    if (temp==7)
      {
        FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
        SetMainCursor();
	return(TtyPutString(" Aborted\r\n"));
      }
     TtyPutString("\r\n");
    if (temp==RightButton)
        vgt = GetVGTfromMenu();
    else
        vgt = FindTopVGT(MouseX, MouseY);
    FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
    SetMainCursor();
    if (vgt==-1) return;
    CreateView( vgt, xCorner, yCorner, xCorner1, yCorner1, 0, 0, 0, TRUE);
}


GetVGTfromMenu()
  {
    /*
     * provide a pop-up menu with the names of the VGTs so
     * that users can pick out the one they want by name
     */
     PopUpEntry vgtPopUp[MAX_VGTS];
     register PopUpEntry *p = vgtPopUp;
     VGT *v = VGTtable;
     short vgt;
     
     for (vgt=0;vgt<MAX_VGTS;vgt++,v++)
       {
         if (v->type == UNUSED) continue;
	 p->menuNumber = vgt;
	 p->string = v->name;
	 p++;
       }
     p->string = NULL;
     p->menuNumber = -1;
     return(popup(vgtPopUp));
  }



DefaultView(vgt, width, height, wXmin, wYmin, zoom, showgrid,
		pWidth, pHeight )
    short vgt, 
	  width, height,	/* zero means user determines it */
	  wXmin, wYmin,		/* offset of world to screen */
	  zoom, showgrid,	/* default viewing parameters */
	  *pWidth, *pHeight;	/* returns the values really used */
  {
     /*
      * Position a default View of the given VGT
      */
    short xCorner, yCorner, xStart, yStart;
    short xCorner1, yCorner1, temp;

    TtyPutString("Creating a default view, corners at B: ");
    SetViewCursor();
    WaitForDownButton();
 
    xCorner = xStart = MouseX;
    yCorner = yStart = MouseY;

    xCorner1 = width>0  ? MouseX+width  : xStart;
    yCorner1 = height>0 ? MouseY+height : yStart;
    FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);

    TtyPutString("and B: ");
    while (MouseButtons!=0)
      {
        ReadMouseInput();
	
	FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
	if (MouseButtons==7)
	  {
            SetMainCursor();
    	    TtyPutString(" Aborted\r\n");
    	    return;
	  }

	CursorUpdate();

	xCorner  = MouseX;
	yCorner  = MouseY;
        xCorner1 = width>0  ? MouseX+width  : xStart;
        yCorner1 = height>0 ? MouseY+height : yStart;

/* Guarantee that (xCorner, yCorner) is upper-left. */

        if (xCorner > xCorner1) SWAP(xCorner, xCorner1);
        if (yCorner > yCorner1) SWAP(yCorner, yCorner1);

	FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
      }

    TtyPutString("\r\n");
    FlashOutline(xCorner,yCorner,xCorner1,yCorner1,AllEdges);
    SetMainCursor();
    CreateView( vgt, xCorner, yCorner, xCorner1, yCorner1, 
      		wXmin, wYmin, zoom, showgrid);
    if (pWidth)  *pWidth  = xCorner1 - xCorner;
    if (pHeight) *pHeight = yCorner1 - yCorner;
  } 

/*******************************************************************/


/*
 * wDeleteView:
 * Gets a mouse click and deletes the indicated window.
 */

static wDeleteView()
{
    short window;

    TtyPutString("Delete view B: ");  
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
     if ((window = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("View not found");
        return;
      }
    DeleteView(window);
    TtyPutString("\r\n");

/* +++ We should also free up all the stuff in the SDF */

    return;
}


/*******************************************************************/


/*
 * wMoveEdges:
 * Adjusts the size of the window whose edge or corner
 * was bugged when the routine was called.  The next mouse coordinates are
 * the new values for that corner.
 * The stretch flag determines if the window should be dragged along
 * the opposite side (stretch==false) or the side moved (stretch==true).
 */

static wMoveEdges(stretch)
short stretch;
{
    short i, window, round;
    short newXmin, newYmin, newXmax, newYmax;
    short oldXmin, oldYmin, oldXmax, oldYmax;
    short newWXmin, newWYmin, x, y;
    int edges;
    BOOLEAN pixelNear();
    register View_PTR w;  

   if (stretch)
       TtyPutString("Stretch viewport edge or corner from B: "); 
   else
       TtyPutString("Move viewport edge or corner from B: ");

    WaitForDownButton();

    x = MouseX;
    y = MouseY;
    if ((window = FindTopView(x, y, TRUE)) == -1)
      {
        TtyBlinkError("edge not found");
        return;
      }

     w = ViewTable + window;

/* Now we have to figure out which edge and corner we are near. */

    newXmin = oldXmin = w->ScreenXmin;
    newXmax = oldXmax = w->ScreenXmax;
    newYmin = oldYmin = w->ScreenYmin;
    newYmax = oldYmax = w->ScreenYmax;
    newWXmin = w->WorldXmin;
    newWYmin = w->WorldYmin;

	/*
	 * determine which edges we are near
	 */
    edges = 0;
    if (pixelNear(w->ScreenXmin, x))
        edges |= LeftEdge;
    if (pixelNear(w->ScreenXmax, x))
        edges |= RightEdge;
    if (pixelNear(w->ScreenYmin, y))
        edges |= TopEdge;
    if (pixelNear(w->ScreenYmax, y))
        edges |= BottomEdge;

    if (edges==0)
      {
        TtyBlinkError("can't find edge or corner");
        return;
      }

    FlashOutline(newXmin,newYmin,newXmax,newYmax,AllEdges);
    TtyPutString("to B: ");
    while (MouseButtons!=0)
      {
        ReadMouseInput();

	FlashOutline(newXmin,newYmin,newXmax,newYmax,AllEdges);
	if (MouseButtons==7)
    	    return(TtyPutString(" Aborted\r\n"));

	if (edges & LeftEdge)     newXmin = min(newXmax,MouseX);
	if (edges & RightEdge)    newXmax = max(newXmin,MouseX);
	if (edges & TopEdge)      newYmin = min(newYmax,MouseY);
	if (edges & BottomEdge)   newYmax = max(newYmin,MouseY);

	CursorUpdate();
	FlashOutline(newXmin,newYmin,newXmax,newYmax,AllEdges);
      }
    FlashOutline(newXmin,newYmin,newXmax,newYmax,AllEdges);

    TtyPutString("\r\n");

    if ((newXmax <= newXmin) || (newYmax <= newYmin))
      {
        TtyBlinkError("illegal stretch");
        return;
      }

/*
 *  must twiddle world coords, if we are moving certain edges.
 */
    if (w->zoom > 0)	round = ( 1 << (w->zoom - 1) );
     else		round = 0;

    if (!stretch && (edges&LeftEdge) )
      {
        if (w->zoom>=0) newWXmin = (newXmin - w->Xconst + round) >> w->zoom;
	else 		newWXmin = (newXmin - w->Xconst) << - w->zoom;
      }

    if (!stretch && (edges&BottomEdge))
      {
        if (w->zoom>=0) newWYmin = (w->Yconst - newYmax + round) >> w->zoom;
	else 		newWYmin = (w->Yconst - newYmax) << - w->zoom;
      }
    
    if (stretch && (edges&RightEdge))
      {
        if (w->zoom>=0) newWXmin += 
		(w->ScreenXmax - newXmax + round) >> w->zoom;
	else 		newWXmin += 
		(w->ScreenXmax - newXmax) << - w->zoom;
      }

    if (stretch && (edges&TopEdge))
      {
        if (w->zoom>=0) newWYmin += 
		(newYmin - w->ScreenYmin + round) >> w->zoom;
	else 		newWYmin += 
		(newYmin - w->ScreenYmin) << - w->zoom;
      }

    DisplayBackground(w);
    w->ScreenXmin = newXmin;
    w->ScreenXmax = newXmax;
    w->ScreenYmin = newYmin;
    w->ScreenYmax = newYmax;
    w->WorldXmin = newWXmin, 
    w->WorldYmin = newWYmin;
    Recalculate(w, w->zoom, newXmin, newXmax, newYmin, newYmax);
    RedrawScreen(oldXmin,oldXmax,oldYmin,oldYmax,w->next);
    RedrawEntireView(window);
}


/*******************************************************************/


/*
 * wCenterPoint:
 * Brings the moused point to the center of the
 * window.
 */

static wCenterPoint(x, y)
short x, y;
{
    short window;
    short worldX, worldY;

    TtyPutString("Point to make center B: ");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
    TtyPutString("\r\n");
    if ((window = FindMouseClick(MouseX, MouseY, &worldX, &worldY)) == -1)
        return;

    ZoomTo( worldX, worldY, window);
}


/*******************************************************************/


/*
 * wZoomIn:
 * Brings the moused point to the center of the window.
 * Zooms in by a factor of 2.  If it is not a graphics
 * window, we bring it to the top, and return its pad number.
 */

static wZoomIn(x, y)
short x, y;
{
    short window;
    short worldX, worldY;

    if ((window = FindMouseClick(x, y, &worldX, &worldY)) == -1)
        return;
    if (ViewTable[window].usage&TTY)
        return(ViewTable[window].vgt);

    ZoomIn( worldX, worldY, window);
    return(-1);
}


/*******************************************************************/


/*
 * wZoomOut:
 * Brings the moused point to the center of the window.
 * Zooms out by a factor of 2.
 */

static wZoomOut(x, y)
short x, y;
  {
    short window;
    short worldX, worldY;

    if ((window = FindMouseClick(x, y, &worldX, &worldY)) == -1)
        return;

    if (ViewTable[window].usage&TTY)
        return(ViewTable[window].vgt);

    ZoomOut( worldX, worldY, window);
    return(-1);
  }



/*******************************************************************/


/*
 * wExpansionDepth:
 * set the expansion depth of a given window
 */
static PopUpEntry depthPopup[] =
  {
    "0", 0,
    "1", 1,
    "2", 2,
    "3", 3,
    "4", 4,
    "5", 5,
    "all", MaxCoord,
    0, 0
  };


static wExpansionDepth()
{
    short window, depth;

    TtyPutString("Select view to change expansion depth B: ");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
    window = FindTopView(MouseX, MouseY, FALSE);
    if (window<0) 
      {
        TtyPutString("View not found\r\n");
	return;
      }
    TtyPutString("\r\nSelect new depth M:");
    depth = popup(depthPopup);
    if (depth>=0)
      {
        ViewTable[window].expansionDepth = depth;
        RedrawEntireView(window);
      }
    TtyPutString("\r\n");
}


/*******************************************************************/


/*
 * ToggleGrid:
 * Toggles the grid flag for a designated (= moused) window.
 */

ToggleGrid()
  {
    short windowId;

    TtyPutString("View to toggle:");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
    TtyPutString("\r\n");
     windowId = FindTopView(MouseX, MouseY, FALSE);
    ViewTable[windowId].showGrid = ! ViewTable[windowId].showGrid;
    RedrawEntireView(windowId);
  }
