/************************************************/
/*						*/
/*	       (C)  COPYRIGHT 1983		*/
/*		BOARD OF TRUSTEES		*/
/*	LELAND STANFORD JUNIOR UNIVERSITY	*/
/*	  STANFORD, CA. 94305, U. S. A.		*/
/*						*/
/************************************************/
 
/*
 *  line.c
 *
 *  This file contains the basic line drawing routines.  Specifically,
 *  it contains XYputLine(), which draws lines by calling XYbrush(),
 *  XYdrawLine(), which draws lines by calling the vector operations,
 *  and XYline(), which attempts to decide which routine will run
 *  faster, and then call it.
 *
 *  XYputLine() is based on a simple encoding of a DDA algorithm.
 *  This proedure will accept the end points of a line, and call a
 *  specified function, passing it the coordinates of each point on
 *  the line.  Typically the Actor function will be XYbrush, and the 
 *  nib variable will be meaningful to it.
 *
 *  XYdrawLine() will (in theory) have the same results as calling
 *  XYline() with XYbrush(), modulo rounding differences.  It
 *  utilizes the (fast) vector routines available on the SUN's.
 *
 *  Update History:
 *	March 1983 -- Written by David Kaelbling.
 */
 
#include "Vgts.h"
#include "sdf.h"
#include "splines.h"
 
/* Imports */
extern XYbrush();
extern DisplayLine();
    
    
/* Exports */
extern XYputLine();
extern XYdrawLine();
extern XYline();
    
/*
 *  Draw lines by placing the pen down at each point along the line.
 */
 
XYputLine( x1, y1, x2, y2, Actor, clip, nib )
	int x1, y1, x2, y2;
	register int (*Actor)();
	struct SubView *clip;
	enum Nib nib;
  {
    int length;
    register int i;
    register int xinc, yinc;
    register int x, y;
    
    /* Compute line length in pixels */
    length = abs( x2 - x1 );
    if ((i = abs( y2 - y1 )) > length)
        length = i;
    
    /* Precompute increments for the inner loop */
    if (length)
      {
	xinc = ((x2 - x1) << 16) / length;
	yinc = ((y2 - y1) << 16) / length;
      }
    else
      {
        xinc = yinc = 0;
      }
    x = (x1<<16) + (1<<15);
    y = (y1<<16) + (1<<15);
    
    /* Plot the line */
    for (i = length + 1;  i;  i--)
      {
	Actor( (short) (x>>16), (short) (y>>16), clip, nib );
	x += xinc;
	y += yinc;
      }
  }

/* Local Definitions for XYdrawLine() */
    
/* Offsets to use with NibCircle2 */
static short DirCircle2[] =
  {   16,	/* Pairs of offsets */
    -1, -2,	     0, -2,	     1, -2,	     1, -1,
     2, -1,	     2,  0,	     2,  1,	     1,  1,
     1,  2,	     0,  2,	    -1,  2,	    -1,  1,
    -2,  1,	    -2,  0,	    -2, -1,	    -1, -1,
  };
    
/* Offsets to use with NibCircle3 */
static short DirCircle3[] =
  {   24,	/* Pairs of offsets */
    -1, -3,	     0, -3,	     1, -3,	     1, -2,
     2, -2,	     2, -1,	     3, -1,	     3,  0,
     3,  1,	     2,  1,	     2,  2,	     1,  2,
     1,  3,	     0,  3,	    -1,  3,	    -1,  2,
    -2,  2,	    -2,  1,	    -3,  1,	    -3,  0,
    -3, -1,	    -2, -1,	    -2, -2,	    -1, -2,
  };

/*
 *  This routine provides services identical to that of XYline() using
 *  XYbrush() as an Actor, but utilizes the (faster) builtin vector
 *  operations of the SUN workstation to achieve greater efficiency.
 */
    
XYdrawLine( x1, y1, x2, y2, clip, nib )
	register short x1, y1, x2, y2;
	struct SubView *clip;
	enum Nib nib;
  {
    register short xinc, yinc, i;
    
    switch (nib)
      {
        case NibSquare0:
	    DisplayLine( x1, x2, y1, y2, clip );
	    break;
	    
	case NibSquare1:
	    xinc = ((x2 > x1) ? -1 : 1);
	    for (yinc = -1;  yinc <= 1;  yinc++)
                DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	    yinc = ((y2 > y1) ? -1 : 1);
	    for (xinc = -1;  xinc <= 1;  xinc++)
	        DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	    XYbrush( x2, y2, clip, nib );
	    break;
    
	case NibSquare2:
	    xinc = ((x2 > x1) ? -2 : 2);
	    for (yinc = -2;  yinc <= 2;  yinc++)
                DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	    yinc = ((y2 > y1) ? -2 : 2);
	    for (xinc = -2;  xinc <= 2;  xinc++)
	        DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	    XYbrush( x2, y2, clip, nib );
	    break;
    
	case NibSquare3:
	    xinc = ((x2 > x1) ? -3 : 3);
	    for (yinc = -3;  yinc <= 3;  yinc++)
                DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	    yinc = ((y2 > y1) ? -3 : 3);
	    for (xinc = -3;  xinc <= 3;  xinc++)
	        DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	    XYbrush( x2, y2, clip, nib );
	    break;
	    
	case NibCircle0:
	    DisplayLine( x1, x2, y1, y2, clip );
	    break;
	    
	case NibCircle1:
	    DisplayLine( x1++, x2++, y1, y2, clip );
	    DisplayLine( x1--, x2--, y1++, y2++, clip );
	    DisplayLine( x1--, x2--, y1--, y2--, clip );
	    DisplayLine( x1++, x2++, y1--, y2--, clip );
	    DisplayLine( x1, x2, y1, y2, clip );
	    break;
	    
	case NibCircle2:
	  {
	    register short *tmp = DirCircle2;
    
	    i = *tmp++;
	    while (i--)
	      {
	        xinc = *tmp++;  yinc = *tmp++;
		DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	      }
	    break;
	  }
	
	case NibCircle3:
	  {
	    register short *tmp = DirCircle3;
    
	    i = *tmp++;
	    while (i--)
	      {
	        xinc = *tmp++;  yinc = *tmp++;
		DisplayLine( x1+xinc, x2+xinc, y1+yinc, y2+yinc, clip );
	      }
	    break;
	  }
    
	/* NB:  Dash cases fall through each other. */
	case NibDash3:
	    DisplayLine( x1-3, x2-3, y1, y2, clip );
	    DisplayLine( x1+3, x2+3, y1, y2, clip );
	    
	case NibDash2:
	    DisplayLine( x1-2, x2-2, y1, y2, clip );
	    DisplayLine( x1+2, x2+2, y1, y2, clip );
    
	case NibDash1:
	    DisplayLine( x1-1, x2-1, y1, y2, clip );
	    DisplayLine( x1+1, x2+1, y1, y2, clip );
	    
	case NibDash0:
	    DisplayLine( x1, x2, y1, y2, clip );
	    break;
	
	/* NB:  Bar cases fall though each other. */
	case NibBar3:
	    DisplayLine( x1, x2, y1-3, y2-3, clip );
	    DisplayLine( x1, x2, y1+3, y2+3, clip );
	    
	case NibBar2:
	    DisplayLine( x1, x2, y1-2, y2-2, clip );
	    DisplayLine( x1, x2, y1+2, y2+2, clip );
	    
	case NibBar1:
	    DisplayLine( x1, x2, y1-1, y2-1, clip );
	    DisplayLine( x1, x2, y1+1, y2+1, clip );
	    
	case NibBar0:
	    DisplayLine( x1, x2, y1, y2, clip );
	    break;
	    
	default:
	    printf("XYDRAWLINE:  Unknown nib type %d.\n\r", nib);
	    break;
      }
  }

/* Local definitions for XYline() */
static short Cost[] =		/* Cost of using XYdrawLine() */
  {
    1,		/* NibSquare0 */
    6,		/* NibSquare1 */
    10,		/* NibSquare2 */
    14,		/* NibSquare3 */
    1,		/* NibCircle0 */
    5,		/* NibCircle1 */
    16,		/* NibCircle2 */
    24,		/* NibCircle3 */
    1,		/* NibDash0 */
    3,		/* NibDash1 */
    5,		/* NibDash2 */
    7,		/* NibDash3 */
    1,		/* NibBar0 */
    3,		/* NibBar1 */
    5,		/* NibBar2 */
    7,		/* NibBar3 */
  };

/*
 *  This routine will accept all of the parameters necessary to
 *  invoke one of the line drawing routines, and then call it.
 *  It attempts to decide which line routine will run faster based
 *  upon the length of the line being drawn, and the complexity
 *  of the nib being used to paint it.
 */
 
XYline( x1, y1, x2, y2, Actor, clip, nib )
	register int x1, y1, x2, y2;
	register int (*Actor)();
	register struct SubView *clip;
	register enum Nib nib;
  {
    register int length;
    
    /* Discard the obviously worthless lines */
    if ((clip->ScreenXmin > max(x1, x2)) ||
        (clip->ScreenXmax < min(x1, x2)) ||
	(clip->ScreenYmin > max(y1, y2)) ||
	(clip->ScreenYmax < min(y1, y2)))
	return( 0 );
    
    /* Compute line length in pixels */
    length = max( abs( x2 - x1 ), abs( y2 - y1 ) );
    
    /* Make a guess */
    if ((Actor == XYbrush) &&
        (length >= (Cost[(int)nib]<<1)) &&
        (max(x1, x2) < clip->ScreenXmax) &&
	(min(x1, x2) > clip->ScreenXmin) &&
	(max(y1, y2) < clip->ScreenYmax) &&
	(min(y1, y2) > clip->ScreenYmin))
        XYdrawLine( x1, y1, x2, y2, clip, nib );
    else
	XYputLine( x1, y1, x2, y2, Actor, clip, nib );
  }
