/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*
 * interp.c - interpret escape sequences as VGTS commands
 *
 * Bill Nowicki September 1982
 */


# include "Vgts.h"
# include "Vioprotocol.h"
# include "remote.h"
# include "chars.h"
# include "client.h"
# include "sdf.h"
# include "pad.h"

extern short Debug;


struct ParseEntry
  {
  	/*
	 * Table for parsing the escape sequences into VGTS requests.
	 * THIS MUST BE KEPT UP TO DATE with the arguments
	 * of the corrsponding routines!
	 */
    enum VgtsRequest req;	/* request type for double-checking */
    short numargs;		/* number of numeric arguments */
    char stringFlag;		/* true if last arg is a string */
  } ParseTable[] =
  {
 /*    Request code   # args   string? */
    VgtsCreateSDF, 	0, 	0,
    VgtsDeleteSDF, 	1, 	0,
    VgtsDefineSymbol,	2,	1,
    VgtsEndSymbol,	3,	0,
    VgtsAddItem,	8,	1,
    VgtsAddCall,	5,	0,
    VgtsDeleteItem,	2,	0,
    VgtsInquireItem,	2,	0,
    VgtsInquireCall,	2,	0,
    VgtsChangeItem,	8,	1,
    VgtsEditSymbol,	2,	0,
    VgtsDeleteSymbol,	2,	0,
    VgtsCreateVGT,	3,	1,
    VgtsDeleteVGT,	2,	0,
    VgtsDisplayItem,	3,	0,
    VgtsCreateView,	8,	0,
    VgtsGetMouseClick,	0,	0,	/* All these are "special" */
    VgtsFindSelectedObject,5,	0,
    VgtsPopup,		0,	0,
    VgtsDefineFont,	1,	1,
    VgtsSetMode,	2,	0,
    VgtsDefaultView,	7,	0,
    (enum VgtsRequest)-1, -1, 0
  };


enum InterpState
  {
  	/*
	 * state of the VGTS interpreter for each user
	 */
    NotThere,		/* user not created yet */
    Transparent,	/* just send chars to the pad */
    GetRequest,		/* after escape, before request code */
    GetParams,		/* getting the scalar parameters */
    GetHalfParam,	/* get the low order half a parameter */
    GetLength,		/* get the length of the string (or escape code) */
    GetLength4,		/* get the hi order 8 of 32 bits of string length */
    GetLength3,		/* get the hi order 8 of 32 bits of string length */
    GetLength2,		/* get the middle 8 of 32 bits of string length */
    GetLength1,		/* get the low order 8 bits of the length */
    GetString,		/* getting a string */
    SkipString,		/* skipping a string because we ran out of memory */
    GetMenuLength,	/* getting string length in Pop up menu string */
    GetMenuString,	/* getting a PopUp menu string */
    GetMenuNumber	/* getting a menu number */
  };


struct InterpUser
  {
    enum InterpState state;	/* state this interpreter is in */
    enum VgtsRequest req;	/* the pending VGTS request */
    short vgt;			/* the pad we are going through */
    short arg[8];		/* scalar parameters */
    int count;			/* number of current parameter */
    int length;			/* string length */
    short menuCount;		/* count menu items */
    char *string;		/* the string we are forming */
    struct TtyPadRec *pad;	/* the pad pointer */
    PopUpEntry menu[16];	/* stores the pop-up menus */
  };


extern short InputPad;		/* current input client */

struct InterpUser *CreateInterp(user)
  {
      /*
       * Initialize the VGTS interpreter for the given user,
       * and return a pointer to it.
       */
    register struct InterpUser *u;
    
    u = (struct InterpUser *)malloc(sizeof(struct InterpUser));
    if (u) 
      {
        u->vgt = user;
	u->pad = TtyPadList + user;
        u->state = Transparent;
      }
     return(u);
  }


InterpPutString( u, s, length, mode )
    register struct InterpUser *u;	/* pointer to Interp user structure */
    char *s;				/* pointer to string */
    int length, 			/* length of the string */
        mode;				/* mode of cooking */
  {
  	/*
	 * Interpret the character according to our state.
	 * Typical finite state machine interpreter, driven
	 * by our input character, and the Parse table to
	 * determine the next state action.
	 */
    register char c;
    struct TtyPadRec *pad;

    if (u==NULL) return;
    pad = u->pad;
    
    if (Debug) printf("Interp got %d bytes\n", length);

    while (length-- > 0)
      {
        c = *s++;
        switch (u->state)
         {
	  case NotThere:		return;

	  case Transparent:
            if (c=='\n' && (mode&LF_Output))
	      	 PadPutChar('\r',pad);
	    if ( c == InterpEscape ) u->state = GetRequest;
 	    else PadPutChar( c, pad);
	    continue;

	  case GetRequest:
	    u->req = (enum VgtsRequest)c;
	    u->state = GetParams;
	    u->count = 0;
	    if ((int)u->req>(int)VgtsMaxCode || (int)u->req < 0 )
	      {
	         if (Debug) printf("Error- request code %d, max is %d\n",
		 	u->req, VgtsMaxCode );
	         u->state = Transparent;
		 PadPutChar(InterpEscape, pad);
		 PadPutChar(c, pad);
		 continue;
	      }
	    if (ParseTable[c].numargs>0) continue;
	    CallVgts(u);
	    continue;

 	  case GetParams:
	    u->state = GetHalfParam;
	    u->arg[u->count] = (c & 0377 ) << 8;
	    continue;

	  case GetHalfParam:
	    u->arg[u->count] += c & 0377;
 	    if (Debug) printf("arg[%d]=%d ", 
	    	u->count, u->arg[u->count] );
	    u->count++;
	    u->state = GetParams;
	    if (u->count<ParseTable[(int)u->req].numargs) continue;
	    if (ParseTable[(int)u->req].stringFlag) u->state = GetLength;
	    else CallVgts(u);
	    continue;
	    
	  case GetLength:
	    if ( (c & 0377)==MAGIC4 ) 
	      {
	    	u->state = GetLength4;
	    	continue;
	      }
	    u->length = c & 0377; 
	    u->count = 0;
	    u->state = GetString;
	    if (u->length == 0) 
	      {
	        u->string = NULL;
	        CallVgts(u);
		continue;
	      }
	    u->string = (char *)malloc(u->length+1);
	    if (u->string==NULL)
	    u->state = SkipString;
    	    u->length = (c & 0377);
	    continue;
	    
	  case GetLength4:
	    u->state = GetLength3;
	    u->length = (c & 0377) << 24;
	    continue;

    	  case GetLength3:
	    u->state = GetLength2;
	    u->length += (c & 0377) << 16;
	    continue;
	    
	  case GetLength2:
	    u->state = GetLength1;
	    u->length += (c & 0377) << 8;
	    continue;
	    
	  case GetLength1:
	    u->length += (c & 0377);
	    u->count = 0;
	    u->state = GetString;
	    if (u->length == 0) 
	      {
	        u->string = NULL;
	        CallVgts(u);
		continue;
	      }
	    u->string = (char *)malloc(u->length+1);
	    if (Debug && u->length> 255)
	      printf("Getting a string of length %d\n", u->length);
	    if (u->string==NULL)
	    	u->state = SkipString;
	    continue;

	  case SkipString:
	    u->count++;
	    if (u->count >= u->length )
		CallVgts(u);
	    continue;

	  case GetString:
	    u->string[u->count++] = c;
	    if (u->count >= u->length )
	      {
	        u->string[u->count] = 0;
		if (Debug) printf("string=%s ",u->string);
		CallVgts(u);
	      }
	    continue;
	    
	  case GetMenuLength:
	    u->length = (c & 0377);
	    u->count = 0;
	    u->state = GetMenuString;
	    if (u->length==0) 
	      {
	      	  /*
		   * we actually do the pop up when
		   * we get a string of length zero
		   */
	        u->menu[u->menuCount].string = NULL;
	        u->menu[u->menuCount].menuNumber = 0;
	        ReturnShort( u->vgt, popup(u->menu), 1 );
		ReleaseMenuStrings(u->menu);
		u->state = Transparent;
		continue;
	      }
	    u->string = (char *)malloc(u->length+1);
	    if (u->string==NULL)
	    	u->state = SkipString;
	    continue;

	  case GetMenuString:
	    u->string[u->count++] = c;
	    if (u->count >= u->length )
	      {
	        u->string[u->count] = 0;
		u->menu[u->menuCount].string = u->string;
		u->state = GetMenuNumber;
	      }
	    continue;
	    
	  case GetMenuNumber:
	    if (Debug) printf("Menu item %d is %s\r\n", c, u->string);
	    u->menu[u->menuCount++].menuNumber = c;
	    u->state = GetMenuLength;
	    continue;
	}
      }
  }


static ReleaseMenuStrings(p)
    PopUpEntry *p;
  {
  	/*
	 * free up the string space used in the given pop-up menu
	 */
    for (;p->string;p++)
      free(p->string);
  }

static CallVgts(u)
    register struct InterpUser *u;
  {
	/*
	 * call one of the Vgts routines
	 */
    short sdf, vgt, item, xmin, ymin, xmax, ymax, buttons;
    LISTTYPE FindSelectedObject(), list;
    char type, typedata;
    short user = u->vgt;

    if ((int)u->req > (int)VgtsMaxCode)
      {
         if (Debug) printf("Error- request code %d, max is %d\n",
	 	u->req, VgtsMaxCode );
         u->state = Transparent;
	 return;
      }

    if (Debug) printf(" call, code %d for user %d\n",
    	u->req, user );

    switch (u->req)
      {
	case VgtsCreateSDF:
		sdf = CreateSDF();
		ReturnShort(user,sdf,1);
    		break;

	case VgtsDeleteSDF:
		sdf = DeleteSDF(u->arg[0],TRUE);
		ReturnShort(user,sdf,1);
    		break;

	case VgtsDefineSymbol:
		item = DefineSymbol(u->arg[0],u->arg[1],u->string);
		ReturnShort(user,item,1);
    		break;

	case VgtsEndSymbol:
		item = EndSymbol(u->arg[0],u->arg[1],u->arg[2]);
		ReturnShort(user,item,1);
    		break;

	case VgtsAddItem:
		item = AddItem(u->arg[0],u->arg[1],u->arg[2],u->arg[3],
			 u->arg[4],u->arg[5],
			 (unsigned char) u->arg[6],
			 (unsigned char) u->arg[7],u->string);
    		break;

	case VgtsDeleteItem:
		FreeString(u->arg[0],u->arg[1]);
		item = DeleteItem(u->arg[0],u->arg[1]);
    		break;

	case VgtsInquireItem:
		item = InquireItem(u->arg[0],u->arg[1],&xmin,&xmax,
			&ymin, &ymax, &typedata, &type, NULL);
		ReturnShort(user,xmin,0);
		ReturnShort(user,xmax,0);
		ReturnShort(user,ymin,0);
		ReturnShort(user,ymax,0);
		ReturnShort(user,typedata,0);
		ReturnShort(user,type,0);
		ReturnShort(user,item,1);
    		break;

	case VgtsInquireCall:
		item = InquireCall(u->arg[0],u->arg[1]);
		ReturnShort(user,item,1);
    		break;

	case VgtsChangeItem:
		if (u->string)
		    FreeString(u->arg[0],u->arg[1]);
		item = ChangeItem(u->arg[0],u->arg[1],u->arg[2],u->arg[3],
			 u->arg[4],u->arg[5],u->arg[6],u->arg[7],u->string);
    		break;

	case VgtsEditSymbol:
		item = EditSymbol(u->arg[0],u->arg[1]);
		ReturnShort(user,item,1);
    		break;

	case VgtsDeleteSymbol:
		item = DeleteSymbol(u->arg[0],u->arg[1]);
		ReturnShort(user,item,1);
    		break;

	case VgtsCreateVGT:
		vgt = CreateVGT(u->arg[0],u->arg[1],u->arg[2],u->string);
		ReturnShort(user,vgt,1);
		if (vgt<0) break;
		ClientTable[vgt].master = user;
		ClientTable[vgt].owner = ClientTable[user].owner;
		ClientTable[vgt].vgt = vgt;
    		break;

	case VgtsDeleteVGT:
		vgt = DeleteVGT(u->arg[0],u->arg[1]);
		ReturnShort(user,vgt,1);
		ClientTable[vgt].master = -1;
    		break;

	case VgtsDisplayItem:
		item = DisplayItem(u->arg[0],u->arg[1],u->arg[2]);
		ReturnShort(user,item,1);
    		break;

	case VgtsCreateView:
		item = CreateView(u->arg[0],u->arg[1],u->arg[2],u->arg[3],
		 u->arg[4],u->arg[5],u->arg[6],u->arg[7]>>8,u->arg[7] & 0377);
		ReturnShort(user,item,1);
    		break;

	case VgtsGetMouseClick:
		WaitForEvent(user);
                u->state = Transparent;
    		break;

	case VgtsFindSelectedObject:
		list = FindSelectedObject(u->arg[0],u->arg[1],u->arg[2],
			u->arg[3],u->arg[4]);
		if (list.NumOfElements == 0)
		  {
		    ReturnShort(user,0,1);
		  }
		else
		  {
		    ReturnShort(user,1,0);
		    ReturnShort(user,list.Header->item,0);
		    ReturnShort(user,list.Header->edgeset,1);
		  }
    		break;
		
	case VgtsPopup:
		  /*
		   * these arguments are handled specially;
		   * there can be an arbitrary number.
		   */
		u->menuCount = 0;
		u->state = GetMenuLength;
		return;
		
	case VgtsDefineFont:
		if (u->string) 
		    item = DefineFont(u->string,u->string+u->arg[0]);
		else item = -1;
		ReturnShort(user,item,1);
		break;

	case VgtsSetMode:
		SetPadMode(user, (u->arg[0]<<16)|(u->arg[1]) );
		break;

	case VgtsDefaultView:
		DefaultView(u->arg[0], u->arg[1], u->arg[2], u->arg[3],
			u->arg[4], u->arg[5], u->arg[6], &xmin, &ymin );
		ReturnShort(user,xmin,FALSE);
		ReturnShort(user,ymin,TRUE);
		break;

	case VgtsAddCall:
		item = AddCall(u->arg[0],u->arg[1],u->arg[2],u->arg[3],
			 u->arg[4]);
    		break;

     }
    u->state = Transparent;
  }


ReturnShort(user,n,pushFlag)
 short user;
 short n;
 char pushFlag;
  {
  	/*
	 * send back a return value to the indicated user
	 * as an escape character followed by two bytes.
	 * if pushFlag is set, the packet is "pushed" out.
	 */
    register struct Client *client = ClientTable + user;

    UserPut(client,InterpEscape);
    UserPut(client,n>>8);
    if (pushFlag)
        UserPush(client,n&0377);
     else
        UserPut(client,n&0377);
    if (Debug) printf("%s value %d to user %d\n", 
    	pushFlag ? "Pushed" : "Returned", n, user);
  }


InterpMouseEscape(vgt, xw, yw, buttons)
  short vgt, xw, yw, buttons;
  {
  	/*
	 * Send back the appropriate escape codes for an Emacs hack.
	 */
      switch (buttons)
	  {
	    case LeftButton:
	        SendEmacsHack(vgt,xw,yw,-1);	/* move cursor */
		break;
		
	    case LeftButton+MiddleButton:
	        SendEmacsHack(vgt,xw,yw,0);	/* null - set mark */
		break;
		
	    case LeftButton+RightButton:
	        SendEmacsHack(vgt,xw,yw, 'W' & 31); /* control W -  kill */
		break;
		
	    case MiddleButton+RightButton:
	        SendEmacsHack(vgt,xw,yw,'Y' & 31); /* control Y - yank */
		break;
		
	    case MiddleButton:
		SelectForInput(vgt);
	        break;

	    case RightButton:
		SelectForInput(manager(InputPad));
		break;		
	   }
  }


ReturnToMaster(master,xw,yw,buttons,vgt)
  {
    ReturnShort(master,xw,0);
    ReturnShort(master,yw,0);
    ReturnShort(master,buttons,0);
    ReturnShort(master,vgt,1);
  }


static WaitForEvent(user)
  {
	/*
	 * The interp client want to wait for an event --
	 * get one off the queue, or else set the waiting flag.
	 */
    ClientTable[user].requestFlag |= SlaveReq;
    ClientTable[user].mode |= ReportClick;
    EventToUser(ClientTable + user);
  }


SendEmacsHack(vgt,x,y,c)
  {
  	/*
	 * send the escape sequence for Emacs to position the
	 * cursor at a given character.
	 * If c is greater than zero, it is another command character
	 * to send.
	 */
    short line, col;
    register struct Client *client = ClientTable + vgt;
    
    if (vgt<1) return;
    PadFindPoint(vgt,TtyPadList[vgt].length,x,y,&line,&col);
    UserPut(client,ESC);
    UserPut(client,'M');
    UserPut(client,col+1);
    if (c<0)
        UserPush(client,line+1);
    else
      {
        UserPut(client,line+1);
        UserPush(client,c); 
      }
  }


