/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1983			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/* execs.c - VGTS executive control routines
 *
 * This file contains the routines to handle the Exec Control
 * option of the View manager in the vgtsexec program.
 *
 * Bill Nowicki April 1983
 *
 * Marvin Theimer, 6/83
 *	Changed the KillProgram and RegisterProcess routines so that they
 *	explicitly specify which process to send a termination request to.
 * Marvin Theimer, 7/83
 *	Changed KillProgram to implement SET_BREAK_PROCESS.  RegisterProcess
 *	is now replaced by the library routine SetBreakProcess and by
 *	SetBanner.
 * Tim Mann, 8/6/83
 *	Fixed exec creation to set up the per-process area.
 *
 */

# include <Vgts.h>
# include <Vioprotocol.h>
# include <Vteams.h>		/* for Root Message and kill program*/
# include <Vprocess.h>
# include "client.h"

extern short MouseX, MouseY, MouseButtons, InputPad, TtySDF, Unique;
extern short Debug;		/* imported from master mux */

extern Exec();			/* imported from exec */
extern struct Buffer *CreateBuffer();

# define C_CreateExec 1
# define C_DeleteExec 2
# define C_KillProgram 3
# define C_BigCreateExec 4
# define C_SetSize 5

char bigMessage[128] = "Create 48 Line Executive";
short bigExecSize = 48;

static PopUpEntry execMenu[] =
    {
      "Delete Executive", C_DeleteExec,
      "Create 28 Line Executive", C_CreateExec,
      "Kill Program", C_KillProgram,
      bigMessage, C_BigCreateExec,
      "Set Alternate Exec Size", C_SetSize,
      0, 0
    };


short ExecControl(pad)
{
  TtyPutString("Exec M: ");
  switch (popup(execMenu))
    {
      case C_CreateExec:
          return(CreateExec(28));
	  break;

      case C_SetSize:
          bigExecSize = TtyGetNumber("Enter number of lines for executives:");
	  sprintf( bigMessage, "Create %d line Executive", bigExecSize );
	  break;

      case C_BigCreateExec:
          return(CreateExec(bigExecSize));
	  break;

      case C_DeleteExec:
          return(DeleteExec());
	  break;

      case C_KillProgram:
          KillProgram();
	  break;

      default:
          TtyPutString("\r\n");
          break;
    }
   return(pad);
}


static KillProgram()
  {
    short view, vgt;
    register struct Client *client; 
    Message msg;
    ExitTeamRequest *reqMsg = (ExitTeamRequest *) msg;

    TtyPutString("Kill Program B: ");
    if (GetMouseInput() == 7)
	return(TtyPutString(" Aborted\r\n"));
     if ((view = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("Viewport not found");
        return;
      }
    vgt = FindTopVGT(MouseX, MouseY);
    if (vgt<0) return;
    client = ClientTable + vgt;
    if (client->termPid)
      {
        DestroyProcess(client->termPid);
	client->termPid = 0;		/* Unregister the break process. */
	TtyPutString("\r\n");
      }
    else TtyBlinkError("No command in that view");
  }


static DeleteExec()
  {
    register struct Client *client; 
    short view, vgt;

    TtyPutString("Delete Exec B: ");
    if (GetMouseInput() == 7)
      {
	TtyPutString(" Aborted\r\n");
        return(InputPad);
      }
     if ((view = FindTopView(MouseX, MouseY, FALSE)) == -1)
      {
        TtyBlinkError("Viewport not found");
        return(InputPad);
      }
    vgt = FindTopVGT(MouseX, MouseY);
    if (vgt<0) return;
    client = ClientTable + vgt;
    if (client->exec)
      {
      	  FlushEvents(client);
          DeleteVGT(vgt,TRUE);
	  PadDelete(vgt);
    	  DestroyProcess(client->exec->execPid);
	  free(client->exec);
	  client->exec = NULL;
          client->owner = 0;
          client->pid = 0;
	  vgt = MakeNotCurrent(vgt);
          TtyPutString("\r\n");
      }
    else TtyBlinkError("No Executive in that view");
    return(vgt);
  }


static CreateExec(size)
    short size;		/* lines in desired pad */
  {
    short xCorner, yCorner, height, width;
    short pad;

    height = ViewHeight(size);
    width = ViewWidth(PadWidth);

    pad = PadInit(TtySDF,Unique++,"Executive", 
    	size, PadWidth);
    if (pad<0 || OpenExecFiles(pad)==0)
      {
        TtyBlinkError("Ran out of memory");
        return(InputPad);
      }

    TtyPutString("Creating an Exec, upper left corner at B: ");
    SetExecCursor();
    WaitForDownButton();
    xCorner = MouseX;
    yCorner = MouseY;
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    while (MouseButtons!=0)
      {
        ReadMouseInput();

	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
	if (MouseButtons==7)
          {
            SetMainCursor();
	    TtyPutString(" Aborted\r\n");
    	    return(InputPad);
      	 }
        xCorner = MouseX;
        yCorner = MouseY;
	CursorUpdate();
	FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    }
    FlashOutline(xCorner,yCorner,xCorner+width,yCorner+height,AllEdges);
    SetMainCursor();

    CreateView( pad, xCorner, yCorner, 
        	xCorner+width, yCorner+height, 0, 0, 0, 0);
    SelectForInput(pad);
    TtyPutString("\r\n");
    return(pad);
  }


OpenExecFiles(vgt)
 short vgt;
  {
  	/*
	 * here we create a fake file for the executive to talk to
	 * one of our VGTs
	 */
    extern File *OpenFile();
    extern RootMessage *RootMsg;
    register struct Client *client = ClientTable + vgt;
    register File *in, *out;
    register struct ExecDescriptor *e;
    ProcessId pid;
    int i;
    Processor_state execState;

    e = (struct ExecDescriptor *)
    	malloc(sizeof(struct ExecDescriptor));

    if (e==NULL) return(0);
    
    if (client->interp==0)
      {
        i = CreateInterp(vgt);
        if (i==NULL) return(0);
        client->interp = i;
      }

    client->exec = e;

    in  = &(e->in);
    out = &(e->out);
    
    in->fileserver = VgtsPid;
    in->fileid = vgt;
    in->state = 0;
    in->lastexception = 0;
    in->type = READABLE|STREAM|VARIABLE_BLOCK;
    in->blocksize = IO_MSG_BUFFER;
    in->tempindex = 0;
    in->readindex = 0;
    in->writeindex = 0;
    in->bytes = 0;
    in->lastblock = 0;
    in->buffer = e->inbuffer;
    
    out->fileserver = VgtsPid;
    out->fileid = vgt;
    out->state = 0;
    out->lastexception = 0;
    out->type = WRITEABLE|STREAM;
    out->blocksize = BlockSize;
    out->tempindex = 0;
    out->readindex = 0;
    out->writeindex = 0;
    out->bytes = 0;
    out->lastblock = 0;
    out->buffer = e->outbuffer;
       
    RootMsg->stdinserver  = VgtsPid;
    RootMsg->stdoutserver = VgtsPid;
    RootMsg->stderrserver = VgtsPid;

    pid = CreateProcess( 8, Exec, e->stack+MainSize);
    if (pid==0) return(0);
    if (ReadProcessState(pid, &execState) == 0) return (0);
    execState.perProcess = (Unspec *) &e->perProcess;
    execState.perProcessLoc = (Unspec **) &PerProcess;
    if (WriteProcessState(pid, &execState) == 0) return (0);
    e->perProcess = *PerProcess;
    e->perProcess.stackSize = sizeof(e->perProcess) + MainSize;

    client->vgt = vgt;
    client->pid = pid;
    client->owner = pid;
    client->master = -1;
    client->exec->execPid = pid;
    client->lineEditBuf = CreateBuffer();
    client->mode = CR_Input+LF_Output+LineBuffer;
    Ready(pid,4,in,out,out,NULL);
    return(pid);
}

