/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 *
 * Interactively execute UNIX programs via the server.
 */ 

#include <sgtty.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <wait.h>
#include <signal.h>
#include <types.h>
#include <stat.h>
#include <server.h>
#include <debug.h>

extern ProcessId MyPid;
extern char *MyEnvironment[];
extern int MyUserId, MyGroupId;
extern char BlockBuffer[];
extern int errno;
extern FileInstance *AllocInstance();
extern FileInstance *GetInstance();
extern FileInstance *FreeInstance();
extern FileInstance *FindInstance();
extern SystemCode ChangeContext();
extern char *FindContext();
extern char *ParseName();
extern char *index();

#define NUM_PATHS	7
#define MAX_PATH_LEN	32
/* The first form of path is used in the MyEnvironment variable */
char MyEnvPath[] = "PATH=.:/usr/local/bin:/usr/ucb:/bin:/usr/bin:/usr/local/commands:/etc:/ng/ng/bin";
/* This second form is used internally for execution searching paths */
static char *SearchPaths[NUM_PATHS+1] = 
  {  
    "/usr/local/bin/",
    "/usr/ucb/",
    "/bin/",
    "/usr/bin/",
    "/usr/local/commands/",
    "/etc/",
    "/ng/ng/bin/",
    "" 				/* makes loop termination easier */
  };

CheckProcesses()
/*
 * Check the status of all the children who wish to report it.
 * Status is updated in only the first instance found for the process.
 */
  { 
    union wait status;
    register int child;
    register ProcessInstance *desc,*desc2;
    int upid;

    while ( (child = wait3( &status, WNOHANG|WUNTRACED, 0 )) > 0)
      {

	/* Search for all descriptors that match this pid */
        if ((desc = (ProcessInstance *)FindInstance( child, PROCESS_INSTANCE )) 
	    == NULL)
          {
	    if (PDebug)
	       /* Status on a child we don't have a descriptor for. */
	        printf( "Process instance not found, UNIX pid = %d\n", child );
	    continue;
	  }

        if (PDebug) 
          printf( "Process instance 0x%x", desc->id );
	if ( desc->state == PD_AWAITING_DEATH )
	  {
	    /* instance has been released thus the client wanted the process
	     * to be killed.  If it is stopped, then strange event occurred.
	     * However, the kill signal must be sent again.  
	     */
            if ( WIFSTOPPED( status ) )
	      {
		if (PDebug) printf( " stopped and SIGKILL sent.\n" );
	        killpg( getpgrp( upid ), SIGKILL );
	      }
	    else
	      {
		/* Descriptor was just around for the SIGCHLD interrupt */
		if (PDebug) printf( " terminated and freed\n" );
	        FreeInstance( desc );
	      }
	  }
	else
	  {
	    /* record the new state in this instance */
	    if ( WIFSIGNALED( status ) )
	        desc->state = PD_KILLED;
            else if ( WIFSTOPPED( status ) )
	        desc->state = PD_STOPPED;
	    else if ( WIFEXITED( status ) )
	        desc->state = PD_DEAD;

	    /* Look for the "other" descriptor for this process */
	    desc->pid = 0;	/* don't find this descriptor */
            if ((desc2 = (ProcessInstance *)FindInstance( child, PROCESS_INSTANCE )) 
	        != NULL)
	      {
		if (PDebug) printf( " and 0x%x", desc2->id );
		desc2->state = desc->state;
	      }

	    if (PDebug) printf( " signal rcvd, state = %d\n", desc->state );
	    desc->pid = child;	/* restore pid in descriptor */
	  }

      } /* while ( child ...*/

  } /* CheckProcesses */

SystemCode ParseArguments( argv, req, pid, segsize )
      register char **argv;
      register CreateInstanceRequest *req;
      ProcessId pid;
      unsigned *segsize;
  {
    /* 
     * Get the program name and argument list.  Fill the array
     * argv with pointers to the beginning of each argument as
     * is required by execve.  The first argument (of course)
     * is the program name.
     */
    char *nameptr;
    register int argc; 
    register char *c;
    SystemCode r;
    unsigned progargslen;
    char *progargs;
	
    /* Get the program name and arguments */
    MoveLong( req->filenamelen, progargslen );

    if ( progargslen > MAX_PROG_ARGS_LEN || progargslen == 0 ) 
	return( BAD_ARGS );
    if ( progargslen <= req->filenameindex ) return( BAD_ARGS );

    if( *segsize < progargslen )
      {
	MoveLong( req->filename, progargs );

	if( (r = MoveFrom( pid, progargs, BlockBuffer, progargslen )) != OK)
		return( r );
	*segsize = progargslen;
      }
    BlockBuffer[ progargslen ] = '\0';	/* terminate */

    if( *BlockBuffer == NULL || strlen(BlockBuffer) > MAX_PROG_NAME_LEN ) 
	return( BAD_ARGS );

    nameptr = &BlockBuffer[ req->filenameindex ];
    for ( argc=0, c=nameptr; c <  (BlockBuffer+progargslen); c++ )
      {
	if ( ++argc > (MAX_PROG_ARGC ) )  return( BAD_ARGS ); 

	*(argv++) = c;			/* assign the argument */
	if ( PDebug ) printf("	Argv[%d] = '%s'\n", argc, argv[-1] );
	while( *c ) c++;		/* find end of this argument 
					   allowing for null arguments */
      } 
	
    *argv = (char *) 0;	 /* terminate the argv list with a NULL ptr*/

    return( OK );

  } /* ParseArguments */
    
SystemCode FindPath( pathname, name )
  char pathname[];
  char *name;
/*
 * Try to execute the named program, searching down SearchPaths.
 * The current directory is tried first, however.
 * If the name begins with '/' or '.', then only the cwd is searched.
 * If it finds it, but is not executable then it returns NO_PERMISSION;
 * the search continues, but if the rest fail then NO_PERMISSION is 
 * returned.
 */
  {
    struct stat st;
    register SystemCode r = NOT_FOUND;
    register int okay;
    register int SearchIndex=0;
      

    strncpy( pathname, name, MAX_PROG_NAME_LEN );
    pathname[MAX_PROG_NAME_LEN] = '\0';		/* may need termination */

    /* imitate shell's search path mechanism */
    if ( index( name, '/' ) != 0 )
	SearchIndex = NUM_PATHS;
    do
      {
	if ( stat( pathname, &st ) != -1 ) 
	  {
	    /* 
	     * This is the backwards (UNIX) way of doing things. 
	     * Owner only checks his access bits.
	     */
	    if ( st.st_uid == MyUserId )
		okay = st.st_mode & 0100;
	    else if (st.st_gid == MyGroupId )
		okay = st.st_mode & 0010;
	    else /* others */
		okay = st.st_mode & 0001;
	    okay = okay && (st.st_mode & S_IFREG);  /* regular file */	
            if ( okay )
		return(OK);
	    else
		r = NO_PERMISSION;
	  }
	strcpy( pathname, SearchPaths[SearchIndex] );
	strcat( pathname, name );
      }
    while( SearchIndex++ < NUM_PATHS );

    return( r );

  } /* FindPath */

SystemCode ExecuteProgram( req, pid, segsize )
    register CreateInstanceRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Programs can only be created by session servers.  Set up the
 * arguments.  Open pipes  and fork.  In the child, we wish to 
 * change the pipe to be correspond to stdin, stdout, and stderr.  
 * Do an execve with the proper arguments.  The parent saves the PID 
 * in the ProcessInstance and continues serving.
 */
  {
    CreateInstanceReply  *reply = (CreateInstanceReply *) req;
    SystemCode r;
    char *shellargv[ MAX_PROG_ARGC+3 ]; 
    char **progargv = &(shellargv[2]);
    register ProcessInstance *writedesc,*readdesc;
    int inpipe[2], outpipe[2];
    register int pipefail;
    ContextId cid;
    char PathName[ MAX_PROG_NAME_LEN + MAX_PATH_LEN + 2]; 

    if ( Session == NULL ) return( ILLEGAL_REQUEST );

    /* change to the appropriate context */
    MoveLong( req->contextid, cid );
    if ( (r = ChangeContext( cid )) ) 
        return( r );

    if ( PDebug ) printf( "Run a program for Pid 0x%x\n",  pid );
    if ( PDebug ) printf( "Context is \"%s\"\n", FindContext( cid ) );

    if ( (r = ParseArguments( progargv, req, pid, &segsize )) != OK )
	return( r );

    if ( (r = FindPath( PathName, progargv[0] )) != OK )
	return( r );
    if (PDebug) printf("PathName = '%s'\n", PathName );


    /* Get two consecutive instances, by convention the write instance
       is first */
    while (1)
      {
        writedesc = (ProcessInstance *) AllocInstance( pid, PROCESS_INSTANCE );
        readdesc = (ProcessInstance *) AllocInstance( pid, PROCESS_INSTANCE );
        if (readdesc == NULL) 
	  {
	    FreeInstance( writedesc ); /* possibly bad */
	    return( NO_SERVER_RESOURCES );
	  }
	if ( writedesc == NULL ) 
	    return( NO_SERVER_RESOURCES ); /*  readdesc must be bad */

        if ( readdesc->id == writedesc->id + 1 ) break;
	FreeInstance( writedesc );
	FreeInstance( readdesc );
      }

tryagain:
    pipefail = 0;
    inpipe[ 0 ] = -1;		/* invalidate values of pipe files */
    inpipe[ 1 ] = -1;
    outpipe[ 0 ] = -1;
    outpipe[ 1 ] = -1;

    /* Set up pipes and fork */
    switch ( req->filemode & ~FEXECUTE )  
      {
	case FREAD:
	  {
	    if ( pipe( inpipe ) == -1 ) pipefail = 1;
	    else
	      {
		readdesc->filetype = READABLE | VARIABLE_BLOCK | STREAM; 
		outpipe[0] = outpipe[1] = -1;  /* mark the outpipe as not being used */
	      }
	    break;
	  }
/* 	case WriteOnly: ???
*	  {
*	    if ( pipe( outpipe ) == -1 ) pipefail = 1;
*	    else
*	      {
*		writedesc->filetype = APPEND_ONLY | VARIABLE_BLOCK | WRITEABLE | STREAM ; 
*		inpipe[0] = inpipe[1] = -1;
*	      }
*	    break;
*	  }
*/
	case FCREATE:
	case FMODIFY:
	case FAPPEND:
	  {
	    if ( pipe( outpipe ) == -1 || pipe( inpipe ) == -1 ) 
		pipefail = 1;
	    else 
	      {
		writedesc->filetype 
		      = WRITEABLE | APPEND_ONLY | STREAM | VARIABLE_BLOCK;
		readdesc->filetype
		      = READABLE  | STREAM | VARIABLE_BLOCK;
	      }
	    break;
	  }

	default:
	  {
	    FreeInstance( (FileInstance *) readdesc );
	    FreeInstance( (FileInstance *) writedesc );
	    return ( INVALID_MODE );
	    break;
	  }
      } /* switch */

    if ( pipefail ) 
      { /* Only reason for a pipe failing is that there are too many
	   open files, so try to clear space in open file tables*/
	if ( inpipe[0] != -1 ) close( inpipe[0] );
	if ( inpipe[1] != -1 ) close( inpipe[1] );
	if ( outpipe[0] != -1 ) close( outpipe[0] );
	if ( outpipe[1] != -1 ) close( outpipe[1] );
	if ( ReclaimInstances( 0 ) ) goto tryagain;
	if ( PDebug ) printf (" ExecuteProgram pipe failed\n");
	FreeInstance( (FileInstance *) readdesc );
	FreeInstance( (FileInstance *) writedesc );
	return( NO_SERVER_RESOURCES );
      }

    if ( (readdesc->pid = (writedesc->pid = fork()) ) == -1) 
      {
	FreeInstance( (FileInstance *) writedesc );
	FreeInstance( (FileInstance *) readdesc );
	close( inpipe[0] );
	close( inpipe[1] );
	close( outpipe[0] );
	close( outpipe[1] );
	return( NO_SERVER_RESOURCES );
      }

    if ( readdesc->pid != 0) 
      {
	/* PARENT */
	if ( PDebug ) printf( "	Program Running, UNIX Pid = %d\n", readdesc->pid );

	/* Following code presumes that a process will always be opened
 	   for writing.  This may be incorrect in the future.
	   Convention is to return read id if only opened for reading
	   and writedesc if opened for writing.
         */

	readdesc->state = PD_RUNNING; 	/* at this point we presume the
					   process is running */

	/* divorce this process from this our group, so we can kill it
	   and its children simultaneously */
	setpgrp( readdesc->pid, readdesc->pid);

	/* Parse the name */
	if ( PathName[0] != '/' )
	  {
	    /* append the current context name to this one */
	    strcpy( BlockBuffer, FindContext( cid ) );
	    strcat( BlockBuffer, PathName );
	    readdesc->name 
	        = ParseName( malloc( strlen( BlockBuffer ) + 2), BlockBuffer, 0 );
	  }
	else
	    readdesc->name 
	        = ParseName( malloc( strlen( PathName ) + 2), PathName, 0 );
	  
	/* Make the pipedesc look like a two way pipe */
	close( inpipe[1] );
        readdesc->pipedesc = inpipe[0];
	readdesc->lastblock = -1;	/* large number */
	readdesc->nextblock = 0;
	readdesc->blocksize = BLOCK_SIZE;  /* ReadInstance depends on this
					      size being what it is */
	readdesc->lastbytes = 0;
	/* must retain last block read to all for repeat requests */
	readdesc->lastdata = (char *)malloc( BLOCK_SIZE );
	readdesc->lastdatasize = 0;
	writedesc->lastreply = OK;

	/* In future execs we don't want these files to be open */
	ioctl( readdesc->pipedesc, FIOCLEX, NULL ); 

	if ( outpipe[1] != -1 )
	  {
	    /* Handle the write part of the "instance" */
	    writedesc->name = readdesc->name;
	    writedesc->state = PD_RUNNING;
	    close( outpipe[0] );
	    writedesc->pipedesc = outpipe[1];
	    writedesc->lastblock = 0;		/* 0 length file */
	    writedesc->nextblock = 0;
	    writedesc->blocksize = BLOCK_SIZE;
	    writedesc->lastbytes = 0;
	    writedesc->lastdata = (char *)0;
	    writedesc->lastdatasize = 0;
	    writedesc->lastreply = OK;
	    ioctl( writedesc->pipedesc, FIOCLEX, NULL );
	    reply->fileid = writedesc->id;
	  }
	else
	  {
	    FreeInstance( writedesc );
	    reply->fileid = readdesc->id;
	  }
	  
	return( QueryInstance( (QueryInstanceRequest *)reply ) );
      }

    /* CHILD */
    alarm(0);		/* don't want a set alarm lying around */
    nice( 4 );		/* Return priority to 0 */

    close( inpipe[0] );   /* don't need these any more */
    close( outpipe[1] ); 

    /* Make the pipe look like stdin, stdout, stderr */
    if ( outpipe[0] != -1 ) 
      {
	dup2( outpipe[0], 0);
	
      }
    else
	close( 0 );
    if ( inpipe[1] != -1 )
      {
	dup2( inpipe[1], 1); dup2( inpipe[1], 2);

      }
    else
      {
	close( 1 ); close( 2 );
      }

    close( inpipe[1] ); 
    close( outpipe[0] ); 
    /* All other files will be closed on the execve */

    /* Attempt to execute the program */
    execve( PathName, progargv, MyEnvironment );

    /* Try it as a shell script "-c" option makes "/bin/sh" pass it
     * to the correct shell.
     */
    shellargv[0] = "/bin/sh";
    shellargv[1] = "-c";

    /* Hack:  The "-c" option takes only the following argument and
	      tries to execute it, any of the other arguments passed
	      to /bin/sh are tossed.  We therefore must put all the 
	      arguments into one string and pass that on to the shell.
	      This is done by replacing the nulls that seperate the
	      strings with spaces, thus the shell takes this as one
	      long string to be parsed.
    */

    { 
	register char **arg;

	for( arg = &(shellargv[3]);  *arg != NULL; arg++ )
	  
	    (*arg)[-1] = ' ';

	shellargv[3] = NULL;  /* terminate argument list after "first" arg */
	
    }

    execve( shellargv[0], shellargv, MyEnvironment );

    /* Execve failed, this should never happen! */
    exit(1);		

  } /* ExecuteProgram */
