/*
 * Unix V file server naming protocol 
 */
#include <Vnaming.h>
#include <naming.h>
#include <types.h>
#include <stat.h>
#include <errno.h>
#include <Venviron.h>
#include <Vsession.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <server.h>
#include <debug.h>

/* Character string name request handlers */
extern SystemCode GetContextId();
extern SystemCode GetContextName();
extern SystemCode GetFileName();

/* Context cache (tree) manipulation routines */
extern SystemCode InitContexts();
extern SystemCode InsertContextNode();
extern DeleteContextNode();
extern SystemCode ChangeContext();
extern char *FindContext();
extern FreeContextTree();			

/* Unix string name handlers */
extern SystemCode GetName();
extern char *ParseName();

/* Imports */
extern FileInstance *GetInstance();
extern char BlockBuffer[];
extern SessionDesc *Session;
extern int errno;
extern int PublicServer;
extern ProcessId MyPid;

char *WellKnownContext[ MAX_WELL_KNOWN_CONTEXTS ] = { NULL };


ContextId CurrentContext = 0;		/* Context Id of DEFAULT_CONTEXT */
char *CurrentContextName = "/";	
ContextNode *RootContextNode = NULL;	/* root of tree cache */

/* Following routines are used to manage the context name cache */
FreeContextTree( cnode )
    ContextNode *cnode;
/*
 * recursively free the context tree.
 */
  { 
    if ( cnode->left != NULL ) FreeContextTree( cnode->left );
    if ( cnode->right != NULL ) FreeContextTree( cnode->right ); 
    free( cnode );
  } /* FreeContextTree */

SystemCode InitContexts( LoginContext )
    char *LoginContext;
/*
 * Free all the nodes in the context cache (tree).
 * Set the CurrentContext to the context name (if non-null)
 * and return its id.  It is assumed that LoginContext starts
 * with a '/'
 */
  {
    extern char *malloc(), *strcpy();
    ContextId cid;
    register int i;

    for( i = 0; i < MAX_WELL_KNOWN_CONTEXTS; i++ )
        WellKnownContext[ i ] = NULL;

    WellKnownContext[ DEFAULT_CONTEXT ] = "/";
	
    /* PUBLIC_CONTEXT is invalid for non-public servers */
    WellKnownContext[ PUBLIC_CONTEXT ] = PublicServer ? PUBLIC_DIR : NULL;
        
    WellKnownContext[ ANY_CONTEXT ] = "/";


    if ( RootContextNode != NULL )
      {
        FreeContextTree( RootContextNode );
	RootContextNode = NULL;
      }

    /* free old context if necessary */
    if ( WellKnownContext[ LOGIN_CONTEXT ] != NULL )
    	free( WellKnownContext[ LOGIN_CONTEXT ] );

    /* assign new context */
    if ( LoginContext == NULL )
      {
	WellKnownContext[ LOGIN_CONTEXT ] = NULL;
	return( ChangeContext( PUBLIC_CONTEXT ) );
      }

    /* store the new non-null login context name */
    WellKnownContext[ LOGIN_CONTEXT ] = 
    	strcpy( malloc( (i = strlen( LoginContext )) + 2 ), LoginContext );

    if ( WellKnownContext[ LOGIN_CONTEXT ][ i - 1 ] != '/' )
      {
        /* must end with a '/' */
	WellKnownContext[ LOGIN_CONTEXT ][ i++ ] = '/';
        WellKnownContext[ LOGIN_CONTEXT ][ i ] = NULL;
      }

    if ( CDebug ) printf( "login context %s\n", LoginContext );
    if ( CurrentContext == LOGIN_CONTEXT )
      {
        /* we changed the name behind the data abstraction's back */
	/* The cid is the same, but the name is different! */
	chdir( LoginContext );
	CurrentContextName = WellKnownContext[ LOGIN_CONTEXT ];
      }

    return( ChangeContext( LOGIN_CONTEXT ) );
    
  } /* InitContexts */    

char *ParseName( NewName, OldName, IsaDirectory )
    char *NewName, *OldName;
    int IsaDirectory;
/*
 * Take OldName and parse out all the extraneous '.', '..', and '/'
 * correctly while transferring into NewName.
 * If IsaDirectory is true, then terminate the name with a '/'
 * otherwise it shouldn't end in a '/'.
 * Always returns its first argument.
 * NOTE: OldName must start with a '/'.
 */
  { 
    register char *np = NewName;
    register char *cp;
    
    *np++ = '/';
    cp = OldName + 1;	/* skip over initial '/' */

    /* parse "name" in unix hierarchical fashion */
    /* Note the extremely clever algorithm */
    while( *cp != NULL )
      {
	/* first get rid of any extra '/'s */
	while( *cp == '/' ) cp++;

	/* determine if this part is '.' or '..' */
        if ( *cp == '.' ) 
	  {
	    /* starts with a '.' */
	    switch( *++cp )
	      {
		case '/': /* go nowhere */
			cp++;
		case NULL: /* same as above, but no need to increment */
			break;

	        case '.': /* is at least "..", so we have to read on */
			if ( *++cp == '/' || *cp == NULL )
			  {
			    /* go up one in the unix directory tree,
			       but don't back over the root */
			    np--;
			    while( np > NewName && *--np != '/' );
			    np++;
			    break; /* switch */
			  }
			/* else the name is "..<foobar>" */
			*np++ = '.';
			/* intentional fall thru to ".<foobar>" */
		default: 
			*np++ = '.';
			while( *cp != NULL ) /* copy name including '/' */
			    if ( (*np++ = *cp++) == '/' )
			        break;
			break;
	      } /* switch */
	  }
	else
	    while( *cp != NULL ) 
	        if ( (*np++ = *cp++) == '/' )
		    break;
      }
	
    if ( IsaDirectory )
      {
        /* always end with a '/' as a convention */
        if ( np[-1] != '/' ) *np++ = '/';
      }
    else
      {
        while ( np[-1] == '/' ) 
	    if ( --np <= NewName )
	      {
	        np++;
		break;
	      }
      }

    *np = NULL;		/* terminate */
    if ( CDebug ) printf( "Parsed name = %s\n", NewName );
    return( NewName );
 
  } /* ParseName */

SystemCode InsertContextNode( Name, NameLength, Cid )
    register char *Name;
    unsigned NameLength;
    ContextId *Cid;
/*
 * Get the context id for the Name and return its value.
 * If the user cannot access the context, then set the return
 * code accordingly.
 * After determining that the directory exists (by way of chdir
 * and stat), have to convert the context name into something
 * reasonable.  This is not done by the "pwd" method, but instead
 * by parsing the name for "/", "..", ".", etc.  The parsing algorithm
 * is quite fast; Thus, we can save a significant amount of time
 * over file system search (pwd) method.
 */
  {
    register ContextNode *cnp,*TmpCnp;
    register ContextId cid;
    struct stat st;
     
    /* should be in the correct context */
    /* try to get to requested context from this one */
    if ( chdir( Name ) != 0 || stat( ".", &st ) != 0 )
      {
        switch ( errno )
          { 
	    case ENOTDIR: /* not a directory. */
            case ENOENT: /* directory is not found. */
		    return( NOT_FOUND );	
		    break;
            case EACCES: 
		    return( NO_PERMISSION );	/* is this right? */
        	    break;
	    default:
		    return( INTERNAL_ERROR );
		    break;
          }
      }

    if ( CDebug ) printf( "InsertContextNode: Name is \"%s\"\n", Name );

    /* at this point, we know client has correct permissions */
    *Cid = CurrentContext = cid = ConsContextId( st.st_dev, st.st_ino );
    if ( CDebug )
        printf( "st.st_dev = 0x%x, st.st_ino = 0x%x\n", st.st_dev, st.st_ino );
    
    /* see if we already have this name in our little cache */
    /* if not insert it */
    cnp = (ContextNode *)calloc( 1, sizeof( ContextNode ) );
    
    TmpCnp = cnp;	/* save this node */
    
    /* everything is OK, insert into the tree */
    if ( RootContextNode == NULL ) /* special case: no tree */
	cnp = RootContextNode = (ContextNode *) TmpCnp;
    else
      { /* search the tree */
        cnp = RootContextNode;
	while( 1 )
	  {
	    if( cnp->contextid < cid)
		if ( cnp->right != NULL )
		    cnp = cnp->right;
		else
		  {
		    cnp->right = TmpCnp;
		    cnp = TmpCnp;
		    break;
		  }
	    else if( cnp->contextid > cid )
		if ( cnp->left != NULL )
		    cnp = cnp->left;
		else
		  {
		    cnp->left = TmpCnp;
		    cnp = TmpCnp;
		    break;
		  }
	    else
	      {
		/* in the tree already */
		free( TmpCnp );
		if ( cnp->name != NULL ) 
		  {
		    /* name is valid */
		    CurrentContextName = cnp->name;
		    return( OK );
		  }
		/* else context name marked as invalid insert new name for 
		       same inode (very rare occurance) */
		break;
	      }
	  } 
      }

    cnp->contextid = cid;

    /* allocate a reasonable length of string and parse the name into it */
    cnp->name = CurrentContextName 
        = ParseName( malloc( NameLength + 2 ), Name, 1 );
    return( OK );

  } /* InsertContextNode */

DeleteContextNode( cid )
    ContextId cid;
/*
 * Since deleting a name from the context cache is such a rare
 * occurance, the actual method is to invalidate the context node
 * by setting the name to NULL.  If the inode is reallocated to
 * another directory, then the InsertContextNode routine will take
 * the appropriate action by validating the node at that time.
 */
  {
    register ContextNode *cnp;
     
    if ( CDebug ) printf( "DeleteContextNode: 0x%x\n", cid );
    if ( cid < MAX_WELL_KNOWN_CONTEXTS ) 
      {
	WellKnownContext[ cid ] = NULL;
        return;
      }

    cnp = RootContextNode;
    while ( cnp != NULL )
      {
	if ( cnp->contextid == cid ) break;
	if ( cnp->contextid < cid )
	    cnp = cnp->right;
	else
	    cnp = cnp->left;
      }

    if ( cnp != NULL )
      { /* invalidate */
        free( cnp->name );
	cnp->name = NULL;
      }

  } /* DeleteContextNode */

char *FindContext( cid )
    register ContextId cid;
/*
 * search for the specified context.
 * returns the name if found, otherwise null.
 */
  {
    register ContextNode *cnp;

    if ( CDebug ) printf( "FindContext: cid = 0x%x\n", cid );

    if ( cid == CurrentContext )
    	return( CurrentContextName );

    if ( cid < MAX_WELL_KNOWN_CONTEXTS )
        return( WellKnownContext[ cid ] );


    cnp = RootContextNode;
    while ( cnp != NULL )
      {
	if ( cnp->contextid == cid ) break;
	if ( cnp->contextid < cid )
	    cnp = cnp->right;
	else
	    cnp = cnp->left;
      }
    if ( cnp == NULL )
	return( NULL );

    return( cnp->name );	/* will be null if invalid */

  } /* FindContext */    
    

SystemCode ChangeContext( cid )
    register ContextId cid;
/*
 * Change the current working directory to the specified context
 * If not possible, then return an appropriate error code.
 */
  {
    register ContextNode *cnp;
    register char *np;
    
    if ( cid == CurrentContext )
    	return( OK );

    if ( (np = FindContext( cid )) == NULL )
        return( INVALID_CONTEXT );

    /* try to change the specified context */
    if ( chdir( np ) != 0 ) 
      {
	/* have to delete this context from the tree */
	DeleteContextNode( cid );

        switch ( errno )
          { 
		
	    case ENOTDIR: /* not a directory! */
            case ENOENT: /* directory is not found! */
		    return( INVALID_CONTEXT );	/* panic if cid = 0 */
		    break;
            case EACCES: 
		    return( NO_PERMISSION );	/* is this right? */
        	    break;
	    default:
		    return( INTERNAL_ERROR );
		    break;
          }
      }
      
    /* Valid context */
    CurrentContext = cid;
    CurrentContextName = np;
    return( OK );

  } /* ChangeContext */

char NameBuffer[ BUFFER_SIZE*2 ];

SystemCode GetName( req, pid, segsize, FileName )
    register NameRequest *req;
    ProcessId pid;
    unsigned *segsize;
    char **FileName;
/*
 * Return the fully qualified pathname of the file.
 */
  {
    extern char *strcat();
    ContextId cid;
    unsigned NameLength;
    char *NamePtr;
    register SystemCode r;
    char *ContextName;

    /* check validity of name length 
       NOTE: that we allow 0 length names which means "." in unix 
     */
    MoveLong( req->namelength, NameLength );
    if( NameLength > BUFFER_SIZE ||  NameLength < req->nameindex ) 
        return( BAD_ARGS );

    /* get the context name */
    MoveLong( req->namecontextid, cid );
    if ( (ContextName = FindContext( cid )) == NULL )
    	return( INVALID_CONTEXT );
    
    /* Was the name sent with the packet? */
    if ( *segsize < NameLength )
      { /* no, have to retrieve from remote host */
	MoveLong( req->nameptr, NamePtr );

	if( (r=MoveFrom( pid, NamePtr, BlockBuffer, NameLength )) != OK)
		return( r );
	*segsize = NameLength;
      }

    BlockBuffer[ NameLength ] = 0; /* terminate name */
    if ( CDebug ) 
        printf( "GetName: \"%s\"\n", &BlockBuffer[ req->nameindex ] );
   
    /* Is the name unix context independent? */
    if ( (*(NamePtr = &BlockBuffer[ req->nameindex ])) == '/' )
    	*FileName = NamePtr;	/* no need for context prefix */
    else
      { 
        /*
	 * NOTE: should be checking for context+name > sizeof(NameBuffer).
	 */
        strcpy( NameBuffer, ContextName );	/* copy context name */
	*FileName = strcat( NameBuffer, NamePtr ); /* append file name */
      }
    
    return( OK );

  } /* GetName */


/* CHARACTER STRING NAME REQUEST HANDLERS */
SystemCode GetContextId( req, pid, segsize )
    ContextRequest *req;
    ProcessId pid;
    unsigned segsize;
/*
 * Find the ContextId for the specified name.
 * Once found, store in a cache that maps ContextIds to names
 * (which is the harder of the two functions).
 */
  {
    register SystemCode r;
    ContextId cid;
    char *NamePtr;
  
    if( Session == NULL ) return( NO_PERMISSION );
 
    Session->inactivity = 0;

    if ( (r = GetName( req, pid, &segsize, &NamePtr )) != OK )
      {
	((ContextReply *)req)->contextid = 0;
        return( r );
      }

    if( CDebug ) printf( "GetContextId: \"%s\"\n", NamePtr );

    if ( (r = InsertContextNode( NamePtr, strlen( NamePtr ), &cid )) != OK )
      {
	((ContextReply *)req)->contextid = 0;
        return( r );
      }
    
    /* valid context */
    if ( CDebug ) printf( "Context id is 0x%x\n", cid );
    MoveLong( cid, ((ContextReply *)req)->contextid );
    MoveLong( MyPid, ((ContextReply *)req)->serverpid );
    /* irrelevant information */
    ((ContextReply *)req)->entrytype = 0;
    ((ContextReply *)req)->instanceid = 0;
    ((ContextReply *)req)->otherinfo = 0;
    
    return( OK );
    
  } /* GetContextId */

SystemCode GetContextName( req, pid )
    ContextRequest *req;
    ProcessId pid;
/*
 * Search for the name of the specified context
 */

  {
    ContextId cid;
    unsigned namelen;
    register char *np;
    register ContextNode *cnp;
    unsigned segsize;
    unsigned segptr;

    MoveLong(req->namelength, segsize);
    req->namelength = 0;

    if ( Session != NULL ) Session->inactivity = 0;

    MoveLong( req->serverpid, cid ); /* cid used as a temporary */
    if ( (ProcessId) cid != MyPid )
	return( NONEXISTENT_SESSION );
    MoveLong( req->contextid, cid );

    if ( CDebug ) printf( "GetContextName: ContextId = 0x%x\n", cid );

    /* Look up context id in the cache */
    if ( (np = FindContext( cid )) == NULL )
        return( INVALID_CONTEXT );
    
    /* enough space to write it? */
    if ( (namelen = strlen( np )) > segsize )
        return( BAD_BUFFER );
  
    if ( CDebug ) printf( "Context name is \"%s\"\n", np );

    /* where to put it */
    MoveLong( req->nameptr, segptr );

    /* how much we are sending */
    MoveLong( namelen, ((ContextReply *)req)->namelength );

    /* context to interpret it in */
    cid = ANY_CONTEXT;
    MoveLong( cid, ((ContextReply *)req)->contextid );
    MoveLong( MyPid, ((ContextReply *)req)->serverpid );
    /* irrelevant information */
    ((ContextReply *)req)->entrytype = 0;
    ((ContextReply *)req)->instanceid = 0;
    ((ContextReply *)req)->otherinfo = 0;

    if ( namelen <= MAX_APPENDED_SEGMENT && Session != NULL )
      { /* put it in the reply packet */
        ((ContextReply *)req)->replycode = OK; 
        ReplyWithSegment( req, pid, np, segptr, namelen );
        return( NOT_AWAITINGREPLY ); 
      }

    /* long name! */
    return( MoveTo( pid, segptr, np, namelen ) );

  } /* GetContextName */

SystemCode GetFileName( req, pid )
    ContextRequest *req;
    ProcessId pid;
/*
 * Search for the instance id in the request and return the name 
 * into the requesting processes segment.
 */

  {
    ContextId cid;
    InstanceId id;
    register FileInstance *desc;
    unsigned namelen;
    unsigned segsize;
    unsigned segptr;

    MoveLong(req->namelength, segsize);
    req->namelength = 0;

    if ( Session != NULL ) Session->inactivity = 0;

    MoveLong( req->serverpid, (ProcessId) cid ); /* cid used as a temporary */
    if ( (ProcessId) cid != MyPid )
	return( NONEXISTENT_SESSION );

    id = req->instanceid;

    if ( CDebug ) printf( "GetFileName: InstanceId = 0x%x\n", id );

    /* Find the instance */
    if ( (desc = GetInstance( id )) == NULL )
        return( INVALID_FILE_ID );
    
    /* The null string is a valid name, thus we must return an error */
    if ( desc->name == NULL )
        return( NOT_FOUND );

    /* enough space to write it? */
    if ( (namelen = strlen( desc->name )) > segsize )
        return( BAD_BUFFER );
  
    if ( CDebug ) printf( "File name is \"%s\"\n", desc->name );

    /* where to put it */
    MoveLong( req->nameptr, segptr );

    /* how much we are sending */
    MoveLong( namelen, ((ContextReply *)req)->namelength );

    /* context to interpret it in */
    cid = ANY_CONTEXT;
    MoveLong( cid, ((ContextReply *)req)->contextid );
    MoveLong( MyPid, ((ContextReply *)req)->serverpid );
    ((ContextReply *)req)->instanceid = id;	/* not necessary to set this */

    /* irrelevant information */
    ((ContextReply *)req)->entrytype = 0;
    ((ContextReply *)req)->otherinfo = 0;

    if ( namelen <= MAX_APPENDED_SEGMENT && Session != NULL )
      { /* put it in the reply packet */
        ((ContextReply *)req)->replycode = OK; 
        ReplyWithSegment( req, pid, desc->name, segptr, namelen );
        return( NOT_AWAITINGREPLY ); 
      }

    /* long name! */
    return( MoveTo( pid, segptr, desc->name, namelen ) );

  } /* GetFileName */

