/*
 * The V UNIX server: a V kernel and V server simulator for VAX/UNIX
 * that provides a subset of UNIX system services to SUN workstations
 * running the V distributed kernel.
 * Copyright (c) 1982 David R. Cheriton, all rights reserved.
 *
 * V kernel simulation code
 */

#include <signal.h>
#include <wait.h>
#include <Venviron.h>
#include <Vio.h>
#include <Vioprotocol.h>
#include <Vethernet.h>
#include <Vikc.h>
#include <net.h>
#include <kernel.h>
#include <debug.h>

ProcessId Logical_id_map[MAX_LOGICAL_ID];
int Net;
NetAddress LocalHost;			/* The servers network address */
ProcessId MyPid;		/* Pid of server */
ProcessId KernelProcessPid;	/* Pid of kernel process for server */
int RootServer = 1;
unsigned char * RecBufPtr = 0;	/* Where to put received segments */
unsigned      *RecBufSizePtr;   /* Max size of received segment allowed */
unsigned	RecBufSize;	/* Remember internally max size  specified */
ProcessId NewSessionPid;	/* Pid used for new session server */
ProcessId TimerPid;		/* Pid for timer pseudo process */
ProcessId RemotePid;		/* Remote process for current operation */
unsigned short CurrentSeqNo = 0x5555; /* Generator for sequence numbers */

struct SendRecord *SendQueue = NULL;  /* all messages which are to be 
					 processed are put in this queue */
struct SendRecord *ReceivePtr = NULL; /* next (or current) message to be
					 processed (received) */
struct SendRecord *SendAlloc();		

/* Signal routines */
void PacketHandler(),AlarmHandler(), DeadChildHandler(), KillSession();

int MyUserId, MyGroupId;    /* Effective user and group id's */
int AlarmSig;		    /* State of the net: REPEAT, RETRANSMIT, DONE 
			       (changed by the alarm handler) */
int KernelOpStatus;	    /* State of kernel operation: REPEAT, DONE, NACK 
			       (changed by a kernel reply */
int ChildrenAreDead;	    /* If SIGCHLD event has occurred */
int SendQueueLock = 0;	    /* Used to sync with alarm handler */
unsigned LocalAddr,RemAddr,TotalLength;
int CurrentOp;		/* Current operation waiting to complete */
int Retrans;		/* Number of retransmissions so far */
int AlarmRate;		/* alarm signal rate in seconds */
InitKernel( reinit )

    int reinit;
  /* Open a connection to the network, set up the network filter,
   * and initialize the pseudo-kernel data structures.
   * If reinit is non-zero, assume the network file is already open
   * and MyPid has been set.
   */
  {
    extern void BadSystemCallHandler();
    extern void DeadProcessHandler();
    extern void TerminalStopHandler();
    extern void KillSession();
    extern int RootServer;
    extern MyUserId, MyGroupId;
    extern geteuid(), getegid();

    RootServer = reinit == 0;
    MyUserId = geteuid();
    MyGroupId = getegid();
    if( reinit == 0 ) /* Open an network file */
      {
	if( (Net = NetOpen( MAIN_SERVER_PID )) == -1 )
	  {
	    printf( "Cannot open network file\n" );
	    exit( 1 );
	  }
	/* Set server Pid */
	MyPid = (GetLogicalHost()<<24) + MAIN_SERVER_PID;
	KernelProcessPid = (GetLogicalHost()<<24) + LOCAL_KERNEL_PROCESS_PID;
	if ( KDebug ) printf( "KernelProcessPid = 0x%x\n", KernelProcessPid );
	NewSessionPid = MyPid;	/* To be incremented for each new session */
	TimerPid = MyPid + 0x8000; /* Random number */
      }
    /* Init storage buffers */
    InitBufMgr();

    SendQueue = ReceivePtr = NULL;

    /* Set interrupt routines */
    sigset( PACKET_RECEIVED,PacketHandler );
    sigset( SIGALRM, AlarmHandler );
    AlarmRate = FAST_ALARM_RATE;
    sigset( SIGCHLD, DeadChildHandler );
    sigset( SIGTSTP, TerminalStopHandler );
    sigset( SIGSYS, BadSystemCallHandler );
    ChildrenAreDead = 0;
    if ( !RootServer )
      {
	sigset( SIGINT, KillSession );    /* assure smooth death of */
	sigset( SIGTERM, KillSession );   /* sessions. */
	sigignore( SIGPIPE );		  /* so session won't die on writes to pipe*/
	/* ABOVE SHOULD CALL A ROUTINE THAT OPERATES ON THE SIGNAL */
      }

    /* Enable signal */
    NetSignal( Net, PACKET_RECEIVED );
    alarm( AlarmRate );
  }

ProcessId ReceiveWithSegment( msg, recbufptr, recbufsizeptr )
   Message msg; unsigned char *recbufptr; unsigned *recbufsizeptr;

 /*
  * Head of Queue for send request awaiting
  *   to be received e pointed to by ReceivePtr.
  *   We take this by copying out the Message
  *   and moving ReceivePtr onto next request.
  *
  * If ReceivePtr is NULL then there are no
  *  send outstanding so we have to pause() 
  *  until one arrives.
  * Note that a pause can be released by any packet arriving.
  */
  {
    register unsigned length;
    register struct SendRecord *msgptr;

    RecBufSize = *recbufsizeptr; /* Save maximum segment size */
    RecBufSizePtr = recbufsizeptr;
    RecBufPtr = recbufptr;
    *recbufsizeptr = 0; /* Set to 0, set otherwise when segment received */

    while(1)
      {
	/* Find the first message which has not been processed
	   it should be marked with msg[0] == 0
	 */
	for( msgptr = SendQueue; (msgptr!=NULL) && (msgptr->msg[0] == 0);
	  	msgptr = msgptr->next );
	if( msgptr != NULL ) break;
	pause();
      }
    SendQueueLock = 1;
    sighold( PACKET_RECEIVED );

    RecBufPtr = 0; /* Flag we are not willing to receive segments now */
    ReceivePtr = msgptr->next;

    /* Copy message */
     {
	register int i;

	for( i=0; i<8; i++ ) msg[i] = msgptr->msg[i];
      } 
    msgptr->msg[0] = 0;		/* mark this msg as processed (received) */
    SendQueueLock = 0;
    sigrelse( PACKET_RECEIVED );

    return( msgptr->srcpid );
  }

ProcessId ReplyWithSegment( msg, pid, src, dest, bytes )
   Message msg;
   ProcessId pid;
   unsigned char *src;
   unsigned dest, bytes;

  /* Check to see if there is a send request
   *  corresponding to this reply.  If there is then
   *  reply to the remote process and remove the
   *  send request from the list of queued messages.
   *
   * If there is no corresponding send to the reply
   *  then zero is returned else the pid.
   *
   */
 {
    register struct SendRecord *ptr, *prevptr;
    register unsigned evensize;
    ProcessId destpid;
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);

    if( SendQueue == NULL ) return( 0 ); /* no corresponding
					    send for reply */

    /* Disable signals to prevent trying to queue new send
     * while we are removing old ones, this is necessary to
     * keep the queue pointers ok.
     */
    SendQueueLock = 1;
    sighold( PACKET_RECEIVED );
    
    if ( KDebug && bytes != 0 ) printf( "ReplyWithSegment\n" );

    /* search for correct send */
    ptr = SendQueue; /*get queue root */

    /* Find the first message that this pid has sent
       NOTE: there should only be one message in the queue from pid */
    while( ptr->srcpid != pid )
      {
	if( ptr->next == NULL ) 
	  { 
	    /* no more messages, thus fail */
	    destpid = 0;
	    goto exit; 
	  }
	prevptr = ptr;
	ptr = ptr->next;
      }

    /* Has this message been processed yet? */
    if( ptr->msg[0] != 0  ) 
      {
	destpid = 0;
	goto exit;
      }
	
    /* Message found so reply to it */
    if( ptr->srcpid != TimerPid ) /* Dont reply to pseudo timer process */
      {
	/* Handle the appending of the data segment if any */
	if( bytes > MAX_APPENDED_SEGMENT  ) bytes = MAX_APPENDED_SEGMENT;
	evensize = bytes + (bytes&1);
	if( bytes ) swab( src, packet->data, evensize );
	NetWrite(remoteReply,Net,SMALL_KPACKET+evensize,&kpacket,ptr->dstpid,pid,0,
		bytes, bytes, dest, ptr->sequenceNo, msg );
      }
    /* Take out of queue and put in free storage */
    if( ptr == SendQueue )
	SendQueue = ptr->next;
    else  /* take out of list */
	prevptr->next = ptr->next;

    destpid = ptr->srcpid;
    SendFree( ptr );
    /* reenable signal */
exit:
    SendQueueLock = 0;
    sigrelse( PACKET_RECEIVED );
	
    return( destpid );
  }
ProcessId Reply( msg, pid )
   Message msg;
   ProcessId pid;
  {
   
   return( ReplyWithSegment(msg, pid, 0, 0, 0) );
  }

ProcessId Forwarder( pid )
    ProcessId pid;
   /* Return the pid of the process that forwarded the message
    * last received from the specified process, which is the
    * sender if the message was not forwarded.
    * Return 0 if the process is not awaiting reply.
    */
 {
    register struct SendRecord *ptr;

    /* Find message in the send queue */

    ptr = SendQueue; /*get queue root */
    while( ptr != NULL )
     {
	if( ptr->srcpid == pid ) return( ptr->forwarder );
	ptr = ptr->next;
      }
    return( 0 );
  }

ProcessId Forward( msg, frompid, topid )
   Message *msg;
   ProcessId frompid, topid;
  /* Forward frompid to topid with the specified message */
  {
    register struct SendRecord *ptr, *prevptr;
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);

    /* Find in send message queue */
    if( SendQueue == NULL ) return( 0 ); /* Empty queue */

    /* Disable signals to prevent trying to queue new send
     * while we are removing old ones, this is necessary to
     * keep the queue pointers ok.
     */
    SendQueueLock = 1;
    sighold( PACKET_RECEIVED );
    ptr = SendQueue; /*get queue root */

    /* find the first message which is from the frompid */
    while( ptr->srcpid != frompid )
      {
	if( ptr->next == NULL )
	  { 
	    topid = 0;
	    goto exit; 
	  }
	prevptr = ptr;
	ptr = ptr->next;
      }

    /* Has this message been "received" yet? */
    if( ptr->msg[0] != 0 ) 
      {
	topid = 0;
	goto exit;
      }

    NetWrite(remoteForward,Net,SMALL_KPACKET,&kpacket,topid,frompid,ptr->dstpid,0,0,
					0, ptr->sequenceNo, msg );

    /* Take out of queue and put in free storage */
    if( ptr == SendQueue ) SendQueue = ptr->next;
    else  /* take out of list */
	prevptr->next = ptr->next;

    SendFree( ptr );
    /* reenable signal */
exit:
    SendQueueLock = 0;
    sigrelse( PACKET_RECEIVED );
	
    return( topid );
  }

ProcessId ValidPid( pid )
   ProcessId pid;
   /*
    * Return non-zero if a valid Pid else 0.
    */
  {
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);
    Message msg;
    unsigned short sequenceno;
    unsigned retranscount;

    ++CurrentSeqNo;
    RemotePid = pid;
    retranscount = 0;

    KernelOpStatus = REPEAT;	/* only needs to be set once, any change
    				   terminates the operation */
    do
      {
	/* NOTE critical race condition if the VAX is loaded
		- we don't know when the Alarm will go off
		- the SUN kernel can reply very quickly, so
		  CurrentOp must be set before doing the write.
         */
	AlarmSig = DONE;
	CurrentOp = VALID_PID_OP;
	NetWrite(remoteReceiveSpecific,Net,SMALL_KPACKET,&kpacket,MyPid,
		pid, 0, 0, 0, 0, CurrentSeqNo, msg );
	AlarmSig = REPEAT;
	while( AlarmSig == REPEAT && KernelOpStatus == REPEAT ) pause();
	CurrentOp = NULL;
	if( ++retranscount > MAX_RETRANSMISSIONS || KernelOpStatus == NACKED )
	  {
	    return( 0 );
	  }
      }
    while( KernelOpStatus != DONE );
     
    return( pid );
  }

MoveTo( dstpid, dstaddr, srcaddr, bytes )
   ProcessId dstpid;
   unsigned dstaddr;
   char *srcaddr;
   unsigned bytes;
  /* Issue a MoveTo packet to destination workstation and wait for
   * acknowledgement. If a response is not received 
   * in the alarm time then we retransmit the request. If after
   * max retransmissions we still have not received a reply then
   * the MoveTo fails and returns RETRY.
   */
  {
    int packets,i,size, retranscount, evensize;
    unsigned dstptr;
    char *srcptr;
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);

    RemotePid = dstpid;
    RemAddr = dstaddr;
    TotalLength = bytes;
    packets = bytes / MAX_APPENDED_SEGMENT;
    ++CurrentSeqNo;
    retranscount = 0;
    if( KDebug ) printf( "MoveTo %d bytes\n", bytes );

    KernelOpStatus = REPEAT;	/* only needs to be set once, any change
    				   terminates the operation */

    do
      {
	/* NOTE critical race condition if the VAX is loaded
		- we don't know when the Alarm will go off
		- the SUN kernel can reply very quickly, so
		  CurrentOp must be set before doing the write.
         */
	AlarmSig = DONE;
	CurrentOp = MOVE_TO_OP;

	/* Transmit data in packets */
	dstptr = dstaddr;
	srcptr = srcaddr;
	for( i=0; i < packets; ++i )
	  {
	    swab( srcptr, packet->data, MAX_APPENDED_SEGMENT );
	    /* NOTE: MyPid is really incorrect, because, there is more
	       than one process running with this "kernel", namely,
	       the KERNEL and MyPid.  OK since KERNEL never sends.
	     */
	    NetWrite(remoteMoveToReq,Net,MAX_APPENDED_SEGMENT+SMALL_KPACKET,
	 &kpacket,MyPid,dstpid,0,MAX_APPENDED_SEGMENT,bytes,dstptr, CurrentSeqNo, 0);
	    dstptr += MAX_APPENDED_SEGMENT;
	    srcptr += MAX_APPENDED_SEGMENT;
	  }
	if( (size = (bytes % MAX_APPENDED_SEGMENT)) > 0 )
	  {
	    evensize = size + (size&1);
	    swab( srcptr, packet->data, evensize );
	    NetWrite(remoteMoveToReq,Net,evensize+SMALL_KPACKET,&kpacket,
			MyPid,dstpid,0,size,bytes,dstptr,CurrentSeqNo,0);
	  }
	AlarmSig = REPEAT;
	while( AlarmSig == REPEAT && KernelOpStatus == REPEAT ) pause();
	
	/* KernelOpStatus set to DONE by network or 
	   AlarmSig set to RETRANSMIT by AlarmHandler */
	CurrentOp = NULL;
	if( retranscount++ > MAX_RETRANSMISSIONS || KernelOpStatus == NACKED )
	  {
	    /* We have done all our retransmissions */
	    return( TIMEOUT );
	  }
      }
    while( KernelOpStatus != DONE );

    return( OK );
  }


MoveFrom( srcpid, src, dest, bytes )

   ProcessId srcpid;
   unsigned src,dest;
   unsigned bytes;
  /* Issue a move from request and wait for response.
   *   Alarm handler makes sure we are not waiting indefinitely
   *   for a response. If we do not get all the data in this length
   *   time the a request is sent out only for the data we have not
   *   yet received.
   */
  {
    struct SmallKernelPacket kpacket;
    register kPacket *packet = &(kpacket.kernelpacket);
    Message msg;

    RemotePid = srcpid;
    RemAddr = src;
    LocalAddr = dest;
    TotalLength = bytes; /* count of bytes we have not get yet */
    ++CurrentSeqNo;
    Retrans = 0;
    if( KDebug ) printf( "MoveFrom %d bytes\n", bytes );

 
    KernelOpStatus = REPEAT;	/* only needs to be set once, any change
    				   terminates the operation */

 
   while( TotalLength != 0 )
      {
	/* NOTE critical race condition if the VAX is loaded
		- we don't know when the Alarm will go off
		- the SUN kernel can reply very quickly, so
		  CurrentOp must be set before doing the write.
         */
	AlarmSig = DONE;
	CurrentOp = MOVE_FROM_OP;

	if( bytes == TotalLength ) /* No data received since last request */
	  {
	    if( ++Retrans > 8 )
	      {
		CurrentOp = NULL;
		return( TIMEOUT );
	      }
	    /* NOTE that MyPid should really be the destination pid of
	       the original message, but this is okay since we only have
	       two processes (KERNEL and SERVER) that could have rcvd
	       the original message and that the KERNEL should not do
	       a MoveFrom
	     */
	    NetWrite(remoteMoveFromReq,Net,SMALL_KPACKET,&kpacket,
	     MyPid,srcpid,0,TotalLength,LocalAddr,RemAddr,CurrentSeqNo,msg);
	  }
	else bytes = TotalLength;

	AlarmSig = REPEAT;
	/* wait for timeout or done */
	while( TotalLength != 0 && (AlarmSig == REPEAT) 
		&& (KernelOpStatus == REPEAT) ) pause();
	CurrentOp = NULL;
	if ( KernelOpStatus == NACKED ) return( TIMEOUT );
      }
    return( OK );
  }


SetPid( logical_id, pid )
     unsigned logical_id; ProcessId pid;

  /* Special version to register this process as the storage server */
  {
    Logical_id_map[logical_id] = pid;
  }

ProcessId GetPid( logical_id )
   unsigned logical_id;

   /* Special version that just returns "active process pid".
    *
    */
  {
    extern ProcessId MyPid;

    return( MyPid );
  }


AnswerGetPid( packet, logicalid, dstpid, seqno )
  struct SmallKernelPacket *packet;
  unsigned logicalid;
  ProcessId dstpid;
  short unsigned seqno;

   /* Answer a GetPid request 
    *
    */
  {
    extern int PublicServer;
    ProcessId retpid;

    /* Only answer GetPid requests if running as a Public Vserver */
    if( !PublicServer || ((unsigned)logicalid > MAX_LOGICAL_ID) ||
		   ((retpid = Logical_id_map[logicalid]) != MyPid))
	return;

    if(NDebug) printf( "Answering remoteGetPid\n" );

    NetWrite(remoteGetPidReply,Net,SMALL_KPACKET,packet,retpid,dstpid,
						logicalid,0,0,0,seqno,0);
  }


void PacketHandler()

 /* Interrupt routine to handle incoming packets
  *  will read all the packtes currently in the
  *  buffer then return after processing them.
  *
  */
  {
    struct KernelPacket kpacket;
    register kPacketWithSegment *packet = &(kpacket.kernelpacket);
    int length,localaddr,remoteaddr;
    int PacketLength = sizeof(struct KernelPacket);
    ProcessId srcpid,dstpid,forwarder;
    short kernelop;
    short seqno;
    Message msg;
    
    while( 1 )
      {
	/* Receive waiting packet */

	kernelop = NetRead(Net,PacketLength,&kpacket,&srcpid,&dstpid,
		&forwarder, &length,&localaddr,&remoteaddr,&seqno,msg);
	/* check if we want it */
      	if( (dstpid == MyPid) || (kernelop == remoteGetPid) 
	     || (dstpid == KernelProcessPid) )
	{
	if (NDebug) 
	    printf( "seqno = %d, srcpid = 0x%x, msg[0] = 0x%x\n", 
			seqno, srcpid, msg[0] );
 	switch( kernelop )
	  {
	    case remoteGetPid:
		    AnswerGetPid( &kpacket, forwarder,srcpid,seqno);
		    break;

	  case remoteSend:
		HandleSend( &kpacket, msg, srcpid, dstpid, forwarder, length, seqno);
		break;

	  case nAck:
		ProcessNack( &kpacket, msg, srcpid,  seqno);
		break;

	  case breathOfLife:
		ProcessBreathOfLife( &kpacket, msg, srcpid, seqno );
		break;

	  case remoteMoveFromReply:
		MoveFromAnswer(&kpacket,srcpid,length,remoteaddr, seqno);
		break;

	  case remoteMoveToReply:
		if( (CurrentOp == MOVE_TO_OP) && (RemotePid == srcpid)
		  && (length == TotalLength) && (RemAddr == localaddr)
		  && seqno == CurrentSeqNo )
		  {
		    KernelOpStatus = DONE;
		  }
		break; 

	  }
	  }
	length = NetReceiveQLength( Net );
	if( length == 0 ) break;
      }
    NetSignal( Net,PACKET_RECEIVED );

  }

HandleSend( packet, msg, srcpid, dstpid, forwarder, length, seqno)
struct KernelPacket *packet;
register long *msg;
ProcessId srcpid, dstpid, forwarder;
unsigned length;
short unsigned seqno;
 /* Queue up send requests for receiver to pick them out.
  *
  */
  {
    register struct SendRecord *ptr;
    ProcessId destpid;	/* who the message was destined for */
    short i;

    SendQueueLock = 1;
    /* check to see if this is a retransmission */
    ptr = SendQueue;
    if( ptr == NULL )
      {
	ptr = SendAlloc();
	SendQueue = ptr; 
      }
    else
      { /* search previous send requests */
	while( (ptr->next != NULL) && (ptr->srcpid != srcpid) )
		{
		if (NDebug)
		printf( "ptr->dstpid = -x%x, ptr->srcpid = 0x%x, ptr->seqno %d\n",
		ptr->dstpid, ptr->srcpid, ptr->sequenceNo );
		ptr = ptr->next;
		}

	if( ptr->srcpid == srcpid )
	  {
	    if( seqno == ptr->sequenceNo )
	      {
		/* transmit breathOfLife to prevent send time out */
		NetWrite(breathOfLife,Net,SMALL_KPACKET,packet,ptr->dstpid,srcpid,0,0,0,0,seqno,msg);
	      }
	    SendQueueLock = 0;
	    return; /* duplicate send so may discard */
	  }
	/* A new send request */
 	/* Check for buffer space for new message */
	if( (ptr->next = SendAlloc()) == NULL )
	  {
	    SendQueueLock = 0;
	    return;
	  }

	ptr = ptr->next;
      }
    ptr->srcpid = srcpid;
    ptr->dstpid = dstpid;
    ptr->forwarder = forwarder;
    ptr->sequenceNo = seqno;
    ptr->next = NULL;
      { /* Copy message */
	register long *mesg = ptr->msg;
	short i = 9;

	while( --i ) *mesg++ = *msg++;
      }
    /* If there aren't any messages which have not been "Received",
       then put this message as first to be "Received".
       "Receiving" - is to attempt to handle a server request.
     */
    if( ReceivePtr == NULL )
      {
	ReceivePtr = ptr;

	/* Try to deliver the segment if any */
	if( RecBufPtr != NULL )
	  {
	    if( KDebug ) printf( "segsize = %d\n", length );
	    if( length > RecBufSize ) length = RecBufSize;
	    *RecBufSizePtr = length;
	    swab(packet->kernelpacket.data,RecBufPtr, length + (length&1) );
	    RecBufPtr = 0; /* Flag to discard subsequent segments. */
	  }
      }
    SendQueueLock = 0;
 } /* HandleSend */

ProcessBreathOfLife( packet, msg, srcpid, seqno )
  struct KernelPacket *packet;
  register long * msg;
  ProcessId srcpid;
  short unsigned seqno;
  /* Handle an incoming breath-of-life packet. */
  {

    if( (CurrentOp == VALID_PID_OP) && (srcpid == RemotePid)
	 && (seqno == CurrentSeqNo) )
      {
	if( KDebug ) printf( "ValidPid OK\n" );
	KernelOpStatus = DONE;
      }
  }

ProcessNack( packet, msg, srcpid, seqno )
  struct KernelPacket *packet;
  register long * msg;
  ProcessId srcpid;
  short unsigned seqno;
  /* Handle a Nack 
   * This means that the process no longer exists.
   */
  {

    if( (CurrentOp != NULL) && (srcpid == RemotePid)
	 && (seqno == CurrentSeqNo) )
      {
	if( KDebug ) printf( "Nack receive from pid 0x%x\n", srcpid );
	KernelOpStatus = NACKED;
      }
  } /* ProcessNack */

MoveFromAnswer(packet,srcpid,length,localaddr, seqno)
 /* Handle a move from put data in right place
  *
  */
   struct KernelPacket *packet;
   ProcessId srcpid;
   unsigned length;
   unsigned localaddr;
   short unsigned seqno;
  {

    if( (CurrentOp == MOVE_FROM_OP) && (RemotePid == srcpid) && (length <= MAX_APPENDED_SEGMENT) &&
	(seqno == CurrentSeqNo)  && (localaddr == LocalAddr) && (length <= TotalLength) )
      {
	swab(packet->kernelpacket.data,LocalAddr, length + (length&1) );

	LocalAddr += length;
	TotalLength -= length;
	RemAddr += length;
	Retrans = 0; /* as long as still getting packets there is hope */
	if( TotalLength == 0 )
	  {
	    KernelOpStatus = DONE;
	  }
      }
  }


void AlarmHandler()
 /* 
  *  Initiate retransmissions by setting global flag to say retransmit .
  *  Queue a time-out message for the server.
  */
  {
    register struct SendRecord *ptr, *msgbuf;
    register IoRequest *msgptr;


    /*
     * Manufacture a time-out message if not interrupting critical part
     * of the kernel.
     */ 
    if( SendQueueLock )
      {
        if( (CurrentOp != NULL) && AlarmSig == REPEAT ) AlarmSig = RETRANSMIT;
	alarm( AlarmRate );
	return;
      }
     sighold( PACKET_RECEIVED );
     if( (msgbuf = SendAlloc()) != NULL )
      {
	msgbuf->srcpid = TimerPid;
	msgbuf->next = NULL;
	msgptr = (IoRequest *) msgbuf->msg;
	msgptr->requestcode = TIMEOUT;

	/* search for end of the SendQueue */
	ptr = SendQueue; /*get queue root */
	if( ptr == NULL )
	    SendQueue = msgbuf;
	else
	  {
	    while( ptr->next != NULL ) ptr = ptr->next;
	    ptr->next = msgbuf;
	  }

	if( ReceivePtr == NULL ) ReceivePtr = msgbuf;
      }
    if( (CurrentOp != NULL) && AlarmSig == REPEAT ) AlarmSig = RETRANSMIT;
    alarm( AlarmRate );
    sigrelse( PACKET_RECEIVED );
  }

void DeadChildHandler()

   /*
    * If root server then do nothing.
    * Else a session must mark that a child has died.
    */
  {
    extern int ChildrenAreDead;
    extern int RootServer;

    /* All children require status check before exiting */
    ChildrenAreDead = 1;

  } /* DeadChildHandler */

void KillSession()
  /* 
   * Kill all process instances associated with this session
   * before we commit suicide.
   */
  {
    extern ReclaimInstances(),CheckProcess();

    sigignore( PACKET_RECEIVED );
    sigignore( SIGALRM );
    sigignore( SIGCHLD );
    sigignore( SIGINT );
    sigignore( SIGTERM );

    if (KDebug) printf( "KillSession\n");
    ReclaimInstances( 1 );    /* release all instances */
    CheckProcesses();

    exit( 0 );

  } /* KillSession */

void TerminalStopHandler()
/*
 * Handles the SIGTSTP signal.
 * Turns-on or off debugging mode.
 */
  {
    extern SetDebugMode();
    static long SavedDebug = -1;	/* all debug bits */

    if ( Debug == 0 )
      {
        SetDebugMode( SavedDebug );
      }
    else
      {
        SavedDebug = Debug;
        SetDebugMode( 0 );
      }
  } /* TerminalStopHandler */


void BadSystemCallHandler()
/*
 * Not really sure what to do other then output debugging information
 */
  {
    if ( GDebug )
      {
        printf( "Bad system call interrupt\n" );
      }
  } /* BadSystemCallHandler */
