/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/


/*
 * Marvin Theimer, 5/83, from code by TPM and DRC.
 * TPM 8/26/83 -- Checks for failure on malloc and calloc, and only
 *	does them the first time the function is called.
 *	Added LoadTeamFromNamedFile.
 */

#include <Vioprotocol.h>
#include <Vprocess.h>
#include <b.out.h>

extern File *Open();

#define TRUE 1
#define FALSE 0
#define NUL '\0'
#define CHUNK 256
#define BUFFERSIZE 1024
#define DEFAULT_STACK 4000
#define DEFAULT_PRIORITY 4
#define DEFAULT_TEAM_PRIORITY 30

/*
 * LoadTeamFromFileInstance:
 * Team file loading function which loads from an already opened file.
 * Modified version of loadteam which takes a file instance rather than a
 * file name.
 */
ProcessId LoadTeamFromFileInstance(fileServer, fileid, errcode)
    ProcessId fileServer;
    InstanceId fileid;
    SystemCode *errcode;
  /*
   * Create a new team with the specified priority, and load the program 
   *   contained in the specified file into it.
   *
   * The file must be in b.out format. The b.out header is checked for a 
   *   valid "magic number."
   * The file is loaded into the new team and the bss segment is zeroed.
   * If LOADSYMBOLS is defined, the symbol table is loaded just above
   *  the bss segment, preceded by a longword containing its length.
   * The number of bytes specified by stacksize is allocated
   *   at the end of the team space as a stack, unless stacksize
   *   is -1, in which case 2048 bytes are allocated.
   * If errors occur during loading, NULL is returned, and a system reply
   *   code describing the error is placed in errcode if it is not null.
   *
   * Team layout:
   *  |kernel|text|data|uninitialized data(bss)|symbols(optional)|stack space|
   */

  {
    File *teamfile;
    int  rsize, bytes, blocksize;
    char *bssptr, *readend, *loadptr, *teamendptr, *GetTeamSize(),
    						   *SetTeamSize();
    static short *databuffer = NULL;
    register char *buffer;
    static short *zeroes = NULL;
    ProcessId teampid;
    struct bhdr header;
    Processor_state teamstate;
    SystemCode r;
    int stacksize;
    unsigned priority, teamPriority;

    if (errcode == NULL) errcode = &r;
    stacksize = DEFAULT_STACK;
    priority = DEFAULT_PRIORITY;
    teamPriority = DEFAULT_TEAM_PRIORITY;

    if ( databuffer == NULL &&
	(databuffer = (short *) malloc(BUFFERSIZE)) == NULL )
      {
	*errcode = NO_MEMORY;
	return (NULL);
      }
    buffer = (char *) databuffer;

    if ( zeroes == NULL &&
	(zeroes = (short *) calloc(BUFFERSIZE/sizeof(short), 
	    sizeof(short))) == NULL )
      {
	*errcode = NO_MEMORY;
	return (NULL);
      }

    if ( (teampid = CreateTeam(priority, 0, 0)) == NULL )
      {
	*errcode = NO_PDS;	/* could also be illegal priority, but ... */
	return (NULL);
      }
    SetTeamPriority(teampid, teamPriority);

    teamfile = OpenFile(fileServer, fileid, FREAD | FBLOCK_MODE, errcode);
    if( *errcode || (teamfile == NULL) )
      {
	DestroyProcess(teampid);
	return (NULL);
      }

    bytes = 0;
    blocksize = teamfile->blocksize;

    /* Read first block of file to get b.out header */
    rsize = Read(teamfile, buffer, blocksize);
    if( rsize < sizeof(struct bhdr) )
      { 
	Close(teamfile);
	DestroyProcess(teampid);
	*errcode = END_OF_FILE;
	return (NULL);
      }
    bytes += rsize;

    /* Copy b.out header to header buffer */
    header = *(struct bhdr *) buffer;
    
    /* Check the header has the legal magic number */
    if( header.fmagic != FMAGIC )
      {
	Close(teamfile);
	DestroyProcess(teampid);
	*errcode = BAD_STATE;
	return (NULL);
      }
    /* Calculate the required sizes and pointers */
    loadptr = GetTeamSize( teampid ); /* Point to start loading */
    /* Beginning of bss area in team */
    readend = bssptr = loadptr + header.tsize + header.dsize;
    /* Pointer to one beyond last byte to be init in read */

    /* Total amount of memory allocated to team */
    teamendptr = readend + header.bsize + stacksize;
	
    /* Set size of destination team */
    if( SetTeamSize(teampid, teamendptr) != teamendptr )
      {
	Close(teamfile);
	DestroyProcess(teampid);  /* don't need to set to null if destroying */
	*errcode = NO_MEMORY;
	return (NULL);
      }

    /* Move remainder of first data block to team code segment */
    buffer += sizeof(struct bhdr);
    rsize -= sizeof(struct bhdr);
    /* In case amount to read is less than that in one block */
    if( loadptr+rsize > readend ) rsize = (int) (readend - loadptr);

    if( (*errcode = MoveTo(teampid,loadptr,buffer,rsize)) != OK )
      {
	Close(teamfile);
	DestroyProcess(teampid);
	return(NULL);
      }
    buffer = (char *) databuffer;
    loadptr += rsize;
    teamfile->block = 1; /* Set to next block number */

    /* Load rest of file */
#ifdef SLOWLOAD
    while( loadptr < readend )
      {
	bytes = (int) (readend - loadptr);
	bytes = (bytes < blocksize) ? bytes : blocksize;

	if( (rsize = Read( teamfile, buffer, bytes)) < bytes )
	  {
	    Close(teamfile);
	    errcode = teamfile->lastexception;
	    DestroyProcess(teampid);
	    return (NULL);
	  }
	if( (*errcode = MoveTo(teampid,loadptr,buffer,rsize)) != OK )
	  {
	    Close(teamfile);
	    DestroyProcess(teampid);
	    return(NULL);
	  }
	loadptr += rsize;
	teamfile->block++; /* move to next file block */
      }
#else
    /* Fast team load using the forwarding trick */
     {
	Message msg;
	IoRequest *req = (IoRequest *) msg;

	req->requestcode = READ_AND_FORWARD;
	req->fileid = teamfile->fileid;
	req->blocknumber = 1;
	req->bufferptr = loadptr;
	req->bytecount = (unsigned) (readend - loadptr);
	Forward( req, teampid, teamfile->fileserver );
	ReceiveSpecific( req, teampid );
	*errcode = (SystemCode) (req->blocknumber>>16); 
				/* Special place for replycode */
	if( *errcode != OK )
	  {
	    Close(teamfile);
	    DestroyProcess(teampid);
	    return( NULL );
	  }
	loadptr = readend;
      }
#endif
    Close(teamfile);

    /* Now zero the bss area. */
    loadptr = bssptr;
    for( bytes = header.bsize; bytes > 0; bytes -= CHUNK)
      {
	if( (*errcode=MoveTo(teampid,loadptr,zeroes,
		((bytes <= CHUNK) ? bytes : CHUNK))) != OK) 
	  {
	    DestroyProcess(teampid);
	    return(NULL);
	  }
	loadptr += CHUNK;
      }

    /* Set pc and stack pointer for new team */
    ReadProcessState(teampid, &teamstate);
    teamstate.USER_STACK_POINTER = teamendptr - 2*sizeof(long);  /* Put two long zeros on stack */
    if( (*errcode=MoveTo(teampid,teamstate.USER_STACK_POINTER,zeroes,2*sizeof(long))) != OK )
      {
	DestroyProcess(teampid);
	return(NULL);
      }
    teamstate.pc = (Unspec) header.entry;
    WriteProcessState(teampid, &teamstate);

    return(teampid);
  }



/*
 * LoadTeamFromNamedFile:
 * Load a team from a file specified by name
 */
ProcessId LoadTeamFromNamedFile(filename, errcode)
    char *filename;
    SystemCode *errcode;
  {
    File *teamfile;
    ProcessId teampid;
    SystemCode r;
    extern SystemCode CreateInstance();
    Message msg;
    register CreateInstanceReply *reply = (CreateInstanceReply *) msg;

    if (errcode == NULL) errcode = &r;

    /* Open the file using CreateInstance since we do not need
     *   a File structure for it */
    *errcode = CreateInstance(filename, FREAD, reply);
    if (*errcode != OK) return(0);

    teampid = LoadTeamFromFileInstance(reply->fileserver,
		reply->fileid, errcode);

    ReleaseInstance(reply->fileserver, reply->fileid, OK);

    return (teampid);
  }

