/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer,  5/83
 */


/*
 * Lower-level support routines for the TCP part of the
 * network server.
 */


#include "net.h"
#include "iptcp.h"


/*
 * Imported routines
 */

extern SystemCode WriteIp();
extern SystemCode QueryInstance();
extern PktBuf DeQueue();




/*
 * HandleOther:
 * Check fin and urg bits of incoming segment as a fcn. of sendAckedFlag.
 * Handle any options present.
 * Queue any data in the segment for the receiver.
 * Reply to CreateInstance request for this connection.
 */

HandleOther(pTcb, curPkt, sendAckedFlag)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    Boolean sendAckedFlag;
  {
    Message msg;
    CreateInstanceReply *repMsg = (CreateInstanceReply *) msg;
    ProcessId pid;

    if (TcpPkt(curPkt->segAdr)->hdr.fin)
	if (sendAckedFlag)
	    pTcb->state = CloseWait;
	else
	  {
	    DiscardSegment(curPkt->segAdr, 
	    	    "fin set and our syn not yet acked.");
	    return;
	  }
    if (TcpPkt(curPkt->segAdr)->hdr.urg)
      {
	pTcb->rcvUrgFlag = True;
	pTcb->rcvUp = TcpPkt(curPkt->segAdr)->hdr.sequenceNumber + 
		TcpPkt(curPkt->segAdr)->hdr.urgentPointer;
      }
    if (curPkt->segDataOffset > StandardByteDataOffset)
	HandleOptions(pTcb, curPkt);
    if (curPkt->segLen > 0)
      {
	QSegment(pTcb, curPkt, sendAckedFlag);
				/* sendAckedFlag here indicates if we have
				   already reached a state where Receive
				   calls can be honored. */
      }
    else
	DiscardSegment(curPkt->segAdr, NULL);
    repMsg->replycode = OK;
    repMsg->fileid = pTcb->netId << 1;
    QueryInstance(repMsg);
    pid = Reply(msg, pTcb->rcvId);
  }




/*
 * HandleOptions:
 * Processes options present in incoming Tcp packet.
 * Assumes that options are present.
 */

HandleOptions(pTcb, curPkt)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
  {
    int i;
    TcpPktPtr p;
    Bit16 *v;
    Boolean done;

    i = 0;
    p = TcpPkt(curPkt->segAdr);
    done = False;
    while (! done)
      {
	switch (p->data[i])
	  {
	    case EndOptionList:
		done = True;
		break;
	    case NopOption:
		i++;
		break;
	    case MaxSegSizeOption:
		i += 2;
		v = (Bit16 *) &(p->data[i]);
		i += 2;
		pTcb->maxPacketDataLength = Min((int) (*v), 
			MaxPacketDataLength);
		NetInstTable[pTcb->netId].blocksize =
			pTcb->maxPacketDataLength;
		break;
	    default:
		done = True;
		break;
	  }
      }
  }




/*
 * FindWhichConnection:
 * Determines which connection is being referred to by the current event.
 * Returns id of Netinst record for that connection.
 * If no connection exists then -1 is returned.
 */

int    FindWhichConnection(
		createInstance, localPort, foreignPort, foreignHost)
    Boolean createInstance;	/* Signals whether this represents a
				   CreateInstance situation or not. */
    Bit16 localPort, foreignPort;
    Bit32 foreignHost;
  {
    int i, indx = -1, indx1 = -1;
    TcpTcbPtr p;

    if (createInstance)
	      {
	        if (foreignHost == 0)
		    return(-1);
		for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
		    if (NetInstTable[i].prot == TCPtype)
		      {
		        p = (TcpTcbPtr) NetInstTable[i].tcbId;
		        if ((localPort == p->localPort) &&
			        (foreignPort == p->foreignSocket.port) &&
			        (foreignHost == p->foreignSocket.host))
			  {
			    indx = i;
			    break;
			  }
		      }
		return(indx);
	      }
    else			/* packet from the network case */
	      {
		for (i = NetInstHead; i != -1; i = NetInstTable[i].next)
		    if (NetInstTable[i].prot == TCPtype)
		      {
		        p = (TcpTcbPtr) NetInstTable[i].tcbId;
			if (p->instState & TCP_CONN_CLOSED)
				/* Connection is already closed. */
			    continue;
		        if ((localPort == p->localPort) &&
			        (foreignPort == p->foreignSocket.port) &&
			        (foreignHost == p->foreignSocket.host))
		          {
			    indx = i;
			    break;
		          }
		        else if ((localPort == p->localPort) &&
				(p->foreignSocket.host == 0))
			  {
			    indx1 = i;
			  }
		      }
		if (indx != -1)
		    return(indx);
		else
		    return(indx1);
	      }
  }




/*
 * NoAccess:
 * Checks if user has access to specified connection.
 *
 * Currently this is a dummy routine.
 */

Boolean     NoAccess()
  {
            return(False);
  }




/*
 * VerifySecurityAndPrecedence:
 * Returns OK if security and precedence specified in event parms.
 * are ok, otherwise returns status indicating which is in error.
 *
 * Currently a dummy routine.
 */

SystemCode VerifySecurityAndPrecedence(pTcb)
    TcpTcbPtr pTcb;		/* Ptr to current connection. */
  {
            return(OK);
  }




/*
 * CloseConnection:
 * Close connection by discarding the incoming segment and then
 * deallocating the connection.
 */

CloseConnection(pTcb, curPkt, errCode)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    CurPktPtr curPkt;		/* Ptr to rec. for current packet. */
    SystemCode     errCode;
  {
    DiscardSegment(curPkt->segAdr, ErrorString(errCode));
    DeallocateTcb(pTcb, errCode);
  }




/*
 * In:
 * Determines whether a sequence number lies within the specified range.
 */

Boolean In(low, s, high)
    Bit32 low, high;		/* Range in which s should lie. */
    Bit32 s;			/* Sequence number to be checked. */
  {
    if (((low <= s) && (s <= high)) ||
	    ((high < low) && ((low <= s) || (s <= high))))
	return(True);
    else
	return(False);
  }




/*
 * CheckSum:
 * Computes the checksum for designated Tcp packet.
 * Optionally also copies the packet to a second one.
 *
 * The checksum consists of the one's complement of the one's 
 * complement sum of the 16-bit words in the tcp packet with
 * the checksum field set to zero.  This sum also includes the
 * pseudo header of the tcp packet which consists of the packet's
 * source and destination address, its length, and its type (TCP).
 */

Bit16 CheckSum(ptr, tcpLength, srcAdr, dstAdr, tcpType)
    TcpPktPtr ptr;		/* Ptr to Tcp packet to calc. checksum
				   for. */
    Bit16 tcpLength;		/* Length of tcp packet. */
    Bit32 srcAdr, dstAdr;	/* Source and destination address of
				   the packet. */
    Bit16 tcpType;		/* Type of the packet.  (Should be
				   TCPprot.) */
  {
    int len, len2;
    register int i;
    register unsigned long sum;
    unsigned short *w1, *w2, *w3, *w4;
    register unsigned short *w;
    Bit16 ckSum;

    ckSum = ptr->hdr.checkSum;
    ptr->hdr.checkSum = 0;
    len = (int) tcpLength / 2;
    w = (unsigned short *) &(ptr->hdr);

    sum = 0;
    len2 = len / 2;
    for (i = 0; i < len2; i++)
        sum += *w++ + *w++;
    if ((len2 << 1) != len)
	sum += *w++;		/* Add in the last (odd) word. */
    if ((tcpLength % 2) != 0)
	sum += ((*w) & 0xff00);
				/* Add in the last (odd) byte. */

    /* Add in the pseudo header. */
    w1 = (unsigned short *) &tcpLength;
    w2 = (unsigned short *) &srcAdr;
    w3 = (unsigned short *) &dstAdr;
    w4 = (unsigned short *) &tcpType;
    sum += (*w1) + (*w2++) + (*w3++) + (*w4);
    sum += (*w2) + (*w3);

    /* Add in the one's complement carry. */
    sum = (sum & 0xffff) + (sum >> 16);
    sum = (sum + (sum >> 16));

    ptr->hdr.checkSum = ckSum;
    return((Bit16) ~sum & 0xffff);
  }




/*
 * InitTcpConnection:
 * Initialize the designated Tcp connection data structures.
 */

InitTcpConnection(pTcb)
    TcpTcbPtr pTcb;
  {
    int     i;

    /*
     * Initialize to "blank" template state.
     */
    InitSafeQueue(&(pTcb->readerQueue), pTcb->ringBufs);
    pTcb->receiveBlocked = False;
    pTcb->netId = -1;
    pTcb->instState = 0;
    pTcb->state = Closed;
    pTcb->rcvId = 0;
    pTcb->wtSignalId = 0;
    pTcb->localPort = 0;
    pTcb->foreignSocket.port = 0;
    pTcb->foreignSocket.host = 0;
    pTcb->originalForeignSocket = pTcb->foreignSocket;
    pTcb->passiveOpen = False;
    pTcb->active = False;
    pTcb->prc = 0;
    pTcb->higherPrcAllowed = True;
    pTcb->security = 0;
    pTcb->waitSignal = False;
    pTcb->maxPacketDataLength = MaxPacketDataLength;
    for (i = SendQueue; i <= SaveQueue; i++)
	InitQueue(&(pTcb->q[i]));
    for (i = 0; i < NumTcpTimeouts; i++)
      {
	StopTimer(pTcb, i);
      }
    pTcb->rcvByteCnt = 0;
    pTcb->rcvQ = Null;
    pTcb->lastReadBuf = Null;
    pTcb->segQBytesAvail = 0;
    pTcb->sndUna = 0;
    pTcb->sndNxt = 0;
    pTcb->sndWnd = 0;
    pTcb->sndUp = 0;
    pTcb->sndWl1 = 0;
    pTcb->sndWl2 = 0;
    pTcb->sndUrgBit = False;
    pTcb->iss = 0;
    pTcb->rcvNxt = 0;
    pTcb->rcvWnd = RcvWindowSize;
    pTcb->delRcvWnd = 0;
    pTcb->rcvUrgFlag = False;
    pTcb->rcvUp = Null;
    pTcb->irs = 0;
    pTcb->numRetransTimeouts = 0;
    pTcb->numOutOfOrderPkts = 0;
  }




/*
 * DeallocateTcb:
 * Resets all fields of the Tcb, returns all queued buffers, replies
 * to all outstanding user calls.
 */

DeallocateTcb(pTcb, statusCode)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    SystemCode     statusCode;	/* Status to return to any outstanding
				   calls. */
  {
    Message msg;
    int i;

    if (InternetDebug)
      {
        printf("***** DeallocateTcb %d*****\n", pTcb->netId);
      }
    /* 
     * Clean up loose ends.
     */
    if (pTcb->state == SynSent)
      {				/* Respond to outstanding createInstance. */
        ((CreateInstanceReply *)msg)->replycode = statusCode;
	Reply(msg, pTcb->rcvId);
      }
    pTcb->state = Closed;
    if (pTcb->waitSignal)
	NotifyUser(pTcb, statusCode);
    if (pTcb->rcvQ != Null)
	ReplyToRead(statusCode, pTcb->rcvId, NULL, NULL, 0);

    if (InternetDebug && !Empty(pTcb->q[SaveQueue]))
      {
	printf("----- Still have packets on the save queue when ");
	printf("deallocating a Tcp connection! -----\n");
      }
    for (i = SendQueue; i <= SaveQueue; i++)
	FlushQueue(pTcb, i);
    for (i = 0; i < NumTcpTimeouts; i++)
	StopTimer(pTcb, i);
    /*
     * Free up the network instance.
     */
    pTcb->instState |= TCP_CONN_CLOSED;
  }





/*
 * SendIp:
 * Sends designated packet to Ip layer.
 * Returns WriteIp status code adjusted for Tcp status values.
 */

SystemCode SendIp(pTcb, packet, fHost)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. 
				   Null if no connection is associated with
				   this send. */
    PktBuf     packet;		/* Ptr to packet buffer to send
				   to Ip. */
    Bit32 fHost;		/* Foreign host to send to. */
  {
    Bit16 CheckSum();
    TcpPktPtr p;		/* Ptr to actual Tcp packet. */
    SystemCode rc;
    char *saveDataptr;		/* Used to save value of 
				   packet->dataptr during WriteIp. */
    int saveLength;		/* Used to save value of
				   packet->length during WriteIp. */

    p = TcpPkt(packet);
    p->hdr.checkSum = CheckSum(p, packet->length,
	    LocalIpHost, fHost, (Bit16) TCPprot);

    saveDataptr = packet->dataptr;
    saveLength = packet->length;
    rc = WriteIp(LocalIpHost, fHost,
		TCPprot, 0, 0, 5, packet, 0, 0);
				/* Note: WriteIp changes the values
				   of dataptr and length since it adds
				   an IP header to the packet. */
    packet->dataptr = saveDataptr;
    packet->length = saveLength;

    if (rc != OK)
      {
	if (InternetDebug)
	  {
	    printf("Return code from WriteIp: %d\n", rc);
	  }
      }
    else if (pTcb != NULL)
	StopTimer(pTcb, AckTimeout);
    return(rc);
  }




/*
 * InitPacketBuffer:
 * Initialize packet buffer.
 * Intended for use with outgoing Tcp packets.
 *
 * Assumes that dOffset is a multiple of 4.
 * Note: the urg and urgentPointer fields of the packet are set at
 * actual send time.
 */

InitPacketBuffer(lPort, fPort, tcpLen, dOffset, packet, seqNum, ackNum,
	    ack, psh, rst, syn, fin, rcvWnd)
    Bit16 lPort, fPort;		/* Local and foreign ports. */
    int tcpLen;			/* Length of Tcp packet. */
    int dOffset;		/* Offset in bytes of data 
				   field. */
    PktBuf packet;		/* Packet buffer to init. */
    Bit32 seqNum, ackNum;	/* Sequence number and ack number resp. */
    Boolean ack, psh, rst, syn, fin;
				/* Equivalent packet bit fields. */
    Bit16 rcvWnd;		/* Receive window size. */
  {
    TcpPktPtr ptr;		/* Ptr to Tcp packet to init. */

    packet->next = Null;
    packet->dataptr = &(packet->data[(MAXPBUFSIZE - sizeof(struct TcpBufType))]);
    packet->length = tcpLen;

    ptr = TcpPkt(packet);
    ptr -> hdr.sourcePort = lPort;
    ptr -> hdr.destinationPort = fPort;
    ptr -> hdr.sequenceNumber = seqNum;
    ptr -> hdr.acknowledgementNumber = ackNum;
    ptr -> hdr.dataOffset = dOffset / 4;
    ptr -> hdr.reserved = 0;
    ptr -> hdr.urg = 0;
    ptr -> hdr.ack = ack;
    ptr -> hdr.psh = psh;
    ptr -> hdr.rst = rst;
    ptr -> hdr.syn = syn;
    ptr -> hdr.fin = fin;
    ptr -> hdr.window = rcvWnd;
    ptr -> hdr.checkSum = Null16;
    ptr -> hdr.urgentPointer = Null16;
  }




/*
 * FlushQueue:
 * Flush designated queue.
 */

FlushQueue(pTcb, queue)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    int     queue;		/* Which queue to flush. */
  {
    PktBuf pkt;

    while (!Empty(pTcb->q[queue]))
      {
	pkt = DeQueue(&(pTcb->q[queue]));
	FreeBuf(pkt);
      }
  }




/*
 * StartTimer:
 * (Re)start specified timer.
 * NOTE: This routine assumes that there will always be a timer 
 * queue record available on the free list.
 */

StartTimer(pTcb, timer, len)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    int    timer;		/* Index of timer to start. */
    int     len;		/* Length of timeout interval. */
  {
    int secs, clicks;

    secs = GetTime(&clicks) - Time0;
    pTcb->timers[timer] = CTime(secs, clicks) + len;
    if (pTcb->timers[timer] < NextTimeout)
      {
        Wakeup(TimerPid);
      }
  }




/*
 * StopTimer:
 * Stop specified timer.
 */

StopTimer(pTcb, timer)
    TcpTcbPtr pTcb;		/* Ptr to tcb of current connection. */
    int    timer;		/* Timer to stop. */
  {
    pTcb->timers[timer] = MaxInt;
  }




/*
 * DiscardSegment:
 * Discard the specified segment.
 */

DiscardSegment(packet, errStr)
    PktBuf packet;
    char *errStr;
  {
    FreeBuf(packet);
    if (InternetDebug && errStr)
      {
	printf("Discarding a Tcp segment: %s\n", errStr);
      }
  }
