/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer,  8/83
 *	First created.
 */


/*
 * Contains routines pertaining to the PUP part of the 
 * network server.
 */


#include "net.h"
#include "pup.h"


/*
 * Imported routines
 */

extern char *SafeMalloc();
extern PktBuf DeQueue();
extern Boolean EnQueueSafe();
extern PktBuf DeQueueSafe();
extern SystemCode PupSendPnet();




/*
 * InitPup:
 * Initialize the PUP module.
 */

InitPup()
  {
  }




/*
 * CreatePupConnection:
 * Creates an PUP connection.
 */

SystemCode CreatePupConnection(rqMsg, eventPid)
    CreateInstanceRequest *rqMsg;
    ProcessId eventPid;
  {
    int netId;
    ProcessId pupPid;
    int PupProcess();
    PupParms *ct = (PupParms *) rqMsg->unspecified;

    netId = FindWhichPupSocket(ct->socket);
    if (netId != -1)
	return(BUSY);	/* This connections is already open. */
    pupPid = Create(2, PupProcess, PupStackSize);
    if (pupPid == 0)
      {
	if (InternetDebug)
	  {
	    printf("Couldn't Create Pup connection process.\n");
	  }
	return(NO_MEMORY);
      }
    if (!Ready(pupPid, 0))
      {
	printf("Couldn't Ready PupProcess.\n");
	ExitInternetServer();
      }
    Forward(rqMsg, eventPid, pupPid);
    return(NO_REPLY);
  }




/*
 * CreatePupInstance:
 * Creates an instance of an PUP "connection".  I.e. it registers the client
 * for the socket specified in rqMsg.
 */

SystemCode CreatePupInstance(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    CreateInstanceRequest *rqMsg;
    ProcessId eventPid;
  {
    PupParms *ct = (PupParms *) rqMsg->unspecified;
    int netId;
    CreateInstanceReply *repMsg = (CreateInstanceReply *) rqMsg;

    netId = AllocNetInst(PUPtype, eventPid, GetPid(0, LOCAL_PID), 
    		MaxPupPacketSize, pTcb);
    if (netId == -1)
      {
	return(NO_SERVER_RESOURCES);
      }
    pTcb->netId = netId;
    pTcb->socket = ct->socket;
    repMsg->fileid = netId << 1;
    QueryInstance(repMsg);
    return(OK);
  }




/*
 * ReleasePupInstance:
 * Releases the specified PUP network instance.
 * Note: the connection is not considered released until both read and write
 * instances have been released.
 */

SystemCode ReleasePupInstance(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    PktBuf pkt;
    Message msg;
    IoReply *repMsg = (IoReply *) msg;
    int indx = rqMsg->fileid >> 1;

    if (rqMsg->fileid & 1)
	NetInstTable[indx].inUse &= ~ READ_INST;
    else
	NetInstTable[indx].inUse &= ~ WRITE_INST;
    if (NetInstTable[indx].inUse != 0)
	return(OK);		/* Can't deallocate connection yet. */
    pTcb->instState = False;	/* Signal that the connection may be
				   deallocated after all its packets have
				   been flushed. */

    /*
     * Get rid of queued packets.
     */
    while (!Empty(pTcb->readyQ))
      {
	pkt = DeQueue(&(pTcb->readyQ));
	FreeBuf(pkt);
      }

    if (pTcb->waitFlag)
				/* Notify the reader of the abort. */
      {
	repMsg->replycode = ABORTED;
	Reply(msg, pTcb->rcvId);
	pTcb->waitFlag = False;
      }

    return(OK);
  }




/*
 * ReadPupInstance:
 * Processes an PUP read request from a user client.
 */

SystemCode ReadPupInstance(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    PktBuf packet;

    if (!(rqMsg->fileid & 1))
      {
	return(NOT_READABLE);
      }

    pTcb->waitFlag = True;
    pTcb->rcvId = eventPid;
    pTcb->bufferPtr = rqMsg->bufferptr;
    /*
     * Check if a packet is already available.
     */
    if (!Empty(pTcb->readyQ))
      {
	packet = DeQueue(&(pTcb->readyQ));
	ReplyToRead(OK, pTcb->rcvId, packet,
		pTcb->bufferPtr, packet->length);
	FreeBuf(packet);
	pTcb->waitFlag = False;
      }
    return(NO_REPLY);
  }




/*
 * WritePupInstance:
 * Processes a user client's write request.
 */

SystemCode WritePupInstance(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    struct pbuf pktBuf;
    PktBuf packet = &pktBuf;
    SystemCode status;
    int i;
    char *t, *f;

    if (rqMsg->fileid & 1)
      {
	return(NOT_WRITEABLE);
      }

    packet->dataptr = &(packet->data[MAXPBUFSIZE - rqMsg->bytecount]);
    packet->length = rqMsg->bytecount;

    if (rqMsg->bytecount > IO_MSG_BUFFER)
      {
	status = MoveFrom(eventPid, packet->dataptr, 
		rqMsg->bufferptr, rqMsg->bytecount);
	if (status != OK)
	    return(status);
      }
    else if (rqMsg->bytecount > 0)
      {
	t = packet->dataptr;
	f = (char *) rqMsg->shortbuffer;
	for (i = 0; i < rqMsg->bytecount; i++)
	    *t++ = *f++;
      }

    status = PupSendPnet(packet);
    return(status);
  }




/*
 * QueryPupFile:
 * Returns state information for the specified PUP instance.
 * The only information of interest is the socket for which the
 * instance is registered.
 */

SystemCode QueryPupFile(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    QueryFileRequest *rqMsg;
    ProcessId eventPid;
  {
    PupParms *qt = (PupParms *) rqMsg->unspecified;

    qt->socket = pTcb->socket;
    return(OK);
  }




/*
 * ModifyPupFile:
 * Dummy routine.  There are no parameters to modify in the current PUP
 * implementation form.
 */

SystemCode ModifyPupFile(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    ModifyFileRequest rqMsg;
    ProcessId eventPid;
  {
    return(REQUEST_NOT_SUPPORTED);
  }




/*
 * RcvPup:
 * Process incoming Pup packet destined for an Pup-level client.
 */

RcvPup(pTcb, packet)
    PupTcbPtr pTcb;
    PktBuf packet;
  {
    EnQueue(&(pTcb->readyQ), packet);
    if (pTcb->waitFlag)
      {
	packet = DeQueue(&(pTcb->readyQ));
	ReplyToRead(OK, pTcb->rcvId, packet,
		pTcb->bufferPtr, packet->length);
	pTcb->waitFlag = False;
	FreeBuf(packet);
      }
  }




/*
 * NextPupTimeout:
 * Returns the time of the next timeout for the specified connection.
 * Currently there are no timeouts for Pup connections - return MaxInt.
 */

int NextPupTimeout(pTcb)
    PupTcbPtr pTcb;
  {
    return(MaxInt);
  }




/*
 * PupTimeout:
 * Currently only invalid instance owner timeouts.
 */

PupTimeout(pTcb, rqMsg, eventPid)
    PupTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    rqMsg->fileid = pTcb->netId << 1;
    ReleasePupInstance(pTcb, rqMsg, eventPid);
    rqMsg->fileid += 1;
    ReleasePupInstance(pTcb, rqMsg, eventPid);
  }




/*
 * PupProcess:
 * Pup connection process.  Handles all actions for this connection 
 * after its initial creation.
 */

PupProcess()
  {
    PupTcbRec pupTcb;
    PupTcbPtr pTcb = &pupTcb;
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    IoReply *repMsg = (IoReply *) msg;
    ProcessId eventPid;
    PktBuf packet;

    InitPupConnection(pTcb);
    while (True)
      {
        while ((packet = DeQueueSafe(&(pTcb->readerQueue))) != NULL)
	  {
	    if (pTcb->instState)
		RcvPup(pTcb, packet);
	    else
		FreeBuf(packet);
	  }

	if (!pTcb->instState)
	  {
	    DeallocNetInst(pTcb->netId);
	    Destroy(0);		/* Commit suicide. */
	  }

	pTcb->receiveBlocked = True;
	eventPid = Receive(msg);
	pTcb->receiveBlocked = False;
	    switch (rqMsg->requestcode)
	      {
	        case CREATE_INSTANCE: 
		    repMsg->replycode = 
			CreatePupInstance(pTcb, rqMsg, eventPid);
		    break;
	        case WRITE_INSTANCE:
	        case WRITESHORT_INSTANCE:
		    repMsg->replycode = 
			WritePupInstance(pTcb, rqMsg, eventPid);
		    break;
	        case READ_INSTANCE:
		    repMsg->replycode = 
			ReadPupInstance(pTcb, rqMsg, eventPid);
		    break;
	        case RELEASE_INSTANCE:
		    repMsg->replycode = 
			ReleasePupInstance(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_FILE:
		    repMsg->replycode = 
			QueryPupFile(pTcb, rqMsg, eventPid);
		    break;
	        case MODIFY_FILE:
		    repMsg->replycode = 
			ModifyPupFile(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_INSTANCE:
		    repMsg->replycode = QueryInstance(rqMsg);
		    break;
	        case NetPktRcvd:
		    repMsg->replycode = OK;
		    break;
	        case NetTimeout:
		    PupTimeout(pTcb, rqMsg, eventPid);
		    repMsg->replycode = OK;
		    break;
		default:
		    repMsg->replycode = REQUEST_NOT_SUPPORTED;
		    break;
	      }
	if (repMsg->replycode != NO_REPLY)
	  {
	    eventPid = Reply(msg, eventPid);
	  }
      }
  }




/*
 * RcvAllPup:
 * Process PUP packets coming in off the physical network.
 * Reroutes packets whose final destination is not the local host.
 * Local-destination packets are handed either to
 * higher level protocols or to PUP user clients.
 */

RcvAllPup(packet)
    PktBuf		packet;
  {
    PupPktPtr			pupPkt;
    PupPort dstPort;
    int netId;
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    PupTcbPtr pupPTcb;
    unsigned length;
    int roundLength;
    short pupSum, ourSum;

    /*
     * Check if this is a good PUP packet.
     */
    pupPkt = (PupPktPtr)(packet->dataptr);
    length = pupPkt->pupLength - PupPacketOverhead;
    if ((length < 0) || (length > 1024))
      {
	if (InternetDebug)
	  {
	    printf("Bad length field in a Pup packet.\n");
	  }
    	FreeBuf(packet);
	return;
      }
    roundLength = RoundUp(length);
    pupSum = * (short *) (pupPkt->data + roundLength);
    ourSum = PupCrc(pupPkt, roundLength + PupPacketOverhead - 2);
    if ((pupSum != ourSum) && (pupSum != -1))
      {
	if (InternetDebug)
	  {
	    printf("Pup Crc error at PUP level.\n");
	    Flush(stdout);
	  }
    	FreeBuf(packet);
	return;
      }
    /*
     * Gateway functions:
     */
    if ((pupPkt->dstPort.host != LocalPnetHost) && (pupPkt->dstPort.host != 0))
      {
        if (InternetDebug)
	  {
	    printf("PUP packet received which was not addressed to the ");
	    printf("local PUP host address.\n");
	    Flush(stdout);
	  }
	FreeBuf(packet);
	return;
      }
    /*
     * Interface to clients of PUP:
     */
    UnpackPupPort(&dstPort, &pupPkt->dstPort);
    netId = FindWhichPupSocket(dstPort.socket);
    if (netId == -1)
      {
	FreeBuf(packet);	/* No PUP instance open for this 
				   protocol. */
	return;
      }
    pupPTcb = (PupTcbPtr) NetInstTable[netId].tcbId;
    if (!EnQueueSafe(packet, &(pupPTcb->readerQueue)))
      {
	FreeBuf(packet);
	return;
      }
    if (pupPTcb->receiveBlocked)
      {
	rqMsg->requestcode = NetPktRcvd;
	Send(msg, NetInstTable[netId].pid);
      }
  }
