/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer,  5/83
 */


/*
 * Contains routines pertaining to the IP part of the 
 * network server.
 */


#include "net.h"
#include "iptcp.h"


/*
 * Imported routines
 */

extern char *SafeMalloc();
extern PktBuf DeQueue();
extern Boolean EnQueueSafe();
extern PktBuf DeQueueSafe();
extern SystemCode IpSendPnet();
extern PktBuf DoReassembly();
extern short IpCrc();




/*
 * InitIp:
 * Initialize the IP module.
 */

InitIp()
  {
    LocalIpHost = 0x24 << 24 | 0x28 << 16 | LocalPnetHost;
  }




/*
 * CreateIpConnection:
 * Creates and IP connection.
 */

SystemCode CreateIpConnection(rqMsg, eventPid)
    CreateInstanceRequest *rqMsg;
    ProcessId eventPid;
  {
    int netId;
    ProcessId ipPid;
    int IpProcess();
    IpParms *ct = (IpParms *) rqMsg->unspecified;

    netId = FindWhichInstance(ct->protocol);
    if (netId != -1)
	return(BUSY);	/* This protocol is already open */
    ipPid = Create(2, IpProcess, IpStackSize);
    if (ipPid == 0)
      {
	if (InternetDebug)
	  {
	    printf("Couldn't Create Ip connection process.\n");
	  }
	return(NO_MEMORY);
      }
    if (!Ready(ipPid, 0))
      {
	printf("Couldn't Ready IpProcess.\n");
	ExitInternetServer();
      }
    Forward(rqMsg, eventPid, ipPid);
    return(NO_REPLY);
  }




/*
 * CreateIpInstance:
 * Creates an instance of an IP "connection".  I.e. it enters the client
 * into the protocol tables, etc.
 */

SystemCode CreateIpInstance(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    CreateInstanceRequest *rqMsg;
    ProcessId eventPid;
  {
    IpParms *ct = (IpParms *) rqMsg->unspecified;
    int netId;
    CreateInstanceReply *repMsg = (CreateInstanceReply *) rqMsg;

    netId = AllocNetInst(IPtype, eventPid, GetPid(0, LOCAL_PID), 
    		MaxIpPacketSize, pTcb);
    if (netId == -1)
      {
	return(NO_SERVER_RESOURCES);
      }
    pTcb->netId = netId;
    pTcb->prot = ct->protocol;
    repMsg->fileid = netId << 1;
    QueryInstance(repMsg);
    return(OK);
  }




/*
 * ReleaseIpInstance:
 * Releases the specified IP network instance.
 * Note: the connection is not considered released until both read and write
 * instances have been released.
 */

SystemCode ReleaseIpInstance(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    PktBuf pkt;
    Message msg;
    IoReply *repMsg = (IoReply *) msg;
    int indx = rqMsg->fileid >> 1;

    if (rqMsg->fileid & 1)
	NetInstTable[indx].inUse &= ~ READ_INST;
    else
	NetInstTable[indx].inUse &= ~ WRITE_INST;
    if (NetInstTable[indx].inUse != 0)
	return(OK);		/* Can't deallocate connection yet. */
    pTcb->instState = False;	/* Signal that the connection may be
				   deallocated after all its packets have
				   been flushed. */

    /*
     * Get rid of queued packets.
     */
    while (!Empty(pTcb->readyQ))
      {
	pkt = DeQueue(&(pTcb->readyQ));
	FreeBuf(pkt);
      }

    if (pTcb->waitFlag)
				/* Notify the reader of the abort. */
      {
	repMsg->replycode = ABORTED;
	Reply(msg, pTcb->rcvId);
	pTcb->waitFlag = False;
      }

    return(OK);
  }




/*
 * ReadIpInstance:
 * Processes an IP read request from a user client.
 */

SystemCode ReadIpInstance(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    PktBuf packet;

    if (!(rqMsg->fileid & 1))
      {
	return(NOT_READABLE);
      }

    pTcb->waitFlag = True;
    pTcb->rcvId = eventPid;
    pTcb->bufferPtr = rqMsg->bufferptr;
    /*
     * Check if a packet is already available.
     */
    if (!Empty(pTcb->readyQ))
      {
	packet = DeQueue(&(pTcb->readyQ));
	ReplyToRead(OK, pTcb->rcvId, packet,
		pTcb->bufferPtr, packet->length);
	FreeBuf(packet);
	pTcb->waitFlag = False;
      }
    return(NO_REPLY);
  }




/*
 * WriteIpInstance:
 * Processes a user client's write request.
 */

SystemCode WriteIpInstance(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    struct pbuf pktBuf;
    PktBuf packet = &pktBuf;
    SystemCode status;
    int i;
    char *t, *f;

    if (rqMsg->fileid & 1)
      {
	return(NOT_WRITEABLE);
      }

    packet->dataptr = &(packet->data[MAXPBUFSIZE - rqMsg->bytecount]);
    packet->length = rqMsg->bytecount;

    if (rqMsg->bytecount > IO_MSG_BUFFER)
      {
	status = MoveFrom(eventPid, packet->dataptr, 
		rqMsg->bufferptr, rqMsg->bytecount);
	if (status != OK)
	    return(status);
      }
    else if (rqMsg->bytecount > 0)
      {
	t = packet->dataptr;
	f = (char *) rqMsg->shortbuffer;
	for (i = 0; i < rqMsg->bytecount; i++)
	    *t++ = *f++;
      }

    status = IpSendPnet(packet);
    return(status);
  }




/*
 * QueryIpFile:
 * Returns state information for the specified IP instance.
 * The only information of interest is the protocol with which the
 * instance is associated.
 */

SystemCode QueryIpFile(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    QueryFileRequest *rqMsg;
    ProcessId eventPid;
  {
    IpParms *qt = (IpParms *) rqMsg->unspecified;

    qt->protocol = pTcb->prot;
    return(OK);
  }




/*
 * ModifyIpFile:
 * Dummy routine.  There are no parameters to modify in the current IP
 * implementation form.
 */

SystemCode ModifyIpFile(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    ModifyFileRequest rqMsg;
    ProcessId eventPid;
  {
    return(REQUEST_NOT_SUPPORTED);
  }




/*
 * RcvIp:
 * Process incoming Ip packet destined for an Ip-level client.
 */

RcvIp(pTcb, packet)
    IpTcbPtr pTcb;
    PktBuf packet;
  {
    EnQueue(&(pTcb->readyQ), packet);
    if (pTcb->waitFlag)
      {
	packet = DeQueue(&(pTcb->readyQ));
	ReplyToRead(OK, pTcb->rcvId, packet,
		pTcb->bufferPtr, packet->length);
	pTcb->waitFlag = False;
	FreeBuf(packet);
      }
  }




/*
 * NextIpTimeout:
 * Returns the time of the next timeout for the specified connection.
 * Currently there are no timeouts for Ip connections - return MaxInt.
 */

int NextIpTimeout(pTcb)
    IpTcbPtr pTcb;
  {
    return(MaxInt);
  }




/*
 * IpTimeout:
 * Currently only invalid instance owner timeouts.
 */

IpTimeout(pTcb, rqMsg, eventPid)
    IpTcbPtr pTcb;
    IoRequest *rqMsg;
    ProcessId eventPid;
  {
    rqMsg->fileid = pTcb->netId << 1;
    ReleaseIpInstance(pTcb, rqMsg, eventPid);
    rqMsg->fileid += 1;
    ReleaseIpInstance(pTcb, rqMsg, eventPid);
  }




/*
 * IpProcess:
 * Ip connection process.  Handles all actions for this connection 
 * after its initial creation.
 */

IpProcess()
  {
    IpTcbRec ipTcb;
    IpTcbPtr pTcb = &ipTcb;
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    IoReply *repMsg = (IoReply *) msg;
    ProcessId eventPid;
    PktBuf packet;

    InitIpConnection(pTcb);
    while (True)
      {
        while ((packet = DeQueueSafe(&(pTcb->readerQueue))) != NULL)
	  {
	    if (pTcb->instState)
		RcvIp(pTcb, packet);
	    else
		FreeBuf(packet);
	  }

	if (!pTcb->instState)
	  {
	    DeallocNetInst(pTcb->netId);
	    Destroy(0);		/* Commit suicide. */
	  }

	pTcb->receiveBlocked = True;
	eventPid = Receive(msg);
	pTcb->receiveBlocked = False;
	    switch (rqMsg->requestcode)
	      {
	        case CREATE_INSTANCE: 
		    repMsg->replycode = 
			CreateIpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case WRITE_INSTANCE:
	        case WRITESHORT_INSTANCE:
		    repMsg->replycode = 
			WriteIpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case READ_INSTANCE:
		    repMsg->replycode = 
			ReadIpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case RELEASE_INSTANCE:
		    repMsg->replycode = 
			ReleaseIpInstance(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_FILE:
		    repMsg->replycode = 
			QueryIpFile(pTcb, rqMsg, eventPid);
		    break;
	        case MODIFY_FILE:
		    repMsg->replycode = 
			ModifyIpFile(pTcb, rqMsg, eventPid);
		    break;
	        case QUERY_INSTANCE:
		    repMsg->replycode = QueryInstance(rqMsg);
		    break;
	        case NetPktRcvd:
		    repMsg->replycode = OK;
		    break;
	        case NetTimeout:
		    IpTimeout(pTcb, rqMsg, eventPid);
		    repMsg->replycode = OK;
		    break;
		default:
		    repMsg->replycode = REQUEST_NOT_SUPPORTED;
		    break;
	      }
	if (repMsg->replycode != NO_REPLY)
	  {
	    eventPid = Reply(msg, eventPid);
	  }
      }
  }




/*
 * RcvAllIp:
 * Process IP packets coming in off the physical network.
 * Reroutes packets whose final destination is not the local host.
 * Reassembles fragmented packets and then hands them either to
 * higher level protocols or to IP user clients.
 */

RcvAllIp(packet)
    PktBuf		packet;
  {
    IpHdrPtr			iphdr;
    int				ihl;
    IpHdrPtr			iph;
    IpAddr ipsrc, ipdst;
    TcpPktPtr p;
    TcpTcbPtr tcpPTcb;
    int netId;
    Message msg;
    IoRequest *rqMsg = (IoRequest *) msg;
    IpTcbPtr ipPTcb;

    /*
     * Check if this is a good IP packet.
     */
    iphdr = (IpHdrPtr)(packet->dataptr);
    if ( IpCrc(packet) ||
         ((iphdr ->version) != CurrentInternetVersion))
      {
	if (InternetDebug)
	  {
	    printf("IpCrc or Version error at IP level.\n");
	  }
    	FreeBuf(packet);
	return;
      }
    /*
     * Gateway functions:
     */
    if (iphdr->dstAdr != LocalIpHost)
      {
        if (InternetDebug)
	  {
	    printf("IP packet received which was not addressed to the ");
	    printf("local IP host address.\n");
	  }
	FreeBuf(packet);
	return;
      }
    /*
     * Reassembly:
     */
    if (FragmentedPacket(iphdr))
      {
        if (InternetDebug)
	  {
	    printf("Fragmented Packet!!!\n"); 
	  }
	packet = DoReassembly(packet);
	if (packet == NULL)
	    return;
	iphdr = (IpHdrPtr)(packet->dataptr);
				/* Need to reset iphdr since packet will
				   have a different value. */
      }
    /*
     * Interface to clients of IP:
     */
    switch (iphdr->prot)
      {
	case TCPprot:
            ihl = (iphdr->ihl) << 2;
	    /*
	     * Strip off the IP header.
	     */
	    packet->dataptr += ihl;
            packet->length = iphdr->tl - ihl;
	    /*
	     * Store source and destination addresses in unspecified fields
	     * of the packet buffer so that Tcp can get at them.
	     */
	    packet->unspecified[0] = (unsigned) iphdr->srcAdr;
	    packet->unspecified[1] = (unsigned) iphdr->dstAdr;
	    /*
	     * Queue the packet in the packet queue of the appropriate Tcp
	     * connection process and notify the process if necessary.
	     */
    	    p = TcpPkt(packet);
	    netId = FindWhichConnection(False, p->hdr.destinationPort,
			p->hdr.sourcePort, iphdr->srcAdr);
	    if (netId == -1)
	      {
	        if (InternetDebug)
		  {
		    printf("Couldn't find a connection for an incoming ");
		    printf("Tcp packet.\n");
		  }
		ReplyToConnectionlessPacket(packet, iphdr->srcAdr);
	        FreeBuf(packet);
		return;
	      }
	    tcpPTcb = (TcpTcbPtr) NetInstTable[netId].tcbId;
	    if (!EnQueueSafe(packet, &(tcpPTcb->readerQueue)))
	      {
	        if (InternetDebug)
		  {
		    printf("EnQueueSafe failed!\n");
		  }
		FreeBuf(packet);
		return;
	      }
	    if (tcpPTcb->receiveBlocked)
	      {
		rqMsg->requestcode = NetPktRcvd;
		Send(msg, NetInstTable[netId].pid);
	      }
	    break;
	default:		/* IP user client */
	    if (InternetDebug)
	      {
		printf("IP packet for an IP user client.\n");
	      }
	    netId = FindWhichInstance(iphdr->prot);
	    if (netId == -1)
	      {
		if (InternetDebug)
		  {
		    printf("Couldn't find a connection for an IP packet.\n");
		  }
		FreeBuf(packet);	/* No IP instance open for this 
					   protocol. */
		return;
	      }
	    ipPTcb = (IpTcbPtr) NetInstTable[netId].tcbId;
	    if (!EnQueueSafe(packet, &(ipPTcb->readerQueue)))
	      {
		FreeBuf(packet);
		return;
	      }
	    if (ipPTcb->receiveBlocked)
	      {
		rqMsg->requestcode = NetPktRcvd;
		Send(msg, NetInstTable[netId].pid);
	      }
	    break;
      }
    return;
  }
