/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, 5/83
 * Marvin Theimer, 8/83
 *	Changed over to use a table for determining to whom packets are
 *	are handed off to.
 */


/*
 * Contains routines pertaining to the physical network part of the 
 * network server.
 * The packet receiving routine actually constitutes
 * a separate process which queues a packet for the main network
 * server process.
 */


#include "net.h"
#include "iptcp.h"


/*
 * Routing constants
 */
#define ClassANet(n)	(n & 0x80000000)
#define Net(n)		(n & 0xff000000)
#define Subnet(n)	(n & 0x00ff0000)
#define SuNet		0x24000000
#define LocalNet	0x00280000
#define MjhSumexSubgateway	077
#define StanfordGateway 076

/*
 * Various other constants
 */

#define AllocDelayTime 50	/* Number of clock ticks to delay before
				   checking to see if a buffer has become
				   available. */


/*
 * Variable declarations
 */

File *FileId;			/* File id for the ethernet device. */

int NumPktDiscards = 0;		/* Number of pkts discarded by reader */


/*
 * Imported routines and tables
 */

extern PktBuf AllocBuf();
extern File *_Open();

extern struct PnetBlock PnetTable[];




/*
 * QueryPnet:
 * Queries the ethernet device.
 */

SystemCode QueryPnet(reqMsg)
    IoRequest *reqMsg;
  {
    reqMsg->requestcode = QUERY_FILE;
    reqMsg->fileid = FileId->fileid;
    Send(reqMsg, FileId->fileserver);
    FileId->lastexception = ((IoReply *)reqMsg)->replycode;
    return(FileId->lastexception);
  }




/*
 * InitPnet:
 * Initialization routine for physical network part of network server.
 */

InitPnet()
  {
    Message msg;
    CreateEnetRequest *req = (CreateEnetRequest *) msg;
    QueryEnetReply *rep = (QueryEnetReply *) msg;
    SystemCode error;

    /*
     * Open an instance to the ethernet device.
     */
    req->requestcode = CREATE_INSTANCE;
    req->filename = "Ethernet";
    req->filenamelen = 9;
    req->filemode = FCREATE + FBLOCK_MODE;
    req->type = ETHERNET;
    req->receivermask = ENET_DEFAULT_MASK;
    FileId = _Open(msg, FCREATE + FBLOCK_MODE, 
			GetPid(DEVICE_SERVER, LOCAL_PID), &error);
    if ((error != OK) || (FileId == NULL))
      {
	printf("ERROR - Failed to open Ethernet instance\n");
	ExitInternetServer();
      }

    error = QueryPnet(msg);
    if (error != OK)
      {
	printf("ERROR - Query of Ethernet device for host address failed.\n");
	ExitInternetServer();
      }
    LocalPnetHost = rep->EthernetHostNumber;
				/* Make physical network address of the
				   local host available. */
  }




/*
 * WritePnet:
 * Writes a packet out on the physical network with specified dest and type.
 */

SystemCode WritePnet(pkt, dst, eType)
    PktBuf pkt;
    PnetAddr dst;
    short unsigned eType;
  {
    EnetBlock *buffer;
    int size;

    /*
     * Add room for the ethernet header. 
     */
    pkt->dataptr -= 4;
    pkt->length += 4;
    if (InternetDebug)
	PacketCheck(pkt);	/* Check if the packet has
				   overwritten its boundaries. */
    /*
     * Fill in the header and send the packet off.
     */
    buffer = (EnetBlock *) (pkt->dataptr);
    buffer->SrcHost = LocalPnetHost;
    buffer->DestHost = dst;
    buffer->EtherType = eType;
    size = Write(FileId, buffer, (pkt->length));
    if (InternetDebug && 
	    ((size != pkt->length) || (FileId->lastexception != OK)))
      {
	printf(
	    "ERROR:           Write packet to ethernet device failed: %s!\n",
	    ErrorString(FileId->lastexception));
      }

    return(FileId->lastexception);
  }




/*
 * RcvPnet:
 * Packet queueing process.
 * Alternately reads a packet from the physical layer and hands
 * it to the appropriate client.  If no packets are available then a
 * spare packet is used to read packets from the physical layer so that
 * they don't clog up the device buffer.  Contents of the spare packet are
 * thrown away until regular packet buffers become available again.
 */

RcvPnet()
{
    EnetBlock *buffer;
    int size;
    PktBuf packet, sparePacket;
    int getNewPkt = 1;
    int NextReassemblyTimeout = ReassemblyTimeoutLen;
    int i;

    sparePacket = AllocBuf();	
				/* There will always be a packet available
				   at this point because it is startup
				   time. */
    while (1)
      {
	if (CurrentTime <= NextReassemblyTimeout)
	  {
	    IpReassemblyTimeout();	/* Check for timed-out IP 
					   fragments. */
	    NextReassemblyTimeout += ReassemblyTimeoutLen;
	  }

        if (getNewPkt)
	  {
	    packet = AllocBuf();
	    if (packet == NULL)
	        packet = sparePacket;
	    getNewPkt = 0;
	  }
	
        buffer = (EnetBlock *) (packet->data);
        size = Read(FileId, buffer, MAXPBUFSIZE);
        if (size > 0)
	  {
	    if (packet == sparePacket)
	      {
	        sparePacket = AllocBuf();
		if (sparePacket == NULL)
		  {
		    NumPktDiscards++;
		    if (InternetDebug)
		      {
			printf("*** Tossing a packet away ***\n");
		      }
		    sparePacket = packet;
				/* Make sparePacket point back at the
				   packet being used as a spare. */
	            continue;
		  }
	      }
	    for (i = 0; i < NumPnetProtocols; i++)
	      {
		if (PnetTable[i].active &&
			(buffer->EtherType == PnetTable[i].prot))
		  {
		    getNewPkt = 1;
		    /*
		     * Strip off the ethernet header.
		     */
		    packet->length = size - 4;
		    packet->dataptr = (char *) (buffer->data);
		    PnetTable[i].rcv(packet);
		    break;
		  }
	      }
	  }
        else if (InternetDebug)
	  {
	    printf("ERROR in Ethernet Read: %d\n", FileId->lastexception);
	  }
      }
}



/*
 * Route:
 * This routine performs the network/subnetwork routing to find a path
 * to the IP destination if one exists. if none exists, returns zero (0)
 */

PnetAddr Route (dest)
    IpAddr	dest;
  {
    if ( Net(dest) == SuNet )
        if ( Subnet(dest) == LocalNet )
	    return ( dest & 0xff );
				/* Use the local dest directly */
	else
	    return ( MjhSumexSubgateway );
				/* Forward to the Sumex gateway */
    else
    	return ( StanfordGateway );
				/* if it's not on SU_NET get it to the 
				   ARPA gateway */
  }




/*
 * PrintPnetStatus:
 * Prints various statistics about the physical network connection.
 */

PrintPnetStatus()
  {
    Message msg;
    QueryEnetReply *repMsg = (QueryEnetReply *) msg;
    SystemCode status;
    
    status = QueryPnet(msg);
    if (status != OK)
      {
	printf("ERROR from QueryPnet: %s.\n",
		ErrorString(status));
      }
    else
      {
	printf("EthernetHostNumber: 0%o\n", repMsg->EthernetHostNumber);
	printf("Number of valid packets: %d\n", repMsg->NumValidPackets);
	printf("Number of collisions: %d\n", repMsg->NumCollisions);
	printf("Number of overflows: %d\n", repMsg->NumOverflows);
	printf("Number of CRC errors: %d\n", repMsg->NumCRCErrors);
	printf("Number of sync errors: %d\n", repMsg->NumSyncErrors);
	printf("Number of timeouts: %d\n", repMsg->NumTimeOuts);
      }
    printf("\n");
    printf("Number of packets discarded by internet reader process: %d\n",
    		NumPktDiscards);
  }
