/*
 * Standard exception handler routine.
 *
 * 28 Feb 83 (mmt) - Abstracted from exceptions.c file so that other exception
 *			handlers can include it.
 *  5 March 83 (WIN) - Fixed bug in MoveFrom: byte count was too large by 1
 * 19 May 83 (mmt) - StandardExceptionHandler made to return error pc value.
 * 12 Aug 83 (TPM) - Added bus error type printing and stack overflow detect.
 */


#include <Venviron.h>
#include <Vprocess.h>
#include <Vexceptions.h>
#include <Vio.h>


#define MaxLookAhead 10
#define TSTART 0x10000

short *StandardExceptionHandler(req, pid, fout)
    register ExceptionRequest *req;
    ProcessId pid;
    File *fout;			/* Print out messages on this file */

    /* Standard exception handling routine.  Prints out some information
     * about the process incurring the exception and returns the pc at
     * which the exception occurred.
     */
  {
    register short *i;
    short codeBuffer[MaxLookAhead];
    register int instCnt, index = 0;
    Processor_state state;
    unsigned stackSize;

    /* Print information about the state of the process */
    Resynch( fout );
    fprintf(fout,"\r\n");

    i = (short *) req->errpc;

    switch( req->type )
      {
      case ZERODIV:
	fprintf(fout, " Divide by zero" );
	break;

      case CHKINST:
	fprintf(fout, " Check instruction" );
	break;

      case TRAPVINST:
	fprintf(fout, " Overflow" );
	break;

      case TRACETRAP:
	fprintf(fout, " Trace bit on" );
	break;

      case PRIVVIOL:
	fprintf(fout, " Privileged instruction" );
	if ((int)i >= TSTART) /* not in kernel space */
	  {
	    /* Load instruction */
	    req->instruction = 0;
	    MoveFrom( pid,(short *)&(req->instruction)+1, i, 2);
	  }
	break;

      case ILLINST:
      case EMU1010:
      case EMU1111:
	fprintf(fout," Illegal instruction" );
	if ((int)i >= TSTART) /* not in kernel space */
	  {
	    /* Load instruction */
	    req->instruction = 0;
	    MoveFrom( pid,(short *)&(req->instruction)+1, i, 2);
	  }
	break;

      case ADDRERROR:
      case BUSERROR:
	if ( req->code & DATA_BIT )
	  {
	    /* Error was not on the instruction fetch */
	    /* find actual address of erroneous instruction */
	    if ((int)i >= TSTART) /* not in kernel space */
	      {
		if ((i - MaxLookAhead*2-(TSTART)) < 0)
		  instCnt = (int)(i - (TSTART))>>1;
		else
		  instCnt = MaxLookAhead;
			/* get code */
		MoveFrom(pid,codeBuffer,i-instCnt, (short *)0 + instCnt);
			/* find instruction */
		for( index=instCnt; (index > 0) &&
			(codeBuffer[index]!= req->instruction);
			index--)
		    ; 	/* empty */

		if (codeBuffer[index] != req->instruction)
		  i = (short *) req->errpc;
		else
		  i -= MaxLookAhead - index;
	      }
	  }

	if( req->type == BUSERROR )
	  {
	    switch (req->buserrortype)
	      {
	      case OUT_OF_RANGE:
	        fprintf(fout, " Address out of range");
	        break;
	      case SPURIOUS:
	        fprintf(fout, " Spurious bus error");
		break;
	      case SYSTEM_SPACE:
		fprintf(fout, " Access to system space");
		break;
	      case PROTECTION:
		fprintf(fout, " Memory protection violation");
		break;
	      case PAGE_INVALID:
	        fprintf(fout, " Access to invalid page");
	        break;
	      case SEG_INVALID:
	        fprintf(fout, " Access to invalid segment");
	        break;
	      case MB_TIMEOUT:
	        fprintf(fout, " Multibus access timeout");
		break;
	      case PARITY:
		fprintf(fout, " Memory parity error");
		break;
	      }
	  }
	else
	  {
	    fprintf(fout, " Odd address access" );
	  }

	if (req->code & DATA_BIT)
	  {
	    if (req->code & RW_BIT)
	        fprintf(fout," on read from");
	    else
        	fprintf(fout," on write to");
	  }
	else
	  {
	    fprintf(fout, " on instruction fetch from");
	    req->instruction = 0;
	  }
	fprintf(fout, " address %x", req->accaddr );
	break;

      case TRAP4:
	fprintf(fout, " Abort trap");
	break;

      case TRAP0:	case TRAP1:	case TRAP2:
      case TRAP3:	case TRAP5:	case TRAP6:
      case TRAP7:	case TRAP8:	case TRAP9:
      case TRAP10:	case TRAP11:	case TRAP12:
      case TRAP13:	case TRAP14:	case TRAP15:
	fprintf(fout, " Trap instruction");
	break;

      case SPURINT:
	fprintf(fout, " Spurious interrupt ");
	break;

      case INT1:	case INT2:	case INT3:
      case INT4:	case INT5:	case INT6:
      case INT7:
	fprintf(fout, " Unexpected interrupt");
	break;

      default: 
	fprintf(fout, " Unknown exception" );
	break;
      }

    fprintf(fout, " in process %x\r\n",pid );
    if( req->instruction != 0 ) 
        fprintf(fout, " Instruction  " );
    fprintf(fout, "  Program counter    Status register\r\n" );
    if( req->instruction != 0 )
        fprintf(fout, "     %4x     ", req->instruction );
    fprintf(fout, "       %6x              %2x\r\n", i, req->status );

    ReadProcessState(pid, &state);
    if (req->status & SUPERVISOR_STATE) 
	fprintf(fout, " In kernel, called from %x\r\n", state.pc);

    /* Check for stack overflow */
    if (state.perProcess != 0)
      {
	stackSize = 0;
	MoveFrom(pid, &stackSize,
	      &(((PerProcessArea *) state.perProcess)->stackSize),
	      sizeof(stackSize));
	if ((unsigned) state.USER_STACK_POINTER <= (unsigned) state.perProcess ||
	      (unsigned) state.USER_STACK_POINTER > (unsigned) state.perProcess + stackSize)
	    fprintf(fout, " Probable stack overflow\r\n");
      }

    Flush( fout );
    return(i);
  }
