/*
 * Vkernel exception server
 * Copyright (c) 1982  Myles Cagney, David Cheriton, Tim Mann, Marvin Theimer
 *
 * ****NOTE****
 * Currently the exception server assumes that its stdout points directly to
 * the kernel device so that it can handle exceptions which occur in output
 * fileservers to which stderr might be pointing.
 *
 *
 * 29 Nov 82 (TPM) - Enlarged stack and fixed some problems.
 *			Still needs work!
 *  6 Dec 82 (TPM) - Now does a Resynch( stderr ) before writing.
 * 28 Feb 83 (MMT) - Revised to incorporate registration facilities.
 *			Forwards to registered exception handlers or calls
 *			the StandardExceptionHandler routine.
 * 21 Apr 83 (WIN) - Checked if printf'ing would cause deadlock.
 * 26 Aug 83 (TPM) - Expanded check to use SameTeam instead of equality test.
 */

#include <Venviron.h>
#include <Vprocess.h>
#include <Vexceptions.h>
#include <Vio.h>


InitExceptionServer()
  {
    extern int ExceptionServer();
    register ProcessId pid;

    if ( (pid = GetPid(EXCEPTION_SERVER, LOCAL_PID)) == 0 )
      {
        if ( pid = Create(3, ExceptionServer, 3000) ) 
          {
	    SetPid( EXCEPTION_SERVER, pid, LOCAL_PID );
	    Ready( pid, 0 );
          }
      }
    return( pid );
  }

ExceptionServer()

    /* Exception server: 
     *
     */
  {
    Message msg;
    register Process_id pid;
    register SystemCode reply;
    register ExceptionRequest *req = (ExceptionRequest *) msg;

    InitHandlers();
    while( 1 )
      {
	pid = Receive( req );
	switch( req->requestcode )
	  {
	    case EXCEPTION_REQUEST:
		reply = HandleException( req, pid );
		break;
	    case REGISTER_HANDLER:
		reply = RegisterHandler( req );
		break;
	    default: 
		reply = ILLEGAL_REQUEST;
		break;
	  }
	if( reply == NO_REPLY ) continue;
	req->requestcode = reply; /* Set the reply code */
	Reply( req, pid );
      }
  }


/*
 * Code for registration of exception handlers.
 */

#define MaxHandlers 10

typedef struct HandlerRecType
  {
    ProcessId handlerPid;
    ProcessId regPid;
    struct HandlerRecType *next;
  }
    HandlerRec;

HandlerRec HandlerRecsArray[MaxHandlers];
				/* Rec. 0 is used as header for the handler
				   list. */
HandlerRec *HandlersHdr = HandlerRecsArray;
HandlerRec *HandlerFreeList;


InitHandlers()
  {
    int i;

    for (i = 1; i < MaxHandlers; i++)
	HandlerRecsArray[i].next = &HandlerRecsArray[i-1];
    HandlerRecsArray[1].next = NULL;

    HandlersHdr->next = NULL;
    HandlerFreeList = &HandlerRecsArray[MaxHandlers - 1];
  }


/*
 * Ancestor:
 * Returns true if pid1 is the ancestor of pid2.
 * If pid1 == pid2 then pid1 is also considered to be an ancestor of pid2.
 */

Ancestor(pid1, pid2)
   register ProcessId pid1, pid2;
  {
    if( pid1 == 0 ) return( 0 );

    while( pid2 != 0 )
      {
	if( pid1 == pid2 ) return( 1 );
	
	pid2 = Creator(pid2);
      }
    return( 0 );
  }


HandlerRec *GetFirstHandlerRec()
  {
    return(HandlersHdr->next);
  }


HandlerRec *GetNextHandlerRec(curPtr)
    HandlerRec *curPtr;
  {
    return(curPtr->next);
  }


EnterHandlerRec(recPtr)
    HandlerRec *recPtr;
  {
    HandlerRec *curPtr, *prevPtr, *newPtr;

    newPtr = HandlerFreeList;
    if (newPtr == NULL)
	return(NO_SERVER_RESOURCES);
    HandlerFreeList = HandlerFreeList->next;

    *newPtr = *recPtr;
    for (curPtr = HandlersHdr->next, prevPtr = HandlersHdr; curPtr != NULL; 
	    prevPtr = curPtr, curPtr = curPtr->next)
      {
	if (Ancestor(curPtr->regPid, newPtr->regPid))
	    break;
      }
    newPtr->next = curPtr;
    prevPtr->next = newPtr;
    return(OK);
  }


RemoveHandlerRec(hPid, rPid)
    ProcessId hPid, rPid;
  {
    HandlerRec *curPtr, *prevPtr;

    for (curPtr = HandlersHdr->next, prevPtr = HandlersHdr; curPtr != NULL; 
	    prevPtr = curPtr, curPtr = curPtr->next)
	if ((curPtr->handlerPid == hPid) && 
		(!rPid || (rPid == curPtr->regPid)))
	  {
	    prevPtr->next = curPtr->next;
	    curPtr->next = HandlerFreeList;
	    HandlerFreeList = curPtr;
	    break;
	  }
  }


int RegisterHandler( req )

  register RegHandlerRequest *req;

  /* Register the requesting process as an exception handler or
     remove the specified handler. */
  {
    int status;
    HandlerRec hRec;

    if (req->regFlag)
      {
	hRec.handlerPid = req->handlerPid;
	hRec.regPid = req->regPid;
	status = EnterHandlerRec(&hRec);
      }
    else
      {
	RemoveHandlerRec(req->handlerPid, req->regPid);
	status = OK;
      }
    return(status);
  }


HandleException( req, pid )

  register ExceptionRequest *req;
  ProcessId pid;

  /* Handle the exception incurred by the requesting process by either
   * forwarding it to its registered handler or else calling the standard
   * exception handling routine.  Registered handlers which are dead are
   * removed from the registry.
   */
  {
    int reply;
    HandlerRec *curPtr = NULL;
    ProcessId forwardPid;
    File *fout;

    /*
     * We must worry about the case when an exception occurs in the stderr
     * fileserver.  In this case assume that stdout of the exception server
     * goes directly to the kernel device and use that as the output file.
     * In fact, we go to the kernel device any time the exception is in
     * a process on the same team as the stderr fileserver, in case it was
     * in a helper process.
     */
    if (SameTeam(stderr->fileserver, pid))
	fout = stdout;
    else
	fout = stderr;

    for (curPtr = GetFirstHandlerRec(); curPtr != NULL; 
	    curPtr = GetNextHandlerRec(curPtr))
	if (Ancestor(curPtr->regPid, pid))
	    break;
    if ((curPtr == NULL) || (curPtr->handlerPid == pid))
      {
	StandardExceptionHandler(req, pid, fout);
        DestroyProcess( pid );
      }
    else
      {
	forwardPid = Forward(req, pid, curPtr->handlerPid);
	if (!forwardPid)
	  {			/* Handler is dead!  Do default handling. */
	    StandardExceptionHandler(req, pid, fout);
	    DestroyProcess(pid);
	    RemoveHandlerRec(curPtr->handlerPid, 0);
				/* Get rid of the rec. for the dead 
				   handler. */
	  }
      }
    return(NO_REPLY);
  }
