/********************************************************/
/*							*/
/*	  Virtual Graphics Terminal Server		*/
/*							*/
/*		(C) COPYRIGHT 1982			*/
/*		BOARD OF TRUSTEES			*/
/*	LELAND STANFORD JUNIOR UNIVERSITY		*/
/*	  STANFORD, CA. 94305, U. S. A.			*/
/*							*/
/********************************************************/

/*------------------------------------------------------------------------

                    STRUCTURED DISPLAY FILE MANAGER

 * This file (stubs.c) is linked with a client running on a Unix system
 * and sends the commands over a byte-stream connection.
 *
 * Bill Nowicki September 1982
 * David Kaelbling March 1983	- Modified to handle splines.
 * Per Bothner March 1983 -Modified to handle raster objects
 * David Kaelbling May 1983 - Fixed the fixes to the splines.
 */


# ifdef VAX
# include <stdio.h>
# else VAX
# include <Vio.h>
# endif VAX

# include <Vgts.h>
# include "remote.h"
# include <chars.h>
# include "splines.h"

short CreateSDF()
  {
    /*
     * Returns the number of an available SDF. by finding an unused
     * entry in the SdfTable.
     */
    SendCode(VgtsCreateSDF);
    return(ReturnValue());
  }



DeleteSDF(i)
  short i;
    {
      /*
       * Remove all symbol definitions from the indicated SDF
       */
      SendCode(VgtsDeleteSDF);
      SendShort(i);
      return(ReturnValue());
    }

/* DefineSymbol(sdf,item,text_ptr) is called to enter item
in the structured display file.  A display record is created containing
this name and a pointer to some text information describing the symbol.
This entry is also added to the hash tables to enable future calls to
find the proper items.  A list of items should follow this definition
item, and all will be linked together using the rightSib field of the
display records.  The list is terminated by a call to EndSymbol which
also fills in the other fields of the symbol definition entry.  */

DefineSymbol(sdf, item, textPtr)
  short sdf;
  short item;
  char *textPtr;
{
    SendCode(VgtsDefineSymbol);
    SendShort(sdf);
    SendShort(item);
    SendString(textPtr);
    return(ReturnValue());
}

/*
 * EndSymbol is called at the end of a list of items defining a
 * symbol.  It closes the named symbol to further additions and fills in
 * some needed information about the symbol that might not have been
 * available at the initial call to DefineSymbol. 
 */

EndSymbol(sdf, item, typeData)
  short sdf;
  short item;
  unsigned char   typeData;
  {
    SendCode(VgtsEndSymbol);
    SendShort(sdf);
    SendShort(item);
    SendShort(typeData);
    return(ReturnValue());
  }

/*
 *  AddItem is the main routine for adding a item to the list of items
 * defining a symbol.  The routine is called with all the necessary
 * information for the display record, and a display record is created
 * with these characteristics.  The global name is used to enter this
 * location in the hash table for future reference.  The global variable
 * "CurrentItem" is used to remember the last item added to the current
 * symbol definition list, and is updated by this routine. 
 */

AddItem(sdf, item, xmin, xmax, ymin, ymax, typedata, type, string)
  short sdf;
  short item,
        xmin,
        xmax,
        ymin,
        ymax;
  unsigned char   type;
  unsigned char   typedata;
  char	*string;
{
    SendCode(VgtsAddItem);
    SendShort(sdf);
    SendShort(item);
    SendShort(xmin);
    SendShort(xmax);
    SendShort(ymin);
    SendShort(ymax);
    SendShort(typedata);
    SendShort(type);
    if (type == SDF_SPLINE)
	SendSpline( (SPLINE *) string,
		   sizeof(SPLINE) +
		    (((SPLINE *)string)->numvert - 1) * sizeof(POINT));
    else if (type == SDF_RASTER)
	SendBlock((short *) string,
	    ((xmax - xmin + (1 + 15)) >> 4) * (ymax - ymin + 1) * sizeof(short));
    else
        SendString(string);
    return(item);
}

/*
 * AddCall is called to insert an instance of one symbol in the
 * definition list of another symbol.  The hashed directory is consulted
 * and a dummy entry is made for this symbol if the definition is not
 * already there. Any future attempts to define this symbol will use this
 * dummy item as the start of the symbol definition list, maintaining
 * consistency with any already existing calls to the symbol.  
 */

AddCall(sdf, item, xmin, ymin, symbolName)
  short sdf;
  short item,
        xmin,
        ymin,
        symbolName;
{
    SendCode(VgtsAddCall);
    SendShort(sdf);
    SendShort(item);
    SendShort(xmin);
    SendShort(ymin);
    SendShort(symbolName);
    return(item);
}

/* DeleteItem: will delete item from the currently open symbol
 * definition, if possible.  The display record will be returned to free
 * storage, and item will be stricken from the hash table. Symbol
 * calls may be deleted just like any other item, but an entire symbol
 * definition must be deleted by using the DeleteSymbol routine.
 */

DeleteItem(sdf, item)
  short sdf;
  short item;
{
    SendCode(VgtsDeleteItem);
    SendShort(sdf);
    SendShort(item);
    return(item);
}

/*
 * InquireItem -- return the values in the SDF record indicated.
 * returns non-zero if the item was found.
 */

InquireItem(sdf, item, xmin, xmax, ymin, ymax, typedata, type, string)
    short sdf;
    short item, *xmin, *xmax, *ymin, *ymax;
    unsigned char   *type;
    unsigned char   *typedata;
    char	*string;
  {
    short temp;

    SendCode(VgtsInquireItem);
    SendShort(sdf);
    SendShort(item);

    temp = ReturnValue();
    if (xmin) *xmin = temp;
    temp = ReturnValue();
    if (xmax) *xmax = temp;
    temp = ReturnValue();
    if (ymin) *ymin = temp;
    temp = ReturnValue();
    if (ymax) *ymax = temp;
    temp = ReturnValue();
    if (typedata) *typedata = temp;
    temp = ReturnValue();
    if (type) *type = temp;
    return(ReturnValue()); 
  }


/*
 * InquireCall -- return the name of the symbol called by the given item.
 * returns zero if item is not found or not a symbol call.
 */

short InquireCall(sdf, item)
    short sdf;
    short item;
  {
      SendCode(VgtsInquireCall);
      SendShort(sdf);
      SendShort(item);
      return(ReturnValue());
  }


/*
 * ChangeItem(sdf,item,xmin,xmax,ymin,ymax,typedata,type,string) is the 
 * only procedure that (possibly) alters the database outside the 
 * currently open symbol.  It merely changes the parameters of an already
 * existent item.  
 */

ChangeItem(sdf,item, xmin, xmax, ymin, ymax, typedata, type, string)
  short sdf;
  short item,
        xmin,
        xmax,
        ymin,
        ymax;
  unsigned char   type;
  unsigned char   typedata;
  char	*string;
{
    SendCode(VgtsChangeItem);
    SendShort(sdf);
    SendShort(item);
    SendShort(xmin);
    SendShort(xmax);
    SendShort(ymin);
    SendShort(ymax);
    SendShort(typedata);
    SendShort(type);
    if (type == SDF_SPLINE)
	SendSpline( (SPLINE *) string,
		   sizeof(SPLINE) +
		    (((SPLINE *)string)->numvert - 1) * sizeof(POINT));
    else if (type == SDF_RASTER)
	SendBlock((short *) string,
	  2 * ((xmax - xmin + 1 + 15) >> 4) * (ymax - ymin + 1));
    else
        SendString(string);
    return(item);
}

/* EditSymbol(item) opens item for editing by
initializing the global variables DefiningSymbol and CurrentItem to
the symbol definition item and the last item in the circular definition
list before the symbol definition, respectively.  This command has the
effect of calling DefineSymbol, inserting all the already existing
entries to the definition list, and waiting for additional insertions
or deletions. The editing process is ended the same way the initial
definition process was ended - a call to EndSymbol. */

EditSymbol(sdf, item)
  short sdf;
  short item;
{
    SendCode(VgtsEditSymbol);
    SendShort(sdf);
    SendShort(item);
    return(ReturnValue());
}


/* DeleteSymbol(item) is used to "unexpand" the definition of a
symbol.  The hash table entry will remain and the symbol definition
item will remain, in order to prevent dangling references to this
symbol by other unknown referees, but all the items within the
definition list will be deleted. */

DeleteSymbol(sdf, item)
  short sdf;
  short item;
{
    SendCode(VgtsDeleteSymbol);
    SendShort(sdf);
    SendShort(item);
    return(ReturnValue());
 }



CreateVGT(sdf, type, topSymbol, name)
    short sdf;
    int type;			/* Type of vgt file. */
    short topSymbol;		/* Top-level symbol */
    char *name;			/* human-readable name */
{
/*
 * CreateVGT:
 * Creates entry for another vgt.
 */
    SendCode(VgtsCreateVGT);
    SendShort(sdf);
    SendShort(type);
    SendShort(topSymbol);
    SendString(name);
    return(ReturnValue());
}

DeleteVGT(vgtId)
    int vgtId;
  {
  	/*
	 * The second argument is assumed to be TRUE at the moment.
	 */
      SendCode(VgtsDeleteVGT);
      SendShort(vgtId);
      SendShort(TRUE);
      return(ReturnValue());
  }



DisplayItem(sdf, item, vgt )
  short sdf;
  short item;
  {
  	/*
	 * Set the currentd top-level item to be displayed
	 */
    SendCode(VgtsDisplayItem);
    SendShort(sdf);
    SendShort(item);
    SendShort(vgt);
    return(ReturnValue());
}



CreateView(vgt, sxmin, symin, sxmax, symax, 
		wxmin, wymin, zoom, showGrid)
  int vgt;
  short sxmin, symin, wxmin, wymin, sxmax, symax, zoom;
  BOOLEAN showGrid;
  {
    SendCode(VgtsCreateView);
    SendShort(vgt);
    SendShort(sxmin);
    SendShort(symin);
    SendShort(sxmax);
    SendShort(symax);
    SendShort(wxmin);
    SendShort(wymin);
    SendShort( (zoom<<8) + (showGrid&0377) );
    return(ReturnValue());
  }




DefaultView(vgt, width, height, wXmin, wYmin, zoom, showGrid,
		pWidth, pHeight )
    short vgt, 
	  width, height,	/* zero means user determines it */
	  wXmin, wYmin,		/* offset of world to screen */
	  zoom, showGrid,	/* default viewing parameters */
	  *pWidth, *pHeight;	/* returns the values really used */
  {
    short temp;

    SendCode(VgtsDefaultView);
    SendShort(vgt);
    SendShort(width);
    SendShort(height);
    SendShort(wXmin);
    SendShort(wYmin);
    SendShort(zoom);
    SendShort(showGrid);
    temp = ReturnValue();
    if (pWidth) *pWidth = temp;
    temp = ReturnValue();
    if (pHeight) *pHeight = temp;
  }



short GetMouseClick( x, y, buttons)
  short *x, *y, *buttons;
  {
  	/*
	 * Wait for and return a mouse click
	 */
    short temp;

    SendCode(VgtsGetMouseClick);
    temp = ReturnValue();
    if (x) *x = temp;
    temp = ReturnValue();
    if (y) *y = temp;
    temp = ReturnValue();
    if (buttons) *buttons = temp;
    temp = ReturnValue();
    return(temp);
  }



LISTTYPE FindSelectedObject( sdf, xHit, yHit, vgt, searchType )
    short sdf, xHit, yHit, vgt;
    char searchType;
  {
	/*
	 * hit detection 
	 */
    static LISTTYPE list;
    static MINREC thing;

    SendCode(VgtsFindSelectedObject);
    SendShort(sdf);
    SendShort(xHit);
    SendShort(yHit);
    SendShort(vgt);
    SendShort(searchType);
    list.NumOfElements = ReturnValue();
    if (list.NumOfElements > 0 )
      {
        list.Header = &thing;
	thing.item = ReturnValue();
	thing.edgeset = ReturnValue();
	thing.next = NULL;
      }
    return(list);
  }

LISTTYPE FindMousedObject( sdf, xHit, yHit, vgt, searchType )
  {
   return(FindSelectedObject( sdf, xHit, yHit, vgt, searchType ));
  }

popup(m)
 PopUpEntry *m;
  {
  	/*
	 * send an entire pop-up menu down the connection
	 */
    SendCode(VgtsPopup);
    while (m->string && strlen(m->string) )
      {
        SendString(m->string);
        putchar(m->menuNumber);
	m++;
      }
     SendString(NULL);
     return(ReturnValue());
  }


DefineFont(name,fileName)
    char *name;
    char *fileName;
  {
  	/*
	 * Declare that we are going to use a font
	 */
    char buf[512];
    int size;
    
    size = strlen(name) + 1;
    strcpy(buf,name);

    SendCode(VgtsDefineFont);
    SendShort(size);

    if (fileName)
      {
        strcpy(buf+size,fileName);
        size += strlen(fileName);
      }
    SendBlock(buf,size);
    return(ReturnValue());
  }


SetMode(mode)
  {
  	/*
	 * set the file mode. i.e. Raw or cooked, etc.
	 */
    SendCode(VgtsSetMode);
    SendShort(mode>>16);
    SendShort(mode);
  }

static SendCode(kind)
  {
  	/*
	 * send the escape character followed by the request code
	 */
     putchar(InterpEscape);
     putchar(kind);
  }

static SendShort(n)
  {
  	/*
	 * send a numerical value as a short,
	 * in big-endian order.
	 */
    putchar( ( (n>>8) & 0377) );
    putchar( (n & 0377) );
  }

static SendString(s)
  register  char *s;
  {
  	/*
	 * Send a string to the remote VGTS.
	 * send the length as a short, then the individual characters.
	 */
     register int i;
     if (s==NULL) 
       {
         putchar(0);
	 return;
       }
     i = strlen(s);
     if (i > 254) i = 254;
     putchar(i);
     while (*s && i-- > 0 ) putchar(*s++);
  }
 
static SendSpline(s, length)
  register SPLINE *s;
  {
    register i;
    register POINT *vert = &(s->head);
	/*
	 * Send a spline of (length) bytes down to the Vgts.
	 */
    putchar(MAGIC4);
    SendShort( length >> 16 );
    SendShort( length & SIXTEEN );
    
    SendShort( s->order );
    SendShort( s->numvert );
    SendShort( (int) (s->nib) >> 16 );
    SendShort( (int) (s->nib) & SIXTEEN );
    SendShort( s->border );
    SendShort( s->closed );
    SendShort( s->filled );
    SendShort( s->opaque );
    SendShort( (int) (s->pat) >> 16 );
    SendShort( (int) (s->pat) & SIXTEEN );
    for (i = 0; i < s->numvert;  i++)
      {
	SendShort( vert[i].x );
	SendShort( vert[i].y );
      }
  }

static SendBlock(s, length)
  register char *s;
  {
    register i;
	/*
	 * Send a block of (length) bytes down to the remote Vgts
	 */
    putchar(MAGIC4);
    SendShort( length >> 16 );
    SendShort( length & SIXTEEN );
    
    for (i = length; i > 0; i--)
        putchar(*s++);
  }

static ReturnValue()
  {
  	/*
	 * gather up a short value to be returned and return it!
	 */
    short temp = 0;
    char got1, got2;

    fflush(stdout);
    while ( (got1 = getchar())!=InterpEscape) 
      switch (got1)
     {
       case LF:
       		continue;
		
	case 3:
		Quit();

       default:
# ifndef VAX
	if (stdin->lastexception || stdin->state)
	   {
	     printf("Error %s on stdin\r\n", 
	     	ErrorString(stdin->lastexception));
	     Quit();
	   }
# endif VAX
        printf( "Sync error - Expected 0%o, got 0%o\r\n", InterpEscape, got1&0377);
        return(-1);
     }
    temp = ( (got1 = getchar()) ) << 8;
    temp += (got2 = getchar()) & 0377;
    return(temp);
  }
