/*
 * error.c - A table of error message strings
 *
 * Bill Nowicki January 1983
 */

# include <Vio.h>

# define ReplyBit 	0x8000
# define SystemBit	0x4000
# define SegmentBit	0x2000
# define ReadBit	0x1000
# define WriteBit	0x0800
# define CodeMask	0x07FF

struct codeEntry
  {
    SystemCode code;
    char *string;
  }
    CodeTable[] =
  {
	0xC001,	"Illegal request",
	0xC002,	"No reply",
	0xC003, "Retry",
	0xC004, "Bad args",
	0xC005, "Bad address",
	0xC006, "Bad buffer",
	0xC007, "No server resources",
	0xC008, "No memory",
	0xC009, "No permission",
	0xC00A, "Power failure",
	0xC00B, "Not found",
	0xC00C, "Busy",
	0xC00D, "Timeout",
	0xC00E, "Bad state",
	0xC00F, "Aborted",
	0xC010, "Nonexistent process",
	0xC011, "Not awaiting reply",
	0xC012, "Bad process priority",
	0xC013, "No process descriptors",
	0xC014, "Invalid file id",
	0xC015, "Not readable",
	0xC016, "Not writeable",
	0xC017, "End of file",
	0xC018, "Invalid mode",
	0xC019, "Mode not supported",
	0xC01A, "Bad byte count",
	0xC01B, "Bad block number",
	0xC01C, "Io break",
	0xC01D, "Internal error",
	0xC01E, "Device error",
	0xC01F, "Request not supported",
	0xC020, "Nonexistent session",
	0xC021, "Duplicate name",
	0xC022, "Server not responding",
	0xC023, "Invalid context",
	0xC024, "Current context invalid",
	0xC025, "Kernel timeout",

/* System Standard Request Codes: to be used requesting all standard
 * system services.
 *
 */
	0x5801, "Create instance",
	0x6002, "Query instance",
	0x6003, "Release instance",
	0x7404, "Read instance",
	0x7805, "Write instance",
	0x6005, "Write short instance",
	0x6006, "Set break process",
	0x6007, "Set instance owner",
	0x6008, "Query file",
	0x6009, "Modify file",
	0x600A, "Set prompt",
	0x5C0B, "Exception request",
	0x600C, "Register handler",
	0x580D, "Get user number",
	0x6010, "Get time",
	0x6012, "Terminate team",
	0x7813, "Lookup server",
	0x7414, "Read and forward",
	0x5815, "Remove file",
	0x5816, "Add context name",
	0x7817, "Delete context name",
	0x7818, "Get context id",
	0x7419, "Get context name",
	0x741A, "Get file name",

/* Kernel opcodes */
	0x4020, "Create process",
	0x6021, "Destroy process",
	0x7422, "Query process",
	0x7823, "Modify process",
	0x4024, "Create team",
	0x6025, "Set team priority",
	0x6026, "Set team size",
	0x4027, "Delay",
	0x6028, "Wakeup",
	0x6029, "Get time",
	0x602a, "Set time",
	0x602b, "Set pid",

	0, "OK",
	0, 0
 };

static char *ShortString(number)
    register number;
  {
  	/*
	 * small and simple hex printer to avoid including printf
	 * in everything.
	 */
    static char buf[7];
    register char *table = "0123456789abcdef";

    buf[0] = '0';
    buf[1] = 'x';
    buf[2] = table[ (number>>12) & 15 ];
    buf[3] = table[ (number>>8 ) & 15 ];
    buf[4] = table[ (number>>4 ) & 15 ];
    buf[5] = table[ (number    ) & 15 ];
    buf[6] = 0;
    return(buf);
  }


char *ErrorString(err)
  SystemCode err;
  {
    static char buf[256];
    register struct codeEntry *p = CodeTable;

    for (;p->string;p++)
      if (p->code==err)
	return(p->string);
	
    strcpy(buf,"Strange code(");
    strcat(buf,ShortString(err));
    strcat(buf,")=");
    if (err&ReplyBit)
      strcat(buf,"Reply+");
    else 
      strcat(buf,"Request+");
    if (err&SystemBit)
      strcat(buf,"System+");
    else 
      strcat(buf,"User+");
    if (err&SegmentBit)
      {
        strcat(buf,"Segment+");
        if (err&ReadBit)
          strcat(buf,"Read+");
        if (err&WriteBit)
          strcat(buf,"Write+");
      }
     strcat(buf,ShortString(err&CodeMask));
     return(buf);
  }


PrintError(err, msg)
  SystemCode err;
  char *msg;
  {
    /*
     * print the message corresponding to the indicated code
     * on the standard error file
     */
    fputs(msg, stderr);
    fputs(": ", stderr );
    fputs( ErrorString(err), stderr );
    putc('\n', stderr);
  }
