/*
 * Machine-independent mouse device routines.
 */


#include "../mi/dm.h"
#include "../../libc/include/Vmouse.h"
#include "interrupt.h"
#include "mouse.h"


/* Imports */
extern Process *Map_pid();
extern SystemCode NotSupported();
extern DeviceInstance *GetDevice();

/* Exports */
extern SystemCode MouseCreate();
extern SystemCode MouseRead();
extern SystemCode MouseWrite();
extern SystemCode MouseRelease();



SystemCode MouseCreate( req, desc )  CreateInstanceRequest *req;
				     DeviceInstance *desc;
  /* Create an instance for the Mouse.
   */
 {
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register DeviceInstance *mouse;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (mouse = GetDevice(MouseInstance)) != NULL )
      {
	if( ValidPid( mouse->owner ) )  return (BUSY);
	MouseInstance = 0;
	mouse->owner = 0;
      }
    /*
     * Initialize the device instance descriptor, can assume all fields
     * are zero except for owner and id.
     */

    desc->readfunc = MouseRead;
    desc->writefunc = MouseWrite;
    desc->modifyfunc = NotSupported;
    desc->queryfunc = NotSupported;
    desc->releasefunc = MouseRelease;

    desc->type = (READABLE+WRITEABLE+FIXED_LENGTH);
    desc->blocksize = sizeof(int) + sizeof(int) + sizeof(short);
    desc->lastbytes = desc->blocksize;

    MouseEvent = 0; /* No outstanding mouse event */
    MouseInstance = desc->id; /* Record the instance for interrupt routine */
    MouseX = 0;
    MouseY = 0;
    MouseButtons = 7;	/* all up */

    /* Initialize the interface */

    StartMouse();

    return( OK );
 }

SystemCode MouseRelease( req, desc )
    IoRequest  *req;
    DeviceInstance *desc;
  /*
   * Release mouse instance 
   */
  {
    IoReply *reply;
    register Process *pd;

    StopMouse();

    if ( (pd = Map_pid(desc->reader)) != NULL ) /* Unblock reader process */
      {
	desc->reader = 0;
	reply = (IoReply *) pd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( pd );
      }
    desc->owner = 0;
    return( OK );
  }

static SystemCode MouseGetEvent( req, desc )
    IoRequest		*req;
    DeviceInstance	*desc;
  {
    register IoReply	*reply = (IoReply *) req;
    register MouseBlock	*buffer;

    if( MouseEvent == 0 ) return( NO_REPLY ); /* Wait until mouse event */

    reply->replycode = OK;
    buffer = (MouseBlock *) reply->shortbuffer;
    buffer->buttons = 7 & ~MouseButtons; /* Invert mouse buttons */
    buffer->xcoordinate = (int) MouseX;
    buffer->ycoordinate = (int) MouseY;
    desc->reader = 0;
    MouseEvent = 0;

    return( OK );
  }

SystemCode MouseRead( req, desc )
    IoRequest		*req;
    DeviceInstance	*desc;

   /* Handle a read instance request for the Mouse interface.
    */
  {

    extern Process *Active;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading.
     */
    if( desc->reader ) if( ValidPid(desc->reader) ) return (RETRY);
    if( req->bytecount != desc->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber != 0 ) return( END_OF_FILE );

    /* Check if event is waiting for the reader */
    disable;
    if( MouseGetEvent(req, desc) == OK ) return (OK);

    /* Wait for event to occur */
    desc->reader = Active->pid;
    Active->state = AWAITING_INT;

    return( NO_REPLY );
  }

SystemCode MouseWrite( req, desc )
    IoRequest		*req;
    DeviceInstance	*desc;

   /* Handle a write instance request to the Mouse
    */
  {
    IoReply	*reply = (IoReply *) req;
    register MouseBlock *buffer;

    if( req->bytecount != desc->blocksize ) return( BAD_BYTE_COUNT );
    if( req->blocknumber != 0 ) return( END_OF_FILE );

    buffer = (MouseBlock *) (req->shortbuffer); /* pointer to X and Y value */

    MouseX = (short) buffer->xcoordinate;
    MouseY = (short) buffer->ycoordinate;
    MouseButtons = 7 & ~(buffer->buttons);	/* Invert to 1 if down */
    MouseEvent = 1;

    return( OK );
  }

SystemCode CheckMouseReader()
  /*
   * Checks if a reader is waiting for the mouse and replies to him.
   */
  {
    register Process *pd;
    register DeviceInstance *desc;

    if (((desc = GetDevice(MouseInstance)) != NULL) &&
    	    ((pd = Map_pid(desc->reader)) != NULL))
      {
        if (MouseGetEvent(pd->msg, desc) == OK)
	    Addready(pd);
      }
  }
