/*
 * V Kernel - Copyright (c) 1982 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 *
 * Kernel device management
 */

#include "../../libc/include/Vio.h"
#include "process.h"
#include "dm.h"
#include "config.h"

/* Imports */
extern Process *Map_pid();
extern DeviceConfigTable DeviceCreationTable[];
extern Process_id GetReply();

/* Exports */
extern ProcessId SendDevice();
extern SystemCode CreateDevice();
extern SystemCode QueryDevice();
extern SystemCode ReadDevice();
extern SystemCode WriteDevice();
extern SystemCode ReleaseDevice();
extern SystemCode SetDeviceOwner();
extern SystemCode ModifyDevice();
extern SystemCode QueryDeviceConfig();
extern DeviceInstance *AllocDeviceInstance();
extern SystemCode NotReadable();
extern SystemCode NotWriteable();
extern SystemCode NotSupported();
extern DeviceInstance *DeviceTable[];
extern DeviceInstance *GetDevice();

ProcessId SendDevice( msg ) register Unspec *msg;

  /* Implement the kernel device server, providing access to kernel
   * devices according to the standard I/O protocol.
   */
  {
    extern Process *Active;
    extern Process_id Device_Server_Pid;
    register IoRequest *req;
    register IoReply *replymsg;
    register DeviceInstance *desc;
    register SystemCode reply;

    req = (IoRequest *)(Active->msg);
    replymsg = (IoReply *)(Active->msg);

    switch( req->requestcode )
      {
	case CREATE_INSTANCE:
	  {
	    reply = CreateDevice( req );
            break;
	  }
  	case QUERY_INSTANCE:
	  {
	    reply = QueryDevice( req );
	    break;
	  }
	case READ_INSTANCE:
	  {
	    reply = ReadDevice( req );
	    break;
	  }
	case WRITE_INSTANCE:
	case WRITESHORT_INSTANCE:
	  {
	    reply = WriteDevice( req );
	    break;
	  }
	case RELEASE_INSTANCE:
	  {
	    reply = ReleaseDevice( req );
	    break;
	  }
	case SET_INSTANCE_OWNER:
	  {
	    reply = SetDeviceOwner( req );
	    break;
	  }
	case MODIFY_FILE:
	  {
	     reply = ModifyDevice( req );
	     break;
	  }
	case QUERY_FILE:
	  {
	     reply = QueryDeviceConfig( req );
	     break;
	  }
	default:
	  {
  	     reply = ILLEGAL_REQUEST;
 	  }
      }
    if( reply == NO_REPLY )
      {
	Removeready(); /* Suspend process */
	Active->blocked_on = Device_Server_Pid;
  	Active->finish_up = (Unspec (*)()) GetReply;
  	Active->returnMessage = (Unspec *) msg;
      }
    else
      {
	replymsg->replycode = reply;
	Copy_out_msg(msg,Active->msg);
      }

    return( Device_Server_Pid );
  }

SystemCode CreateDevice( req ) register CreateInstanceRequest *req;

  /* Create an instance of the specified device.
   * Most of the work is performed by a device-dependent routine.
   */
  {
    register DeviceInstance *desc;
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register SystemCode (*func)(), r;
    short unsigned index;
    short unsigned FindDevice();

    if( (desc=AllocDeviceInstance()) == NULL ) return( NO_SERVER_RESOURCES );

    if( (index = FindDevice(req->type)) > MAX_DEV_TYPE )
	return( NOT_FOUND );

    func = DeviceCreationTable[index].CreateFunc;

    if( (r = (*func)(req, desc)) != OK )
      {
	desc->owner = 0;
	return( r );
      }
    reply->fileid = desc->id;

    return( QueryDevice( reply ) );
  }

short unsigned FindDevice( type ) short unsigned type;
  {
    register short	i;

    for ( i = 0; i < MAX_DEV_TYPE; i++ )
      if ( DeviceCreationTable[i].type == type ) return( i );
    return( MAX_DEV_TYPE+1 );
  }

SystemCode QueryDevice( req ) register QueryInstanceRequest *req;

  /* Satisfy a query instance request from a process.
   */
  {
    extern Process_id Device_Server_Pid;
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register DeviceInstance *desc;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );

    reply->fileserver = Device_Server_Pid;
    reply->blocksize = desc->blocksize;
    reply->filetype = desc->type;
    reply->filelastbytes = desc->lastbytes;
    reply->filelastblock = desc->lastblock;
    reply->filenextblock = desc->nextblock;

    return( OK );
  }

SystemCode ReadDevice( req ) IoRequest *req;

  /* Handle a read instance request to a kernel device.
   */
  {
    register DeviceInstance *desc;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );

    return( (*desc->readfunc)( req, desc ) );
  }

SystemCode WriteDevice( req ) IoRequest *req;

  /* Handle a write instance request to a kernel device.
   */
  {
    register DeviceInstance *desc;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );

    return( (*desc->writefunc)( req, desc ) );
  }


SystemCode ModifyDevice( req ) IoRequest *req;

  /* Handle a modify file request to a kernel device.
   */
  {
    register DeviceInstance *desc;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );

    return( (*desc->modifyfunc)( req, desc ) );
  }

SystemCode QueryDeviceConfig( req ) IoRequest *req;

  /* Handle a query file request to a kernel device.
   */
  {
    register DeviceInstance *desc;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );

    return( (*desc->queryfunc)( req, desc ) );
  }

SystemCode ReleaseDevice( req ) IoRequest *req;

  /* Handle a release instance request to a kernel device.
   */
  {
    extern Process *Pd_bundle[], *Active;
    register DeviceInstance *desc;
    Process *pd;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );
    pd = Map_to_pd(desc->owner);
    if( pd->team != Active->team ) 
	return( NO_PERMISSION );

    return( (*desc->releasefunc)( req, desc ) );
  }

SystemCode SetDeviceOwner( req ) IoRequest *req;

  /* Handle a set instance owner request to a kernel device.
   */
  {
    extern Process *Pd_bundle[], *Active;
    register DeviceInstance *desc;
    Process *pd;

    if( (desc = GetDevice(req->fileid)) == NULL ) return( NOT_FOUND );
    pd = Map_to_pd(desc->owner);
    if( pd->team != Active->team ) 
	return( NO_PERMISSION );

    desc->owner = req->instanceowner;

    return( OK );
  }

SystemCode NotSupported( req, desc ) register IoRequest *req;
					register DeviceInstance *desc;
  /* Dummy function for requests not supported by device.
   */
  {
    return( REQUEST_NOT_SUPPORTED );
  }

SystemCode NotReadable( req, desc ) register IoRequest *req;
					register DeviceInstance *desc;
  /* Dummy read function for non-readable devices.
   */
  {
    return( NOT_READABLE );
  }

SystemCode NotWriteable( req, desc ) register IoRequest *req;
					register DeviceInstance *desc;
  /* Dummy write function for non-writeable devices.
   */
  {
    return( NOT_WRITEABLE );
  }

DeviceInstance *DeviceTable[MAX_DEVICES];

DeviceInstance *AllocDeviceInstance()

  /* Allocate a pair of consecutive free device instance
   * descriptors from the device table and return a pointer
   * to the first one. The instance ids are set to the next
   * "generation" for these descriptors.
   */
  {
    extern Process *Active;
    extern DeviceInstance *DeviceTable[];
    InstanceId	id;
    static int index = -2;
    register DeviceInstance *desc;
    register char *p;
    register unsigned i;

    for( i = 0; i < MAX_DEVICES; i += 2 )
      {
	index += 2;
	if( index >= MAX_DEVICES ) index = 0;

	desc = DeviceTable[index];

        if( Map_pid(desc->owner) == NULL &&
	    Map_pid((desc+1)->owner) == NULL )
	  {
	    id = desc->id + MAX_DEVICES; /* Next generation */
	    p = (char *) desc;	/* Now zero the descriptors */
	    for( i = 0; i < sizeof(DeviceInstance)*2; ++i ) *p++ = 0;
	    desc->id = id;
	    (desc+1)->id = id+1;
	    desc->owner = Active->pid;  /* Only set owner for first desc */
	    return (desc);
	  }
      }
    return( NULL );
  }

DeviceInstance *GetDevice( id ) register InstanceId id;

  /* Locate the kernel device descriptor with the corresponding
   * instance id if any and return a pointer to this descriptor.
   * Return NULL if no such descriptor.
   */
  {
    extern DeviceInstance *DeviceTable[];
    register DeviceInstance *desc;

    desc = DeviceTable[id & (MAX_DEVICES-1)];

    return( (desc->id == id) ? desc : NULL );
  }

