/*
 * V Kernel - Copyright (c) 1982 by David Cheriton
 * (Transliterated from Zed and Verex Kernel)
 *
 * Basic Process Implementation
 *
 * Provides operations to add and remove processes from
 * the ready queue:  Addready(pid) and Removeready().
 * Kernel synchronization is achieved by only switching processes
 * upon exit from the kernel.  Exit from a kernel trap can
 * always cause a process switch; exit from an interrupt service
 * routine can only cause a switch if the
 * preempted process was not running in kernel mode.
 * Interrupt routines are assumed to only access local data
 * structures plus add processes to the ready queues.
 * The latter is synchronized by disabling interrupts.
 * There is assumed to be an idle process that is always ready so
 * that the ready queue is never empty.
 */

#include "process.h"

/* Ready queue head and active process pointer */
Process  *Readyq_head;
Process  *Active;


Process *Map_pid( pid ) register Process_id pid;
  {
    extern Process *Pd_bundle[];
    register Process *pd;

    return( (pd=Map_to_pd(pid))->pid == pid ? pd : NULL );
  }

Removeready()

  /* Remove the active process from the ready queue.
   * It continues running (in the kernel) until the kernel is exited.
   * Note: This restores interrupts to the same level as was set
   * when it was called.
   */
  {
    extern Process *Active;
    register  Process *pd, *tmp_pd;
    Interrupt_mask old_mask;

    pd = Active;
    tmp_pd = (Process *) &Readyq_head;  /* link is first field of a pd */
    old_mask = Mask_interrupts();
    /* Search for Active in the ready queue */
    while( tmp_pd->link != pd ) tmp_pd = tmp_pd->link;
    tmp_pd->link = pd->link;

    Restore_interrupt_mask( old_mask );
  }

Addready( pd ) register  Process *pd;

  /* Add the specified process to the ready queue.
   * Note: this must restore interrupts to the same level as
   * the level when it was called.
   */
  {
    register Process *curr_pd, *tmp_pd;
    register short unsigned prio;
    Interrupt_mask old_mask;

    prio = pd->priority;
    old_mask = Mask_interrupts();
    pd->state = READY;

    /* Add pd to the ready queue */
    curr_pd = (Process *) &Readyq_head;
    while( (tmp_pd = curr_pd->link)->priority <= prio ) curr_pd = tmp_pd;

    pd->link = tmp_pd;
    curr_pd->link = pd;
    Restore_interrupt_mask( old_mask );
  }
