/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann, Andrew Shore
 * (Transliterated from Zed and Verex Kernel)
 *
 * Serial Interface Handling Routines by Andrew Shore May-June 1982
 *	to handle the CadLinc-keytronix keyboard and demultiplex 
 *	keystrokes from mouse squeaks -- implements the `mouse-device'
 *
 * Cadlinc mouse device code debugged by Marvin Theimer and prepared for final 
 * 	installation: July 30 1982.
 *
 * Serial lines are now devices.  Tim Mann, November 4, 1982.
 *
 * Mouse device code changed to reflect the possibility of both SMI and
 * Cadlinc mice.  This effects only the serial line interrupt routine.
 * - Marvin Theimer, 2/83.
 */

#include "interrupt.h"
#include "cbuff.h"
#include "../mi/dm.h"
#include "../mi/mouse.h"
#include "../../libc/include/Vserial.h"


/* Imports */
extern Process *Map_pid();
extern SystemCode NotSupported();
extern DeviceInstance *GetDevice();
extern int CbuffGet();
extern CbuffPut();
extern CbuffFlush();

extern Process *Active;
extern short WsType;

/* Exports */
extern SystemCode SerialCreate();
extern SystemCode SerialRead();
extern SystemCode SerialWrite();
extern SystemCode SerialRelease();


/* Routines */
Init_serial_interface( linenum ) 
    int linenum;
  /* 
   * Initialize the specified serial interface.
   */
  {
    extern int Asm_SerialInterrupt();

    lineservice( Asm_SerialInterrupt );
    linereset( linenum );
    linearmtx( linenum );
    linearmrx( linenum );

    if (WsType == CADLINC && linenum == 0)
	lineput( 0, '`' );  /* make sure mouse is off */	
  }

/* Kernel Serial interface support routines for the Device Server */

/* Pointers to instances for serial lines */
DeviceInstance	*SerialInInstance[2];
DeviceInstance	*SerialOutInstance[2];

SystemCode SerialCreate( req, desc )  
    CreateSerialInstanceRequest *req;
    DeviceInstance *desc;
  /*
   * Create instances for a serial line.  The instance returned
   *  is used for output to the line.  The instance with the next
   *  larger id is used for input; its parameters may be obtained
   *  by doing a QueryInstance.
   */
 {
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register DeviceInstance *line;
    register int linenum = (req->linenum) & 1;

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (line = SerialInInstance[linenum]) != NULL )
      {
	if( ValidPid( line->owner ) )  return (BUSY);
	SerialInInstance[linenum] = 0;
	line->owner = 0;
      }

    if( (line = SerialOutInstance[linenum]) != NULL )
      {
	if( ValidPid( line->owner ) )  return (BUSY);
	SerialOutInstance[linenum] = 0;
	line->owner = 0;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner.
     */

    /* Output descriptor */
    desc->readfunc = NotSupported;
    desc->writefunc = SerialWrite;
    desc->modifyfunc = NotSupported;
    desc->queryfunc = NotSupported;
    desc->releasefunc = SerialRelease;

    desc->type = WRITEABLE+STREAM;
    desc->blocksize = 1;
    desc->devicedesc = (char *) linenum;
		 /* Only device-specific info is which line */
    SerialOutInstance[linenum] = desc;

    /* Input descriptor */
    desc++;
    desc->owner = Active->pid;
    desc->readfunc = SerialRead;
    desc->writefunc = NotSupported;
    desc->modifyfunc = NotSupported;
    desc->queryfunc = NotSupported;
    desc->releasefunc = SerialRelease;

    desc->type = READABLE+STREAM+VARIABLE_BLOCK;
    desc->blocksize = IO_MSG_BUFFER;
    desc->devicedesc = (char *) linenum;
    SerialInInstance[linenum] = desc;
		 /* Record the instance for interrupt routine */

    Init_serial_interface(linenum);
    CbuffFlush(linenum);

    return( OK );
 }


SystemCode SerialRelease( req, desc )
    IoRequest  *req;
    DeviceInstance *desc;
  /*
   * Release serial line instance
   */
  {
    IoReply *reply;
    register Process *pd;

    if( Map_pid(desc->reader) != NULL ) /* Unblock reader process */
      {
	desc->reader = 0;
	reply = (IoReply *) pd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( pd );
      }

    if( Map_pid(desc->writer) != NULL ) /* Unblock writer process */
      {
	desc->writer = 0;
	reply = (IoReply *) pd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( pd );
      }

    desc->owner = 0;
    return( OK );
  }


SerialReturnChar(reply, desc, c)
    IoReply		*reply;
    DeviceInstance	*desc;
    char		c;
  {
    reply->replycode = OK;
    reply->shortbuffer[0] = c;
    reply->bytecount = 1;
    desc->nextblock++;
    desc->reader = 0;
  }


SystemCode SerialRead( req, desc )
    IoRequest		*req;
    DeviceInstance	*desc;

   /* Handle a read instance request for a serial line
    */
  {
    extern Process *Active;
    register unsigned bytecount;
    register int c, i;
    register int linenum = (int) desc->devicedesc;
    register IoReply *reply = (IoReply *) req;
    register char *sbuff = &reply->shortbuffer[0];

    bytecount = req->bytecount;
    reply->bytecount = 0;
    /*
     * Reply for RETRY if already a waiting reader process.
     * This effectively provides busy-wait queuing for reading. 
     */
    if( desc->reader ) if( ValidPid(desc->reader) ) return( RETRY );

    if( bytecount > desc->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
    if( req->blocknumber != desc->nextblock ) return( BAD_BLOCK_NO );

    disable;
    /* Try to get characters from the buffer */
    for (i = 0; i < bytecount; i++)
      {
	if ( (c = CbuffGet(linenum)) == CBUFF_EMPTY ) break;
	*sbuff++ = c;
      }

    if (i != 0) 
      {
	reply->replycode = OK;
	reply->bytecount = i;
	desc->nextblock++;
	return (OK);  /* One or more characters returned */
      }

    /* Wait for a character to come in */
    desc->reader = Active->pid;
    Active->state = AWAITING_INT;

    return( NO_REPLY );
  }


SerialXmitChar(reply, desc, linenum)
    IoReply		*reply;
    DeviceInstance	*desc;
    short		linenum;
  {
    desc->lastblock++;
    lineput( linenum, reply->shortbuffer[0] );
		  /* character is still in msg */
    reply->bytecount = 1;
    reply->replycode = OK;
  }


SystemCode SerialWrite( req, desc )
    IoRequest		*req;
    DeviceInstance	*desc;

   /* Handle a write instance request to a serial line
    */
  {
    extern Process *Active;
    IoReply *reply = (IoReply *) req;
    register int linenum = (int) desc->devicedesc;
    register unsigned bytecount;

    bytecount = req->bytecount;
    reply->bytecount = 0;

    if( bytecount > desc->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return ( OK );
/*  Disabled to simplify multi-process access to the console.
    if( req->blocknumber != desc->lastblock+1 ) return( BAD_BLOCK_NO );
*/
    disable;
    if (!linereadytx(linenum))
      {
	desc->writer = Active->pid;	/* Wait until UART ready */
	Active->state = AWAITING_INT;

	return (NO_REPLY);
      }

    SerialXmitChar(reply, desc, linenum);
    return( OK );
  }

Call_inthandler(SerialInterrupt)   /* Macro expansion to interrupt-invoked
				    * Call to SerialInterrupt
				    */


SerialInterrupt()

  /*
   * Handle an interrupt from the serial device.
   *   Checks both lines to see which generated the interrupt.
   */
  {
    register Process *pd;
    register unsigned char c;
    register char c1;
    register DeviceInstance *desc;
    register int	linenum;
    static short	squeakCount = 0;	/* bytes left in squeak */
    static short	significantSqueak = 0;	/* flag to say if this squeak
                                                 * represents an `event' */

    linenum = 0;
    for (linenum = 0; linenum < 2; ++linenum)
      {
	if( linereadyrx(linenum) ) /* Receiver interrupt */
	  {
	    c = lineget( linenum );

	    if( (linenum == 0) && (WsType == CADLINC) )
              {
		if((squeakCount == 0) && (0x80 <= c) && (c <= 0x87)) 
							/* squeak start */
		  {
		    squeakCount = 3;
		    if( (c & 07) != MouseButtons )
		      {
		        significantSqueak = 1;
			MouseButtons = c & 07;
		      }
		  }
		else if( (squeakCount == 1) && c )
		  {
		    c1 = (char) c;
		    MouseX += (short) c1;
		    significantSqueak = 1;
		  }
		else if(( squeakCount == 2) && c )
		  {
		    c1 = (char) c;
		    MouseY += (short) c1;
		    significantSqueak = 1;
		  }
	      }
	    if ((linenum == 0) && squeakCount)	/* Within a Cadlinc
						   mouse squeak */
	      {
		if( ((--squeakCount) == 0) && significantSqueak )
		  {
		    significantSqueak = 0;
		    MouseEvent = 1;
		  }
	      }
	    else
	      {
		/* not a mouse interrupt -- unblock serial line reader 
					    or queue character for it */
		if( WsType == CADLINC && linenum == 0 && c == ('S'|0x80) )
			lineput( 0, '`' );  /* turn off CadLinc mouse */

		if ( (desc = SerialInInstance[linenum]) != NULL )
		  {
		    if ( (pd = Map_pid(desc->reader)) != NULL )
		      {
			/* Reader waiting -- give it the character */
			SerialReturnChar(pd->msg, desc, c);
		        Addready( pd );
		      }
		    else /* Reader not waiting -- queue the character */
			CbuffPut(linenum, c);
		  }		
	      }	
	  }

	if( linereadytx( linenum ) ) /* Transmitter interrupt */
	  {
	    if( (desc = SerialOutInstance[linenum]) != NULL
		&& (pd = Map_pid(desc->writer)) != NULL )
	      {
		SerialXmitChar(pd->msg, desc, linenum );
		Addready( pd );
	      }
	    else lineresettxint( linenum );
	  }
      }
  }
