/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 *
 * Memory mapping routines--machine dependent
 */

#include "../mi/process.h"
#include "memory.h"
#include "interrupt.h"
#include "../../libc/include/Vexceptions.h"
#include <b.out.h>

#define min(a, b) ((a) < (b) ? (a) : (b))

PageMapEntry OnboardFreePageList = NOPAGE;   /* Free list of pages */
PageMapEntry MultibusFreePageList = NOPAGE;
int FreePages = 0;
struct bhdr KernelBheader = {0},
	    TeamBheader = {0};		/* Force to data segment */
Page *FreeKernelSpace;		/* First unused page in kernel space */

Process *Map_pid();
int Asm_MultibusParityError(), MultibusParityError();

Init_memory_mapping()
  /*
   * Setup the memory mapping as required.
   * We assume the memory mapping has been initialized by the Sun PROM
   *   monitor in the standard way:
   * Addresses 0 to MBMEM (1 megabyte) are mapped to the Sun onboard or P2
   *   bus memory, if that much memory exists.  If there is more than one
   *   megabyte of P2 bus memory, the additional memory is not mapped.
   * Addresses MBMEM to TLIMIT are mapped to ordinary Multibus memory,
   *   if that much exists.
   * Addresses TLIMIT to MBIO are mapped to devices in the Multibus 
   *   memory address space.  A frame buffer is usually located at
   *   TLIMIT.
   * Addresses MBIO to (slightly below) MEMLIMIT are mapped to the
   *   Multibus I/O address space.
   *
   * We assume the loader program has placed the kernel at KSTART and
   *   the first team at TSTART, with the team's b.out header just
   *   below TSTART, and the kernel's b.out header just below that.
   *
   * This routine maps memory as follows:
   * The kernel is assigned addresses between 0 and TSTART.  Any unused
   *   memory in this area is added to the free list.
   * Addresses from TSTART to TLIMIT are available to teams.  The first
   *   team is initially given just enough mapped memory in this area
   *   to hold its code and data.  All other memory is added to the free
   *   list.
   * The area from TLIMIT to MEMLIMIT is left mapped as the Sun monitor
   *   has it, to permit access to devices in the Multibus memory and I/O
   *   space by all teams.
   *
   * The free list is actually maintained as two separate lists, one for
   *   onboard and the other for Multibus memory, since Multibus memory is
   *   much slower.  Pages are allocated from the onboard list until it
   *   is empty, then from the Multibus list.
   */
  {
    int onboardPages, multibusPages;
    char *ttop, *ktop, *cptr;
    int ctx, i;
    PageMapEntry freepg, oldpg;
    Page *page;
    Segment *seg;

    /* Plug Multibus parity error interrupt location (level 4) */
    *( (int (**)()) INT4 ) = Asm_MultibusParityError;

    /* Copy the b.out headers into kernel data space */
    TeamBheader = *((struct bhdr *) (TSTART - sizeof(struct bhdr)));
    KernelBheader = *((struct bhdr *) (TSTART - 2*sizeof(struct bhdr)));

    ttop = (char *) TSTART + TeamBheader.tsize + TeamBheader.dsize
				     + TeamBheader.bsize + INIT_STACK;
    ktop = (char *) KSTART + KernelBheader.tsize + KernelBheader.dsize 
				     + KernelBheader.bsize;

    /* Zero the bss area in case the loader didn't do it */
    cptr = (char *) TSTART + TeamBheader.tsize + TeamBheader.dsize;
    for (i = TeamBheader.bsize; i; i--) *cptr++ = '\0';

    cptr = (char *) KSTART + KernelBheader.tsize + KernelBheader.dsize;
    for (i = KernelBheader.bsize; i; i-- ) *cptr++ = '\0';

    FreeKernelSpace = uptopage(ktop);

    GetMemorySize(&onboardPages, &multibusPages);

    for (ctx = 0; ctx < NUMCTXS; ctx++) /* Protect kernel from users */
      {
	setcontext(ctx);
	for (seg = 0; seg < (Segment *) TSTART; seg++)
	  {
	    segmap(seg) = RWX___ | (segmap(seg) & (PHYSSEGS-1));
	  }
      }

    /* Unmap segments from contexts other than 0 */
    /* Leave frame buffer and Multibus I/O in all contexts */
    for (ctx = 1; ctx < NUMCTXS; ctx++)
      {
	setcontext(ctx);
	for (seg = (Segment *) TSTART; 
		    seg < (Segment *) TLIMIT; seg++)
	    segmap(seg) = UNUSED;
      }
    setcontext(0);

    /* Construct free list of pages */

    /* First, unmap unused kernel memory from kernel segments */
    for (page = uptopage(ktop); page < ((Page *) TSTART); page++)
	FreePage(page);

    /* Next, free unused onboard memory (up to MBMEM) */
    for ( page = uptopage(ttop); 
	  page < (Page *) min(onboardPages*PAGESIZE, MBMEM); 
	  page++ )
	FreePage(page);

    /* Unmap nonexistent pages from onboard area */
    for ( ; page < (Page *) MBMEM; page++)
	pgmap(page) = NOPAGE;

    /* Now free any Multibus memory found */
    for ( ; page < (Page *) (MBMEM + multibusPages*PAGESIZE); page++)
	FreePage(page);

    /* Unmap nonexistent pages from Multibus area */
    for ( ; page < (Page *) TLIMIT; page++)
	pgmap(page) = NOPAGE;

    /* Free unused segments.  Leave frame buffer and Multibus I/O. */
    for (seg = (Segment *) uptoseg(ttop); seg < (Segment *) TLIMIT; seg++)
	segmap(seg) = UNUSED;

    /* Add the rest of onboard memory to the free list, if
     *  there was more than 1 megabyte */
    oldpg = pgmap(TESTPAGE);
    for (freepg = MBMEM/PAGESIZE; freepg < onboardPages; freepg++)
      {
	pgmap(TESTPAGE) = freepg | ONBOARD;
	FreePage(TESTPAGE);
      }
    pgmap(TESTPAGE) = oldpg;

  }

char *SetTeamSize(pid, size)
    Process_id pid;
    char *size;
  /*
   * Set the memory size for a team, mapping or unmapping pages as needed.
   */
  {
    extern Process *Pd_bundle[], *Active;
    register Process *pd;
    char *KSetTeamSize();

    if (pid == 0)
	pd = Active;
    else if ( (pd=Map_to_pd(pid))->pid != pid )
	return (0);

    return (KSetTeamSize(pd->team, size));
  }

char *KSetTeamSize(td, size)
    register Team *td;
    char *size;
  /*
   * Called by SetTeamSize to do the actual work.
   */
  {
    register Page *firstUnused, *newFirstUnused;
    int oldcxt, fail;

    /* Range check on size */
    if (size < (char *) TSTART) size = (char *) TSTART;
    if (size >= (char *) TLIMIT) size = (char *) TLIMIT;

    oldcxt = getcontext();	/* Temporarily switch contexts */
    setcontext(td->team_space.context);

    firstUnused = uptopage(td->team_space.size);
    newFirstUnused = uptopage(size);

    fail = 0;
    if (firstUnused <= newFirstUnused)
      {
	/* More memory needed */
	for (; firstUnused < newFirstUnused; firstUnused++)
	  {
	    if (fail = AllocatePage(firstUnused)) break;
	  }
      }
    else
      {
	/* Less memory needed */
	for (firstUnused--; firstUnused >= newFirstUnused; firstUnused--)
	  {
	    FreePage(firstUnused);
	    if (firstUnused == uptoseg(firstUnused))
	      {
		segmap(firstUnused) = UNUSED;  /* Release the phys seg */
	      }
	  }
      }

    setcontext(oldcxt);

    if (fail)
      {
	return (td->team_space.size = ((char *) firstUnused));
      }
    else
      {
	return (td->team_space.size = size);
      }

  }



FreePage(pgaddr)
    Page *pgaddr;
  /*
   * Add a page to the free list, unmapping it from the current context
   */
  {
    PageMapEntry freepg;

    freepg = pgmap(pgaddr);	 /* Get map entry pointing to this page */
    if ( (freepg & PAGETYPE) == ONBOARD )
      {
	/* Point page to old list top */
	pgaddr->nextpage = OnboardFreePageList;
	OnboardFreePageList = freepg;
      }
    else
      {
	pgaddr->nextpage = MultibusFreePageList;
	MultibusFreePageList = freepg;
      }
    pgmap(pgaddr) = NOPAGE;	 /* Unmap page */
    FreePages++;
  }


AllocatePage(pgaddr)
    Page *pgaddr;
  /*
   * Map a page into the current context at the given address, if one
   *  can be found.  Return 0 for success, 1 for failure.
   */

  {
    PageMapEntry newpg;
    int segno, gcdone = 0;

    /* Find a page, if available */
    if ( (newpg = OnboardFreePageList) == NOPAGE &&
	 (newpg = MultibusFreePageList) == NOPAGE )
      {
	GarbageCollect();
	gcdone++;
	if ( (newpg = OnboardFreePageList) == NOPAGE &&
	     (newpg = MultibusFreePageList) == NOPAGE ) return (1);
      }

    /* Make sure there is a physical segment mapped to this logical seg */
    if ((segmap(pgaddr) & SEGMAP) == UNUSED)
      {
	/* Find a physical segment with no page mapped into this
	 *  slot.  This page slot is guaranteed to be the first in the
	 *  segment, and if it is empty, the segment is guaranteed
	 *  to be unused. This works since we always fill segments from
	 *  low addresses to high, and always unmap empty segments.
	 */
mapphys:
	for (segno = 0; segno < PHYSSEGS; segno++)
	  {
	    segmap(pgaddr) = RWXRWX | segno;
	    if (pgmap(pgaddr) == NOPAGE) break;
	  }
	if (segno == PHYSSEGS)	  /* No segs available */
	  {
	    segmap(pgaddr) = UNUSED;
	    if (gcdone)
	      {
		return (1);
	      }
	    else
	      {
		GarbageCollect();
		gcdone++;
		goto mapphys;  /* go back and try again */
	      }
	  }
      }

    /* Map in the page */
    pgmap(pgaddr) = newpg;
    FreePages--;

    /* Remove it from the free list */
    if ( (newpg & PAGETYPE) == ONBOARD )
	OnboardFreePageList = pgaddr->nextpage;
    else
	MultibusFreePageList = pgaddr->nextpage;

    return (0);
  }


GarbageCollect()
  /*
   * Search through the list of teams, deallocating any memory
   * assigned to teams that have died.
   */
  {
    extern Team *Teams;
    register Team *td;
    register Page *page;
    register Segment *seg;
    int oldcxt;

    oldcxt = getcontext();
    td = Teams;
    while (td != NULL)
      {
	if ((td->team_space.size > (char *) TSTART)
				     && !ValidPid(td->team_root))
	  {
	    setcontext(td->team_space.context);

	    for (page = (Page *) TSTART;
		page < uptopage(td->team_space.size); page++)
	      {
		FreePage(page);
	      }

	    for (seg = (Segment *) TSTART;
		seg < (Segment *) uptoseg(td->team_space.size); seg++)
	      {
		segmap(seg) = UNUSED;  /* Release the phys seg */
	      }

	    td->team_space.size = (char *) TSTART;
	  }

	td = td->link;
      }

    setcontext(oldcxt);
  }


InterTeamCopy(dest, dteam, src, steam, bytes)
    Team *dteam, *steam;
    char *dest, *src;
    long unsigned bytes;
  /*
   * This routine is used by MoveFrom and MoveTo to copy bytes between
   *   different teams.
   */

  {
    short oldcxt;
    SegMapEntry oldxfs, xfs;
    int sbytes;

    oldcxt = getcontext();		/* Save old context */
    setcontext(dteam->team_space.context);
    oldxfs = segmap(XFERSEG);

    while (bytes > 0)
      {
	/* Set up to move all the desired bytes from one source segment */
	sbytes = ((char *) uptoseg(src+1)) - src;
	sbytes = min(sbytes, bytes);

	/* Map segment containing source bytes into xfer segment */
	setcontext(steam->team_space.context);
	xfs = segmap(src);
	setcontext(dteam->team_space.context);
	segmap(XFERSEG) = xfs;

	/* Move the bytes */
	Copy_bytes(dest, XFERSEG + ((unsigned) src) % SEGSIZ, sbytes);
	bytes -= sbytes;
	src += sbytes;
	dest += sbytes;

      }

    segmap(XFERSEG) = oldxfs;
    setcontext(oldcxt);

  }



AllocateMultibusMemory(size, multibusAddress, mappedAddress)
    unsigned size;
    unsigned *multibusAddress;
    Page **mappedAddress;
  /*
   * Allocate 'size' bytes of contiguous Multibus memory and map it into
   *   kernel space.  Returns the Multibus address of the block in
   *   'multibusAddress', and the address within kernel space in
   *   'mappedAddress'.
   * This routine may only be called during kernel initialization, after
   *   Init_memory_mapping().
   */
  {
    Page *p;
    PageMapEntry newpg;

    /*
     * Memory was stacked on Multibus free list, so taking pages off
     *   one by one gives contiguous memory with steadily decreasing
     *   addresses.  Thus we fill the requested block from back to front.
     */
    p = FreeKernelSpace + (size - 1)/PAGESIZE;	/* Start of last page
    						 * in block */
    if (p >= (Page *) TSTART) Kabort("Not enough kernel space");
    for (; p >= FreeKernelSpace; p--)
      {
        newpg = MultibusFreePageList;	/* Find a multibus page */
	if (newpg == NOPAGE) Kabort("No Multibus memory");
	pgmap(p) = newpg;		/* Map it in */
	FreePages--;
	MultibusFreePageList = p->nextpage;
      }

    /* Set up return values */
    *mappedAddress = FreeKernelSpace;
    FreeKernelSpace += (size + PAGESIZE - 1)/PAGESIZE;
    *multibusAddress = (newpg & 0x0FFF) << 11;
  }

	
GetMemorySize(onboardPages, multibusPages)
    int *onboardPages, *multibusPages;
  /*
   * Probe to determine size of on-board and Multibus memory on 
   *  a Sun. Returns the amount of each type of memory, in pages.
   *  Multibus memory is assumed to start at Multibus address 00000.
   */
  {
    PageMapEntry oldPage;
    register int opages, mpages, *p;

    /* Save old contents of page map entry we are using */
    oldPage = pgmap(TESTPAGE);

    /* Test onboard pages until one doesn't work */
    opages = TSTART/PAGESIZE;  /* Don't test kernel memory */
    while (opages < MAX_ONBOARD_PAGES)
      {
	pgmap(TESTPAGE) = ONBOARD | opages;
	if (probe(TESTPAGE, 0))
	    opages++;	/* more memory here */
	else
	    break;	/* no more */
      }
	
    /* Test Multibus pages until an error occurs,
     *  writing into all locations of each page found
     *  to initialize it to correct parity.
     */
    mpages = 0;
    while (mpages < MAX_MULTIBUS_PAGES)
      {
	pgmap(TESTPAGE) = MULTMEM | mpages;
	if (probe(TESTPAGE, 1))
	    mpages++;	/* more memory here */
	else
	    break;	/* no more */

	for (p = (int *) TESTPAGE; p < (int *) (TESTPAGE + PAGESIZE); p++)
	    *p = -1;  /* Must not be *p = 0 because that would generate 
		       * a clrw instruction, which reads before writing! */
      }

    *onboardPages = opages;
    *multibusPages = mpages;

    pgmap(TESTPAGE) = oldPage;
  }


/*
 * Interrupt handler for Multibus parity error interrupts.
 */
MultibusParityError()
  {
    Kabort("Multibus parity error");
  }

Call_inthandler(MultibusParityError);



probe(address, writefirst)
    short *address;
    int writefirst;
  /*
   * Look for writeable memory at address.  If writefirst == 1,
   *   write before reading to initialize memory; else preserve
   *   contents of this location.
   */
  {
    register short *Zero = (short *) 0;
    register short z, x;
    int BusErrorOnProbe(), (*oldBusErrorVector)();

    /* Set bus error vector */
    oldBusErrorVector = *( (int (**)()) BUSERROR );
    *( (int (**)()) BUSERROR ) = BusErrorOnProbe;

    z = *Zero;  /* Save old contents of address 0 */
		
    if (writefirst) 
	*address = 0x5c5c;	/* Initialize the test location */

    x = ~*address;		/* Read from it */
    *address = x;		/* Write back the complement */

    if (*address == x && *Zero == z) 
      {
	/* Success */
	*address = ~x;
	*( (int (**)()) BUSERROR ) = oldBusErrorVector;
	return (1);
      }
    else
      {
        /* Failure */
	asm("ProbeFailed:");
	*( (int (**)()) BUSERROR ) = oldBusErrorVector;
	*Zero = z;
        return (0);
      }

    asm("	.globl BusErrorOnProbe");
    asm("BusErrorOnProbe:");
    asm("	addl	#14, sp");
    asm("	jmp	ProbeFailed");

  }


/* Memory protection types.  Maps from one of 16 protection
 *  codes, defined in memory.h, into individual bits */
char permission[16] =
  {
    0,			/*  0 */
    SX,			/*  1 */
    SR,			/*  2 */
    SR|SX,		/*  3 */
    SR|SW,		/*  4 */
    SR|SW|SX,		/*  5 */
    SR|UR,		/*  6 */
    SR|SW|UR,		/*  7 */
    SR|UR|SW,		/*  8 */
    SR|SW|UR|UW,	/*  9 */
    SR|SW|UR|UX,	/* 10 */
    SR|SW|UR|UW|UX,	/* 11 */
    SR|SX|UR|UX,	/* 12 */
    SR|SW|SX|UR|UX,	/* 13 */
    SR|SW|SX|UX,	/* 14 */
    SR|SW|SX|UR|UW|UX	/* 15 */
  };


int DiagnoseBusError(req)
    ExceptionRequest *req;
  /*
   * Diagnose the cause of a bus error.  Used by exception handler.
   */
  {
    SegMapEntry smentry, attempt;

    if (req->accaddr > ADDRLIMIT)
      {
	/* Garbage address */
	return (OUT_OF_RANGE);
      }

    if (req->accaddr > MEMLIMIT)
      {
	/* Address outside mapped region */
	if (req->status & SUPERVISOR_STATE)
	    return (SPURIOUS);
	else
	    return (SYSTEM_SPACE);
      }

    /* Get segment map entry for this address */
    smentry = segmap(req->accaddr) & SEGMAP;

    if (smentry == UNUSED) return (SEG_INVALID);

    /* Determine type of access */
    if (req->code & SUPER_BIT)
      {
	if (req->code & PROGRAM_BIT) 
	    attempt = SX;	/* Supervisor execute */
	else if (req->code & RW_BIT)
	    attempt = SR;	/* Supervisor read */
	else
	    attempt = SW;	/* Supervisor write */
      }
    else
      {
	if (req->code & PROGRAM_BIT) 
	    attempt = UX;	/* User instruction */
	else if (req->code & RW_BIT)
	    attempt = UR;	/* User read */
	else
	    attempt = UW;	/* User write */
      }

    if ((attempt & permission[PermissionField(smentry)]) == 0)
        return (PROTECTION);

    /* Check page type */
    switch (pgmap(req->accaddr) & PAGETYPE)
      {
      case ONBOARD:
        return (PARITY);

      case NONEXIS:
        return (PAGE_INVALID);

      case MULTMEM:
      case MULTIO:
        return (MB_TIMEOUT);
      }
  }
