/*
 * V Kernel - Copyright (c) 1982 by David Cheriton, Tim Mann
 *
 * Kernel Console support routines for the device server
 *
 * The "console" provides access to the default keyboard and display
 *  for standalone programs.  The association between this logical
 *  device and physical device(s) depends on the workstation's
 *  type and configuration.
 */

#include "../../libc/include/Venviron.h"
#include "../../libc/include/Vserial.h"
#include "../../libc/include/Vteams.h"
#include "../mi/dm.h"
#include "../mi/process.h"
#include "machine.h"
#include "cbuff.h"

#define	BELL '\007'

/* Imports */
extern Process *Map_pid();
extern SystemCode NotSupported();
extern SystemCode SerialCreate();
extern SystemCode SerialRead();
extern SystemCode SerialWrite();
extern SystemCode SerialRelease();
extern DeviceInstance *GetDevice();
extern short WsType;
extern short fbmode();
extern short emt_mayget();

extern Process *Active;

/* Exports */
extern SystemCode ConsoleCreate();
extern SMIKeyboardPoll();  /* called by timer routine */

/* Private */
static SystemCode FbWrite();
static SystemCode FbRelease();
static SystemCode SMIKeyboardRead();
static SystemCode SMIKeyboardRelease();
DeviceInstance *ConsoleInInstance, *ConsoleOutInstance;
	  /* used only if different from serial line 0 */

/* Routines */
SystemCode ConsoleCreate( req, desc )  
    CreateSerialInstanceRequest *req;
    DeviceInstance *desc;
  {
  /*
   * Create instances for the console.  The instance returned is
   *  used for console output.  Instance id + 1 is used for console
   *  input.  Its parameters can be obtained by doing a QueryInstance.
   */
    register CreateInstanceReply *reply = (CreateInstanceReply *) req;
    register DeviceInstance *inst;
    SystemCode code;

    if ( WsType != SMI )
      {
        /* Open serial line 0 for keyboard and possibly display */
        req->linenum = 0;  /* Use line 0 */
        if ( (code = SerialCreate(req, desc)) != OK ) return (code);

        /* Decide whether to use serial line or framebuffer for output,
	 *  based on information from the Sun PROM monitor  */
        switch (fbmode(2))
          {
          case 1:  /* Framebuffer selected */
	    desc->writefunc = FbWrite;
	    break;

          case 0:  /* No framebuffer */
	    break;

          case -1: /* Either framebuffer not selected, or fbmode() didn't work */
	    if (fbmode(1) == 1)
	      {
	        /* fbmode() works--use serial line */
	        fbmode(-1);  
	        break;
	      }
	    else
	      {
	        /* fbmode() doesn't work--look at config register */
	        if ( GetConfig() & (1<<10) )
		    desc->writefunc = FbWrite;
	      }
          }
        return (OK);
      }

    /* If WsType == SMI, the console is unrelated to serial device 0 */

    if (req->filemode != FCREATE) return (MODE_NOT_SUPPORTED);

    if( (inst = ConsoleInInstance) != NULL )
      {
	if( ValidPid( inst->owner ) )  return (BUSY);
	ConsoleInInstance = 0;
	inst->owner = 0;
      }

    if( (inst = ConsoleOutInstance) != NULL )
      {
	if( ValidPid( inst->owner ) )  return (BUSY);
	ConsoleOutInstance = 0;
	inst->owner = 0;
      }

    /*
     * Initialize the device instance descriptors, can assume all fields
     * are zero except for id and first instance's owner.
     */

    /* Output descriptor */
    desc->readfunc = NotSupported;
    desc->writefunc = FbWrite;  /* Assume all SMIs have a framebuffer */
    desc->modifyfunc = NotSupported;
    desc->queryfunc = NotSupported;
    desc->releasefunc = FbRelease;

    desc->type = (WRITEABLE+STREAM);
    desc->blocksize = sizeof(char);  /* assume character by character for now */
    desc->lastblock = -1;	     /* indicate file is empty */
    desc->lastbytes = desc->blocksize;
    ConsoleOutInstance = desc;

    /* Input descriptor */
    desc++;
    desc->owner = Active->pid;
    desc->readfunc = SMIKeyboardRead;
    desc->writefunc = NotSupported;
    desc->modifyfunc = NotSupported;
    desc->queryfunc = NotSupported;
    desc->releasefunc = SMIKeyboardRelease;

    desc->type = (READABLE+STREAM);
    desc->blocksize = sizeof(char);  /* assume character by character for now */
    desc->lastblock = -1;	     /* indicate file is empty */
    desc->lastbytes = desc->blocksize;

    ConsoleInInstance = desc; /* Record the instance for interrupt routine */

    CbuffFlush(CBUFF_SMI);	/* Flush any buffered characters */

    return( OK );

  }


static SystemCode FbWrite( req, desc )
    IoRequest		*req;
    DeviceInstance	*desc;
    /*
     * Handle a write instance request for the frame buffer
     */

  {
    extern Process *Active;
    IoReply *reply = (IoReply *) req;
    register char c;
    extern short WsType;
    register unsigned bytecount;

    bytecount = req->bytecount;
    reply->bytecount = 0;
    if( bytecount > desc->blocksize ) return( BAD_BYTE_COUNT );
    if( bytecount == 0 ) return( OK );
/* Disable block number checking for now */
/*    if( req->blocknumber != desc->lastblock+1 ) return( BAD_BLOCK_NO );*/

    c = req->shortbuffer[0];
    if (c == BELL)
      {
        /* Warning: hack follows */
	if (WsType == CADLINC) 
	    lineput( 0, '\004' );  /* For CadLinc keyboards */
	else if (WsType == STANFORD)
	    lineput( 0, BELL );    /* For terminals */

	/* hack ends */
      }
    else
	emt_putchar( c );

    desc->lastblock++;
    reply->bytecount = 1;

    return( OK );
  }



static SystemCode FbRelease(req, desc)
    IoRequest  *req;
    DeviceInstance *desc;
  {
    desc->owner = 0;
    return (OK);
  }



SMIKeyboardPoll()
  {
    /* Called by timer interrupt if workstation type == SMI */
    register short c;
    register Process *pd;
    register DeviceInstance *desc;

    if ( (desc = ConsoleInInstance) != NULL &&
		(c = emt_mayget()) != -1 )
      {
	if ( (pd = Map_pid(desc->reader)) != NULL )
	  {
	    /* Reader waiting -- give it the character */
	    SMIKeyboardReturnChar(pd->msg, desc, c);
	    Addready( pd );
	  }
	else /* Reader not waiting -- queue the character */
	    CbuffPut(CBUFF_SMI, c);
      }

  }

static SystemCode SMIKeyboardRead( req, desc )
    IoRequest		*req;
    DeviceInstance 	*desc;
/*
 * Handle a read instance request for the SMI keyboard
 */
  {
    extern Process *Active;
    register char c;
    register IoReply *reply = (IoReply *) req;
    register unsigned bytecount;

    bytecount = req->bytecount;
    reply->bytecount = 0;

    /* Reply RETRY if already a waiting reader */
    if ( desc->reader && ValidPid(desc->reader) ) return (RETRY);

    if ( bytecount > desc->blocksize ) return (BAD_BYTE_COUNT);
    if ( bytecount == 0 ) return (OK);
    if ( req->blocknumber != desc->nextblock ) return (BAD_BLOCK_NO);

    disable;
    /* Try to get a character from the buffer */
    if ( (c = CbuffGet(CBUFF_SMI)) != CBUFF_EMPTY )
      {
	SMIKeyboardReturnChar(reply, desc, c);
	return (OK);
      }

    /* Wait for character to come in */
    desc->reader = Active->pid;
    Active->state = AWAITING_INT;

    return (NO_REPLY);
  }

static SMIKeyboardReturnChar(reply, desc, c)
    IoReply		*reply;
    DeviceInstance	*desc;
    char		c;
  {
    reply->replycode = OK;
    reply->shortbuffer[0] = c;
    reply->bytecount = 1;
    desc->nextblock++;
    desc->reader = 0;
  }



static SystemCode SMIKeyboardRelease( req, desc )
    IoRequest  *req;
    DeviceInstance *desc;
  /*
   * Release SMI keyboard (console) instance
   */
  {
    IoReply *reply;
    register Process *pd;

    if( Map_pid(desc->reader) != NULL ) /* Unblock reader process */
      {
	desc->reader = 0;
	reply = (IoReply *) pd->msg;
	reply->replycode = END_OF_FILE;
	reply->bytecount = 0;
	Addready( pd );
      }
    desc->owner = 0;
    return( OK );
  }


CreateStdIo( pd )
    Process *pd;
  /* Create a console device instance for the root team's standard i/o */
  {
    Message msg;
    register CreateInstanceRequest *req = (CreateInstanceRequest *) msg;
    register CreateInstanceReply *reply = (CreateInstanceReply *) msg;
    register RootMessage *rmsg = (RootMessage *) pd->msg;
    Process *oldActive;
    extern Process_id Kernel_Process_Pid;

    req->requestcode = CREATE_INSTANCE;
    req->type = CONSOLE;
    req->filemode = FCREATE;
    req->filenamelen = 0;

    oldActive = Active;
    Active = pd;	/* needed for CreateDevice */
    if (CreateDevice( req ) != OK) Kabort("Error creating std i/o");
    Active = oldActive;

    rmsg->rootflags = STDOUT_APPEND | STDERR_APPEND;
    rmsg->stdinserver = rmsg->stdoutserver = rmsg->stderrserver = 
    	reply->fileserver;
    rmsg->stdinfile = reply->fileid + 1;
    rmsg->stdoutfile = rmsg->stderrfile = reply->fileid;
    rmsg->kernelpid = Kernel_Process_Pid;  /* has nothing to do with stdio */
    rmsg->nameserver = rmsg->contextid = 0;
  }
