/*
 * ----------------------------------------------------------------
 *
 * state.h - the state vector for each user
 * by Bill Nowicki February 1981
 *
 * Copyright (c) 1981 by Stanford University
 *
 *
 * This structure defines the state of each user.
 * note that using unsigned 32 bit integers for byte IDs
 * might fail for streams longer than a few billion characters,
 * which will take about one month of full speed output.
 *
 * ----------------------------------------------------------------
 */

# ifndef MaxUsers
# define MaxUsers 4		/* maximum number of users */
# endif  MaxUsers

# ifndef TicksPerSecond
# define TicksPerSecond 1	/* clock interrupts per second */
# endif  TicksPerSecond

# define IdleTimeout  60		/* idle timeout (in seconds) */

# define ADATATimeout 1			/* timeout for ADATAs (in seconds) */
# define ADATATransmitLimit 25		/* maximum times to retransmit */

# define NAMEREQTimeout 2
# define NAMEREQTransmitLimit 5 	/* ditto for name requests */

# define RFCTimeout 2
# define RFCTransmitLimit 6		/* ditto for connection requests */

# define DallyTimeout 1
# define DallyTransmitLimit 6		/* ditto for dallies! */

# define PURGETimeout 15		/* periodically purge routing table entries */

# define FromBufferSize 32	/* size of buffer from user to network */
# define ToBufferSize 512	/* size of buffer from network to user */

# include <puplib.h>
# include <puppacket.h>
# include <pupconstants.h>

typedef unsigned long ByteID;

enum RTP_states 
   { 
	Closed, 		/* initial condition, ask for a host name */
	ReadingHostName, 	/* read keyboard input */
	LookingUpHostName,	/* waiting for name server to respond */
	RFCOut,			/* sent an RFC, waiting for one back */
	OpenState,		/* normal state */
	EndIn,			/* closed by remote host */
	EndOut,			/* closed by user */
	Dally,			/* waiting for END reply */
	GatewayLookup,		/* looking up telnet-gateway */
	GatewayRFC,		/* set telnet-gateway an RFC */
	GatewayMSG,		/* waiting for the Gateway message LF */
	GatewayHostSent		/* waiting for + or - */
   };

enum ModemStates
  {
  	/*
	 * States in which a modem can be in; scanned a few times a second.
	 */
    Idle,		/* hung up, waiting for a ring */
    Hangup,		/* Ready to disconnect */
    OffHook		/* Off-hook, ready for input */
  };


struct UserState 
 {
   enum RTP_states state;
   enum ModemStates modem;

   unsigned char ImmediateDestination;	/* actually network specific */
 
   ByteID ConnectionID;		/* user number is encoded in this as follows: */
# define GetUser(ID) (ID & 0777)
# ifdef VAX
# define NextConnection(USER) (UserTable[USER].ConnectionID + 01000)
# else  VAX
# define NextConnection(USER) ( (Random<<10)+USER)
# endif VAX
 
   struct Port SourcePort, DestPort;
   
   ByteID InterruptID;			/* ID of last interrupt */

   ByteID ByteID_received;		/* the last byte we received */
   ByteID ByteID_to_user;		/* the last byte sent to the user */

   char BufferToUser [ToBufferSize];
 
   ByteID ByteID_from_user;	/* the last byte we got from the user */
   ByteID ByteID_acked;		/* the last byte acked by remote host */
 
   short TransmitCount, 		/* number of times retransmitted */
         TimeoutCount,		/* counts ticks for timeouts */
         CommandMode;		/* true if last character was telnet escape */
 
   char BufferFromUser [FromBufferSize];
   char FlagsFromUser [FromBufferSize];		/* Right now only has Data/Mark flag */
  } 
 UserTable [ MaxUsers ];


#define Gateway		GatewayLookup		/* define for telnet-gateway */

#define BytesLeft (ToBufferSize - (thisUser->ByteID_received - thisUser->ByteID_to_user) )

#define TIPCTRL		0237
#define TIPSTAT		0240
#define MARK_CODE	0200
#define TIMING_MARK	5
#define TIMING_REPLY	6

struct PupPacket PacketBuffer;

#define NetQueueSize 64		/* number of entries in the network queue */

struct NetQueueEntry
 {
  uchar UserNumber;
  uchar PupPacketType;
 } NetQueue[ NetQueueSize ];

#define MaxLines MaxUsers
   char Line_Ready[MaxLines];

int NetQueueIn, NetQueueOut;	/* pointers into NetQueue */

int ClockTick();
