/*
 * ------------------------------------------------
 *
 * mut.c - multi-user PUP telnet
 * by Bill Nowicki, March 1981
 *
 * Copyright (c) 1981 by Stanford University
 *
 * Special version for use with VGTS
 *
 * ------------------------------------------------
 */

# include "state.h"
# include "chars.h"
# include <ctype.h>
# include <puproute.h>

 /*
  * the following macro puts one character on the user's network queue,
  * With the flags as indicated (0=data, MARK_CODE=MARK)
  */

# define PutChar(character, code)  thisUser -> BufferFromUser[ (thisUser->ByteID_from_user) & (FromBufferSize-1) ] \
     = character; \
 thisUser -> FlagsFromUser[ (thisUser->ByteID_from_user++) & (FromBufferSize-1) ] \
      = code;

# define MaxPupBytes 512		/* a large number of bytes in a PUP */

static short NumberOfUsers = MaxUsers;
static short PurgeTimeoutCount = 0;
static long Random, Time;		/* clock ticks for random #s and time */

static struct Port ControlPort = {0, 0, 0};
static struct Port NameLookupPort = { 0, 0, MISCSERVICES};
static char closedMessage[] = "\r\n[Connection closed by remote host]\r\n";
static char abortMessage1[] = "\r\n[Connection aborted - ";
static char abortMessage2[] = "]\r\n";

extern Net OurNetNumber;
 
InitializeMUT( numberOfUsers )
 {
  /*
   * Initialize everything in MUT
   */
   
   int userNumber;
   register struct UserState *thisUser = UserTable;   
   
   NumberOfUsers = numberOfUsers;
   PurgeTimeoutCount = 0;
   Time = 0;
   Random = emt_ticks();

   for ( userNumber = 0; userNumber < NumberOfUsers; userNumber++, thisUser++)
     {
      thisUser -> ConnectionID = userNumber + ( 1 << 24);
      thisUser->modem = Idle;
      NewConnection( userNumber );
     }
     
   InitRoutingTable();
   NetQueueIn = NetQueueOut = 0;
 }

      
NewConnection( userNumber )
 {
  register struct UserState *thisUser = UserTable+userNumber;
  register int connID = NextConnection(userNumber);
  static char welcome[] = "\r\n\r\nWelcome to SUNet. Enter host name: ";
  
  thisUser -> ConnectionID     =
  thisUser -> InterruptID      = 
  thisUser -> ByteID_received  = 
  thisUser -> ByteID_to_user   = 
  thisUser -> ByteID_from_user = 
  thisUser -> ByteID_acked     = connID;
  thisUser -> SourcePort.host  = GetHost();
  thisUser -> SourcePort.net   = GetNet();
  thisUser -> SourcePort.socket= connID;
  thisUser -> state = ReadingHostName;
  thisUser -> TransmitCount = 
  thisUser -> TimeoutCount  = 
  thisUser -> CommandMode   = 0;
  
  MutSetBanner(userNumber,"");
  UserEnqueue( BSP_DATA,  userNumber, welcome, sizeof(welcome) - 1);
 }


UserRead( userNumber, character ) 
 int userNumber;
 char character;
{
   /*
    * a character has been read from a user 
    */
 register struct UserState *thisUser = UserTable+userNumber;
 static char deleteSequence[] = "\b \b";
 static char crlf[] = "\r\n";
 static char connectionClosed[] = "\r\n[Connection Closed]\r\n";

 switch ( thisUser -> state)
  {
   case Closed:
   case ReadingHostName:
	character &= 0177;
	if (thisUser->CommandMode)
	  switch (character)
	    {
	      case 'e':
	      case 'E':
	 	ExitMUT();
	 	return;

	      case 'd':
	      case 'D':
	        DeleteMUT(userNumber);
	        return;

	      case 'o':
	      case 'O':
	     	OpenSmall();
		thisUser->CommandMode = 0;
	    	return;

	      case 'b':
	      case 'B':
	     	OpenBig();
		thisUser->CommandMode = 0;
	    	return;
	    }
	switch (character) 
        {
	 case BS:
	 case DEL:
	   if (thisUser->ByteID_acked == thisUser->ByteID_from_user) break;
	   UserEnqueue( BSP_DATA,  userNumber, deleteSequence, sizeof(deleteSequence) );
	   thisUser->ByteID_from_user--;
	   break;

	 case NAK:
	 case ETB:
	   while (thisUser->ByteID_acked != thisUser->ByteID_from_user) 
	      {
	       UserEnqueue( BSP_DATA,  userNumber, deleteSequence, sizeof(deleteSequence) );
	       thisUser->ByteID_from_user--;
	      }
	   break;
	   
	 case CR:
	 case LF:
	   if (thisUser->ByteID_acked == thisUser->ByteID_from_user)
	     {
	          /*
		   * Don't bother connecting if we have a null host name
		   */
	       thisUser -> state = Closed;
	     }
	    else {
	    	     /*
		      * here we actually open the connection, starting
		      * with a name lookup request
		      */
         	   UserEnqueue( BSP_DATA,  userNumber, crlf, sizeof(crlf)-1 );
         	   thisUser -> state = LookingUpHostName;
		   thisUser -> TransmitCount = 0;
		   thisUser -> DestPort = NameLookupPort;
	           NetEnqueue( userNumber, NAMEREQ );
		 }
	   break;

	 case EOT:
	   thisUser->modem=Hangup;
	   break;
	   
	 case TELNET_ESCAPE:
	    break;

	 default:	
	      /*
	       * echo other random characters
	       */
	   UserEnqueue( BSP_DATA,  userNumber, &character, 1);
           PutChar(character, 0);
	} /* character switch */
     thisUser->CommandMode = character==TELNET_ESCAPE;
     break;

   case LookingUpHostName:
   case RFCOut:
       /*
        * throw away characters we get while looking up the host name
	*/
	character &= 0177;
	if (thisUser->CommandMode)
        switch ( islower(character) ?  toupper(character) : character)
	 {
	  case 'C':
	     thisUser -> state = EndOut;
	     return;

	  case 'E':
	     ExitMUT();
	     return;

	  case 'D':
	     thisUser -> state = EndOut;
	     DeleteMUT(userNumber);
	     return;

	  case 'O':
	     OpenMUT();
	     thisUser->CommandMode = 0;
	     return;

	  case 'B':
	     OpenBig();
	     thisUser->CommandMode = 0;
	     return;

	  default:
	     thisUser->CommandMode = 0;
	     return;
	 }
      thisUser->CommandMode = (character == TELNET_ESCAPE);
      break;

   case OpenState:
   case GatewayHostSent:
        /*
	 * Decode the local commands, or just queue up a character
	 */
      if (thisUser->CommandMode)
       {
	character &= 0177;
        switch ( (int)character)
	 {
	  case 'd':
	  case 'D':
	     DeleteMUT(userNumber);

	  case 'c':
	  case 'C':
	     thisUser -> state = EndOut;
             NetEnqueue( userNumber, RTP_END);
	     UserEnqueue( BSP_DATA,  userNumber, connectionClosed, sizeof(connectionClosed) );
	     return;

	  case TELNET_ESCAPE:
	     PutChar( character, 0 );
	     thisUser->CommandMode = 0;
             NetEnqueue( userNumber, BSP_ADATA);
	     return;

	  case 'e':
	  case 'E':
	     ExitMUT();
	     return;

	  case 'b':
	  case 'B':
	     OpenBig();
	     thisUser->CommandMode = 0;
	     return;

	  case 'o':
	  case 'O':
	     OpenMUT();
	     thisUser->CommandMode = 0;
	     return;
	  }
	}
      if ( ! (thisUser->CommandMode = (character&0177)==TELNET_ESCAPE) )
        {
          PutChar( character, 0 );
          NetEnqueue( userNumber, BSP_ADATA);
        }
      break;

   case EndIn:
   case Dally:
     break;
  } /* state switch */
}


UserPutChar( userNumber, character ) 
 int userNumber;
 char character;
{
   /*
    * Similar to UserRead, but do not enqueue up a packet
    */
 register struct UserState *thisUser = UserTable+userNumber;

 if (thisUser->state == OpenState)
  {
     PutChar( character, 0 );
  }
 else
     UserRead( userNumber, character );
}


UserEnqueue( markFlag, userNumber, string, count) 
 unsigned char markFlag;
 int userNumber;
 char *string;
 int count;
{
   /*
    * enqueue a character string for userNumber.
    * If the output buffer is empty, 
    * Actually output the first character (reply if mark byte)
    */
 register struct UserState *thisUser = UserTable+userNumber;
 register char *buffer = thisUser -> BufferToUser;
 register int  charCount;
 register char *from = string;

 if (count==0) return;

 if ( markFlag == BSP_AMARK ) 
     {
       /*
        * handle a Mark byte - send a timing reply back
	*/
       thisUser->ByteID_received++;
       thisUser->ByteID_to_user++;

       if ( (*from &= ~MARK_CODE) == TIMING_MARK )
         {
           PutChar(TIMING_REPLY, MARK_CODE);
           NetEnqueue( userNumber, BSP_ADATA);
	 }
       return;
     }

 for ( charCount=count; charCount; charCount-- )
  {
   thisUser->ByteID_received++;
   thisUser->ByteID_to_user++;
   TelnetPutChar( *from++, userNumber );
  }
 TelnetFlush(userNumber);
}



NetRead( pupPacket ) 
 struct PupPacket *pupPacket;
{
   /*
    * a packet has been read from the network 
    */
 register struct PupPacket *packet = pupPacket;
 register struct UserState *thisUser;
 struct Port destPort;
 short userNumber, pupsum, checksum();
 static char badNews[] = "[Name not found]\r\n";
 register unsigned int length = packet->PupLength - PUPACKOVER;
 int roundlen = roundup( length );

 if (length<0 || length>1024) return;
 pupsum = * (short *)(packet->PupData+roundlen);
 if ( pupsum != checksum( packet, roundlen + PUPACKOVER - 2)
   && pupsum != -1) return; 

 if (packet->PupType == TIPCTRL)
   {
	/*
	 * Handle "TIP control" packets
	 */
      PortUnpack( &packet->PupSrc, &ControlPort);
      NetEnqueue( 0, TIPSTAT );
      return;
   }
 
 PortUnpack( &packet->PupDst, &destPort);

 if (packet->PupType == GWINFREP)
   {
    struct Port sourcePort;
	/*
	 * update information in the Routing table using received info 
	 */
	if (OurNetNumber==0) SetNetNumber(destPort.net);
	PortUnpack( &packet->PupSrc, &sourcePort);
	UpdateRoutingTable( RoutingTable, &ConnectedNets, packet->PupData, 
	       packet->PupLength - PUPACKOVER, &sourcePort);
     return;
   }
 
 userNumber = GetUser( destPort.socket );

 if ( userNumber < 0 || userNumber >= NumberOfUsers ) 
  { 
    /* printf("Bad user number: %d\r\n", userNumber); */
     return;
  }


 thisUser = UserTable + userNumber;

 if ( thisUser -> ConnectionID != destPort.socket) return; 
 
 /*
  * possibly we should check the whole source and destination ports
  */

 switch ( thisUser -> state )
  {
   case Closed:
   case ReadingHostName:
      break;

   case LookingUpHostName:
   case GatewayLookup:
      switch ( packet->PupType)
        {
	 case NAMERESP:
	   PortUnpack( packet->PupData, &thisUser->DestPort);
	   if (thisUser->DestPort.socket==0) 
	   	thisUser->DestPort.socket = TELNET;

	   if (OurNetNumber==0) SetNetNumber(destPort.net);
	   if ( puproute( &thisUser->DestPort,
	           &thisUser->ImmediateDestination, NULL ) )
	       {
	          static char badNews[] = 
		  	"\r\n[You can't get there from here]\r\n";
		  UserEnqueue( BSP_DATA,userNumber,badNews,sizeof(badNews)-1);
		  thisUser->state = Closed;
	       }
	   else 
	       {
	          NetEnqueue( userNumber, RFC);
	          thisUser -> state = thisUser->state==GatewayLookup ? 
	             GatewayRFC : RFCOut;
		}
	   break;

	 case LOOKUPERR:
# ifdef Gateway
	   if (OurNetNumber==0) SetNetNumber(destPort.net);
           if ( thisUser -> state == LookingUpHostName)
	     {
    	         /*
    	          * If it is not a valid PUP host name, try the Arpanet gateway
    	          */
    	       thisUser -> state = GatewayLookup;
     	       thisUser -> TransmitCount = 0;
    	       thisUser -> DestPort = NameLookupPort;
	       NetEnqueue( userNumber, NAMEREQ );
	     }
	   else
# endif Gateway
            {
 	       UserEnqueue( BSP_DATA,  userNumber, badNews, sizeof( badNews )-1 );
	       thisUser -> state = Closed;
	     }
	   break;

	} /* switch on Pup Type */
      break;

   case RFCOut:
   case GatewayRFC:
      switch ( (int)  packet->PupType )
        {
	  case RFC:
  	    PortUnpack( packet->PupData, &thisUser->DestPort);
	    if ( thisUser->state==GatewayRFC )
	      {
	           /*
		    *  Now we execute the MRC telnet gateway protocol.
		    *  first step is to wait for a message coming back.
		    */
	         thisUser->state = GatewayMSG;
	      }
	     else
	      {
	         thisUser -> ByteID_from_user = thisUser->ByteID_acked = thisUser ->ConnectionID;
	         /* printf("Connected to host %o, socket %O\r\n",
	            thisUser->DestPort.host, thisUser->DestPort.socket); */
	         thisUser -> state = OpenState;
	      }
	    thisUser -> ByteID_received  = thisUser->ByteID_to_user = thisUser ->ConnectionID;
	    NetEnqueue( userNumber, BSP_ACK);
	    break;

	  case RTP_ABORT:
	    UserEnqueue( BSP_DATA,  userNumber, abortMessage1, sizeof(abortMessage1) - 1);
	    UserEnqueue( BSP_DATA,  userNumber, packet->PupData+2, 
	           packet->PupLength - PUPACKOVER - 2);
	    UserEnqueue( BSP_DATA,  userNumber, abortMessage2, sizeof(abortMessage2) - 1 );
	    thisUser -> state = Closed;
	    break;

	} /* switch on Pup Type */
      break;
    
   case GatewayMSG:
   case GatewayHostSent:
   case OpenState:
     {
      register int pupID  = getlong( packet->PupID );

      switch ( packet->PupType )
        {
	 case BSP_ADATA:
	 case BSP_DATA:
	    if ( pupID != thisUser->ByteID_received || length > BytesLeft ) 
	       {
	          /* printf( "Threw away bad ADATA, PupID=%O,  received=%O\r\n",
		    pupID, thisUser->ByteID_received); */
	       }
	      /*
	       * This is the usual operation here
	       * put the characters from the packet into 
	       * the buffer for the user
	       */
            else switch ( thisUser->state )
	        {
		  case GatewayMSG:
		         {
			   register char *pd = packet->PupData;

			   thisUser -> ByteID_received += length;
                           thisUser->ByteID_to_user = thisUser -> ByteID_received;
			   for ( ; pd <= packet->PupData + length;)

			      if (*pd++ == LF ) 
			        {
			          thisUser->state = GatewayHostSent;
				  PutChar( CR, 0 );
				  NetEnqueue( userNumber, BSP_ADATA );
				}
			   break;
			 }

		  case GatewayHostSent:
		         {
			   register char *pd = packet->PupData;

			   for ( ; pd <= packet->PupData + length;pd++)
			      if (*pd == '+' || *pd == '-' ) 
			        {
			          thisUser->state = OpenState;
				  *pd = ' ';
	                          UserEnqueue( BSP_DATA,  userNumber, packet->PupData, length);
				}
			   break;
			 }

		  case OpenState: UserEnqueue( BSP_DATA,  userNumber, packet->PupData, length);
		}
	    if ( thisUser->ByteID_from_user==thisUser->ByteID_acked)
	     {
	        thisUser -> TimeoutCount  = 
	        thisUser -> TransmitCount = 0;
	     }
	    if ( packet->PupType == BSP_ADATA )
	      NetEnqueue( userNumber, BSP_ACK);
	    break;
	    
	 case BSP_AMARK:
	 case BSP_MARK:
	    if ( pupID == thisUser->ByteID_received && BytesLeft > 0 )
	       {
	         packet->PupData[0] |= MARK_CODE;
	         UserEnqueue( BSP_AMARK,  userNumber, packet->PupData, length);
	       }
	    if ( packet->PupType == BSP_AMARK )
	      NetEnqueue( userNumber, BSP_ACK);
	    break;

	 case BSP_ACK:
	    if ( (pupID - thisUser->ByteID_acked) < 0)
	       {
	         /* printf("Got a strange ACK, off by %d\r\n", 
		       pupID - thisUser->ByteID_acked ); */
	       }
	     else {
	            thisUser->ByteID_acked = getlong( packet->PupID );
		    thisUser->TransmitCount = 0;
	          }
            break;
	    
	 case BSP_INTR:
	    if ( (thisUser -> InterruptID - pupID) >  1) break; 
	    thisUser -> InterruptID = pupID; 
	    NetEnqueue( userNumber, BSP_INTRR);
	    break; 

	 case RTP_ABORT:
	    if ( pupID != thisUser->ConnectionID) break;
	    UserEnqueue( BSP_DATA,  userNumber, abortMessage1, sizeof(abortMessage1) - 1 );
	    UserEnqueue( BSP_DATA,  userNumber, packet->PupData+2, length - 2);
	    UserEnqueue( BSP_DATA,  userNumber, abortMessage2, sizeof(abortMessage2) - 1);
	    thisUser->state = Closed;
	    break;

	 case RTP_END:
	    if ( pupID != thisUser->ConnectionID) break;
	    UserEnqueue( BSP_DATA,  userNumber, closedMessage, sizeof(closedMessage) - 1 );
	    thisUser -> state = Dally;
	    thisUser -> TransmitCount = 0;
	    NetEnqueue( userNumber, RTP_ENDR);
	    break;
	} /* switch on Pup Type */
      break;
     }

   case EndOut:
      switch ( (int)  packet->PupType )
        {
	 case RTP_END:
	 case RTP_ENDR:
           NetEnqueue( userNumber, RTP_ENDR);
           thisUser -> state = Closed;
	   break;
	} /* switch on Pup Type */
      break;

   case Dally:
      switch ( (int)  packet->PupType )
        {
	 case RTP_ENDR:
	    thisUser-> state = Closed;
	    break;
	 case RTP_END:
	    NetEnqueue( userNumber, RTP_ENDR);
	    break;
	} /* switch on Pup Type */
      break;

  } /* state switch */
}

NetEnqueue( userNumber, packetType)
 int userNumber;
 uchar packetType;
 {
    /*
     * send a packet of the given type for the given user 
     * (or queue it up if ethernet is busy)
     *
     */
   int empty;
   
  empty = ( NetQueueIn == NetQueueOut );

  NetQueue[ NetQueueIn  ].UserNumber  = userNumber;
  NetQueue[ NetQueueIn++].PupPacketType     = packetType;

  NetQueueIn %= NetQueueSize;
  if ( empty )
    NetPacket( userNumber, packetType);
 }


NetDone() 
{
  /*
   * the network has sent a packet and is ready for another 
   * if the net queue is non-empty, send another one!
   */
   
  if (NetQueueOut == NetQueueIn) return;

  NetQueueOut++;  
  NetQueueOut %= NetQueueSize;
  
  if (NetQueueOut != NetQueueIn)
       NetPacket( NetQueue[NetQueueOut].UserNumber, NetQueue[NetQueueOut].PupPacketType);
}


NetPacket( userNumber, pupType )
 int userNumber;
 uchar pupType;
{
 /*
  * Format a packet for the given user and  send it on the network
  */
  
  register struct UserState *thisUser = UserTable + userNumber;
  
  PacketBuffer.PupType = pupType;
  PacketBuffer.PupTransport = 0;
  PacketBuffer.PupLength = PUPACKOVER;
  
  PortPack( &thisUser->SourcePort, &PacketBuffer.PupSrc);
  PortPack( &thisUser->DestPort,   &PacketBuffer.PupDst);
 
  /* printf("Sending packet type 0%o for user %d\r\n", pupType, userNumber); */
  switch ( (int) pupType)
   {
    case RFC:
        PacketBuffer.PupID = makelong(thisUser->ConnectionID);
        PortPack( &thisUser->SourcePort, PacketBuffer.PupData);
        PacketBuffer.PupLength += sizeof( PacketBuffer.PupDst );
        break;

    case NAMEREQ:
    	  /* 
	   * broadcast a name lookup request
	   */
        thisUser->ImmediateDestination = 0;
        if (thisUser -> state == GatewayLookup)
	   {
	      /*
	       * Here we lookup the name of the telnet-gateway port
	       */
            register char *pupData = PacketBuffer.PupData;
	    static char telnetGateway[] = "Telnet-Gateway";
	    register char *tgp = telnetGateway;

            while (*tgp) *pupData++ = *tgp++;
            PacketBuffer.PupID      = makelong(thisUser->ByteID_acked);
	    PacketBuffer.PupLength += sizeof(telnetGateway)-1;
	    break;
	   }

    case BSP_ADATA:
      {
        register char *pupData = PacketBuffer.PupData;
	register int counter;

        PacketBuffer.PupID = makelong(thisUser->ByteID_acked);

	for ( counter=thisUser->ByteID_acked; counter<thisUser->ByteID_from_user;)
	  { 
	    register char character = thisUser-> BufferFromUser [ counter 
	      & (FromBufferSize-1) ];
	    
	    if (thisUser-> FlagsFromUser [ counter & (FromBufferSize-1) ]
	            & MARK_CODE ) 
	      {
	        if (counter==thisUser->ByteID_acked)
		   {
		     PacketBuffer.PupType = BSP_AMARK;
		     *pupData = character;     
		     counter++;
		   }
	        break;
	      }
	    *pupData++ = character;
	    counter++;
	  }
	if (pupType==NAMEREQ)
	  {
	    *pupData++ = 0;
	    MutSetBanner(userNumber,PacketBuffer.PupData);
	  }
	PacketBuffer.PupLength += counter - thisUser->ByteID_acked;
	/* printf("packet had %d data bytes\r\n", PacketBuffer.PupLength - PUPACKOVER);*/
        break;
      }

    case BSP_ACK:
     {
      register short *allocation = (short *) (PacketBuffer.PupData);
/***  allocation[ 2 ] = BytesLeft;
      allocation[ 1 ] = (allocation[2]+MaxPupBytes-1)/MaxPupBytes;
 ****/
      allocation[ 2 ] = 1024;
      allocation[ 1 ] = 4;
      allocation[ 0 ] = MaxPupBytes;
      PacketBuffer.PupID = makelong(thisUser->ByteID_received);
      PacketBuffer.PupLength += 6;
      break;
     }

    case RTP_END:
    case RTP_ENDR:
      PacketBuffer.PupID = makelong(thisUser->ConnectionID);
      break;

    case BSP_INTRR:
      PacketBuffer.PupID = makelong(thisUser -> InterruptID++);
      break;
      
     case TIPSTAT:
	/*
	 * return a "Tip Status" packet.
	 * This is not associated with any user.
	 */
	SendStatus( &PacketBuffer );
	return;
   } /* switch on pup Type */
   
   thisUser -> TimeoutCount = 0;
   
   NetWrite( &PacketBuffer, thisUser -> ImmediateDestination );
}


SendStatus( pup )
 register struct PupPacket *pup;
  {
    unsigned char immediateDestination;
    register char *p = pup->PupData;
    int userNumber;
    register struct UserState *thisUser = UserTable;   

    puproute( &ControlPort, &immediateDestination );
    PortPack( &ControlPort, &pup->PupDst );
    
    for (userNumber=0; userNumber < NumberOfUsers; userNumber++, thisUser++)
      {
      	  /*
	   * Pack in all the user's state information until we run out of room.
	   * This packing is very machine dependent.
	   */
        *(enum RTP_states *)p = thisUser->state;  p+= sizeof(enum RTP_states);
        PortPack( &thisUser->DestPort, p); p += sizeof( pup->PupDst );
	pup->PupLength += sizeof( pup->PupDst ) + sizeof(enum RTP_states);
	if (pup->PupLength > 500) break;
      }
    NetWrite( pup, immediateDestination );
  }


ClockTick()
  {
   /*
    * this routine is called once on every clock tick 
    */
    
   int userNumber;
   register struct UserState *thisUser = UserTable;   
 
   Time++; Random++;

   if ( PurgeTimeoutCount++ > PURGETimeout*TicksPerSecond )
       {
       /***   PurgeRoutingTable( RoutingTable, &ConnectedNets );
         ***** Uncomment when it works!!! ******/
          PurgeTimeoutCount = 0;
       }

   for (userNumber=0; userNumber < NumberOfUsers; userNumber++, thisUser++)
    switch ( (int) thisUser->state) 
     {
      case Closed:
         if (thisUser->ByteID_to_user == thisUser->ByteID_received)
	     NewConnection( userNumber );
 	 break;

      case ReadingHostName:
         break;

      case LookingUpHostName:
      case GatewayLookup:
         if (++thisUser->TimeoutCount < NAMEREQTimeout*TicksPerSecond) break;
	 thisUser->TimeoutCount = 0;
         if (thisUser->TransmitCount++ > NAMEREQTransmitLimit)
	    {
	    	/*
		 * Egad, no name server?  Probably the hardware bug
		 */
	      static char badnews[] = "[Ethernet interface failed - reinitializing]\r\n";
	      UserEnqueue( BSP_DATA,  userNumber, badnews, sizeof(badnews) - 1 );
	      thisUser -> state = Closed;
	      NetReset();
	    }
          else  NetEnqueue( userNumber, NAMEREQ);
	 break;

      case GatewayMSG:
          break;


      case RFCOut:
      case GatewayRFC:
         if (++thisUser->TimeoutCount < RFCTimeout*TicksPerSecond) break;
	 thisUser->TimeoutCount = 0;
         if (thisUser->TransmitCount++ > RFCTransmitLimit)
	    {
	      static char badnews[] = "[Host does not respond]\r\n";
	      static char badgate[] = "[No such Host or Telnet-gateway]\r\n";
	      if (thisUser->state==GatewayRFC)
	            UserEnqueue( BSP_DATA,  userNumber, badgate, sizeof(badgate) - 1 );
	       else UserEnqueue( BSP_DATA,  userNumber, badnews, sizeof(badnews) - 1 );
	      thisUser -> state = Closed;
	    }
          else if (thisUser->state!=GatewayMSG) NetEnqueue( userNumber, RFC);
	  break;

      case OpenState:
      case GatewayHostSent:
         if (++thisUser->TimeoutCount < 
	      ( (thisUser->ByteID_from_user == thisUser->ByteID_acked &&
	         ( thisUser->TransmitCount == 0 || 
		   thisUser->ByteID_received != thisUser->ByteID_to_user) ) ?
	        IdleTimeout*TicksPerSecond : ADATATimeout*TicksPerSecond) )
	         break;
	 thisUser->TimeoutCount = 0;
         if (thisUser->TransmitCount++ > ADATATransmitLimit)
	    {
   	      static char badnews[] = "[Host seems to have crashed]\r\n";
	      UserEnqueue( BSP_DATA,  userNumber, badnews, sizeof(badnews) - 1 );
	      thisUser -> state = Closed;
	    }
          else NetEnqueue( userNumber, BSP_ADATA);
	 break;

      case EndOut:
         if (++thisUser->TimeoutCount < DallyTimeout*TicksPerSecond) break;
	 thisUser->TimeoutCount = 0;
         if (thisUser->TransmitCount++ > DallyTransmitLimit)
	    thisUser->state = Closed;
         NetEnqueue( userNumber, RTP_END);
	 break;

      case Dally:
         if (++thisUser->TimeoutCount < DallyTimeout*TicksPerSecond) break;
	 thisUser->TimeoutCount = 0;
         if (thisUser->TransmitCount++ > DallyTransmitLimit)
	    thisUser-> state = Closed;
         NetEnqueue( userNumber, RTP_ENDR);
	 break;
     }
  }


/*
 *
 * Pup Routing Table routines
 *
 * Stolen from Jeffrey Mogul @ Stanford	20 July 1981
 *
 * The algorithms in this module are based upon a Xerox memo
 * entitled "Gateway Information Protocol (revised)", by Ed
 * Taft, May 30, 1979, filed on <Pup>GatewayInformation.press.
 */

/*
 * This should be meant merely to get us off the ground; once on the air,
 * the "right" thing to do is to update this by listening to broadcast
 * GatewayInfo Pups, and to purge entries more than RTETIMEOUT seconds
 * old.
 * 
 * However, this will work as a static router if every generator of
 * GatewayInfo pups sends a true routing table.  Until it does, this
 * should probably NOT be a broadcast request, but instead should go to
 * the properly working gateways.
 */

InitRoutingTable()
  {
    struct PupPacket packetBuffer;
    register struct PupPacket *packet = &packetBuffer;
    static struct Port gatewayPort = { 0, 0, GATEWAYINFO};

    packet->PupType = GWINFREQ;
    packet->PupLength = PUPACKOVER;

    PortPack( &gatewayPort, &packet->PupDst );
    PortPack( &UserTable[0].SourcePort, &packet->PupSrc );

    NetWrite( packet, 0);
}


SetNetNumber(n)
  {
  	/*
	 * sets the global OurNetNumber to be the indicated value.
	 *  Also sets it in each user's source port.
	 */
   register int userNumber;
   register struct UserState *thisUser = UserTable;   
      
   OurNetNumber = n;
   for ( userNumber = 0; userNumber < NumberOfUsers; userNumber++, thisUser++)
     {
       thisUser -> SourcePort.net = n;
     }
  }
