/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, 8/83
 *	Added line editing, put in a separate file.
 * Tim Mann, 9/8/83
 *	Don't self-insert control characters.
 * Tim Mann, 9/16/83
 *	Fixed EOF check, deleted useless capitalization features
 */

/*
 * GetLine:
 * Returns next input line.  Performs line editing ala emacs-style.
 * If inputLineLength > 0 then the line is displayed to allow it to be edited.
 *
 * Returns -1 if EOF, or the number of characters in the line.
 *
 * At most maxLineLength characters will be handled.  Overflow characters are
 * lost.
 */


#include <Vio.h>
#include <Vgts.h>
#include "sh.h"


#define DEL 127
#define BS 8
#define ESC 27

#define CTRL_A 1
#define CTRL_B 2
#define CTRL_D 4
#define CTRL_E 5
#define CTRL_F 6
#define CTRL_K 11
#define CTRL_T 20
#define CTRL_U 21
#define CTRL_W 23

#define ESC_B 226
#define ESC_C 227
#define ESC_D 228
#define ESC_F 230
#define ESC_H 232
#define ESC_L 236
#define ESC_U 245


/*
 * NOTE:
 * These macros assume that the character passed in is resp. a lower (upper)
 * case alphabetic letter.
 */
#define UpperCaseLetter(c) (c - 'a' + 'A')
#define LowerCaseLetter(c) (c - 'A' + 'a')


int GetLine(inputLine, inputLineLength, maxLineLength,
	printFlag, inFile, outFile, l)
    register char *inputLine;
    int inputLineLength, maxLineLength;
    int printFlag;
    File *inFile, *outFile;
    LineRec *l;
 {
    int ch, i;

    ModifyPad(outFile, CR_Input+LF_Output);
				/* Set input cooking to that expected by
				   this routine. */

    /* Initialize data structures. */
    l->line = inputLine;
    l->lineLength = maxLineLength;
    l->printChars = printFlag;
    l->in = inFile;
    l->out = outFile;
    l->escFlag = 0;

    /* If line is non-empty, then display it for the user to edit. */
    l->cursor = 0;
    if (inputLineLength > 0)
      {
	l->endLine = inputLineLength;
	PrintLine(0, l);
	EndOfLine(l);
      }
    else
      {
	l->endLine = 0;
      }

    /* Read user's input and act appropriately. */
    while (1)
      {
        Flush(l->out);
        ch = getc(l->in);
	if (ch == EOF)
	  {
	    if (l->in->lastexception != END_OF_FILE) 
		fprintf(l->out, "Error on read: %s, state=0x%x\n", 
	  		ErrorString(l->in->lastexception), l->in->state );
	    return(-1);
	  }
	if (l->escFlag)
	  {
	    l->escFlag = 0;
	    ch = ch | 0x80;
	  }
	if ((ch == '\r') || (ch == '\n'))
	  {
	    break;
	  }
	switch (ch)
	  {
	    case DEL:
	    case BS:
	        BackSpace(l);
	        break;
	    case CTRL_A:
	        BeginningOfLine(l);
	        break;
	    case CTRL_B:
	        BackwardChar(l);
	        break;
	    case CTRL_D:
	        DeleteChar(l);
	        break;
	    case CTRL_E:
	        EndOfLine(l);
	        break;
	    case CTRL_F:
	        ForwardChar(l);
	        break;
	    case CTRL_K:
	        DeleteLine(l);
	        break;
	    case CTRL_T:
	        ReverseChars(l);
	        break;
	    case CTRL_U:
	        DeleteEntireLine(l);
	        break;
	    case CTRL_W:
	    case ESC_H:
	        BackwardsDeleteWord(l);
	        break;

	    case ESC:
		l->escFlag = 1;
		break;
	    case ESC_B:
	        BackwardWord(l);
		break;
	    case ESC_D:
	        DeleteWord(l);
		break;
	    case ESC_F:
	        ForwardWord(l);
		break;
#ifdef DELETED
	    case ESC_C:
	        CapitalizeWord(l);
		break;
	    case ESC_L:
	        LowerCaseWord(l);
		break;
	    case ESC_U:
	        UpperCaseWord(l);
		break;
#endif DELETED

	    default:
		if (ch >= ' ' && ch < DEL)
		    SelfInsert(ch,l);
		break;
	  }
      }

    l->line[l->endLine] = '\0';
    fprintf(outFile,"\r\n");
    Flush(outFile);
    return(l->endLine);
  }




/*
 * WordChar:
 * Returns 1 if ch is a character that can be part of a word, otherwise 0.
 */

int WordChar(ch)
    char ch;
  {
    if (((ch >= 'a') && (ch <= 'z')) ||
        ((ch >= 'A') && (ch <= 'Z')) ||
	(ch == '_'))
      {
	return(1);
      }
    else
      {
	return(0);
      }
  }


/*
 * AtStartOfWord:
 * Returns 1 if the cursor is at the start of a word.
 */

int AtStartOfWord(l)
    LineRec *l;
  {
    if (!WordChar(l->line[l->cursor]))
      {
	return(0);
      }
    if ((l->cursor == 0) || (!WordChar(l->line[l->cursor-1])))
      {
	return(1);
      }
    else
      {
	return(0);
      }
  }


/*
 * PrintLine:
 */

PrintLine(col, l)
    int col;
    LineRec *l;
  {
    int cnt = 0;

    if (l->printChars)
      {
        /* Print the line. */
        while (col < l->endLine)
          {
	    cnt++;
	    putc(l->line[col++], l->out);
          }
	/* Back up the cursor to its original pos. */
	while (cnt--)
	  {
	    putc('\b', l->out);
	  }
      }
  }


/*
 * SelfInsert:
 */

SelfInsert(ch, l)
    char ch;
    LineRec *l;
  {
    int i;

    /* If the input line is already of max. length then discard the last 
       char. */
    if (l->endLine >= l->lineLength)
      {
	l->endLine--;
      }

    for (i = l->endLine-1; i >= l->cursor; i--)
      {
	l->line[i+1] = l->line[i];
      }
    l->line[l->cursor] = ch;
    l->endLine++;
    PrintLine(l->cursor, l);
    ForwardChar(l);
  }


/*
 * BackSpace:
 */

BackSpace(l)
    LineRec *l;
  {
    if (l->cursor > 0)
      {
        BackwardChar(l);
	DeleteChar(l);
      }
  }


/*
 * DeleteChar:
 */

DeleteChar(l)
    LineRec *l;
  {
    int i;

    if (l->cursor < l->endLine)
      {
	for (i = l->cursor; i < l->endLine; i++)
	  {
	    l->line[i] = l->line[i+1];
	  }
	l->line[l->endLine-1] = ' ';	/* This will ensure that the character
				   on the display will get erased. */
	PrintLine(l->cursor, l);
	l->endLine--;
      }
  }


/*
 * DeleteLine:
 */

DeleteLine(l)
    LineRec *l;
  {
    while (l->endLine > l->cursor)
      {
	DeleteChar(l);
      }
  }


/*
 * DeleteEntireLine:
 */

DeleteEntireLine(l)
    LineRec *l;
  {
    BeginningOfLine(l);
    DeleteLine(l);
  }


/*
 * BeginningOfLine:
 */

BeginningOfLine(l)
    LineRec *l;
  {
    while (l->cursor > 0)
      {
	BackwardChar(l);
      }
  }


/*
 * EndOfLine:
 */

EndOfLine(l)
    LineRec *l;
  {
    while (l->cursor != l->endLine)
      {
	ForwardChar(l);
      }
  }


/*
 * ForwardChar:
 */

ForwardChar(l)
    LineRec *l;
  {
    if (l->cursor < l->endLine)
      {
	if (l->printChars)
	  {
	    putc(l->line[l->cursor], l->out);
	  }
	l->cursor++;
      }
  }


/*
 * BackwardChar:
 */

BackwardChar(l)
    LineRec *l;
  {
    if (l->cursor > 0)
      {
	l->cursor--;
	if (l->printChars)
	  {
	    putc('\b', l->out);
	  }
      }
  }


/*
 * ReverseChars:
 */

ReverseChars(l)
    LineRec *l;
  {
    char c1, c2;

    if (l->cursor > 1)
      {
	c1 = l->line[l->cursor-2];
        c2 = l->line[l->cursor-1];
	BackSpace(l);
	BackSpace(l);
	SelfInsert(c2, l);
	SelfInsert(c1, l);
      }
  }


/*
 * BackwardWord:
 */

BackwardWord(l)
    LineRec *l;
  {
    BackwardChar(l);
    while ((l->cursor > 0) && !WordChar(l->line[l->cursor]))
      {
	BackwardChar(l);
      }
    while ((l->cursor > 0) && WordChar(l->line[l->cursor]))
      {
	BackwardChar(l);
      }
    if (!WordChar(l->line[l->cursor]))
      {
	ForwardChar(l);
      }
  }


/*
 * ForwardWord:
 */

ForwardWord(l)
    LineRec *l;
  {
    ForwardChar(l);
    while ((l->cursor != l->endLine) && !WordChar(l->line[l->cursor]))
      {
	ForwardChar(l);
      }
    while ((l->cursor != l->endLine) && WordChar(l->line[l->cursor]))
      {
	ForwardChar(l);
      }
  }


/*
 * DeleteWord:
 */

DeleteWord(l)
    LineRec *l;
  {
    while ((l->cursor != l->endLine) && !WordChar(l->line[l->cursor]))
      {
	DeleteChar(l);
      }
    while ((l->cursor != l->endLine) && WordChar(l->line[l->cursor]))
      {
	DeleteChar(l);
      }
  }


/*
 * BackwardsDeleteWord:
 */

BackwardsDeleteWord(l)
    LineRec *l;
  {
    if (!AtStartOfWord(l))
      {
	BackwardWord(l);
      }
    DeleteWord(l);
  }


#ifdef DELETED
/*
 * CapitalizeWord:
 */

CapitalizeWord(l)
    LineRec *l;
  {
    char ch;

    if (!AtStartOfWord(l))
      {
        BackwardWord(l);
      }
    ch = l->line[l->cursor];
    if ((ch >= 'a') && (ch <= 'z'))
      {
	DeleteChar(l);
	SelfInsert(UpperCaseLetter(ch), l);
      }
    else
      {
	ForwardChar(l);
      }
    while ((l->cursor != l->endLine) && WordChar(l->line[l->cursor]))
      {
	ch = l->line[l->cursor];
	if ((ch >= 'A') && (ch <= 'Z'))
	  {
	    DeleteChar(l);
	    SelfInsert(LowerCaseLetter(ch), l);
	  }
	else
	  {
	    ForwardChar(l);
	  }
      }
  }


/*
 * LowerCaseWord:
 */

LowerCaseWord(l)
    LineRec *l;
  {
    char ch;

    if (!AtStartOfWord(l))
      {
        BackwardWord(l);
      }
    while ((l->cursor != l->endLine) && WordChar(l->line[l->cursor]))
      {
	ch = l->line[l->cursor];
	if ((ch >= 'A') && (ch <= 'Z'))
	  {
	    DeleteChar(l);
	    SelfInsert(LowerCaseLetter(ch), l);
	  }
	else
	  {
	    ForwardChar(l);
	  }
      }
  }


/*
 * UpperCaseWord:
 */

UpperCaseWord(l)
    LineRec *l;
  {
    char ch;

    if (!AtStartOfWord(l))
      {
        BackwardWord(l);
      }
    while ((l->cursor != l->endLine) && WordChar(l->line[l->cursor]))
      {
	ch = l->line[l->cursor];
	if ((ch >= 'a') && (ch <= 'z'))
	  {
	    DeleteChar(l);
	    SelfInsert(UpperCaseLetter(ch), l);
	  }
	else
	  {
	    ForwardChar(l);
	  }
      }
 }
#endif DELETED