/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * V Prototype command interpreter
 * David Cheriton, Tim Mann, Marvin Theimer.
 *
 * Marvin Theimer, 6/83
 *	Substantially rewritten.  The new version of the exec makes use
 *	of a teamserver which manages all team loading, terminating, and 
 *	querying.  Concurrent programs are now owned by the teamserver, 
 *	and thus don't go away when the exec dies.
 * Marvin Theimer, 6/83
 *	Changed team loading structure.  Programs are now loaded by a call to
 *	a single routine LoadNewTeam.
 * Marvin Theimer, 7/83
 *	Changed the team loading structure again.  Programs are now loaded
 *	and set running by a routine called ExecProg, which does the 
 *	directory search stuff in addition to calling LoadNewTeam and
 *	actually setting the program running.
 *	Also changed the directory search path mechanism to use the string
 *	PATH instead of a loop through hardwired string names.
 *	Removed various dead code segments and variable defns.
 * Bill Nowicki, 7/83
 *	Added slight hooks to display banners in VGTSexec.
 * Tim Mann 8/5/83
 *      The 'cd' command is now built-in again, and the exec knows about the
 *      'login' command and does a cd to [home] after it runs (ugly hack to
 *	avoid having all of the login command built-in).
 * Tim Mann 8/19/83
 *	Changed calling sequence and added EOF detection to support the
 *	"serverexec".  Exec() now returns on EOF.
 * Tim Mann 8/24/83
 *	Tries to recover if current context is invalid.  Knows about
 *	'logout' command.  Added [bin] stuff.
 * Marvin Theimer, 8/27/83
 *	Merged the sh (shell) program with the exec.  Thus the exec now 
 *	handles io redirection, pipes, aliases, input line editing, and
 *	supports a history mechanism.
 * Tim Mann, 9/8/83
 *	Do ModifyPad here instead of in LoadProg, plus some minor cleanup.
 * Tim Mann, 9/16/83
 *	Redid i/o redirection and pipe stuff.
 */

#include <Venviron.h>
#include <Vsession.h>
#include <Vteams.h>
#include <Vioprotocol.h>
#include <Vgts.h>
#include <Vexceptions.h>
#include <Vteams.h>
#include "sh.h"
#include <chars.h>

#ifdef XEXEC
#define EXEC_LOGO "\f\r\n\nExperimental V System Executive, Version 4.0\r\n\n"
#else
#define EXEC_LOGO "\f\r\n\nV System Executive, Version 4.0\r\n"
#endif


ProcessId RunProgs();
ProcessId SetUpPipe();
SystemCode RedefineEnvironment();

extern short ExecDebug;
extern SystemCode AddContextName(), ChangeDirectory();
extern ProcessId Kernel_Process_Pid;

Exec(in, out, err, shIn)
    File *in, *out, *err, *shIn;
  {
    char *args[MaxArgs];		/* array of argument pointers */
    char lineBuf[LineLength];
    char banner[LineLength];		/* temp. banner string */
    char **argv;			/* pointer into args array */
    ProcessId newpid = 0;
    Message msg, rmsg, dmsg;
    SystemCode error;
    char *errstr;
    unsigned i, concurrent, login, logout;
    int inLen, inLen1, argc;
    ProcessId localTeamServerPid, teamServerPid;
    File *loadFile;
    ProcessId homepid; ContextId homecid;
    HistoryRec hRec, *h = &hRec;
    AliaseRec aliases[MaxAliases];
    LineRec lRec, *l = &lRec;
    register RootMessage *rtMsg = (RootMessage *) rmsg;
	/* new team's root message */
    RootMessage *debugRtMsg = (RootMessage *) dmsg;
	/* root message for new team's postmortem debugger. */

    if (shIn == NULL)
	shIn = in;    /* read commands from "in" unless "shIn" is non-null */

    ModifyPad(out, CR_Input+LF_Output);
				/* Set input cooking to that expected by
				   this exec. */

    SetBreakProcess(in, 0);
    SetVgtBanner(in, "");
    
    localTeamServerPid = GetPid(TEAM_SERVER, LOCAL_PID);
    if (localTeamServerPid == 0)
      {
	fprintf(out, "ERROR - couldn't locate team server for this host.\n");
	Flush(out);
	exit();
      }

    InitAliases(aliases);
    InitHistory(h);
    InitBuiltInCmds();

    ChangeDirectory("[home]");

    fprintf(out, EXEC_LOGO);

    while (1)
      {

errLabel:
	/* Continue here to get the next command, or 
	 * on an error in a command */
	teamServerPid = localTeamServerPid;

	SetVgtBanner(in, "");
	fprintf(out,"! ");
	Flush(out);

	/* Get the next command line.  Editing and historical references
	   are provided. */
	inLen = 0;
	while (1)
	  {
	    /* Get new input (editing is provided). */
	    inLen = GetLine(lineBuf, inLen, LineLength,
	    		TRUE, shIn, out, l);
	    if (inLen == -1)
	      {
		/* EOF has been signalled. */
		return;
	      }
	    else if (inLen == 0)
	      {
		/* No input received - start over. */
		goto errLabel;
	      }
	    else if (lineBuf[inLen-1] == BELL)
	      {
		fprintf(out, " XXX\n");
		goto errLabel;
	      }

	    /* Check for history references.  If history is invoked, then
	       redisplay the designated line for further editing. */
	    inLen1 = DoHistory(lineBuf, h);
	    if (inLen1 == -1)	/* Bad history ref. - start over. */
	      {
		fprintf(out, "Command not part of recent history.\n");
		goto errLabel;
	      }
	    else if (inLen1 == 0)
	      {
		/* No history ref. - get out of input loop. */
		break;
	      }
	    else	      
	      {
		/* History ref. - display designated line for user to edit. */
	        fprintf(out, "! ");
		inLen = inLen1;
		continue;
	      }
	  }

	/* Replace aliases. */
	if (!DoAliases(lineBuf, aliases, &error))
				/* Returns 1 if should continue - i.e. NOT an
				   alias defn. */
	  {
	    if (error != OK)	/* error only has meaning on an alias defn. */
	      {
		fprintf(out, "ERROR: no room for a new alias defn.\n");
	      }
	    continue;		/* Done with this cmd. */
	  }

	/* Parse the command line into separate words. */
        if ( (argc = ParseLine(lineBuf,args, MaxArgs))==0 ) 
	    continue;

	argv = args;

	/* Handle built-in commands */
	if (BuiltInCmd(args, h, aliases, out, &error))
	  {
	    if (error)		/* error only has a meaning if a built-in cmd
				   was encountered. */
	        fprintf( out, "%s: %s\n", lineBuf, ErrorString(error) );

	    continue;		/* Done with this cmd. */
	  }

	/* Handle the special case of login and logout commands - which need 
	   extra processing. */
	login = logout = 0;
	if ( strcmp(args[0], "login") == 0 )
	    login++;
	else if ( strcmp(args[0], "logout") == 0 )
	    logout++;

	/* Set default values */
	concurrent = 0;
	rtMsg->stdinserver = in->fileserver;
	rtMsg->stdinfile = in->fileid;
	rtMsg->stdoutserver = out->fileserver;
	rtMsg->stdoutfile = out->fileid;
	rtMsg->stderrserver = err->fileserver;
	rtMsg->stderrfile = err->fileid;
	rtMsg->rootflags = 0;
	rtMsg->nameserver = PerProcess->nameserver;
	rtMsg->contextid = PerProcess->contextid;
	rtMsg->kernelpid = Kernel_Process_Pid;

	*debugRtMsg = *rtMsg;

	/* Check for command environment redefinitions. */
	error = RedefineEnvironment(&argc, args, &concurrent, &teamServerPid,
		rtMsg, out);
	if (error != OK)
	    goto errLabel;

	sprintf(banner,"%.100s (loading)", argv[0] );
	SetVgtBanner(in,banner);

	newpid = RunProgs(argv, concurrent, teamServerPid,
		rtMsg, debugRtMsg, out, &error);
	if( newpid == 0 )
	  {
	    switch (error)
	      {
		case NOT_FOUND:
		    errstr = "File '%s' not found";
		    break;
		case NO_PERMISSION:
		    errstr = "Permission denied on '%s'";
		    break;
		case NO_MEMORY:
		    errstr = "Not enough memory for '%s'";
		    break;
		case BAD_STATE:
		case END_OF_FILE:
		    errstr = "File '%s' not runnable";
		    break;
		case CURRENT_CONTEXT_INVALID:
		    fprintf(out, "Current context is invalid\n");
		    /* Check if [home] is invalid, too */
		    GetContextId("[home]", &homepid, &homecid);
		    if (!ValidPid(homepid))
		      {
			fprintf(out, "[home] context is invalid\n");
			fprintf(out, "Setting [home] to [public]\n");
			DeleteContextName("[home]");
			AliasContextName("[home]", "[public]");
		      }
		    /* Check if [bin] is invalid, too */
		    GetContextId("[bin]", &homepid, &homecid);
		    if (!ValidPid(homepid))
		      {
			fprintf(out, "[bin] context is invalid\n");
			fprintf(out, "Setting [bin] to [public]\n");
			DeleteContextName("[bin]");
			AliasContextName("[bin]", "[public]");
		      }
		    fprintf(out, "Changing current context to [home]\n");
		    ChangeDirectory("[home]");
		    errstr = "Try again";
		    break;
		default:
		    errstr = "Error on file '%s': %s";
		    break;
	      }
	    fprintf( out, errstr, lineBuf, ErrorString(error) );
	    fprintf( out, "\r\n" );
	    goto errLabel;
	  }

	  /*
	   * This is when the new loaded team actually runs.
	   * We wait for it to terminate by doing a ReceiveSpecific on the
	   * rootPid of the team, providing it is not running
	   * concurrently.
	   */
	if ( concurrent )
	  {
	    fprintf( out, "pid = %x\n", newpid );
	    Flush( out );
	  }
	else
	  {
	    SetBreakProcess(in, newpid);
	    SetVgtBanner(in, argv[0]);
	    newpid = ReceiveSpecific(msg, newpid);
	    Resynch(in);
	    Resynch(out);

	    /* HACK WARNING: if the command executed was 'login', we
	     *   force a cd to [home] here, since login sets [home] to
	     *   the newly created context.  Unfortunately, we can't tell
	     *   if the login failed.
	     * We also check after a 'logout' to see if the current context 
	     *   or home context have become invalid.
	     */
	    if (login) ChangeDirectory("[home]");
	    if (logout)
	      {
		/* Check if [home] is invalid */
		GetContextId("[home]", &homepid, &homecid);
		if (!ValidPid(homepid))
		  {
		    DeleteContextName("[home]");
		    AliasContextName("[home]", "[public]");
		  }
		/* Check if [bin] is invalid, too */
		GetContextId("[bin]", &homepid, &homecid);
		if (!ValidPid(homepid))
		  {
		    DeleteContextName("[bin]");
		    AliasContextName("[bin]", "[public]");
		  }
		/* Check if current context is invalid */
		if (!ValidPid(PerProcess->nameserver))
		    ChangeDirectory("[home]");
	      }
	    fprintf( out, "\r\n" );
	  }
      }
  }




/*
 * InitBuiltInCmds:
 */

InitBuiltInCmds()
  {
  }




/*
 * BuiltInCmd:
 * Checks if args represents a builti-in shell command.  If so, it is 
 * executed and the function returns TRUE.  Otherwise FALSE is returned.
 */

int BuiltInCmd(args, h, aliases, out, error)
    char *args[];
    HistoryRec *h;
    AliaseRec aliases[];
    File *out;
    SystemCode *error;
  {

    *error = OK;

    if (strcmp(args[0], "history") == 0)
      {
        PrintHistory(h, out);
        return(TRUE);
      }
    if (Equal(args[0], "alias"))
      {
        PrintAliases(aliases, out);
	return(TRUE);
      }
    if (Equal(args[0], "unalias"))
      {
	RemoveAlias(args[1], aliases);
	return(TRUE);
      }

    if ( strcmp(args[0], "cd") == 0 )
      {
	if (args[1] == NULL)
	    *error = ChangeDirectory("[home]");
	else
	    *error = ChangeDirectory(args[1]);
	return(TRUE);
      }

    return(FALSE);
  }




/*
 * ProgDelim:
 * Returns true if the argument passed in is a program delimiter symbol.
 */

int ProgDelim(arg)
    char *arg;
  {
    if ((arg == NULL) ||
	    (strcmp(arg, ">") == 0) ||
	    (strcmp(arg, "<") == 0) ||
	    (strcmp(arg, "@") == 0) ||
	    (strcmp(arg, "&") == 0) ||
	    (strcmp(arg, "|") == 0))
	return(TRUE);
    else
	return(FALSE);
  }




/*
 * RunProgs:
 * Parses the argv array for programs to run and pipes to set up.
 * This is a recursive routine that calls itself again if a pipe is
 * encountered.
 * The routine cleans up after itself if an error occurs while trying to
 * set up sequence of programs and pipes.
 */

ProcessId RunProgs(argv, concurrent, teamServer, rtMsg, debugRtMsg, 
	shOut, error)
    char **argv;
    int concurrent;
    ProcessId teamServer;
    register RootMessage *rtMsg, *debugRtMsg;
    File *shOut;
    SystemCode *error;
  {
    Message msgx, msgy;
    register RootMessage *rtMsg1 = (RootMessage *) msgx;
    register CreateInstanceReply *pipeReply = (CreateInstanceReply *) msgy;
    ProcessId newpid;
    int argc1 = 0;

    while (!ProgDelim(argv[argc1]))
      {
	argc1++;
      }
    if ((argv[argc1] != NULL) && (strcmp(argv[argc1], "|") == 0))
      {
        *error = SetUpPipe(pipeReply, shOut);
	if (*error != OK) return(0);
        argv[argc1++] = NULL;
	*rtMsg1 = *rtMsg;
	rtMsg1->stdoutserver = pipeReply->fileserver;
	rtMsg1->stdoutfile = pipeReply->fileid;
	rtMsg1->rootflags |= STDOUT_REDIRECTED + RELEASE_OUTPUT;
	newpid = LoadProg(argv, concurrent, teamServer, 
		rtMsg1, debugRtMsg, error);
	if (*error != OK) return(0);
	SetOwnership(rtMsg1, newpid);
	Reply(rtMsg1, newpid);

	rtMsg->stdinserver = pipeReply->fileserver;
	rtMsg->stdinfile = pipeReply->fileid + 1;
	rtMsg->rootflags |= STDIN_REDIRECTED + RELEASE_INPUT;
	newpid = RunProgs(&argv[argc1], concurrent, teamServer, 
		rtMsg, debugRtMsg, shOut, error);
	if (*error != OK) return(0);
      }
    else
      {
	argv[argc1] = NULL;
	newpid = LoadProg(argv, concurrent, teamServer, rtMsg, debugRtMsg, 
		error);
	if (*error != 0) return (0);
	SetOwnership(rtMsg, newpid);
	Reply(rtMsg, newpid);
      }
    return(newpid);
  }


/*
 * SetOwnership:
 * Make the new team the owner of any stdio instances it is supposed
 *   to release
 */
SetOwnership(rtMsg, newpid)
    register RootMessage *rtMsg;
    ProcessId newpid;
  {
    if (rtMsg->rootflags & RELEASE_INPUT)
        SetInstanceOwner(rtMsg->stdinserver, rtMsg->stdinfile, newpid);
    if (rtMsg->rootflags & RELEASE_OUTPUT)
        SetInstanceOwner(rtMsg->stdoutserver, rtMsg->stdoutfile, newpid);
    if (rtMsg->rootflags & RELEASE_ERR)
        SetInstanceOwner(rtMsg->stderrserver, rtMsg->stderrfile, newpid);
  }
    
