/*	vedit.c:	 V editor - main program			*/
/*			Kenneth Brooks, Nov. 1982			*/
/*									*/
/* This is a display editor which runs under the V kernel and VGTS. 	*/
/* Its commands resemble emacs: it is not line-oriented.  It can 	*/
/* handle multiple buffers in multiple windows, and uses its invoking	*/
/* window for interaction text, such as search strings.  The mouse can	*/
/* be used to make selections and cut and paste.			*/
	
/*
 * RJN - 4/83 - Added tag function to ved.  Follows same conventions
 *		as unix tag file.
 * RJN - 8/83 - wrote code to handle context naming conventions of V.
 *	 	Deleted the ved's banners and put file name in vgt banner.
 *		Command vgt displays the current context in its banner.
 *		Fixed a few bugs with file names and changed backups to
 *		to do a copy operation (versus remote execution).
 * TPM 8/31/83  Corrected arguments to GetPid().
 */
#include <Venviron.h>
#include <Vioprotocol.h>
#include "Vgts.h"
#include "vedit.h"
#include "chardef.h"
#include "ansipad.h"
#include "menu.h"

/* The following variables are the current buffer record, and correspond
   to the record type BUFFER.  They are not expressed as a record because
   they occur all over the program, and making them record elements would
   make this code vastly uglier.  */

char	filename[80];
Rowrec	*rows;
Marklink Marklist;	/* a quick way of finding all the permanent Marks */
Mark	headmark, endmark, regionmark;
Mark	curmark, mousemark;
Pos	curpos;
short	modified = 0;	/* the modified bit for the current buffer */
short	selectionexists = 0;	/* selection flag */
short	incommandwindow = 0;	/* on when stdin is selected for input */
Padtype	pad;		/* pad's output file descriptor and handle in general */
File	*infile;	/* pad's input file descriptor */
Process_id keyproc;	/* this pad's keyboard input watcher */
/* end of the current buffer record */

     /* thisbuffer stores the buffer i.d. of the current active buffer */
int	thisbuffer = -1;	/* -1 means no buffer exists */

Chunk	readchunk;
Mark	tempmark;	/* used by BlockInsert and BlockDelete, also
			 * usable by anything that just uses single-char
			 * inserts and deletes
			 */
Mark	eolmark;	/* for use with PreAdjust/AdjustDisplay */
Vedmsg	commandmsg;
int	debug = 0;	  /* a bit vector controlling various printout */
int	backupoption = 1; /* if 1, .BAK files are made when writing */
SystemCode paderr;
Pos	zeropos = {0,0};
Pos	eolpos;	  /* stores a parameter for AdjustDisplay */

     /* the data type string, when used in ved, uses the (pointer, length) */
     /* representation - null termination is provided for debugger 	   */
     /* compatibility, but is not important.				   */
char	searchstring[80], repstring[80], linebuf[81], filename2[80];
int	searchlen, replen;

int	wishcol;   /* the column where ^N or ^P would like to land */
int	oktoquit = 0;	/* signal that the user intends to quit */
int	addmarkfail = 0;	/* signal that Addmark calloc failed */
Process_id mainproc = 0;/* the editor process */
Process_id rootproc;	/* the underlying process that survives crashes */

/* RJN -- 4/22/83 -- Replaced constant with appropriate defintion */
struct BUFTABLE buffers[Maxbufs] = {  
         /* numbers are default window positions */
	{NULL, 160, 170, NULL},
	{NULL, 330, 10,  NULL},
	{NULL, 15,  15,  NULL},
	{NULL, 10,  330, NULL},
	{NULL, 325, 325, NULL},
	{NULL, 180, 20,  NULL},
	{NULL, 20,  190, NULL},
	{NULL, 140, 320, NULL},
	{NULL, 330, 150, NULL}
	};

# define Keyprocstack 1024
extern ReaderProcess();
extern Mainloop();
extern File *OpenName();
extern InitFixedMenu();

main(argc,argv)  int argc;  char **argv;
{ int n;
  Process_id donepid;
  Message msg;  
  static char TmpName[ MAX_FILE_NAME_LEN ];

  rootproc = GetPid(ACTIVE_PROCESS, LOCAL_PID);
  ModifyPad(stdout, LF_Output);		/* I'll do the echoing */
  if (argc > 2) sscanf(argv[2],"%x",&debug);
  if (debug) {printf("debug is %x\n",debug);  Flush(stdout);}

  if (argc < 2) {
    readchunk = 0;
    filename[0] = '\0';
    NewMsg("no file");
/*^*/ myprint(8)("No file: headmark.chunk=%x\n",headmark.chunk);
    }
  else {
    TmpName[0]= '\0';
    strncat( TmpName, argv[1], MAX_FILE_NAME_LEN );
    readchunk = ReadFile( OpenName( filename, TmpName ) );
    }

  SetContextBanner();
  InitFixedMenu( );

  n = NewBuffer(readchunk);

  if (!n) return;	/* user aborted or no memory */

  while (!oktoquit) {
    mainproc = Create(20, Mainloop, 2000 );
    if (mainproc == 0) {
	printf("Failed to create mainproc!!\n");
	Flush(stdout);
	}
    Ready(mainproc,0);
    do {
      donepid = ReceiveSpecific(msg, mainproc);
      if ( donepid != 0 ) {
	switch (msg[0]) {
	    case CREATE_KEYPROC:
		keyproc = Create(9, ReaderProcess, Keyprocstack);
		Ready(keyproc, 2, msg[1], infile);
		Reply(msg, mainproc);
		break;
	    case DESTROY_KEYPROC:
		DestroyProcess(keyproc);
		Reply(msg, mainproc);
		break;
	    case RESTART_MAINPROC:
		DestroyProcess(mainproc);
		donepid = 0;
	    }  /* end switch */
	}
      }
     while (donepid != 0);

    if (oktoquit) return;	/* and exit from ved */
    SelectPad(stdout);
    printf("\nEditor crash!  Shall I try to save the current buffer? ");
    if (Yesno() > 0) WriteFile(filename,headmark,endmark);
    printf("\nTry to continue? ");
    if (Yesno() <= 0) return;
    SelectPad(pad);
    }
  }

/* Mainloop: a separate process so we can restart it after a strange crash */
/* We waitfor mouse or keyboard action, and act on it.			   */
Mainloop()
{ int n, c;
  Process_id pid;
  int msgtype, bufid;

  n = 0;
  while(1) {
    BackToPad();
    if (selectionexists) PadCursorOff(pad);
      else  PadCursorOn(pad);
    UpdateBanner();
    Flush(pad);  RedrawPad(pad);
    pid = Receive(&commandmsg);

    Reply(&commandmsg, pid);
    msgtype = commandmsg.type;  bufid = commandmsg.bufid;

    if (bufid != thisbuffer) PickBuffer(bufid);

/*^*/ myprint(8)("Msg type %d  buffer %d  x=%o\n",msgtype,bufid,
	  commandmsg.x);  Flush(stdout);

    if (msgtype == Mouse)
      {
	/* take action on menu driven command */
        c = MouseAction(&commandmsg);
	if ( c !=  INVALID_COMMAND )
	    if (selectionexists) n = SelectionKeyAction( c );
	    else n = KeyAction(c);
      }
    else if (msgtype == Key) {
	if (!quoted && !hiquote)  c = StateMachine(commandmsg.x);
	else c = commandmsg.x;
	if (c >= 0) {
	    if (selectionexists)  n = SelectionKeyAction(c);
	    else  n = KeyAction(c);
	    }
	else n=0;
	}
    else n=0;
    if (n == -1) {
/*^*/	myprint(255)("Good-bye\n");
	oktoquit = 1;
	return;
	}
    }
}


/* ReaderProcess: read chars and mouse clicks from the pad and
   send them to mainproc.
   One of these exists for each editing pad - whenever a pad is not selected,
   the process reading it just remains blocked on input.  Parameters are
   buffer index and pad input file descriptor. */
ReaderProcess(mybuf, myinpad)  int mybuf;  File *myinpad;
{ Vedmsg msg;
  short x, y, buttons;
  register int n, j, k;
  char charbuf[20];
  short LastButtons = 0;

  while (1) {
    n = GetEvent(myinpad, &x, &y, &buttons, charbuf);
    if (n < 0) {
	printf("Pad file error: %s", ErrorString(myinpad->lastexception));
	continue;
	}

    msg.code = 0;
    msg.bufid = mybuf;
    if (n > 0) {	/* keyboard message */
/*^*/myprint(8)("Key %o bytecount %d   ", charbuf[0], n);
	for (j=0; j < n; j++) {
/*^*/myprint(8)("(%d) ", j);
	    msg.type = Key;
	    msg.x = charbuf[j];
	    k = Send(&msg, mainproc);
	    if (k == 0) printf("reader %d send failure!\n",mybuf);
	    }
	}
    else {	/* mouse message */
	/* send down transitions only for all buttons, but the Left */
	/* The left is sent on the up transition */
/*^*/myprint(0x10)("Lastbuttons %d, buttons %d\n", LastButtons, buttons );
	if ( (buttons != 0 && buttons == LeftButton)
	     || LastButtons == MiddleButton
	     || LastButtons == RightButton )
	  {
	    msg.buttons = buttons ? buttons : LastButtons; 
	    msg.type = Mouse;
	    msg.x = x;
	    msg.y = y;
	    k = Send(&msg, mainproc);
	    if (k == 0) printf("reader %d send failure!\n",mybuf);
	  }
	LastButtons = buttons;
	}
    }
  }


/* Yesno: Get a y or n answer from the user.  y returns 1, n returns 0,	*/
/* Return returns 2 ("default answer"), ^G returns -1,			*/
/* all others ask "y or n please." 					*/
int Yesno()
{ register char ch;

 SelectPad(stdout);
 incommandwindow = 1;
 while (1) {
  Flush(stdout);
  ch = getchar();
  if (ch == '\07') {	/* control-G */
    return(-1);
    }
  if (ch == 'y' || ch == 'Y') {
    putchar(ch);
    Flush(stdout);
    return(1);
    }
  if (ch == 'n' || ch == 'N') {
    putchar(ch);
    Flush(stdout);
    return(0);
    }
  if (ch == '\r') return(2);
  else printf("\nyes or no? ");
  }
 }

/*********************  Multi-buffering functions  **************************/

/* NewBuffer: given a Chunk pointer, which may be NULL, creates a buffer with
   the first available buffer id number, and displays it in a window in the
   appropriate place.  The new buffer will be the current one, so the previous
   current buffer should be stored before NewBuffer is called.  The new
   window is selected for input.  If pad creation fails, or there is not
   enough memory, returns 0, else 1.
 */
int NewBuffer(contents)  Chunk contents;
{ register int bufid;
  register BUFFER *buf;
  static char *PadNameFormat = "Ved %d";
  char PadName[ 10 ];
  Message msg;

  /* find an unused buffer i.d. */
  for (bufid = 0;  bufid < 9 && buffers[bufid].pad != 0;  bufid++) ;
  if (bufid == 9) {
    NewMsg("No more buffers!");  Beep();
    return(0);
    }

  /* create a place to store its variables */
  buf = (BUFFER *) calloc(1, sizeof(struct BUFFER));
  buffers[bufid].ptr = buf;
  /* create the row descriptor array */
  rows = (Rowrec *) calloc(Nrows+1,sizeof(struct Rowrec));
  
  /* create the Marklist for this buffer's Marks */
  Marklist = NULL;
  Addmark(&curmark);
  Addmark(&endmark);
  Addmark(&regionmark);
  Addmark(&mousemark);
  Addmark(&eolmark);
  Addmark(&tempmark);
  
  /* see if we're out of memory */
  if (buf == NULL || rows == NULL || addmarkfail) {
    NewMsg("Out of memory!  Buffer not created!");
    if (buf != NULL) free(buf);
    if (rows != NULL) free(rows);
    goto Bailout;
    }

  /* establish the buffer start and end marks */
  if (contents)  InitBuffer(contents, Findend(contents));
   else  ClearBuffer();

  /* create a window and keyboard watcher process for it */

  sprintf( PadName, PadNameFormat, bufid + 1 );	/* gotta get the # right */
  pad = OpenPad(PadName, Nrows+1, Ncols, &paderr);
  if (pad == NULL || paderr != OK) {
    Msg("   Aborted");
    free(rows);
    free(buf);
Bailout:
    if (thisbuffer >= 0)  SelectBuffer(thisbuffer);
    addmarkfail = 0;
    return(0);
    }
  ModifyPad(pad,ReportTransition);  /* Totally raw input with mouse */

  infile = OpenFile(pad->fileserver,pad->fileid,FREAD,&paderr);
  if (infile == NULL || paderr != OK) {
    PrintError(paderr,"can't open input: ");
    free(rows);  free(buf);  goto Bailout;
    }
  else {
    if (mainproc == 0) {	/* initial buffer 1st time */
	keyproc = Create(9, ReaderProcess, Keyprocstack);
	Ready(keyproc,2,bufid,infile);
	}
    else {
	msg[0] = CREATE_KEYPROC;
	msg[1] = bufid;
	Send(msg, rootproc);
	}
    }

  /* display and establish page and cursor marks */
  Display(headmark);
  Setcursor(headmark, zeropos);
  modified = selectionexists = 0;
  thisbuffer = bufid;

  SelectPad(pad);
  incommandwindow = 0;
/*^*/ myprint(0x800)("New buffer %d chunk %x keyproc %x\n",
	bufid,headmark.chunk,keyproc);
  return(1);
  }



/* DeleteBuffer: delete the current buffer and all it entails.  Leaves the
   next-lower-numbered buffer selected and returns its number.  Returns -1
   if the last buffer was deleted.  */
int DeleteBuffer()
{ register int n;
  Message msg;

  FreeText(headmark.chunk);		/* free the chunks */
  free(rows);				/* free the row descriptor array */
  msg[0] = DESTROY_KEYPROC;
  Send(msg, rootproc);			/* get the keyproc destroyed */
  Close(pad);
  buffers[thisbuffer].pad = 0;
  if (buffers[thisbuffer].ptr != NULL)
      free(buffers[thisbuffer].ptr);	/* free the buffer record */
  n = PrevBuffer(thisbuffer);
/*^*/ myprint(0x800)("\nPrevBuffer returned %d for buffer #%d", n, thisbuffer );
  if (n < 0)  return(-1);
  SelectBuffer(n);
  return(n);
  }

/* StoreBuffer: save the current buffer into an inactive buffer record. */
StoreBuffer(bufid)  int bufid;
{ register BUFFER *buf;
  buf = buffers[bufid].ptr;
  strcpy(buf->filename, filename);
  buf->rows = rows;
  buf->Marklist = Marklist;
  buf->headmark = headmark;  buf->endmark = endmark;
  buf->regionmark = regionmark;  buf->curmark = curmark;
  buf->mousemark = mousemark;
  buf->curpos = curpos;
  buf->modified = modified;  buf->selectionexists = selectionexists;
  buf->pad = pad;  buf->infile = infile;
  buf->keyproc = keyproc;

  buffers[bufid].pad = pad;
/*^*/ myprint(0x800)("stored filename '%s'\n",buf->filename);
  }

/* SelectBuffer: make an inactive window into the current editing window.
   Fetches its buffer and selects it for input. */
SelectBuffer(bufid)  int bufid;
{
  FetchBuffer(bufid);
  thisbuffer = bufid;
  wishcol = curcol;
  SelectPad(pad);
  incommandwindow = 0;
  }

/* RJN -- 4/22/83 -- created FindBuffer */
/* Find the Buffer which has the associated file name associated with it. 
 * It returns the Buffer index or -1 on failure.
 * filename must be null terminated.
 */
int FindBuffer(filename) 
    char *filename;
  {
    register char *fnp, *bnp;
    register int i;
    register struct BUFTABLE *bp;
    extern struct BUFTABLE buffers[];
    
    /* search all valid buffers for a match */
    for( i = 0, bp = buffers; i < Maxbufs; i++,bp++ )
      {
        if ( bp->pad != 0 )
	  {
	    fnp = filename;
	    bnp = bp->ptr->filename;
	    while (*bnp == *fnp && *fnp) fnp++,bnp++;
	    if (*bnp == *fnp) break;
	  }
      }
    return( i < Maxbufs ? i : -1 );
  } /* FindBuffer */
	      
    
/* FetchBuffer: make an inactive buffer into the current one.  Does not
   actually select it for input, this may be transient. */
FetchBuffer(bufid)  int bufid;
{ register BUFFER *buf;

  if (buffers[bufid].ptr)  buf = buffers[bufid].ptr;
  else {
    ErrorMsg("FetchBuffer: nonexistent buffer");
    return;
    }

  strcpy(filename, buf->filename);
  rows = buf->rows;
  Marklist = buf->Marklist;
  headmark = buf->headmark;  endmark = buf->endmark;
  regionmark = buf->regionmark;  curmark = buf->curmark;
  mousemark = buf->mousemark;
  curpos = buf->curpos;
  modified = buf->modified;  selectionexists = buf->selectionexists;
  pad = buf->pad;  infile = buf->infile;
  keyproc = buf->keyproc;

/*^*/ myprint(0x800)("fetched buffer %d stored with name '%s' to name '%s'\n",
		bufid, buf->filename, filename);
  }


/* PrevBuffer: given a buffer i.d. find the next-lower-numbered buffer.
   Will find the same old buffer if it is valid and if no others exist.
   Returns -1 if no buffers exist.  */
int PrevBuffer(n)  int n;
{ register int b;

  for (b = n-1; b >= 0; b--)
    if (buffers[b].pad != 0)  return(b);
  for (b = 8; b >= n; b--)
    if (buffers[b].pad != 0)  return(b);
  return(-1);
  }

/* PickBuffer: the command routine for the keypad keys 1-9.  Selects that
   buffer if there is such a buffer, otherwise beeps.  */
PickBuffer(b)  int b;
{
  if (buffers[b].pad == 0)  {
    Beep();  /* macro, must be enclosed in braces here */
    }
  else {
    Flush(pad);  RedrawPad(pad);
    StoreBuffer(thisbuffer);
    SelectBuffer(b);
    wishcol = curcol;
    }
  }


/* PickWindow: Ask the user to indicate an editor window, by clicking	*/
/* in it with the mouse.  If mode is nonzero a keystroke will abort it	*/
/* and be returned as the negative of its ASCII value;			*/
/* ^G will do that regardless of mode.					*/
/* Stdin/stdout are not used and must not be selected, or the ^G key	*/
/* cannot be read.  PickWindow returns a buffer number.			*/
int PickWindow(mode)  int mode;
{ register int bufid;
  Process_id pid;
  Vedmsg msg;
  char ch;

  while (1) {
    Flush(stdout);
    pid = Receive(&msg);  /* await a mouse click from one of my pads */
    Reply(&msg, pid);
    if (msg.type == Mouse) {
	bufid = msg.bufid;
	return(bufid);
	}
    else {
	ch = msg.x;
	if (ch == 0) continue;	/* NUL is ignored, just for simplicity */
	if (ch == '\07' || mode != 0) return(-ch);
	}
    /* else we just ignore the keystroke */
    } /* end while (1) */
  }

/* SelfInsert: insert this character at the cursor, and advance the cursor. */
SelfInsert(ch) unsigned char ch;
{ 
  PreAdjust(curmark,curpos,&eolmark,&eolpos);
  TextInsertChar(ch);
  if (MarkEQ(curmark,eolmark)) {
    eolmark.cp++;
    Advance(&eolmark);
    }
  AdjustDisplay(curmark,eolmark,&curpos,&eolpos);
  if (currow == Nrows) Scroll();
  Setcursor(curmark,curpos);
  Forespace();
  }

/* DeleteForward: the ^D function.  Here because it is opposite to SelfInsert.
 */
DeleteForward()
{ Mark m;
  char ch;

  if (Atend(curmark))  {Beep(); return;}
  ch = *curmark.cp;
  if (ch == '\n') {
    m = curmark;
      /* Forespace would be nice here, but we mustn't Scroll */
    curmark.cp++;  Advance(&curmark);
    curpos = Makepos(currow+1,0);
    PreAdjust(curmark,curpos,&eolmark,&eolpos);
    Backspace();
    }
  else {
    PreAdjust(curmark,curpos,&eolmark,&eolpos);
    }
  TextDeleteForward();
  AdjustDisplay(curmark,eolmark,&curpos,&eolpos);
  if (currow >= Nrows)  Scroll();  /* delete \n on full row Nrows-1 */
  Setcursor(curmark,curpos);
  }


/* Forespace: advance the cursor by 1 character */
/* returns 1 on success, 0 on failure because at end */
int Forespace()
{ char ch;
  int newcol;
  int newsize;

    if (Atend(curmark))  { Beep(); return(0);}
    ch = *curmark.cp++;
    Advance(&curmark);
    newcol = CountWidth(ch,curcol);
/*^*/myprint(0x2)("Forespace: curcol %d, newcol %d, chars '%c%c'\n",
	curcol, newcol, ch, *curmark.cp);
    if (Atend(curmark)) newsize = 0;
    else newsize = charsize[*curmark.cp];
    if (newcol + newsize > Ncols) newcol = 0;
    if (newcol > 0)
	for (; curcol<newcol; curcol++)  PadCursorForward(pad);

    else {
	currow++;
	curcol = -newcol;
	if (currow >= Nrows) Scroll();
	PadPosition(pad, currow, curcol);
	}
    wishcol = curcol;
    return(1);
    }

/* Backspace: backspace the cursor by 1 character */
int Backspace()
{ char ch;
  if (Atstart(curmark)) {Beep(); return(0);}
  if (currow == 0 && curcol == 0) Backscroll();
  Retract(&curmark);
  ch = *curmark.cp;
  if (ch >= ' ' && ch < '\177' && curcol != 0) {  /* printing character */
    PadCursorBackward(pad);
    curcol--;
    }
  else {
    if (curcol == 0) currow--;
    curcol = Markcol(currow, curmark);
    PadPosition(pad, currow, curcol);
    }
  wishcol = curcol;
  return(1);
  }

/* Downaline: the down-arrow function. */
Downaline()
{
  if (!rows[currow+1].exists) {Beep(); return;}
  if (currow+1 == Nrows) Scroll();
  currow++;
  PosSetcursor(Makepos(currow,wishcol));
  }

/* Upaline: the up-arrow function. */
Upaline()
{
  if (currow == 0 && PageAtHead()) {Beep(); return;}
  if (currow == 0) Backscroll();
  currow--;
  PosSetcursor(Makepos(currow,wishcol));
  }

Mark nonMark = {NULL, NULL};  /* meaning "not found" */

/* Search: given a string and its length (NOT the null-termination convention)
   find its first occurrence in the text after (but not at) curmark.  If found,
   curmark and mousemark are set to the beginning and end, respectively, of the
   found string, and Search returns 1.  Otherwise they are unchanged and Search
   returns 0. Not defined for strings of length 0.  When search succeeds, any
   existing selection is deselected since curmark and mousemark are moved.
 */
int Search(str,len)  char *str;  int len;
{ register char *searchpt, *scanpt, *stringpt;
  char *searchecp, *scanecp;
  register Chunk srchunk, scanchunk;
  register int i;

  if (Atend(curmark)) return(0);
  srchunk = curmark.chunk;  searchpt = curmark.cp + 1;
  searchecp = Chunkend(srchunk);

  while (1) { /* main search loop */
    for (; searchpt < searchecp; searchpt++) { /* in-chunk loop for speed */
      if (*searchpt == *str) { /* found first character */
	stringpt = str; scanpt = searchpt; scanchunk = srchunk;
	scanecp = searchecp;
	for (i=0; i<len && (*scanpt++ == *stringpt++); i++) {
	  if (scanpt >= scanecp) { /* i.e. our find crosses a chunk boundary */
	    if (!scanchunk->next) break; /*not return, might be end of a find*/
	    scanchunk = scanchunk->next; scanpt = scanchunk->text;
	    scanecp = Chunkend(scanchunk);
	    }
	  }
	if (i == len)  {  /* found! */
	    if (selectionexists) Deselect();
	    curmark = Makemark(srchunk, searchpt);
	    mousemark = Makemark(scanchunk, scanpt);
	    return(1);
	    }
	}  /* end of "found first character" */
      }  /* end in-chunk loop */
   /* main search: next chunk */
    if (!srchunk->next)  return(0);
    srchunk = srchunk->next;  searchpt = srchunk->text;
    searchecp = Chunkend(srchunk);
    }  /* end while(1) */
  }

/* ReverseSearch: just like Search only backward.  Just like Search, will not
   find an instance starting right at the cursor.  Will find an instance
   starting before the cursor and extending past it.  */
int ReverseSearch(str,len)  char *str;  int len;
{ register char *searchpt, *scanpt, *stringpt;
  char *searchecp, *scanecp;
  register Chunk srchunk, scanchunk;
  register int i;

  if (Atstart(curmark)) return(0);
  srchunk = curmark.chunk;  searchpt = curmark.cp - 1;
  searchecp = Chunkend(srchunk);

  while (1) { /* main search loop */
    for (; searchpt >= srchunk->text; searchpt--) { /* in-chunk loop */
      if (*searchpt == *str) { /* found first character */
	stringpt = str; scanpt = searchpt; scanchunk = srchunk;
	scanecp = searchecp;
	for (i=0; i<len && (*scanpt++ == *stringpt++); i++) {
	  if (scanpt >= scanecp) { /* i.e. our find crosses a chunk boundary */
	    if (!scanchunk->next) break;
	    scanchunk = scanchunk->next; scanpt = scanchunk->text;
	    scanecp = Chunkend(scanchunk);
	    }
	  }
	if (i == len)  {  /* found! */
	    if (selectionexists) Deselect();
	    curmark = Makemark(srchunk, searchpt);
	    mousemark = Makemark(scanchunk, scanpt);
	    return(1);
	    }
	}  /* end of "found first character" */
      }  /* end in-chunk loop */
   /* main search: next chunk */
    if (!srchunk->prev)  return(0);
    srchunk = srchunk->prev;  searchecp = Chunkend(srchunk);
    searchpt = searchecp - 1;
    }  /* end while(1) */
  }


/* SelFind:  To be called with the output of Search or ReverseSearch.	*/
/* If the search succeeded, the text found is displayed and selected,	*/
/* otherwise the message "not found" is printed.			*/
SelFind(foundit)  int foundit;
{
  BackToPad();
  if (foundit) {
    DispSetcursor(curmark);
    Select(curmark, mousemark);
    }
  else NewMsg("Not found");
  }

/* Beep: the simplest available signal that something is wrong.	*/
Beep()
{ putchar('\007');
  Flush(stdout);
}

/* ErrorMsg: united here so that it can easily be redefined */
ErrorMsg(s) char *s;
{ printf(s);  Flush(stdout);
}

/*************************** Debugging Subroutines **************************/

DumpChunks(chunk)  Chunk chunk;
{ for ( ; chunk!=NULL; chunk=chunk->next)
    printf("Chunk %x (%c%c%c%c): %d chars, end %x, prev %x, next %x\n",
        chunk,chunk->text[0],chunk->text[1],chunk->text[2],chunk->text[3],
	chunk->length,chunk->text+chunk->length,
        chunk->prev,chunk->next);
  Flush(stdout);
  }

/* SixRows: dumps the data for the first 6 Rowrecs			*/
SixRows()
{ register int i;
  putchar('\n');
  for (i=0; i<6; i++) DumpRow(i);
  Flush(stdout);
  }

LastRows()
{ register int i;
  putchar('\n');
  for (i=Nrows-5; i<=Nrows; i++) DumpRow(i);
  Flush(stdout);
  }

DumpRow(n)  int n;
{ register Rowrec *r = rows + n;
  register char ch;
  printf("Row %d  ", n);
  if (r->exists) {
    ch = *r->cp;
    if (ch == '\n') ch = 01;	/* control-A, a down arrow */
    printf("'%c'  chunk %x cp %x  length %d ",ch,r->chunk,r->cp,r->length);
    if (r->continues) putchar('<');
    if (r->continued) putchar('>');
    }
  else putchar('-');
  putchar('\n');
  }
