/*	textual.c: 	Functions to insert and delete text 		*/

#include "vedit.h"

/* This is a good place to document policy on marks.  A Permanent Mark is a
variable of type Mark whose value endures from command to command; A
Normal Mark is a permanent mark to which a pointer is kept on the Marklist:
hence it will be corrected in the standard way after an insertion or deletion.
curmark and endmark and regionmark are normal marks; beginmark is not.
beginmark is changed only by special test, and the first chunk is seldom
changed or relinquished.
*/

static Marklink marklinkp;

/* TextInsertChar: the textual (as opposed to visual, or graphical) aspect of
   inserting a character.  All normal marks are left pointing to the character
   they used to point to - a mark on the insertion spot ends up before it.
   insertion is done at the cursor (curmark). */
TextInsertChar(ch)  char ch;
{ register Chunk ichunk;
  register char *icp,*p,*q;
  char *zcp;
  Mark *m;
  int yes, rowcount;

  modified = 1;
  ichunk = curmark.chunk;  icp = curmark.cp;  zcp = Chunkend(ichunk);

  if (ichunk->length == CHUNKSIZE) {
    if ((icp - ichunk->text) <= CHUNKSPLITPOINT) { /* 70% rule */
/*^*/ myprint(0x200)("Early split: ");
	SplitChunk(ichunk,icp+1);  /* handles all marks */
	zcp = icp+1;
	}
    else {
/*^*/ myprint(0x200)("Late split: ");
	SplitChunk(ichunk,ichunk->text + CHUNKSPLITPOINT);
	icp = ichunk->next->text + ((icp - ichunk->text) - CHUNKSPLITPOINT);
	ichunk = ichunk->next;
	zcp = Chunkend(ichunk);
	}
    }

  for (p=zcp, q=zcp+1; p > icp; ) *--q = *--p;
  ichunk->length++;
  *icp = ch;
/*^*/ myprint(0x80)("TextInsert: just before markfixing  ");
/* A mark sitting after the insertion point gets adjusted by 1 */
  for (m = Firstmark(ichunk,&yes); yes; m = Nextmark(ichunk, &yes) ) {
    if (m->cp > icp) m->cp++;
    }
/*^*/ myprint(0x80)("markfixing done\n");
  if (Atend(curmark)) endmark.cp++;
  if (endmark.cp != Chunkend(endmark.chunk))  
	printf("Inconsistent endmark\n");

  }


/* TextDeleteForward: the textual aspect of deleting a character.  There is
   no DeleteBackward - you move backward then delete forward if need be.
   Returns 1 if OK, 0 if can't delete because mark = end of file.   Deletion
   is at the cursor. */

int TextDeleteForward()
{ register Chunk dchunk;
  register char *dcp,*p,*q;
  char *zcp, ch;
  Mark *m;
  int yes, dcflag;

  dchunk = curmark.chunk;  dcp = curmark.cp;  zcp = Chunkend(dchunk);
  ch = *dcp;
  if (Atend(curmark))  return(0);
  if (dcp == zcp) {printf("TextDeleteForward error: at end of chunk!\n");
		   return(0);}

  modified = 1;
  for (p = dcp, q = dcp+1; q < zcp; )  *p++ = *q++;
  dchunk->length--;  zcp--;
  if ((dchunk->length == 0) && (dchunk->prev || dchunk->next)) {
    DeleteChunk(dchunk);
    dcflag = 1;
    }
  else {
/*^*/ myprint(0x80)("TextDelete: markfixing about to begin  ");
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	if (m->cp > dcp) m->cp--;
	else if (m->cp == zcp && m->chunk->next) {
	    m->chunk = m->chunk->next;
	    m->cp = m->chunk->text;
	    }
	}
    dcflag = 0;
    }
  if ((!dcflag) && dchunk->prev)  Scavenge(dchunk->prev);
  return(1);
  }


/* BlockInsert: (purely textual) Given a Mark and a pointer to the first
chunk of a piece of straytext, inserts that text here.  Usually "Display" is
the appropriate graphical counterpart.  "straytext" has the following
properties:  It has the same chunk structure as buffer text, but there are
no row descriptors for it.  Its last chunk->next is NULL.  It is not linked
into the main text.  There are (by current plans) no permanent marks pointing
into it.  BlockInsert is intended to work whether imark is in a chunk, on a
chunk boundary, at beginning or end of text, or in a null text.  The nchunk
pointer may not be null, but it may point to a null text.  Since an insertion
moves all Marks on the insertion point to the end of the insertion,
BlockInsert returns the Mark of the beginning of the insertion.  */

Mark  BlockInsert(imark, nchunk)  Mark imark;  Chunk nchunk;
{ Chunk ichunk,echunk,lastchunk;
  register char *icp;
  Mark *m;

  modified = 1;
  ichunk = imark.chunk;  icp = imark.cp;
  
 {
  /* find the end of the block being inserted */
    for (lastchunk = nchunk; lastchunk->next; lastchunk = lastchunk->next);

  /* prepare the main text to receive it: make sure imark is on a chunk
     boundary */
    if (icp == ichunk->text) {echunk = ichunk; ichunk = ichunk->prev;}
    else if (!Atend(imark)) {SplitChunk(ichunk,icp); echunk = ichunk->next;}
    else echunk = NULL;   /* when we insert at end of text */

/* link in the new text */
    if (Atstart(imark)) {
	headmark.chunk = nchunk; headmark.cp = nchunk->text;
	ichunk = nchunk;
	}
    else { ichunk->next = nchunk; nchunk->prev = ichunk; }

    if (Atend(imark)) {
        for (marklinkp = Marklist; marklinkp; marklinkp = marklinkp->next) {
	    m = marklinkp->pmark;
	    if (m->cp == imark.cp) { /* in particular, fix endmark */
		m->chunk = lastchunk;  m->cp = Chunkend(lastchunk);
		}
	    }
	}
    else { lastchunk->next = echunk;  echunk->prev = lastchunk; }

  /* tempmark is on the Marklist so it will survive Scavenges */
    tempmark = Makemark(nchunk, nchunk->text);
    Scavenge(ichunk);
    if (ichunk->prev) Scavenge(ichunk->prev);
    if (echunk) {
	Scavenge(echunk);
	if (echunk->prev) Scavenge(echunk->prev);
	}
    return(tempmark);
    }
  } /* end BlockInsert */


/* BlockDelete: given a pair of marks (which must be in order!) delete the
   text between them and return it as a sparetext - stripped of 
   marks.  For large deletes this involves the creation of two new chunks:
   the chunks containing the beginning and end of the block are each split.  */

Chunk  BlockDelete(bmark,emark)  Mark bmark,emark;
{ Chunk bchunk,	/* beginning of deletion - the part that remains */
        echunk, /* end of deletion - the part that remains */
	nchunk, /* beginning of deletion - the deleted part */
	lchunk, /* last chunk of the deleted part */
	chunk;
  Mark *m;
  char *bcp,*ecp,*zcp;
  int yes, dcflag;

  modified = 1;
  if (MarkEQ(bmark,emark)) { ErrorMsg("BlockDelete of nothing");  return(NULL);}

  bchunk = bmark.chunk; echunk = emark.chunk;
  bcp = bmark.cp; ecp = emark.cp;  zcp = Chunkend(bchunk);

       /* simple version: part of a single chunk */
  if (bchunk == echunk) {
    nchunk = CutSubchunk(bchunk,bcp,ecp);
    }

  else {  /* multichunk version */
/*^*/ myprint(0x200)("Big Blockdelete from (%x,%x) to (%x,%x): ",
	bchunk,bcp,echunk,ecp);
    nchunk = CutSubchunk(bchunk,bcp,zcp);
    nchunk->next = bchunk->next;  nchunk->next->prev = nchunk;
    nchunk->prev = NULL;
/*^*/ myprint(0x200)("second CutSubchunk.. ");
    lchunk = CutSubchunk(echunk,echunk->text,ecp);
    lchunk->prev = echunk->prev;  lchunk->prev->next = lchunk;
    lchunk->next = NULL;
    if (lchunk->length == 0) { /* no empty tail on sparetext */
	lchunk = lchunk->prev;  free(lchunk->next);
	lchunk->next = NULL;
	}
    bchunk->next = echunk;  echunk->prev = bchunk;

/*^*/ myprint(0x200)("\n   mark movement.. ");
  /* clean the marks out of any interior chunks */
    if (nchunk != lchunk) {
      for (chunk = nchunk->next; chunk; chunk = chunk->next)
	for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) )
	  if (m->chunk == chunk)  {m->chunk = echunk;  m->cp = echunk->text;}
      }

  /* move any marks on deletion spot to head of echunk */
    for (m = Firstmark(bchunk,&yes); yes; m = Nextmark(bchunk,&yes) ){
	if (m->cp == bcp) m->cp = echunk->text;
	}
    } /* end multichunk version */
/*^*/ myprint(0x200)("end of mark movement  ");

  /* delete an empty chunk - unless it's the only one */
  dcflag = 0;
  if (bchunk->prev || bchunk->next) 
    if (bchunk->length == 0) {DeleteChunk(bchunk); dcflag = 1;}
  if (echunk != bchunk && (echunk->prev || echunk->next))
    if (echunk->length == 0) DeleteChunk(echunk);

  if (!dcflag) {
    Scavenge(bchunk);
    Scavenge(bchunk->prev);
    }
/*^*/ myprint(0x200)("Done\n");
  return(nchunk);
  }


/* SplitChunk: given the chunk to split, and the cp at which to split it,
   splits it into two chunks, properly linked.  Handles all marks on the
   Marklist properly.  Not currently usable on unlinified text. */
SplitChunk(chunk,cp)  Chunk chunk;  char *cp;
{ register char *p,*q,*zcp;
  Chunk nchunk;
  int k,yes;
  Mark *m;
  
  zcp = Chunkend(chunk);
  k = zcp - cp;
  if (k==0) {
	printf("Inappropriate SplitChunk at beginning of chunk\n");  return;}
  else if (k==chunk->length) {
	printf("Inappropriate SplitChunk at end of chunk\n");  return;}
/*^*/ myprint(0x200)("Split %x at %x  ",chunk,cp);
  nchunk = MakeChunk(chunk, k);

/*^*/ myprint(0x200)("copy beginning to chunk %x  ",nchunk);
  for (p=cp, q = nchunk->text; p < zcp; ) *q++ = *p++;
  chunk->length -= k;
  nchunk->next = chunk->next;
  if (chunk->next) nchunk->next->prev = nchunk;
  chunk->next = nchunk;
/*^*/ myprint(0x200)("copy done\n");
 
  for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) ) 
    if (m->cp >= cp) {
	m->chunk = nchunk;
	m->cp = nchunk->text + (m->cp - cp);
	}
  }


/* DeleteChunk: handles pointer-pushing and mark-fixing when we delete a
   chunk.  The chunk should already be empty or otherwise unwanted before
   DeleteChunk is called.  Eventually the freeing task will be passed to
   Scavenger. */

DeleteChunk(dchunk)  Chunk dchunk;
{ Chunk pchunk;
  Mark *m;
  int yes;

  if (dchunk == headmark.chunk) {
    if (!dchunk->next) {ErrorMsg("DeleteChunk: solitary chunk!"); return;}
    headmark.chunk = dchunk->next;  headmark.cp = headmark.chunk->text;
    headmark.chunk->prev = 0;
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	m->chunk = headmark.chunk;  m->cp = headmark.cp;
	}
    free(dchunk);
    return;
    }
  else if (!dchunk->prev) {ErrorMsg("DeleteChunk: losing head of chain!");
    return;
    }  
/*^*/ myprint(0x200)("Deleting chunk %x  ",dchunk);
  pchunk = dchunk->prev;
  pchunk->next = dchunk->next;
  if (dchunk->next) dchunk->next->prev = pchunk;
  
  if (pchunk->next)
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	m->chunk = pchunk->next;
	m->cp = m->chunk->text;
	}
  else
    for (m = Firstmark(dchunk,&yes); yes; m = Nextmark(dchunk,&yes) ) {
	m->chunk = pchunk;
	m->cp = Chunkend(pchunk);
	}
/*^*/ myprint(0x200)("deleted.\n");
  free(dchunk);
  }


/* CutSubchunk: chop out a section of an old chunk and return it as a new
   one.  Any marks collapse into the deletion point: the new chunk contains
   no marks.  */
Chunk  CutSubchunk(chunk,bcp,ecp)  Chunk chunk;  char *bcp,*ecp;
{ Chunk nchunk;
  Mark *m;
  register char *p,*q;  char *zcp;
  int k,yes;

  k = ecp - bcp;  zcp = Chunkend(chunk);
  nchunk = MakeChunk(NULL, k);
  for (p = bcp, q = nchunk->text;  p < ecp; )  *q++ = *p++;
  for (p = bcp, q = ecp;  q < zcp; )  *p++ = *q++;
  chunk->length -= k;

  if (ecp == zcp && chunk->next) {
    for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) )
	if (m->cp >= bcp) {m->chunk = chunk->next; m->cp = m->chunk->text;}
    }
  else
    for (m = Firstmark(chunk,&yes); yes; m = Nextmark(chunk,&yes) )
      if (m->cp > bcp) {
   	if (m->cp < ecp) m->cp = bcp;
    	else m->cp -= k;
    	}
  return(nchunk);
  }


/***************************  Mark list subroutines *************************/

/* Mark loop helpers.  Firstmark should be called once, followed by calls to
   Nextmark, until yes == 0.  They first retrieve any marks from the Marklist
   which point into this chunk, then any row descriptors which do so.  The 
   first two fields in a Rowrec look just like a Mark, so it is safe to coerc
   a Rowrec pointer into a (Mark *) type pointer and fool the outside 
   routine.  */

static Marklink marklp;
static int	markrow;
static int	donemarks;

Mark * Firstmark(chunk, yes)  Chunk chunk;  int *yes;
{ Mark *pmark;

  donemarks = 0;
  for (marklp = Marklist; marklp != NULL; marklp = marklp->next) {
/*^*/ myprint(0x40)("Fm: chunk=%x, cp=%x   ",marklp->pmark->chunk,
		marklp->pmark->cp);
    if (marklp->pmark->chunk == chunk) {
	*yes = 1;
	return(marklp->pmark);
	}
    }

  donemarks = 1;
  for (markrow=0; markrow <= Nrows && rows[markrow].chunk != chunk; markrow++);
  *yes = (markrow <= Nrows && rows[markrow].exists);
  pmark = (Mark *) &rows[markrow];
  return(pmark);
  }

Mark *Nextmark(chunk, yes)  Chunk chunk;  int *yes;
{ Mark *pmark;

  if (donemarks)  { markrow++;  goto A;}
  for (marklp = marklp->next; marklp != NULL; marklp = marklp->next) {
/*^*/ myprint(0x40)("Nm: chunk=%x, cp=%x  ",marklp->pmark->chunk,
		marklp->pmark->cp);
    if (marklp->pmark->chunk == chunk) {
	*yes = 1;
	return(marklp->pmark);
	}
    }

  donemarks = 1;
  for (markrow=0; markrow <= Nrows && rows[markrow].chunk != chunk; markrow++);

A:*yes = (markrow <= Nrows && rows[markrow].chunk == chunk &&
	rows[markrow].exists);
  pmark = (Mark *) &rows[markrow];
  return(pmark);
  }


/* Addmark: called from vedit.c (main or an initializer) to build the Marklist.
   The Marklist is rather static, but room is left open for user-defined marks
   to be added at runtime.  Addmark adds a single mark to it.  The mark whose
   pointer is in p MUST be a global or static variable.  */
Addmark(p)  Mark *p;
{ Marklink link;

  if (addmarkfail) return;
  link = (struct marklink *) calloc(1,sizeof(struct marklink));
  if (link == NULL) {addmarkfail = 1;  return;}
  link->pmark = p;
  link->next = Marklist;
  Marklist = link;
  }


/* Scavenge: checks a chunk to see if it is small to be merged with the next 
   chunk, and if so, merges it.  Prevents editing from producing a
   progressively less efficient structure.  */

Scavenge(chunk)  Chunk chunk;
{ Chunk nextchunk;
  Mark *m;
  char *p,*q;
  int yes,k,n,offset;

/*^*/ myprint(0x80)("Scavenge of %x ",chunk);    
  if (chunk == NULL) return;

  nextchunk = chunk->next;
  if (nextchunk) n = chunk->length + nextchunk->length;
  if (nextchunk && n < CHUNKFILLPOINT ) {
/*^*/ myprint(0x80)("in action  ");
    k = nextchunk->length;
    for (p = chunk->text + chunk->length, q = nextchunk->text;
		q < Chunkend(nextchunk); )
	*p++ = *q++;
    chunk->length += k;
    offset = q - p;
	/* Yes, I subtracted pointers in different chunks! It works. */
    
    for (m = Firstmark(nextchunk,&yes); yes; m = Nextmark(nextchunk,&yes) ) {
	m->chunk = chunk;
	m->cp -= offset;  /* directly corrects the character pointer */
	}

    DeleteChunk(nextchunk);
/*^*/ myprint(0x80)("Scavenge deleted %x\n",nextchunk);
    }
  }
