/*
 *  misc.c
 *
 *  This file contains the miscellaneous routines used to manipulate
 *  data structures, performs useful common tasks, etc.
 *
 *  David Kaelbling, April 1983
 */
 
 
/* Includes */
# ifdef VAX
# include "stdio.h"
# else
# include "Vio.h"
# endif
# include "Vgts.h"
# include "splines.h"
# include "draw.h"
 
 
/* Imports */
extern DebugItemDescriptor();
extern DebugObjectList();
extern DebugLinks();
extern double sin();
extern double cos();
 
 
/* Exports */
extern InvertString();		/* Flip the higher order bit in a string */
extern Highlight();		/* Highlight a menu selection. */
extern HighlightStatic();	/* Highlight a static menu selection. */
extern SelectMenuObject();	/* Put a selected symbol into the menu */
extern UnSelectMenuObject();	/* Undo the above. */
extern Quit();			/* Clean up the window manager and exit */
extern RedrawActivelist();	/* Redisplay the main drawing area. */
extern NewObject();		/* Add a new object to the display list */
extern RevertToCKP();		/* Back off a transaction. */
extern DecrementRef();		/* Decrement reference count on an item */
extern Checkpoint();		/* Make a checkpoint. */
extern NewGroup();		/* Create a new group name */
extern RotateObject();		/* Rotate an object about a point. */
extern ScaleObject();		/* Expand/Contract an object about a point. */
extern CopyObject();		/* Make a copy of an object. */
extern MoveObject();		/* Move an object. */
extern EraseObject();		/* Erase (delete) an object. */
extern LowerObject();		/* Lower an object. */
extern RaiseObject();		/* Raise an object. */
    
    
/* Local definitions */
static int CopyCount = 0;	/* Unique tag for copied group names. */
# define ArrowChar	'\031'	/* Displays as a right-arrow character. */

/*
 *  This routine is used to put a string into inverse video, and visa
 *  versa.  Text is inverted using the recognized convention of setting
 *  the higher order bit to 1 to indicate inverse video.
 */
 
InvertString( s )
	char *s;
  {
    while (*s)
      {
	*s ^= 0200;
	s++;
      }
  }

/*
 *  This function will toggle the highlighting of a menu selection.
 */
 
Highlight( cmd )
	MenuData *cmd;
  {
    short x1, x2, y1, y2;
    
    /* Toggle the highlighting of the string. */
    InvertString( cmd->string );
    
    /* Display the new string */
    EditSymbol( sdf, menuSymbol );
    InquireItem( sdf, cmd->itemno, &x1, &x2, &y1, &y2, 0, 0, NULL );
    ChangeItem( sdf, cmd->itemno, x1, x2, y1, y2,
	0, SDF_SIMPLE_TEXT, cmd->string );
    if ((cmd->string[0]) & 0200)
      {
	/* Insert some filler so that entire menu line is blacked out. */
	AddItem( sdf, cmd->itemno + 1, x1, x2 + 1, y1 - 3, y2,
		AllEdges, SDF_OUTLINE, NULL );
      }
    else
      {
	/* Already inverted, delete filler */
	DeleteItem( sdf, cmd->itemno + 1 );
      }
    EndSymbol( sdf, menuSymbol, menuVgt );
  }

/*
 *  This routine will toggle the hilighting of a static menu option.
 */
 
HighlightStatic( cmd )
	MenuData *cmd;
  {
    short x1, x2, y1, y2;
    
    /* Toggle the highlighting of the string. */
    InvertString( cmd->string );
    
    /* Display the new string */
    EditSymbol( sdf, menuSymbol );
    InquireItem( sdf, cmd->itemno, &x1, &x2, &y1, &y2, 0, 0, NULL );
    ChangeItem( sdf, cmd->itemno, x1, x2, y1, y2,
	0, SDF_SIMPLE_TEXT, cmd->string );
    if ((cmd->string[0]) & 0200)
      {
	/* Insert some filler so that entire menu box is blacked out. */
	AddItem( sdf, cmd->itemno + 1, x1 - 1, x2 + 1, y1 + 16, y2 - 1,
		BLACK, SDF_FILLED_RECTANGLE, NULL );
	AddItem( sdf, cmd->itemno + 2, x1 + 48, x2 + 1, y1 - 3, y2 - 1,
		BLACK, SDF_FILLED_RECTANGLE, NULL );
	AddItem( sdf, cmd->itemno + 3, x1 - 1, x2 + 1, y1 - 4, y2 - 1,
		AllEdges, SDF_OUTLINE, NULL );
	AddItem( sdf, cmd->itemno + 4, x1 - 1, x2 + 1, y1 - 3, y1 - 2,
		AllEdges, SDF_OUTLINE, NULL );
      }
    else
      {
	/* Already inverted, delete filler */
	DeleteItem( sdf, cmd->itemno + 1 );
	DeleteItem( sdf, cmd->itemno + 2 );
	DeleteItem( sdf, cmd->itemno + 3 );
	DeleteItem( sdf, cmd->itemno + 4 );
      }
    EndSymbol( sdf, menuSymbol, menuVgt );
  }

/*
 *  This routine will put an arrow next to an entry in the Object menu.
 */
 
SelectMenuObject( num )
	short num;
  {
    short x1, x2, y1, y2;
    
    /* Insert the arrow into the second character of the string, */
    /* preserving any highlighting while we do it. */
    Objects[num].string[1] =
	ArrowChar | (Objects[num].string[0] & 0200);
	
    /* Update the string in the Vgts */
    EditSymbol( sdf, menuSymbol );
    InquireItem( sdf, Objects[num].itemno, &x1, &x2, &y1, &y2, 0, 0, NULL );
    ChangeItem( sdf, Objects[num].itemno, x1, x2, y1, y2, 0,
	    SDF_SIMPLE_TEXT, Objects[num].string );
    
    /* Fix off by one bug in redisplay code */
    AddItem( sdf, Objects[num].itemno + 1, x1, x2+1, y1-3, y2,
	    RightEdge, SDF_OUTLINE, NULL );
    
    EndSymbol( sdf, menuSymbol, menuVgt );
  }

/*
 *  This routine will undo SelectMenuObject.
 */
 
UnSelectMenuObject( num )
	short num;
  {
    short x1, x2, y1, y2;
    
    /* Restore the second character of the string to a space, */
    /* preserving highlighting by copying the first character. */
    Objects[num].string[1] = Objects[num].string[0];
    
    /* Update the string in the Vgts */
    EditSymbol( sdf, menuSymbol );
    InquireItem( sdf, Objects[num].itemno, &x1, &x2, &y1, &y2, 0, 0, NULL );
    ChangeItem( sdf, Objects[num].itemno, x1, x2, y1, y2, 0,
	    SDF_SIMPLE_TEXT, Objects[num].string );
    
    /* Work around off by one bug in Vgts redisplay code */
    DeleteItem( sdf, Objects[num].itemno + 1 );
    
    EndSymbol( sdf, menuSymbol, menuVgt );
  }

/*
 *  Tidy up the window manager and punt.
 */
 
Quit()
  {
    /* If the main drawing VGT was created, destroy it. */
    if (mainVgt)
        DeleteVGT( mainVgt, 1 );
    mainVgt = 0;
    
    /* If the menu VGT was created, destroy it too. */
    if (menuVgt)
        DeleteVGT( menuVgt, 1 );
    menuVgt = 0;
    
    /* If an SDF was allocated, release it. */
    if (sdf)
        DeleteSDF( sdf );
    sdf = 0;
    
    /* Restore the TTY cooking */
    ResetTTY();
    
    /* Stop processing */
    exit();
  }

/*
 *  This internal routine will add a single object to the display.
 */
DisplaySingleItem( q, refresh, dx, dy )
	ITEM_LIST_PTR *q;
	short refresh;
	short dx, dy;
  {
    /* Add the item to the current display list. */
    EditSymbol( sdf, mainSymbol );
    q->call = AddCall( sdf, item++, dx, dy, q->itemdesc->symbol );
    EndSymbol( sdf, mainSymbol, (refresh ? mainVgt : 0) );
  }

/*
 *  This internal routine is called each time an object is created.
 *  It will allocate a symbol for the new object.
 */
  
CreateSymbol( q, dx, dy )
	ITEM_LIST_PTR *q;	/* Entry in the activelist to display */
	short dx, dy;		/* Relative offset of the object */
  {
    ITEM_LIST_PTR *g;		/* Pointer for following group chains */
    register ITEM_DESCRIPTOR *id;	/* Optimize dereferencing. */
    
    /* Bomb Proofing */
    if (q == NULL)
      {
	printf("Internal Error:  CreateSymbol on a NULL Item.\n\r");
	return;
      }
    
    /* Display the new object */
    id = q->itemdesc;
    if (id->type == TextObj)
      {
	/* Text. */
	id->symbol = DefineSymbol( sdf, item++, "new text symbol" );
        id->number = AddItem( sdf, item++, id->xmin+dx, id->xmax+dx,
		id->ymin + dy + id->base, id->ymax + dy,
		FontData[id->subtype].refnumber, SDF_TEXT, id->data );
	EndSymbol( sdf, id->symbol, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* A spline of some sort. */
	id->symbol = DefineSymbol( sdf, item++, "new spline symbol" );
	
	/* Since this is a new object, get bounding box & number. */
	id->number = AddItem( sdf, item++, dx, 0, dy, 0,
		0, SDF_SPLINE, id->data );
	InquireItem( sdf, item - 1, &(id->xmin), &(id->xmax),
	    	&(id->ymin), &(id->ymax), 0, 0, 0 );
	EndSymbol( sdf, id->symbol, 0 );
      }
    else
      {
	/* Construct the group */
	id->symbol = DefineSymbol( sdf, item++, "new group symbol" );
	g = (ITEM_LIST_PTR *) id->data;
	while (g)
	  {
	    AddCall( sdf, item++, g->dx, g->dy, g->itemdesc->symbol );
	    g = g->next;
	  }
	EndSymbol( sdf, id->symbol, 0 );
      }
  }

/*
 *  This routine will rebuild the Groups data structure, which consists
 *  of a list of pointers to all groups reachable throught the activelist.
 */
 
RebuildGroups()
  {
    register ITEM_LIST_PTR *p;
    register int numgr;
    
    /* Scan through the active list. */
    for (numgr = 0, p = activelist->prev;  p;  p = p->next)
      {
	/* Is this a group? */
	if (p->itemdesc->type == GroupObj)
	  {
	    /* Yes.  Attempt to cache the pointer to it. */
	    if (numgr < MAXGROUP)
	      {
		Groups[numgr++] = p;
	      }
	    else
	      {
		printf("Internal Error: Too many groups to rebuild cache.\n\r");
		return;
	      }
	  }
      }
    
    /* Wipe the rest of the entries. */
    while (numgr < MAXGROUP)
      {
	Groups[numgr++] = NULL;
      }
  }

/*
 *  This routine will wipe the drawing area and redisplay the
 *  activelist from scratch.
 */
RedrawActivelist( refresh )
	register short refresh;
  {
    register ITEM_LIST_PTR *p;
    
    /* Delete the old drawing area. */
    DeleteSymbol( sdf, mainSymbol );
    
    /* Create the new one. */
    mainSymbol = DefineSymbol( sdf, item++, "Draw main symbol" );
    AddItem( sdf, 0, 0, 850, 0, 1100, AllEdges, SDF_OUTLINE, NULL );
    EndSymbol( sdf, mainSymbol, 0 );
    if (Debug)
	DebugObjectList( "Redraw activelist", activelist, 0 );
    
    /* Update the Vgts */
    for (p = activelist->prev;  p;  p = p->next)
      {
	DisplaySingleItem( p, 0, p->dx, p->dy );
      }
    
    /* Update the screen */
    if (refresh)
	DisplayItem( sdf, mainSymbol, mainVgt );
  }

/*
 *  Add a new object to the active display list.  Note that there are
 *  several display lists -- those used by RevertToCKP to back off a 
 *  set of updates, and the the current list (activelist).
 */
 
NewObject( dataptr, typeP, subtypeP, td, x1, x2, y1, y2, b, nameP,
	   refresh, deltax, deltay )
	char *dataptr;			/* New itemdesc->data entry. */
	enum ObjTypes typeP;		/* itemdesc->type entry. */
	short subtypeP, td;		/* subtype and typedata fields. */
	short x1, x2, y1, y2, b;	/* Bounding box and baseline */
	char *nameP;			/* name field. */
	short refresh, deltax, deltay;
  {
    ITEM_LIST_PTR *ptr;
    ITEM_DESCRIPTOR *id;
    
    /* We have a new current object.  Make it so. */
    if ((id = (ITEM_DESCRIPTOR *) malloc( sizeof(ITEM_DESCRIPTOR) )) == NULL)
      {
	printf("Ran out of memory -- couldn't create a new object.\n\r");
	if (dataptr)
	    free( dataptr );
	return;
      }
    
    /* Initialize all the fields, append to main item list */
    if (itemlist)
	itemlist->next = id;
    id->next = NULL;
    id->prev = itemlist;
    itemlist = id;
    id->data = dataptr;
    id->type = typeP;
    id->subtype = subtypeP;
    id->typedata = td;
    id->number = 0;
    id->xmin = x1;
    id->xmax = x2;
    id->ymin = y1;
    id->ymax = y2;
    id->base = b;
    id->name = nameP;
    id->refs = 1;
    
    /* Load the font if necessary. */
    if ((typeP == TextObj) && (!(FontData[subtypeP].loaded)))
      {
	printf("\n\rLoading font '%s', please wait ..",
		FontData[subtypeP].fontname);
# ifndef VAX
	RedrawPad( stdout );
# endif
	FontData[subtypeP].refnumber =
	    DefineFont( FontData[subtypeP].fontname, NULL );
	printf(".. done.\n\r");
	if ((FontData[subtypeP].refnumber < 0) ||
	    (FontData[subtypeP].refnumber == 255))
	  {
	    printf("Internal Error. Can't load font %s (%d).\n\r",
		    FontData[subtypeP].fontname, subtypeP );
	    printf("    return code = %d\n\r",
		    FontData[subtypeP].refnumber );
	  }
	else
	    FontData[subtypeP].loaded = 1;
      }

    /* Append to active list */
    if ((ptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) )) == NULL)
      {
	printf("Out of memory.  Couldn't allocate item list pointer.\n\r");
	if (dataptr)	free( dataptr );
	if (nameP)	free( nameP );
	free( id );
	CurrentObject = NULL;
	return;
      }
    CurrentObject = ptr;
    ptr->prev = activelist->next;
    ptr->itemdesc = id;
    ptr->next = NULL;
    ptr->dx = deltax;
    ptr->dy = deltay;
    if (activelist->prev == NULL)
	activelist->prev = ptr;
    if (activelist->next)
	(activelist->next)->next = ptr;
    activelist->next = ptr;
    CreateSymbol( ptr, deltax, deltay );
    DisplaySingleItem( ptr, refresh, 0, 0 );
    if (Debug)
      {
	printf("NewObject:  Just created\n\r");
	DebugItemDescriptor( id, 0 );
	DebugLinks( activelist );
      }
  }

/*
 *  This routine will decrement the reference count for an entry
 *  in the main item descriptor list.  It will, if necessary,
 *  also free the storage associated with unused items.
 */
 
DecrementRef( q )
	register ITEM_DESCRIPTOR *q;
  {
    ITEM_LIST_PTR *p, *ptmp;
    
    if (q == NULL)
	return;

    /* Do the easy part.  Decrement the reference count. */
    q->refs -= 1;
    if (q->refs > 0)
	return;
	
    /* Free the storage */
    if (q->type == GroupObj)
      {
	/* Decrement the count for items in the group recursively */
	p = (ITEM_LIST_PTR *) q->data;
	while (p)
	  {
	    DecrementRef( p->itemdesc );
	    ptmp = p;
	    p = p->next;
	    free( ptmp );
	  }
      }
    else if (q->data)
	free( q->data );
    if (q->name)
	free( q->name );
      
    /* Fix up the links */
    if (q->prev)
	q->prev->next = q->next;
    if (q->next)
	q->next->prev = q->prev;
    if (itemlist == q)
	itemlist = q->prev;
    DeleteSymbol( sdf, q->symbol );
    free( q );
  }

/*
 *  Checkpoint switches backup display lists, making a backup copy of
 *  the activelist, and freeing any unreferenced objects in the central
 *  object list.
 */
    
Checkpoint()
  {
    register ITEM_LIST_PTR *p, *ptmp, *pold;
    
    /* Do the dirty bit */
    modified = 1;
    
    /* Deallocate the old backup list, freeing unused memory. */
    CurrentBackup = (CurrentBackup + 1) % NumBackup;
    if (Debug)
      {
	printf("Checkpoint:  Deallocating backuplist %d\n\r", CurrentBackup);
	DebugLinks( backuplists[CurrentBackup] );
      }
    p = backuplists[CurrentBackup]->prev;
    while (p)
      {
	ptmp = p->next;
	DecrementRef( p->itemdesc );
	free( p );
	p = ptmp;
      }
    
    if (Debug)
      {
	printf("Activelist:\n\r");
	DebugLinks( activelist );
      }
    
    /* Build the new backup list */
    backuplists[CurrentBackup]->prev = NULL;
    p = activelist->prev;
    pold = NULL;
    while (p)
      {
	if ((ptmp = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
		== NULL)
	  {
	    printf("Out of memory.  Can't make a checkpoint.\n\r");
	    backuplists[CurrentBackup]->next = pold;
	    return;
	  }
	if (pold == NULL)
	    backuplists[CurrentBackup]->prev = ptmp;
	ptmp->prev = pold;
	if (pold)
	    pold->next = ptmp;
	
	/* Copy over the data */
	ptmp->itemdesc = p->itemdesc;
	p->itemdesc->refs += 1;
	ptmp->dx = p->dx;
	ptmp->dy = p->dy;
	ptmp->next = NULL;
	
	pold = ptmp;
	p = p->next;
      }
    backuplists[CurrentBackup]->next = pold;
  }

/*
 *  RevertToCKP will abort a transaction, reverting back to the previous
 *  backup list.
 */
 
RevertToCKP()
  {
    register ITEM_LIST_PTR *tmp;
    
    /* Set the dirty bit */
    modified = 1;
    
    /* Swap display lists */
    tmp = activelist;
    activelist = backuplists[CurrentBackup];
    backuplists[CurrentBackup] = tmp;
    CurrentBackup = (CurrentBackup + NumBackup - 1) % NumBackup;
    CurrentObject = NULL;
    
    /* Display this version of the main drawing area */
    RebuildGroups();
    RedrawActivelist( 1 );
  };

/*
 *  This procedure will update the object list to reflect the creation
 *  of a new group.  It will also update the active group list.  If
 *  this is an original group (i.e., not a copy of an existing group),
 *  then items are transfrerred off of the activelist.
 */
 
NewGroup( q, original, refresh )
	ITEM_LIST_PTR *q;
	short original, refresh;
  {
    ITEM_LIST_PTR *gptr;
    
    if (Debug)
	printf("New Group:  %d, original %d, refresh %d\n\r",
		q, original, refresh );
    
    /* Update the active list */
    gptr = (ITEM_LIST_PTR *) q->itemdesc->data;
    if (original)
      {
	/* For an original group, move objects off of the active list. */
        while (gptr)
          {
	    ITEM_LIST_PTR *tmp;
	
	    /* Verify that items are on activelist */
	    tmp = activelist->prev;
	    while ((tmp) && (tmp->itemdesc != gptr->itemdesc))
	        tmp = tmp->next;
	    if (tmp == NULL)
	      {
		/* Only groups can be specified when not top level items */
		if (gptr->itemdesc->type != GroupObj)
		  {
		    printf("Internal Error:  Newgroup item not active.\n\r");
		    DebugItemDescriptor( gptr->itemdesc, 0 );
		  }
		else
		    gptr->itemdesc->refs += 1;
	      }
	    else
	      {
		/* Update pointers within activelist */
		if (tmp->prev)
		    tmp->prev->next = tmp->next;
		else
		    activelist->prev = tmp->next;
		if (tmp->next)
		    tmp->next->prev = tmp->prev;
		else
		    activelist->next = tmp->prev;
        	free( tmp );
	      }
	    gptr = gptr->next;
	  }
      }
    else
      {
	/* Update the reference counts for a non-original group. */
	while (gptr)
	  {
	    gptr->itemdesc->refs += 1;
	    gptr = gptr->next;
	  }
      }
    
    /* Creating a group can raise/lower objects within it. */
    RebuildGroups();
    if (refresh)
	RedrawActivelist( 1 );
  }

/*
 *  This routine will delete an object from the activelist and from
 *  the screen.
 */
 
EraseObject( victim, refresh )
	ITEM_LIST_PTR *victim;
	short refresh;
  {
    register short rebuild;
    
    /* Print Debugging Information */
    if (Debug)
      {
	printf("Erase Object:\n\r");
	DebugItemDescriptor( victim->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    /* Update the screen */
    EditSymbol( sdf, mainSymbol );
    DeleteItem( sdf, victim->call );
    EndSymbol( sdf, mainSymbol, (refresh ? mainVgt : 0) );
    
    /* Remove any dangling pointers. */
    CurrentObject = NULL;
    rebuild = (victim->itemdesc->type == GroupObj);
    
    /* Update the activelist */
    DecrementRef( victim->itemdesc );
    if (victim->prev)
	victim->prev->next = victim->next;
    if (victim->next)
	victim->next->prev = victim->prev;
    if (activelist->prev == victim)
	activelist->prev = victim->next;
    if (activelist->next == victim)
	activelist->next = victim->prev;
    free( victim );
    if (rebuild)
	RebuildGroups();
  }

/*
 *  This primitive will raise an object to the top of the display.
 *  This operation is useful when opaque filling is used.
 */
 
RaiseObject( id, refresh )
	ITEM_LIST_PTR *id;
	short refresh;
  {
    if (Debug)
      {
	printf("Raise object\n\r");
	DebugItemDescriptor( id->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    if (activelist->next == NULL)
      {
	printf("Internal Error:  Null end cell in activelist.\n\r");
	return;
      }
      
    /* If it's already there, don't bother playing with the links. */
    CurrentObject = id;
    if (activelist->next == id)
      {
	if (refresh)
	    DisplayItem( sdf, mainSymbol, mainVgt );
	return;
      }
	
    /* Update the links in the activelist */
    if (id->prev)
        id->prev->next = id->next;
    else
	activelist->prev = id->next;
    if (id->next)
	id->next->prev = id->prev;
    else
	printf("Internal Error:  Raise already raised object.\n\r");
    activelist->next->next = id;
    id->prev = activelist->next;
    id->next = NULL;
    activelist->next = id;
    
    /* Redisplay? */
    if (refresh)
      {
	if (id->itemdesc->type == GroupObj)
	  {
	    /* Lots of work.  Might as well start from scratch. */
	    RedrawActivelist( 1 );
	  }
	else
	  {
	    /* Not so much work.  Delete and add the item back. */
	    EditSymbol( sdf, mainSymbol );
	    DeleteItem( sdf, id->call );
	    EndSymbol( sdf, mainSymbol, mainVgt );
	    DisplaySingleItem( id, 1, id->dx, id->dy );
	  }
      }
  }

/*
 *  This routine will change the display offset of an object in the
 *  activelist.  It is assumed that a call to RaiseObject will be made
 *  in conjunction with a call to this routine, as the Vgts will
 *  implicitly raise a modified item to the top of its display list.
 */
 
MoveObject( p, deltax, deltay, refresh )
	ITEM_LIST_PTR *p;
	short deltax, deltay;
	short refresh;
  {
    /* Don't die because of bad pointers. */
    if (p == NULL)
      {
	printf("Internal Error:  Move NULL object.\n\r");
	return;
      }
    
    /* Debugging information requested? */
    if (Debug)
      {
	printf("Move Object %x by (%d, %d)\n\r", p, deltax, deltay );
	DebugItemDescriptor( p->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    /* Do the actual work */
    CurrentObject = p;
    p->dx += deltax;
    p->dy += deltay;
    if (!refresh)
	return;
    if (p->itemdesc->type != GroupObj)
      {
	/* Move a single item.  Try incremental screen updates. */
	EditSymbol( sdf, mainSymbol );
	DeleteItem( sdf, p->call );
	EndSymbol( sdf, mainSymbol, mainVgt );
	DisplaySingleItem( p, 1, p->dx, p->dy );
      }
    else
      {
	/* Move a group.  Lots of work. */
	RedrawActivelist( 1 );
      }
  }

/*
 *  This primitive will lower an object to the bottom of the display list.
 *  This operation is useful when opaque filling is used.
 */
 
LowerObject( id, refresh )
	ITEM_LIST_PTR *id;
	short refresh;
  {
    if (Debug)
      {
	printf("Lower object\n\r");
	DebugItemDescriptor( id->itemdesc, 0 );
	DebugLinks( activelist );
      }
    
    if (activelist->prev == NULL)
      {
	printf("Internal Error:  Null start cell in activelist.\n\r");
	return;
      }
      
    /* If it's already there, don't bother playing with the links. */
    CurrentObject = id;
    if (activelist->prev == id)
      {
	if (refresh)
	    DisplayItem( sdf, mainSymbol, mainVgt );
	return;
      }
	
    /* Update the links in the activelist */
    if (id->prev)
        id->prev->next = id->next;
    else
	printf("Internal Error:  Lower already lowered object.\n\r");
    if (id->next)
	id->next->prev = id->prev;
    else
	activelist->next = id->prev;
    activelist->prev->prev = id;
    id->next = activelist->prev;
    id->prev = NULL;
    activelist->prev = id;
    
    /* Redisplay? */
    if (refresh)
      {
	/* Lots of work.  Might as well start from scratch. */
	RedrawActivelist( 1 );
      }
  }

/*
 *  This routine will copy an object.  Groups are copied with a shallow
 *  copy, so changing the elements of the root group will change all of
 *  its instances.  Other objects are copied with a deep copy, so each
 *  copy is independent.
 */
 
CopyObject( p, deltax, deltay, refresh )
	ITEM_LIST_PTR *p;
	short deltax, deltay;
	short refresh;
  {
    register ITEM_DESCRIPTOR *id;
    char *string;
    SPLINE *sptr, *newsptr;
    POINT *pptr, *newpptr;
    register short i;
    ITEM_LIST_PTR *idptr;
    
    /* Do a little checking */
    if ((p == NULL) || (p->itemdesc == NULL))
      {
	printf("Internal Error:  Bad pointers to CopyObject.\n\r");
	return;
      }
    
    /* Print debugging information */
    if (Debug)
      {
	printf("Copy Object %x with translation (%d, %d)\n\r",
		p, deltax, deltay );
	DebugItemDescriptor( p->itemdesc, 0 );
      }
    
    /* For non-groups, just invoke NewObject() properly. */
    id = p->itemdesc;
    if (id->type == TextObj)
      {
	/* Text.  Make a new copy of the string, and use new offsets. */
	if ((string = (char *) malloc( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't copy any text.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->data );
	NewObject( string, TextObj, id->subtype, id->typedata,
		id->xmin + deltax + p->dx, id->xmax + deltax + p->dx,
		id->ymin + deltay + p->dy, id->ymax + deltay + p->dy,
		id->base, NULL, refresh, 0, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* Splines.  Copy spline data structure. */
	sptr = (SPLINE *) id->data;
	if ((newsptr = (SPLINE *) malloc( sizeof(SPLINE) +
		sptr->numvert * sizeof(POINT) )) == NULL)
	  {
	    printf("Out of memory.  Can't copy a spline-type object.\n\r");
	    return;
	  }
	newsptr->order = sptr->order;
	newsptr->numvert = sptr->numvert;
	newsptr->nib = sptr->nib;
	newsptr->border = sptr->border;
	newsptr->closed = sptr->closed;
	newsptr->filled = sptr->filled;
	newsptr->opaque = sptr->opaque;
	newsptr->pat = sptr->pat;
	
	pptr = &(sptr->head);
	newpptr = &(newsptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	  {
	    newpptr[i].x = pptr[i].x + deltax + p->dx;
	    newpptr[i].y = pptr[i].y + deltay + p->dy;
	  }
	NewObject( newsptr, id->type, 0, 0, 0, 0, 0, 0, 0, NULL,
		refresh, 0, 0 );
      }
    else
      {
	/* Check that this is possible. */
	if (Groups[MAXGROUP - 1])
	  {
	    printf("Sorry, no room for more groups.\n\r");
	    return;
	  }
	  
	/* A group.  Construct a new group with old group as contents. */
	if ((string = (char *) malloc ( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't copy group name.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	sprintf( string, "%s, copy %d", id->name, CopyCount++ );
	if ((idptr = (ITEM_LIST_PTR *) malloc(sizeof(ITEM_LIST_PTR))) == NULL)
	  {
	    printf("Out of memory.  Can't copy group contents.\n\r");
	    free( string );
	    return;
	  }
	idptr->prev = NULL;
	idptr->next = NULL;
	idptr->itemdesc = id;
	idptr->flag = 0;
	idptr->dx = p->dx;
	idptr->dy = p->dy;
	idptr->call = 0;
	NewObject( idptr, GroupObj, 0, CopyCount, 0, 0, 0, 0, 0,
		string, 0, deltax, deltay );
	NewGroup( CurrentObject, 0, refresh );
      }
  }

/*
 *  This internal routine is a helper routine for RotateObject().
 *  It will rotate a single point about (x0, y0), given the sine and
 *  cosine of the angle.
 */
 
RotatePoint( x, y, sine, cosine, x0, y0 )
	short *x, *y;
	double sine, cosine;
	short x0, y0;
  {
    register short newx, newy;
    
    if (Debug)
	printf("    (%d, %d) rotates to ", *x, *y );
    
    /* Compute the distance of the point from the center of rotation. */
    newx = x0 + (cosine * (*x - x0) + sine * (*y - y0));
    newy = y0 + (cosine * (*y - y0) - sine * (*x - x0));
    *x = newx;
    *y = newy;
    
    if (Debug)
	printf(" (%d, %d)\n\r", *x, *y );
  }

/* 
 *  This internal routine is a helper routine for ScaleObject().
 *  It will scale a single point about (x0, y0), given the scale factor.
 */
 
ScalePoint( x, y, factor, dummy, x0, y0 )
	short *x, *y;
	double factor, dummy;
	short x0, y0;
  {
    register short newx, newy;
    
    if (Debug)
	printf("    (%d, %d) scales to ", *x, *y );
    
    /* Scale the point. */
    newx = ((*x - x0) * factor) + x0;
    newy = ((*y - y0) * factor) + y0;
    *x = newx;
    *y = newy;
    
    if (Debug)
	printf(" (%d, %d)\n\r", *x, *y );
  }

/*
 *  This routine will transform and raise an object.  Currently, the two
 *  transformation are rotation about a point, and scaling.
 */
 
TransformObject( p, fact1, fact2, x0, y0, dx, dy, refresh, erase, ptrans )
	ITEM_LIST_PTR *p;
	double fact1, fact2;	/* Point Transform factors. */
	short x0, y0;		/* Center of transformation. */
	short dx, dy;		/* Cumulative offset. */
	short refresh, erase;
	register (*ptrans)();	/* Point transformation routine. */
  {
    register ITEM_DESCRIPTOR *id;
    char *string;
    SPLINE *sptr, *newsptr;
    POINT *pptr, *newpptr;
    short i, x, y;
    ITEM_LIST_PTR *idptr, *gptr, *newgptr;
    
    
    /* For non groups, transform about the proper point. */
    id = p->itemdesc;
    if (id->type == TextObj)
      {
	/* Text.  Transform about the alignment point. */
	if ((string = (char *) malloc( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't transform any text.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->data );
	switch (id->typedata)
	  {
	    /* Transform about the positioning point. */
	    case PositionLeft:
		x = id->xmin + p->dx + dx;
		y = id->ymin + p->dy + dy;
		ptrans( &x, &y, fact1, fact2, x0, y0 );
		x = (id->xmin + p->dx + dx) - x;
		y = (id->ymin + p->dy + dy) - y;
		break;
	    
	    case PositionCenter:
		x = ((id->xmin + id->xmax) / 2) + p->dx + dx;
		y = id->ymin + p->dy + dy;
		ptrans( &x, &y, fact1, fact2, x0, y0 );
		x = (((id->xmin + id->xmax) / 2) + p->dx + dx) - x;
		y = (id->ymin + p->dy + dy) - y;
		break;
	    
	    case PositionRight:
		x = id->xmax + p->dx + dx;
		y = id->ymin + p->dy + dy;
		ptrans( &x, &y, fact1, fact2, x0, y0 );
		x = (id->xmax + p->dx + dx) - x;
		y = (id->ymin + p->dy + dy) - y;
		break;
	  }
	dx -= x;
	dy -= y;
	if (erase)
	    EraseObject( p, refresh );
	NewObject( string, TextObj, id->subtype, id->typedata,
		id->xmin + p->dx + dx, id->xmax + p->dx + dx,
		id->ymin + p->dy + dy, id->ymax + p->dy + dy,
		id->base, NULL, refresh, 0, 0 );
      }
    else if (id->type != GroupObj)
      {
	/* Splines.  Transform all of the points. */
	sptr = (SPLINE *) id->data;
	if ((newsptr = (SPLINE *) malloc( sizeof(SPLINE) +
		sptr->numvert * sizeof(POINT) )) == NULL)
	  {
	    printf("Out of memory.  Can't transform a spline object.\n\r");
	    return;
	  }
	newsptr->order = sptr->order;
	newsptr->numvert = sptr->numvert;
	newsptr->nib = sptr->nib;
	newsptr->border = sptr->border;
	newsptr->closed = sptr->closed;
	newsptr->filled = sptr->filled;
	newsptr->opaque = sptr->opaque;
	newsptr->pat = sptr->pat;
	
	pptr = &(sptr->head);
	newpptr = &(newsptr->head);
	for (i = 0;  i < sptr->numvert;  i++)
	  {
	    newpptr[i].x = pptr[i].x + dx + p->dx;
	    newpptr[i].y = pptr[i].y + dy + p->dy;
	    ptrans( &(newpptr[i].x),&(newpptr[i].y),fact1,fact2,x0,y0 );
	  }
	if (erase)
	    EraseObject( p, refresh );
	NewObject( newsptr, id->type, 0, 0, 0, 0, 0, 0, 0, NULL,
		refresh, 0, 0 );
      }
    else
      {
	/* Groups.  Transform all of the objects within the group. */
	if ((string = (char *) malloc ( MAXLEN + 1 )) == NULL)
	  {
	    printf("Out of memory.  Can't transform group name.\n\r");
	    return;
	  }
	for (i = MAXLEN + 1;  i--;)
	    string[i] = '\0';
	strcpy( string, id->name );
	
	/* Transform the contents. */
	gptr = (ITEM_LIST_PTR *) id->data;
	if ((newgptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
	    == NULL)
	  {
	    printf("Out of memory.  Can't transform group contents.\n\r");
	    free( string );
	    return;
	  }
	newgptr->prev = NULL;
	newgptr->next = NULL;
	dx += p->dx;
	dy += p->dy;
	while (gptr)
	  {
	    TransformObject( gptr, fact1, fact2, x0, y0,
		dx + gptr->dx, dy + gptr->dy, 0, 0, ptrans );
	    if ((idptr = (ITEM_LIST_PTR *) malloc( sizeof(ITEM_LIST_PTR) ))
		== NULL)
	      {
		printf("Out of memory.  Can't transform a group.\n\r");
		return;
	      }
	    idptr->itemdesc = CurrentObject->itemdesc;
	    idptr->dx = 0;
	    idptr->dy = 0;
	    idptr->call = 0;
	    idptr->flag = 0;
	    idptr->next = NULL;
	    if (newgptr->next)
		newgptr->next->next = idptr;
	    else
		newgptr->prev = idptr;
	    idptr->prev = newgptr->next;
	    newgptr->next = idptr;
	    gptr = gptr->next;
	  }

	if (erase)
	    EraseObject( p, 0 );
	RedrawActivelist( 0 );
	NewObject( newgptr->prev, GroupObj, 0, 0, 0, 0, 0, 0, 0,
		string, 0, 0, 0 );
	NewGroup( CurrentObject, 1, 0 );
	if (refresh)
	    RedrawActivelist( 1 );
      }
  }

/*
 *  This routine will rotate and raise an object.
 */
 
RotateObject( p, theta, x0, y0, refresh, erase )
	ITEM_LIST_PTR *p;
	double theta;		/* Angle of rotation. */
	short x0, y0;		/* Center of rotation. */
	short refresh, erase;
  {
    double sine, cosine;
    
    /* Check parameters. */
    if (p == NULL)
	return;
    
    if (Debug)
      {
	printf("Rotate object %d through %f about (%d, %d)\n\r",
		p, theta, x0, y0);
	DebugItemDescriptor( p->itemdesc, 0 );
      }
    
    /* Transform the object. */
    sine = sin( theta );
    cosine = cos( theta );
    TransformObject( p, sine, cosine, x0, y0, 0, 0,
	refresh, erase, RotatePoint );
  }

/*
 *  This routine will scale and raise an object.  To do this, it invokes
 *  TransformObject(), which will transform an object based on the
 *  point transformation routine passed to it.
 */
 
ScaleObject( p, factor, x0, y0, refresh, erase )
	ITEM_LIST_PTR *p;
	double factor;		/* Scale factor */
	short x0, y0;		/* Center of transformation */
	short refresh, erase;
  {
    /* Check parameters. */
    if (p == NULL)
	return;
    
    if (Debug)
      {
	printf("Scale object %d by %f about (%d, %d)\n\r",
		p, factor, x0, y0);
	DebugItemDescriptor( p->itemdesc, 0 );
      }
    
    /* Transform the object. */
    TransformObject( p, factor, (double) 0, x0, y0, 0, 0,
	refresh, erase, ScalePoint );
  }
