/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */

/*
 * This file, symbols.c, is part of the V debugger.  It contains the routines
 * needed to find and return the symbol table entries stored on the file
 * produced by ld68.  The symbols themselves are cached for quicker access so
 * major portions of the code deal with handling of the cache.  This file
 * includes InitCache, QueryCache, FindSymByValue, FindSymByName, GetSymbolRec,
 * getsymval, PrintSym, and FindRegister.
 */

#include <Venviron.h>
#include <b.out.h>
#include <Vio.h>
#include "Vdb.h"

extern ElementOf(), bufput(), numout();


/*
 * Declarations for a self-organizing symbol table cache.
 * The cache is maintained as a linked list of records which contain
 * symbol value ranges and the associated symbol table entry.
 * All references result in placing the cache entry at the beginning of the
 * list.
 */

typedef struct
  {
    long lowAddress;
    long highAddress;
    int offset;
    long next;
    SymRec symEntry;
  } CacheType;

CacheType Cache[CacheSize];

long HighAddress;
int NumberCacheHits = 0, NumberCacheMisses = 0;
int NumCallsByVal = 0, NumCallsByName = 0;

SymRec UsrSymbol;


InitCache()
  {
    int i;

    for (i = 0; i < CacheSize; i++)
      {
	Cache[i].lowAddress = 0;
	Cache[i].highAddress = 0;
	Cache[i].next = i + 1;
      }
    Cache[CacheSize-1].next = 0;
  }




QueryCache()
  {
    printf("Cache hits: %d,    Cache misses: %d,    Total: %d\n",
    	NumberCacheHits, NumberCacheMisses, 
	NumberCacheHits + NumberCacheMisses);
    printf("Cache hit ratio: %d\n", 
	(NumberCacheHits * 100) / (NumberCacheHits + NumberCacheMisses));
    printf("Number of calls to FindSymByValue: %d\n", NumCallsByVal);
    printf("Number of calls to FindSymByName: %d\n", NumCallsByName);
  }



/*  FindSymByValue  is passed a long value "v" and
 *  a pointer to a file containing the symbol table. It searches the
 *  file linearly for the symbol whose value is closest to, but not
 *  greater than, v.  It then returns a pointer to a sym structure which
 *  contains the type, length, and value of the newfound symbol and a pointer
 *  to a string containing the symbol's name.
 *  
 *  The routine also discovers the offset of the symbol table entry from the
 *  beginning of the file.  PrintSym can then use that entry to
 *  to find the symbol in the file and copy the information.
 *
 *  In the next generation this should be changed to read in a block
 *  of the symbol table at a time and search through that.  It would
 *  help if the symbols were kept in numerical order, however.
 */


SymRec *FindSymByValue ( v, symfileptr )
    register long v;
    File *symfileptr;
  {
    register int current, best;			/* Offsets in symbol file. */
    long bestvalue, currentvalue;
    int length;					/* Length in bytes of symbol. */
    register int i, j, k;

    if ((v < (long) Origin) || (v > (long) DebugeeTeamSize))
				/* Is this value even reasonable? */
      return(NULL);

    NumCallsByVal += 1;
    /*
     * Look for v in the cache.
     */
    for (k = -1, j = 0, i = Cache[0].next; i != 0; 
    		k = j, j = i, i = Cache[i].next)
      {
	if ((Cache[i].lowAddress <= v) && (v < Cache[i].highAddress))
	  {
	    bestvalue = Cache[i].lowAddress;
	    best = Cache[i].offset;
	    NumberCacheHits += 1;
	    Cache[j].next = Cache[i].next;
				/* Place cache hit entry at front of cache. */
	    Cache[i].next = Cache[0].next;
	    Cache[0].next = i;
	    break;
	  }
      }
    if (i == 0)			/* Couldn't find v in the cache. */
      {
        NumberCacheMisses += 1;
	bestvalue = 0;
	fseek( symfileptr, SYMPOS, 0 );
				/* Start at the beginning of the file */
	HighAddress = (long) DebugeeTeamSize;
    
	for (current = 0; current < filhdr.ssize; 
			current += SYMLENWOUTNAME + length ) 
	  {						  
	    fseek( symfileptr, 1, 1 );		/* Skip the type. */
	    length = getc( symfileptr );		/* Get the length. */
	    currentvalue = getsymval( symfileptr );	/* Get the value. */
	    fseek( symfileptr, length+1, 1 );	/* Skip to end of entry */
    
	    if (bestvalue < currentvalue && currentvalue <= v)
	      {
		best = current;
		bestvalue = currentvalue;
	      }
	    if ((v < currentvalue) && (currentvalue < HighAddress))
		HighAddress = currentvalue;
	    if( (length & 1) == 0 )			
				/* Ld68 has the symbol entries
				   on even byte boundaries. */
	      {
		fseek( symfileptr, 1, 1 );
		current += 1;
	      }
	  }
	/* j points to the last record on the cache list. */
	Cache[j].lowAddress = bestvalue;
	Cache[j].highAddress = HighAddress;
	Cache[j].offset = best;
	GetSymbolRec(&(Cache[j].symEntry), symfileptr, best);
	Cache[k].next = 0;	/* Put cache entry at front of cache. */
	Cache[j].next = Cache[0].next;
	Cache[0].next = j;
      }

    /* The first entry in the cache now contains the right symbol. */
    if (bestvalue != 0 && v - bestvalue < LargestOffset)
      {
	offset = best;
	return(&(Cache[Cache[0].next].symEntry));
      }
    else
	return(NULL);
  }




/* Given a symbol name, FindSymByName finds its type, length, and value.
 *  A ptr to the stored symbol entry is returned.
 *  Note the parallel between this and FindSymByValue.
 */


struct sym *FindSymByName ( name, symfileptr )
    char *name;
    File *symfileptr;
  {
      register int current;
      int length, found, i;

    NumCallsByName += 1;
      fseek( symfileptr, SYMPOS, 0 );	/* Start at the beginning of 
					   the file */
    
      found = 0;
      current = 0; 			/* Current offset from SYMPOS. */

      while( current < filhdr.ssize && !found )

        {						  
          fseek( symfileptr, 1, 1 );		/* Skip the type. */
          length = getc( symfileptr );		/* Get the length. */
          fseek( symfileptr, 4, 1 );		/* Skip the value. */

          i=0;
	  if (strlen(name) == length)
	    {
              found = 1;	/* This could be the symbol! */
              while( i<=length && found )
				/* Do all length characters match? */
	        {
                  found =  ( getc( symfileptr ) == name[i++] );
		}
	    }
	  else
	    found = 0;
          fseek( symfileptr, length+1 - i, 1 );	
				/* Skip uncompared characters. */

          if( !found )
            {
              if( (length & 1) == 0 )
				/* The next entry must, by ld68,
				   begin at an even boundary. */
                {
                  current += 1;
                  fseek( symfileptr, 1, 1 );
                }
	      current += SYMLENWOUTNAME + length;
            }
        }

      if( found )
        {
	  GetSymbolRec(&UsrSymbol, symfileptr, current);
	  return(&(UsrSymbol.symbol));
        }
      else
	  return(NULL);
  }


/*
 * GetSymbolRec:
 * Stores the designated symbol table entry at symptr.
 * Adjusts the symbol type for the debugger's tastes.
 */

GetSymbolRec(symptr, symfileptr, offset)
    SymRec *symptr;
    File *symfileptr;
    int offset;
  {
    int i;

    fseek(symfileptr, SYMPOS + offset, 0);
    symptr->symbol.stype = getc(symfileptr);
    switch (symptr->symbol.stype & 7)
      {
	case TEXT:	symptr->symbol.stype = INSTTYPE; break;
	case BSS:
	case DATA:	symptr->symbol.stype = LONGTYPE; break;
	default:	symptr->symbol.stype = 0; break;
      }
    symptr->symbol.slength = getc(symfileptr);
    symptr->symbol.svalue = getsymval(symfileptr);
    for (i = 0; ((i < MAXSYMLEN) && (i < symptr->symbol.slength)); i++)
	symptr->symName[i] = getc(symfileptr);

    symptr->symName[i] = '\0';
  }


/* Getsymval returns the value of the symbol stored on the symbol file.
 */

long getsymval( fileptr )
  File *fileptr;

{
  long value;
  value = getc( fileptr ) << 24;
  value |= ( getc( fileptr ) << 16 ) & 0xFF0000;
  value |= ( getc( fileptr ) << 8 ) & 0xFF00;
  value |= getc( fileptr ) & 0xFF;
  return( value );
}


/* Given a value "v", look for a symbol whose value is 'closest' to "v".
 * If one is found, output the name of that symbol, otherwise, output the
 * numeric value of "v" in the current radix.
 */

PrintSym ( v, fileptr )
    register long v;
    File *fileptr;
  {
      SymRec *symptr;
      int i;

      symptr = FindSymByValue(v, fileptr);
      if (symptr)
        {
	  for (i = 0; symptr->symName[i] != '\0'; i++)
				/* Print the symbol. */
	      bufput(symptr->symName[i]);

          if (symptr->symbol.svalue < v) 
            {
	      bufput('+');			  /* And any offset from it. */
              numout(v - symptr->symbol.svalue);
            }
        }
      else
        numout(v);
  }


/*
 * Look up a register designator string in the Vdb symbol table. 
 * Return a pointer to the appropriate location storing the value of
 * the designated register.
 */
long *FindRegister(string)
    char *string;
{
    if ((string[0] != '%') || (string[3] != '\0'))
	return(NULL);
    switch (string[1])
      {
        case 'd':	if (ElementOf(string[2], "01234567"))
		            return(&SavedRegisters[(string[2] - '0')]);
			else
			    return(NULL);

	case 'a':	if (ElementOf(string[2], "01234567"))
			    return(&SavedRegisters[8 + (string[2] - '0')]);
			else
			    return(NULL);

	case 'f':	if (string[2] == 'p')
			    return(&SavedRegisters[FP]);
	    		else
			    return(NULL);

	case 's':	switch (string[2])
	      		  {
			    case 'p':	return(&SavedRegisters[SP]);
			    case 'r':	return(&StatusRegister);
			    default:	return(NULL);
	      		  }

	case 'p':	if (string[2] == 'c')
			    return((long *) &ProgramCounter);
	    		else
			    return(NULL);

	default:	return(NULL);
      }
}
