/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */


/* This file, stack.c, is part of the V debugger.  It contains the routines
 * necessary to implement the StackDump command.
 */


#include <b.out.h>
#include <Vio.h>
#include "Vdb.h"




extern char *StackBottom;

extern PrintAddress(), GetMemLong();
extern short GetInstrWord();


char *kernelname[] =
  {
    "Forward",
    "GetPid",
    "MoveFrom",
    "MoveTo",
    "ReceiveSpe",
    "RereadMsg",
    "Send",
    "ReplyWithS",
    "ReceiveWit",
    "endofnames"
  };



/* StackDump prints the sequence of calls that led to the routine in which
 *  execution is halted, giving the address of each call and the current values
 *  of the first MAXNUMPARAMSTOPRINT parameters of each routine.
 *  Variables ending in "fp" hold various values of the frame pointer.  For
 *  full details of the C calling conventions for the SUN, contact your
 *  nearest SUN User's Guide, Chapter 2.
 */


StackDump( disassembledCode, maxLines )
	char *disassembledCode;
	int maxLines;
  {
    char *currentfp, *returnpc, *nextfp, *correctcurrentfp, *correctnextfp;
    unsigned short lastinstr, currentinstr;
    SystemCode error = OK;

    cbfp = &OutputBuffer;
    PrintMessage( "Halted at", ProgramCounter );

    currentfp = (char *) SavedRegisters[FP];
    if( currentfp == 0 )
      {
	printf("\nFrame pointer's value is 0. Either execution hasn't begun\n");
	printf(" or the frame pointer hasn't been set correctly.\n");
	return;
      }

/* In the following cases, the frame pointer, a6, has either not been updated
 * or been updated prematurely.  In each case, the current frame pointer
 * should indicate the word above the top of the stack. This occurs in the
 * kernel routines where, for efficiency's sake, linking and unlinking with
 * a6 isn't done; at the beginning of a routine, just before the link statement
 * has been executed; and at the end of a routine, when the unlk has been
 * executed but the return has not.
 */

    if( InKernelRoutine( disassembledCode ) || AtStartofRoutine( disassembledCode ) || AtEndofRoutine() )
      {
	correctcurrentfp = (char *) SavedRegisters[SP] - STACKENTRYSIZE;
	correctnextfp = currentfp;

	PrintParams( correctcurrentfp, correctnextfp, &error );
        if( error != OK )
	  {
	    printf( "\nFrame pointer holds invalid address: %x\n", correctcurrentfp );
	    return;
	  }
	
	if( --maxLines == 0 ) return;

	returnpc = (char *) GetMemLong( (char *) SavedRegisters[SP], &error );
	if( error != OK ) 
	  {
	    printf( "\nStack pointer holds invalid address: %x\n", SavedRegisters[SP]);
	    return;
	  }

	PrintMessage( "Called from", returnpc - SubCallLength( returnpc ) );
      }

    nextfp = (char *) GetMemLong( currentfp, &error );
    if( error != OK ) 
      {
	printf( "\nFrame pointer holds invalid address: %x\n", currentfp );
	return;
      }

    while( nextfp != 0  )
      {
	PrintParams( currentfp, nextfp, &error );
	if( error != OK )
	  {
	    printf( "\nFrame pointer holds invalid address: %x\n", currentfp );
	    return;
	  }

	if( --maxLines == 0 ) return;

        returnpc = (char *) GetMemLong( currentfp + STACKENTRYSIZE, &error );
	if( error != OK ) 
	  {
	    printf( "\nFrame pointer holds invalid address: %x\n", currentfp );
	    return;
	  }

	PrintMessage( "Called from", returnpc - SubCallLength( returnpc ) );  
	currentfp = nextfp;
	nextfp = (char *) GetMemLong( currentfp, &error );
	if( error != OK ) 
	  {
	    printf( "\nFrame pointer holds invalid address: %x\n", currentfp );
	    return;
	  }
      }

    PrintParams( currentfp, nextfp, &error );

    if( error != OK ) 
      printf( "\nFrame pointer holds invalid address: %x\n", currentfp );
  }



/* Determine the symbolic representation of the address and print it.
 */

PrintMessage( message, address )
	char *message, *address;
  {
    printf( "%s ", message );
    PrintAddress( address );
  }



/* Print the first MAXNUMPARAMSTOPRINT parameters of the routine whose frame is
 * indicated by currentfp.  Note that the parameters are stored 2 words below the
 * frame pointer.  Refer to the Sun User's Guide for details of the C calling
 * conventions.
 */

PrintParams( currentfp, nextfp, error )
	char *currentfp, *nextfp;
	SystemCode *error;
  {
    int i, nparams;
    char *paramaddress, *stackaddress;
    long param;

    if( nextfp != 0 )
      nparams = GetNumofParams( currentfp, error );
    else
      nparams = GetNumofProcessParams( currentfp );

    if( *error != OK ) return;

    putchar('(');
    stackaddress = currentfp + 2 * STACKENTRYSIZE;

    if( nparams != 0 )
      {
	paramaddress = stackaddress;
	param = GetMemLong( paramaddress, error );
	if( *error != OK ) return;
	putchar(' ');
	printnum( param );

	for( i = 2; i <= nparams; i++ )
	  {
	    paramaddress += STACKENTRYSIZE;
	    param = GetMemLong( paramaddress, error );
	    if( *error != OK ) return;
	    printf(", ");
	    printnum( param );
	  }
      }
    printf( " )    Stack Address: %x\n", stackaddress );
  }



/* Get the number of parameters by finding the instruction at the return
 * address of the calling routine.  The compiler's convention is to pop
 * the stack at that point by adding the appropriate number of bytes to the
 * stack pointer: 4 if there was one parameter, 8 if there were two, etc. 
 * Unfortunately, there are two separate add instructions, and this whole
 * routine is quite compiler dependent.
 */


GetNumofParams( currentfp, error )
	char *currentfp;
	SystemCode *error;
  {
    char *returnpc;
    unsigned short instruction;
    int nameindex;
    int count;

    returnpc = (char *) GetMemLong( currentfp + STACKENTRYSIZE, error );
    if( *error != OK ) 
      {
	printf( "\nFrame pointer holds invalid address: %x\n", currentfp );
	return( -1 );
      }

    instruction = GetInstrWord( returnpc, error );
    if( *error != OK ) return( -1 );

    switch( instruction )
      {
	case ADDQL4 :  return( 1 );
	case ADDQL8 :  return( 2 );
	case ADDL   :  count = GetMemLong( returnpc + 2, error ) / 4;
			if( *error != OK )
			  {
			    printf( "\nReturn address holds invalid address: %x\n", returnpc );
			    return( -1 );
			  }
		       if( count > MAXNUMPARAMSTOPRINT ) count = MAXNUMPARAMSTOPRINT;
		       return( count );
        default     :  return( 0 );
      }
  }




GetNumofProcessParams( currentfp )
	char *currentfp;
  {
    int nparams;
    nparams = (int) ( StackBottom - currentfp ) / STACKENTRYSIZE  - 2;
    return( nparams > MAXNUMPARAMSTOPRINT ?  MAXNUMPARAMSTOPRINT : nparams );
  }




#define SameString !strcmp

/* Given the string produced by the disassembler, which begins with the symbolic
 * address expression, isolate the name of the current routine and check it against
 * the list of kernel routines.  Note that because the disassembler truncates 
 * symbols to ten characters, the kernel name table above has truncated entries.
 * The index of the entry in the table is returned, but not currently used.
 */

InKernelRoutine( string )
	char *string;
  {
    char routinename[ MAXSYMLEN ];
    int i, j;

    i = 0;
    while( (*string != '\0') && (*string != '+') && (*string != ' ') &&
							(*string != '\t') )
	routinename[i++] = *string++;
    routinename[i] = '\0';

    j = 0;
    while( ( !SameString( routinename, kernelname[j] ) ) && !SameString( 
		kernelname[j], "endofnames" ) )
	j++;
    return( SameString( kernelname[j], "endofnames" ) ?  0 : j );
  }


/* Given the string returned by the disassembler, which begins with the symbolic
 * address representation, assume that we're at the start of a routine if no
 * offset has been calculated by the disassembler--i.e. the name is followed by
 * a blank, not a plus sign.  This works because the entries in the symbol table
 * are currently routine names only.  If this should change, it may be better
 * to get inaccurate stackdumps at the start of a routine rather than in the middle.
 */

AtStartofRoutine( string )
	char *string;
  {
    while( (*string != ' ') && (*string != '+') && (*string != '\t') )
	string++;
    return( *string == ' '  ||  *string == '\t' );
  }


/* The end of a routine occurs when the current instruction is an RTS and
 * the previous instruction was an UNLK A6.
 */

AtEndofRoutine()
  {
    SystemCode error;
    unsigned short currentinstr, lastinstr;
    currentinstr = GetInstrWord( ProgramCounter, &error );
    lastinstr = GetInstrWord( ProgramCounter - 1, &error );
    return( ( currentinstr == RTS ) && ( lastinstr == UNLKA6 ) );
  }


/* SubCallLength determines which possible subroutine call instruction
 * was used just before a given return address: JSR, BSR with a short
 * offset, or BSR with a long offset, and returns the number of bytes
 * in the instruction.
 */

SubCallLength( returnpc )
	char *returnpc;
  {
    SystemCode error;

    if( (GetInstrWord( returnpc - JSRLENGTH, &error ) & JSRMASK ) == JSRINST )
	return( JSRLENGTH );

    else if( (GetInstrWord( returnpc - BSRLONGLENGTH, &error ) & BSRLONGMASK )
								 == BSRINST )
	return( BSRLONGLENGTH );

    else if( (GetInstrWord( returnpc - BSRSHRTLENGTH, &error ) & BSRSHRTMASK )
								 == BSRINST )
	return( BSRSHRTLENGTH );

    else
	return( 0 );
  }
