/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */

/* This file, memoryaccess.c, is part of the V debugger.  It maintains a
 * buffer containing a copy of INSTRBLK bytes of program memory.  It includes:
 * GetFromMemBuffer, GetMemByte, GetMemWord, GetMemLong, GetInstrWord, PutMem,
 * PutMemData (the replacement for PutMemByte, PutMemWord, PutMemLong, and
 * PutMemString), InvalidMemBuffer, ValidInstrPc, ValidAddress, 
 * and QueryMemBuffer.
 */


#include <b.out.h>
#include <Vio.h> 
#include "Vdb.h"


unsigned char MemBuffer[INSTRBLK] = { 0 };
				/* Buffer for program memory. */
char *bfrbegpc, *bfrendpc;	/* First and last pc's in current MemBuffer. */
int BfrValid = 0;		/* Signals that mem. buffer may be invalid. */


/* GetFromMemBuffer ensures that the byte pointed to by pc is in
 * MemBuffer unless an error condition occurs. It reads the next
 * INSTRBLK bytes starting at that pc if *pc is not already in MemBuffer,
 * returning an error code if, for some reason, the contents of pc can
 * not be obtained.
 * Other routines which must access program memory do so through the
 * MemBuffer--GetFromMemBuffer is the only one with a MoveFrom instruction,
 */

SystemCode GetFromMemBuffer( pc )
	char *pc;
  {
    SystemCode error;
    int blocksize, bytestoend;

    if (BfrValid && (bfrbegpc <= pc) && (pc <= bfrendpc))
      return(OK);

    if( (pc < Origin) || (pc >= DebugeeTeamSize) )
      return(BAD_ADDRESS);

    bytestoend = (int)( DebugeeTeamSize - (int) pc );
    if( bytestoend < INSTRBLK )
      blocksize = bytestoend;
    else
      blocksize = INSTRBLK;

    error = MoveFrom( CurExceptionPid, MemBuffer, pc, blocksize );
    if (error != OK)
      {
        BfrValid = 0;
        printf("ERROR--while trying to access debuggee's memory\n %s\n", ErrorString( error ) );
      }
    else
      {
        bfrbegpc = pc;
        bfrendpc = pc + blocksize - 1;
        BfrValid = 1;			/* Valid mem. buffer */
      }
    return(error);
  }



/* Fetch the byte at loc. */

char GetMemByte( loc, error )
	char *loc;
	SystemCode *error;
  {
    if( (*error = GetFromMemBuffer(loc)) != OK)	return(-1);
    return( MemBuffer[ loc - bfrbegpc ] );
  }


/* Fetch the word at loc. */

short GetMemWord( loc, error )
	char *loc;
	SystemCode *error;
  {
    short word;

    if( (*error = GetFromMemBuffer(loc)) != OK ) return(-1);
    word = ( (short) MemBuffer[ loc - bfrbegpc] ) << 8;
    if( (*error = GetFromMemBuffer(loc+1)) != OK ) return(-1);
    word |= ( (short) MemBuffer[ loc+1 - bfrbegpc ] ) & 0xFF;
    return( word );
  }


/* Fetch a long at loc. */

long GetMemLong( loc, error )
	char *loc;
	SystemCode *error;
  {
    long lword;
    if( (*error = GetFromMemBuffer(loc) ) != OK ) return(-1);
    lword = ( (long) MemBuffer[ loc - bfrbegpc ] ) << 24;
    if( (*error = GetFromMemBuffer(loc+1) ) != OK ) return(-1);
    lword |= ( ( (long) MemBuffer[ loc+1 - bfrbegpc ] ) << 16 ) & 0xFF0000;
    if( (*error = GetFromMemBuffer(loc+2) ) != OK ) return(-1);
    lword |= ( ( (long) MemBuffer[ loc+2 - bfrbegpc ] ) << 8 ) & 0xFF00;
    if( (*error = GetFromMemBuffer(loc+3) ) != OK ) return(-1);
    lword |= ( (long) MemBuffer[ loc+3 - bfrbegpc ] ) & 0xFF;
    return( lword );
  }


/*
 * Determines if the instruction is a breakpoint instruction and returns
 * its true value.
 */

short GetInstrWord( pc, error )
    short *pc;
    SystemCode *error;
  {
    int bpno;
    short instrVal;

    instrVal = GetMemWord(pc, error);
    if ( instrVal == BPINST )
      {
	bpno = FindUserBreakpoint(pc);
	if (bpno != 0)
	  instrVal = bp[bpno].oldinst;
      }
    return( instrVal );
  }


/*
 * Here to PutMem data
 */
int PutMem( type, adrs, data )
	short type;
	char *adrs;
	long data;
{
    short bpno;
    char byte;
    short word;

    switch (type)
      {
	case 0:			/* Default PutMem */
	case LONGTYPE:
	    return(PutMemLong(adrs, &data));

	case CHARTYPE:
	case BYTETYPE:
	    byte = (char) data;
	    return(PutMemByte(adrs, &byte));

	case WORDTYPE:
	case INSTTYPE:
	    bpno = FindUserBreakpoint( adrs);
	    if ( bpno != 0 )
	      {
		bp[ bpno ].oldinst = (short) data;
		return(2);
	      }
	    else
	      {
		word = (short) data;
		return(PutMemWord(adrs, &word));
	      }

	default:
	    return( -1 );
      }
}

/*
 * Function to deposit data in team space.
 */

PutMemData( address, data, nbytes)
	char *address;
	char *data;
	int nbytes;
  {
    SystemCode error;
    int i, indx;

    if( address < Origin  ||  address + nbytes - 1 >= DebugeeTeamSize )
      return( -1 );
    error = MoveTo( CurExceptionPid, address, data, nbytes );
    if( error != OK ) return( -1 );

    /* Deposit data at loc in MemBuffer. */
    for (i = 0; i < nbytes; i++)
	if (BfrValid && (bfrbegpc <= address) && (address <= bfrendpc))
	  {
	    indx = address++ - bfrbegpc;
            MemBuffer[indx] = *data++;
	  }
     return( nbytes );
  }



/*
 * InvalidMemBuffer:
 * Invalidates the memory buffer cache.
 */

InvalidMemBuffer()
  {
    BfrValid = 0;
  }


/*
 * Checks if pc points to a legal instr.
 */

ValidInstrPc(pc)
    short *pc;
  {
    int legal;

    if ((((char *)pc) >= Origin) && 
        (((char *)pc) < DebugeeTeamSize) && (!(((long)pc) & 1)))
      {
        dasm(pc, Symfile, &legal);
	if (legal)
	    return(1);
      }
    return(0);
  }


ValidAddress( address )
	char *address;
  {
    return( address >= Origin  &&  address < DebugeeTeamSize );
  }



QueryMemBuffer()
  {
    printf("BfrValid %d  bfrbegpc %x  bfrendpc %x \n",
	BfrValid, bfrbegpc, bfrendpc );
  }
