/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

 
/*
 * Marvin Theimer, Eric Berglund,  5/83
 */

/* This file, ioutilities.c, is part of the V debugger.  It contains routines
 * for reading and parsing user input, and some short routines used for output
 * to the screen and to a buffer.  Included in this file are ReadInputLine,
 * ParseToGetCommand, ParseToGetArg, ParseForSymbol, ParseForSymbolValue,
 * NumericConstant, ElementOf, bufput, numout, printnum, and PrintAddress.
 */


#include <b.out.h>
#include <Vio.h>
#include "Vdb.h"

#define MAX_EOFS_ALLOWED	2

extern struct sym *FindSymByName();
extern long *FindRegister();
short ElementOf();
char *ParseForSymbolValue();


char *SymTerms = "|+-,";	/* Characters which delimit the end of a
				   symbol.  NOTE: '\0' may also delimit
				   the end of a symbol. */
char *DisplaySyms = "/\\@;=";	/* Characters which denote display commands. */
char *InputTerms = "\r\n/\\@;=";
				/* Characters which delimit the end of an
				   input line. */
char UserString[80];		/* holds user-specified string */



/*
 * ReadInputLine:
 * Returns user's next input line.
 */

ReadInputLine(buffer)
    char *buffer;
  {
    register char *sptr = buffer;
    char c;
    int len = 0;
    int inString = 0;
    int eofCount = 0;

    c = getchar();
    if( c == EOF && stdin->lastexception == BAD_BLOCK_NO )
      {
	Resynch( stdin );
      }
    ungetc( c, stdin );

    do
      {
	c = getchar();

	if( c == EOF )
	  {
	    eofCount++;
	    printf("Hit EOF\n");
	    if( eofCount > MAX_EOFS_ALLOWED ) return( -1 );
	  }

	else if ((c == ' ') && !inString)
	    putchar(c);		/* Echo blanks but don't input them. */
        else if ((c == '\b') || (c == '\177'))
	  {			/* Backspace or DEL */
	    putchar('\b');
	    putchar(' ');
	    putchar('\b');
	    if (sptr > buffer)
	      {
	        len--;
		sptr--;
	      }
	  }
	else if (c == '\25')
	  {			/* Ctrl-u */
	    while (len--)
	      {
	        putchar('\b');
		putchar(' ');
	        putchar('\b');
	      }
	    len = 0;
	    sptr = buffer;
	  }
        else if ((c == '\n') || (c == '\r'))
	  {
            putchar('\n');
	    len++;
	    *sptr++ = c;
	  }
	else
	  {
	    putchar(c);
	    len++;
	    *sptr++ = c;
	    if (c == '"')
		inString = !inString;
	  }
	Flush(stdout);
      }
    while (!ElementOf(c, InputTerms));
    *sptr = '\0';
    return(len);
  }


/*
 * ParseToGetCommand:
 * Returns the command specified in buffer.
 * Zeroes the command in the input buffer by putting ARGDELIM '\0' at
 * the starting point of the command in the buffer.  This is done
 * so that ParseToGetArg() will not mistake the command as an 
 * additional argument.
 * NOTE: assumes that buffer is terminated by '\0'.
 */

enum DbCmds ParseToGetCommand(buffer, len)
    char *buffer;
    int len;			/* Length of input buffer. */
  {
    char *cmd, c, c1;

    if (len <= 0)
	return(NoCmd);

    /*
     * Find the beginning of the command and make cmd point to it.
     */
    for (cmd = &buffer[len - 1]; cmd > buffer; cmd--)
				/* Terminate the loop pointing to the front of the
				   beginning of the input line if no
				   expression precedes the command. */
      {
        if (ElementOf(*cmd, DisplaySyms))
				/* The command is a display command */
	    break;
	else if (*cmd == ARGDELIM)
				/* Action commands are preceded by ARGDELIM */
	  {
	    cmd++;		/* Inc to point to beginning of command. */
	    break;
	  }
      }
    c = *cmd;
    c1 = *(cmd + 1);
    *cmd++ = ARGDELIM;
    *cmd = '\0';

    switch (c)
      {
	case '\n':
        case '\r':	return(NoCmd);
	case '/':	return(DisplayNextCmd);
	case '\\':	return(DisplayPrevCmd);
	case '@':	return(DisplayPtrCmd);
	case ';':	return(SingleStepCmd);
	case 'n':	return(DisplayNextPageCmd);
	case 'p':	return(DisplayPrevPageCmd);
	case '=':	return(DisplayEqCmd);
	case 'd':	return(DisplayRegCmd);
	case 'r':	if (c1 != 'r')
				return(ReplaceCmd);
			else
				return(ReplaceRegCmd);
	case 'g':	if (c1 != 'b')
			    return(GoCmd);
			else
			    return(GoBrkptCmd);
	case 'x':	if (c1 != 'x')
				return(SingleStepCmd);
			else
				return(SingleStepSbrCmd);
	case 'y':	return(SingleStepSbrCmd);
	case 'b':	return(BpCmd);
	case 'f':	return(SearchCmd);
	case 'm':	return(MaskCmd);
	case 'i':	return(IRadixCmd);
	case 'o':	if (c1 == 'f')
				return(OffsetCmd);
			else if(c1 == 'r')
				return(ORadixCmd);
			else
				return(BadCmd);
	case 't':	if (c1 != 't')
				return(TmpTypeCmd);
			else
				return(TypeCmd);
	case 's':	if (c1 != 'p')
				return(StackTraceCmd);
			else
				return(StopProcessCmd);
	case 'h':	return(HelpCmd);
	case 'q':	return(QuitCmd);
	case 'w':	return(StatsCmd);
	default:	printf("%c? ", c );
			return(BadCmd);
      }
  }


/*
 * ParseToGetArg:
 * Returns the specified input argument in arg and reports whether it is
 * bad (-1), non-existent (0), or OK (1).
 * If no argument exists then arg is returned with value 0.
 * Assumes that the input buffer contains only arguments and ends 
 * with ARGDELIM '\0'.
 */

int ParseToGetArg(buffer, n, arg)
    char *buffer;		/* User's input line. */
    int n;			/* Index of argument to get. */
    long *arg;			/* Value of argument is returned here. */
  {
    int i;
    char *ptr, c;
    long symvalue;
    char del, nxtDel;

    *arg = 0;			/* Default return value is 0. */

    /*
     * Get to the location of the desired arg. in the input buffer.
     */
     i = 1;
     ptr = buffer;
     while (i < n)
      {
        c = *ptr++;
	if (c == '\0')
	    return(0);
	else if (c == ARGDELIM)
	    i++;
      }
    if ((*ptr == '\0') || (*ptr == ARGDELIM))
	return(0);
    /*
     * Parse the desired argument symbol by symbol.
     */
    del = '+';
    while (del != ARGDELIM)
      {
        ptr = ParseForSymbolValue(ptr, &symvalue);
	if (!ptr)
	    return(-1);
	switch (del)
	  {
	    case '|': 
	        *arg |= symvalue;
		break;
	    case '+':
		*arg += symvalue;
		break;
	    case '-':
		*arg -= symvalue;
		break;
	  }
	del = *ptr++;
      }
    return(1);
  }



/*
 * Scan buffer for the next symbol and place it in symbol.  Return the 
 * position of the delimiting character in buffer.
 * NOTE: assumes that buffer is terminated by a symbol terminator char.
 */

char *ParseForSymbol(buffer, symbol, maxlen)
    char *buffer, *symbol;
    int maxlen;
  {
    char c;

    c = *buffer++;
    while(!ElementOf(c, SymTerms))
      {
	if (maxlen-- <= 0)
	    break;
	*symbol++ = c;
        c = *buffer++;
      }
    *symbol = '\0';
    return(buffer-1);
  }


/*
 * Scan buffer for the next symbol, set the value of the symbol,
 * and return the position of the delimiting character.
 * A bad symbol yields a 0 return ptr.
 * NOTE: Assumes that buffer terminates with ARGDELIM '\0' .
 */
char *ParseForSymbolValue(buffer, symvalue)
	register char *buffer;
	register long *symvalue;
  {
    char c, *bufptr;
    char symbol[40];
    char *sptr = symbol;
    char *sptr1 = UserString;
    register struct sym *symadr;
    long *regadr;

    /*
     * insert characters into symbol until we hit an argument terminator
     */
    *symvalue = 0;
    bufptr = ParseForSymbol(buffer, symbol, 40);

    if(symbol[0] == '\0' ) /* case of reading nothing */
	return(bufptr);

    SymType = 0;		/* Default symbol type */

    if ( symbol[0] == '"')      /* if starts with '"' it must be a string
				   - put it in UserString */
      {
        SymType = STRTYPE;
	sptr++;
	*symvalue = (long) UserString;
	while ( *sptr && (*sptr != '"'))
	  {
	    if (*sptr == '\\')
		sptr++;
	    *sptr1++ = *sptr;
	    sptr++;
	  }
	*sptr1 = '\0';
	return(bufptr);
      }

    else if (symbol[0] == '\'')	/* if starts with ' it must be a character
				   constant - return the character's value. */
      {
        SymType = CHARTYPE;
        *symvalue = (int) symbol[1];
	return(bufptr);
      }

    else if ( NumericConstant( symbol, symvalue, InputRadix ) )
        return(bufptr);

    /*
     * Assume what we have is a symbol and look for it in the symbol tables
     */
    else if (strcmp(symbol, ".") == 0)
      {
        *symvalue = (long) Dot;
	return(bufptr);
      }
    else if (symbol[0] == '%')
      {				/* Symbol may be a Vdb symbol. */
	regadr = FindRegister(symbol);
	if (regadr)
	  {
	    SymType = LONGTYPE;
	    *symvalue = *regadr;
	    return(bufptr);
	  }
      }

    symadr = FindSymByName( symbol, Symfile );

    if( symadr )
      {
        SymType = symadr->stype;	/* remember symbol type */
	*symvalue = symadr->svalue;		/* return its value */
	return(bufptr);
      }
    else
	return(NULL);
  }



/* NumericConstant tests whether a particular string can represent a
 * numeral in the specified radix.  If it can, the value of the number
 * is returned in symbolvalue.
 */

#define IsDigit( c, radix )  !( c<'0' || (radix<=10 && c>'0'+radix-1) ||\
					 (radix>10 && c>'9' && c>'A'+radix-11))


NumericConstant( symbolstring, symbolvalue, radix )
	char *symbolstring;
	long *symbolvalue;
	short radix;
  {
    char c;
    int digit;

    *symbolvalue = 0;
    c = *symbolstring++;
    if( c >= 'a' && c <= 'z' ) c = c - 'a' + 'A';

    while( IsDigit( c, radix ) )
      {
	if( c <= '9' )
	  digit = c - '0';
	else
	  digit = c - 'A' + 10;
	*symbolvalue = *symbolvalue * radix + digit;
	c = *symbolstring++;
	if( c >= 'a' && c <= 'z' ) c = c - 'a' + 'A';
      }

    return( c == '\0' );	/* If this is the end of the string, we have
				   a number. */
  }



/*
 * Test character set membership.
 */
short ElementOf(c,string)
	register char c, *string;
  {
    register char m;

    while(( m = *string++ ))
	if( c == m ) return(1);
    return(0);
  }



/* All output is routed through bufput, one character at a time.
 *   As characters are placed in the output buffer, we keep track of the
 *   column position.
 */

bufput(c)
    register char c;
  {
      register int lcol;
   
      lcol = cbfp->col;			/* Cbfp is a pointer to the current
					   output buffer. */
      if ( c == '\t' )
					/* Tabbing. */
          lcol = ( ( lcol + 8 ) / 8 ) * 8;
      else
          lcol++;
      cbfp->col = lcol;
      cbfp->buf[ cbfp->pnt++ ] = c;	
  }


/* Numout converts a number to its ascii representation in the current
 *   output radix, and sends it to the output buffer.
 */

numout( number )
  unsigned long number;
{
  register unsigned short c;

  c = number % OutputRadix;
  if( (number /= OutputRadix) != 0 )
    numout( number );
  c &= 0xFF;
  c += (c > 9) ? ( 'A' - 10 ) : '0';
  bufput(c);
}




/*
 * Print a number in the current output radix.  Return the length of the 
 * string printed.
 */
int printnum(number)
long number;
{
    OutputBuffer.pnt= 0;
    OutputBuffer.col= 1;
    cbfp = &OutputBuffer;
    numout( number );		/* output value of arg */
    bufput('\0');
    printf("%s", OutputBuffer.buf);
    return(OutputBuffer.col);
}

/*
 * Print address at pc in a suitable format.
 */
PrintAddress( pc )
long pc;
{
register struct sym *sp;

    OutputBuffer.pnt= 0;
    OutputBuffer.col= 1;
    cbfp = &OutputBuffer;
    PrintSym( pc, Symfile );
    bufput('\0');
    printf("%s ", OutputBuffer.buf);
}
