/************************************************************************/
/*									*/
/*			(C) COPYRIGHT 1983				*/
/*			BOARD OF TRUSTEES				*/
/*			LELAND STANDFORD JUNIOR UNIVERSITY		*/
/*			STANFORD, CA. 94305, U.S.A.			*/
/*									*/
/************************************************************************/

/*
 * Marvin Theimer, Eric Berglund,  5/83
 */


/*
 * Top level routine for Vdb.
 *
 * Vdb is an assembler-level debugger for C programs running under the 
 * V system on MC68000's.  It expects the V exec to provide initial
 * program setup of itself and the program to debug, and to provide file 
 * io to the .r load file of the program to be debugged.  The load file is
 * expected to contain the symbol table produced by ld68.
 *
 * This file includes the routines main, TryToLoadTeam, SetUpArguments,
 * SetDefaults, ExitDebugger, and Vdebug.
 */


#include <b.out.h>
#include <Vexceptions.h>
#include <Vprocess.h>
#include <Vteams.h>
#include <Vgts.h>
#include <Vioprotocol.h>
#include <Vdirectory.h>
#include "Vdb.h"

extern RootMessage *RootMsg;	/* My root message */

ProcessId DebugeePid;		/* Pid of program to debug. */
ProcessId DebugeeRootPid;

int StopForSure;		/* True at start or access error. */
ProcessId VgtsPid;		/* guess */
int VgtsPresent;		/* True if the Vgts is the stdin fileserver */
int saveMode;			/* Used to save the file mode of stdout if
				   the Vgts is being used. */
#define DebuggerPadMode		(CR_Input+LF_Output)

short *CurBrkPc;		/* pc of last breakpoint stopped at. */
int CurBrkIndx = 0;		/* breakpoint index of last breakpoint
				   stopped at. */

char *StackBottom;		/* Used by StackDump to determine how many
				   parameters a process has. */
RegHandlerRequest regReq;
ProcessId ExceptionServerPid;


/*
 * Imported routines
 */

extern File *OpenPad(), *OpenFile();
extern short *StandardExceptionHandler();
extern File *dasminit();
extern char GetMemByte();
extern short GetInstrWord();
extern int dasm();
extern SetBreakpoint(), ResetBreakpoint(), ClearBreakpoint(),
	RemoveBreakpoint(), FindUserBreakpoint(), InvalidMemBuffer(),
	CommandLoop(), ValidAddress();
extern ProcessId LoadProg();


/*
 * main:
 * Main debug program.
 * This is where all the interactions with the ExceptionServer and the 
 * process's kernel processState are handled.
 */


main(argc, argv)
    int argc;
    char **argv;
  { 
    ProcessId pid, debuggerPid, watcherPid;
    Message	msg;
    RootMessage dRootMsg;
    ExceptionRequest *req = (ExceptionRequest *) msg;
    CreateInstanceRequest *debugReq = (CreateInstanceRequest *) msg;
    char *fName[128];

    ProcessBlock processBlk;
    int i;
    short error;
    char nameBuf[80];
    char *name;			/* Name of program to debug. */
    ProcessId symFileServer;	/* Pid of server from which debugee program
				   was loaded. */
    InstanceId symFileId;	/* File instance of file from which debugee
				   program was loaded. */

    int nArg;			/* Current argument to parse. */
    int debugIoPad = FALSE;	/* True if an io pad should be created for
				   the debugger. */
    int postDebug = FALSE;	/* True if doing a postmortem debug. */
    SystemCode loadError;
    SystemCode moveError;

    PerProcessArea debugeePerProcess;
    PerProcessArea *debugeePerProcessAddr;
    char *assumedStackBottom;
    extern DeathWatcher();

    debuggerPid =  GetPid(ACTIVE_PROCESS, LOCAL_PID);

    /*
     * Parse command flags.
     */
    nArg = 1;
    while (argv[nArg][0] == '-')
      {
	if (strcmp(argv[nArg], "-d") == 0)
	  {
	    debugIoPad = TRUE;
	    nArg++;
	  }
	if (strcmp(argv[nArg], "-p") == 0)
	  {
	    postDebug = TRUE;
	    nArg++;
	  }
      }

    /*
     * Set the debugee program's io.
     */
    dRootMsg = *RootMsg;
    
    VgtsPid = GetPid(VGT_SERVER, LOCAL_PID );
    if ( VgtsPresent = (VgtsPid == FileServer( stdin )) )
      {
	if (debugIoPad)
	  {
	    stdout = OpenPad("debugger",24,80,&error);
	    if (error != OK)
	      {
		printf("ERROR - couldn't open new output pad\n%s\n", ErrorString(error));
		ExitDebugger();
	      }
	    stdin = OpenFile(stdout->fileserver, stdout->fileid, FREAD, &error);
	    if ((error != OK) || (stdin == NULL))
	      {
		printf("ERROR -couldn't open new pad for input.\n%s\n", ErrorString(error));
		ExitDebugger();
	      }
	    SelectPad(stdout);
	  }
	saveMode = QueryPad( stdout );
	ModifyPad(stdout, DebuggerPadMode );
      }

    /*
     * If this is a normal debugging session then
     * try to find the program to debug and then load it.
     * If this is a postmortem debugging session then
     * get the current exception pid from the forwarded exception msg
     * and get the program counter from a call to the 
     * StandardExceptionHandler.
     */
    if (!postDebug)
      {
	name = argv[nArg];

        CurExceptionPid = LoadProg( &argv[nArg], FALSE, 0,
		&dRootMsg, &dRootMsg, &loadError );


	if (CurExceptionPid == NULL)
	    ExitDebugger();

	ProgramCounter = 0;	/* Signals that we should use the value
				   obtained from QueryProcessState. */
      }
    else
      {
	CurExceptionPid = Receive(msg);
	ProgramCounter = StandardExceptionHandler(req, CurExceptionPid, stdout);
      }

    printf("\n\nWelcome to the V debugger.\n\nType q to quit.\n\n");

    /*
     * Get process information for process which incurred the exception.
     */
    QueryProcessState(CurExceptionPid, &processBlk);
    DebugeeRootPid = processBlk.team_root;
    DebugeeTeamSize = processBlk.team_size;
    for (i = 0; i < 16; i++)
        SavedRegisters[i] = (long) processBlk.proc_state.regs[i];
    if (ProgramCounter == 0)
	ProgramCounter = (short *) processBlk.proc_state.pc;
    StatusRegister = (long) processBlk.proc_state.sr;

    assumedStackBottom = DebugeeTeamSize;

    /*
     * Get access to the symbol table file for the debugee program.
     * Then initialize the disassembler and get to the user symbol table. 
     */
    AccessSymFile( DebugeeRootPid, &symFileServer, &symFileId );
    Symfile = dasminit( symFileServer, symFileId );

    /*
     * Enter the debugger to allow the user to examine state and set 
     * breakpoints.
     */
    SetDefaults();
    StopForSure = 1;

    Vdebug( NOT_ATBREAKPOINT, NOT_EXCEPTION );

    StopForSure = 0;
    processBlk.proc_state.sr = (short) StatusRegister;
    processBlk.proc_state.pc = (char *) ProgramCounter;
    for (i = 0; i < 16; i++)
        processBlk.proc_state.regs[i] = (char *) SavedRegisters[i];
    WriteProcessState(CurExceptionPid, &processBlk.proc_state);

    /*
     * Register the debug handler with the exception server.
     */
    ExceptionServerPid = GetPid(EXCEPTION_SERVER, LOCAL_PID);
    if (ExceptionServerPid == 0)
      {
        printf("ERROR--couldn't find the exception server's pid.\n");
        ExitDebugger();
      }

    regReq.requestcode = REGISTER_HANDLER;
    regReq.regFlag = 1;
    regReq.handlerPid = debuggerPid;
    regReq.regPid = DebugeeRootPid;

    Send(&regReq, ExceptionServerPid);
    if ( regReq.requestcode != OK )
      {
        printf("ERROR--couldn't register the debugger to handle exceptions.\n");
	printf("This is often due to an exception server bug that can be\n");
	printf("circumvented by rebooting the workstation.\n");
        ExitDebugger();
      }

    /*
     * Set the debugee program running.
     * First must reset the mode of pad correctly, if necessary.
     */
    if (VgtsPresent && !debugIoPad)
      {
	ModifyPad(stdout, saveMode);
      }

    Reply(&dRootMsg, CurExceptionPid);

    /*
     * Create process to watch for death of the debugee.
     */
    watcherPid = Create( 1, DeathWatcher, 500 );
    Ready( watcherPid, 1, DebugeeRootPid );

    /*
     * Loop forever on exception event msgs.
     */
    while (1)
      {
	CurExceptionPid = Receive(msg);

	QueryProcessState(CurExceptionPid, &processBlk);
	if( !StopProcessFlag  &&  Creator(CurExceptionPid) != debuggerPid )
	  SetTeamPriority( CurExceptionPid, STOP_TEAM_PRIORITY );

	if (req->requestcode == TERMINATE_TEAM)
	    ExitDebugger();

	Flush( stdin );

	DebugeeTeamSize = processBlk.team_size;
	for (i = 0; i < 16; i++)
	    SavedRegisters[i] = (long) processBlk.proc_state.regs[i];
	ProgramCounter = (short *) processBlk.proc_state.pc;
	StatusRegister = (long) processBlk.proc_state.sr;
	debugeePerProcessAddr = (PerProcessArea *) processBlk.proc_state.perProcess;
	moveError = MoveFrom( CurExceptionPid, &debugeePerProcess, debugeePerProcessAddr, sizeof( debugeePerProcess ) );
	if( moveError != OK )
	  StackBottom = assumedStackBottom;
	else
	  StackBottom = (char *)debugeePerProcessAddr + debugeePerProcess.stackSize;

	if (VgtsPresent && !debugIoPad)
	  {
	    saveMode = QueryPad(stdout);
	    ModifyPad(stdout, DebuggerPadMode );
	  }

	if (req->type == TRAP3)
	    Vdebug( ATBREAKPOINT, NOT_EXCEPTION );
	else if ( req->type == TRACETRAP )
	    Vdebug( NOT_ATBREAKPOINT, NOT_EXCEPTION );
	else
	  {
	    ProgramCounter = StandardExceptionHandler(req, CurExceptionPid, stdout);
	    Vdebug( NOT_ATBREAKPOINT, EXCEPTION );
	  }
	if (VgtsPresent && !debugIoPad)
	    ModifyPad(stdout, saveMode);
	processBlk.proc_state.sr = (short) StatusRegister;
	processBlk.proc_state.pc = (char *) ProgramCounter;
        for (i = 0; i < 16; i++)
            processBlk.proc_state.regs[i] = (char *) SavedRegisters[i];
	WriteProcessState(CurExceptionPid, &processBlk.proc_state);
	SetTeamPriority( CurExceptionPid, processBlk.team_priority );

	Reply(&dRootMsg, CurExceptionPid);	/* NOTE: We must reply with dRootMsg
				   because RereadMessage in TeamRoot() is
				   expecting this message in spite of
				   exception messages due to e.g. 
				   single-stepping. */
      }
  }

/*
 * Queries the team server for information about the team with root pid rootPid
 * and returns the file server and file id of the file from which the team
 * was loaded.
 */
AccessSymFile(rootPid, fileServer, fileId)
	ProcessId rootPid;
	ProcessId *fileServer;
	InstanceId *fileId;
  {
    Message msg;
    DescriptorRequest *reqMsg = (DescriptorRequest *) msg;
    DescriptorReply *repMsg = (DescriptorReply *) msg;
    char buffer[sizeof(TeamDescriptor)];
    TeamDescriptor *desc = (TeamDescriptor *) buffer;
    ProcessId teamServerPid;

    reqMsg->requestcode = NREAD_DESCRIPTOR;
    reqMsg->namecontextid = rootPid;
    reqMsg->segmentptr = buffer;
    reqMsg->segmentlen = sizeof(TeamDescriptor);
    reqMsg->dataindex = 0;
    teamServerPid = GetPid(TEAM_SERVER, LOCAL_PID);
    Send(msg, teamServerPid);
    if ( repMsg->replycode != OK )
      {
	printf("ERROR - on access to team server: %s\n", 
		ErrorString(repMsg->replycode));
	ExitDebugger();
      }
    *fileServer = desc->fileserver;
    *fileId = desc->fileid;
  }



/*
 * Initialization routine for Vdb--sets default values for debug parameters.
 */
SetDefaults()
  {
    int i;

    SsSbrFlag = 0;		/* Not singlestepping. */
    SsFlag = 0;
    for ( i = 0 ; i < BPMAX ; i++ ) 
      {
	cbptr = &bp[ i ];
	cbptr->pc = 0;		/* No breakpoints set. */
	cbptr->count = 0;
      }
    LargestOffset = DEFOFFSET;
    InputRadix = DEFRADIX;
    OutputRadix = DEFRADIX;
    TypeOutMode = 0;		/* No default type out mode yet. */
    TempTypeOutMode = 0;
    ActualTypeOutMode = 0;
    StringLength = 80;		/* Default string length. */
    StopProcessFlag = 0;
  }



/*
 * ExitDebugger:
 * Exits the debugger and debugee program.
 */

ExitDebugger()
  {
    Flush( stdout );
    if( DebugeeRootPid != NULL ) DestroyProcess(DebugeeRootPid);

    /* Unregister the debugger. */
    regReq.regFlag = 0;
    Send( &regReq, ExceptionServerPid );
    exit();
  }


/*
 * Helper process to detect and report the death of the debugee.
 */

DeathWatcher( debugeePid )
	ProcessId debugeePid;
  {
    Message msg;
    ExceptionRequest *req = (ExceptionRequest *) msg;

    ReceiveSpecific( msg, debugeePid );

    req->requestcode = TERMINATE_TEAM;
    Send( req, Creator(0) );
  }
   



/* This is the prime entry to the debugger.
 */

Vdebug( atBrkpt, trueException )
	int atBrkpt, trueException;
  {
    short curpc;
    short *brkpc;
    int legal;
    int curpcbp;
    short bpno = 0;
    int atSsbp = 0;
    int atTrbp = 0;
    int stopAtBrkpt;
    SystemCode error;

    if( !ValidAddress( ProgramCounter ) )
      {
	printf( "\nWarning: PC value is illegal address: %x\n", ProgramCounter );
	printf( "PC value has been changed to Origin. Can be reset with rr.\n");
	Flush(stdout);
	ProgramCounter = (short *) Origin;
	atBrkpt = 0;
	StopForSure = 1;
	CurBrkIndx = 0;
	SsFlag = 0;
	SsSbrFlag = 0;
      }

    StatusRegister &= ~TRACEBIT;		/* clear trace bit in sr */

    if (CurBrkIndx)		/* Replace breakpt. at instr. just executed.
				   */
      {
	ResetBreakpoint(CurBrkIndx, CurBrkPc);
	CurBrkIndx = 0;
	bpno = FindUserBreakpoint(ProgramCounter - 1);
				/* Find out if this is a user-set brkpt, too. */
	if ( !SsFlag && !trueException && !bpno )
	    return;
      }

    if (atBrkpt)
      {
	ProgramCounter--;		/* Back up over brk pt instr. */
	if (ProgramCounter == bp[SSBP].pc)
				/* Subroutine ss */
	  {
	    atSsbp = 1;
	    RemoveBreakpoint(SSBP);
	  }
	if (ProgramCounter == bp[TRBP].pc)
				/* Trap ss */
	  {
	    atTrbp = 1;
	    RemoveBreakpoint(TRBP);
	  }

	bpno = FindUserBreakpoint(ProgramCounter);
				/* Find out if this is a user-set brkpt. */
        
	stopAtBrkpt = !( (SsSbrFlag && !atSsbp) || 
			 (atTrbp && !SsFlag && !bpno) ||
			 (!atSsbp && !atTrbp && !bpno) );
				/* DONT stop if stepping over subroutines and
				   at an interior breakpoint, or if stepping
				   over a trap instr. and not single stepping
				   or at a normal breakpoint, or if at a
				   breakpoint which has been removed earlier. */
      }
    else
      {
	stopAtBrkpt = SsFlag || trueException || StopForSure;
	bpno = FindUserBreakpoint(ProgramCounter);
				/* Find out if this is a user-set brkpt. */
      }

    if (stopAtBrkpt)
	  {				/* stop at this breakpoint */
	    Flush( stdout );
	    dasm(ProgramCounter, Symfile, &legal);
					/* Display the current instr. */
	    printf("B%d: %s %s\n", bpno, HexBuffer.buf, OutputBuffer.buf);
	    if (SsFlag) SsCount--;	/* count down single step */
	    if (bpno)		 	/* if breakpoint, */
	      {
		cbptr = &bp[bpno];	/* remember appropriate pointer */
		if( cbptr->count > 0 ) cbptr->count--;
					/* count down the breakpoint counter */
	      }
	    else
		cbptr = NULL;
	    Dot = (char *) ProgramCounter;	/* Dot defaults to here */

	    if ( (SsFlag && SsCount == 0) || (cbptr && cbptr->count == 0) ||
			trueException || StopForSure )
		CommandLoop(&bpno);		/* process commands */
	    Flush(stdout);
	  }

/* we are about to return to the calling program. set breakpoints or
 * turn on tracing.
 */

    curpc = GetInstrWord( ProgramCounter, &error );
    if( error != OK )
      {
	printf( "PC is illegal address: cannot resume execution.\n" );
	ExitDebugger();
      }

    if (bpno)			/* If there is a brkpt. at the current pc
				   location then we have to set it AFTER
				   having executed the actual instr.  Do this
				   by setting the trace bit and setting the
				   brkpt at the next instr. cycle. */
      {
	StatusRegister |= TRACEBIT;
	CurBrkIndx = bpno;
	CurBrkPc = ProgramCounter;
      }

    if (SsFlag)
      {
	StatusRegister |= TRACEBIT;
	if ((SsSbrFlag && (((curpc & 0xFF00) == BSRINST) ||
		           ((curpc & 0xFFC0) == JSRINST))) ||
	    ((curpc & 0xFFF0) == TRAPINST))
	  {			/* If single-stepping over subroutines or at
				   a trap instr. then use a brkpt to step
				   over them. */
	    brkpc = ProgramCounter + dasm( ProgramCounter, Symfile, &legal );
						/* set bp at return */
	    if ((curpc & 0xFFF0) == TRAPINST)
		SetBreakpoint(TRBP, brkpc);
	    else
		SetBreakpoint(SSBP, brkpc);
	    StatusRegister &= ~TRACEBIT;/* Don't want the trace bit on when going
				   over traps or subroutines. */
	  }
      }

    if (bpno)
      ClearBreakpoint(bpno);
    InvalidMemBuffer();			/* Invalidate mem. buffer cache
					   since prog. execution may change
					   things. */

}

