#include <Vgts.h>
#include "checkers.h"

/*
 * This file contains routines responsible for the display of the
 * checker board.
 */


/* Variables and constants: */

/* Manifests for the position and size of the board value indicator. */
#define VALUE_INDICATOR_X 17*SQUARE_SIZE/2
#define VALUE_INDICATOR_Y 3*SQUARE_SIZE/2
#define VALUE_INDICATOR_WIDTH SQUARE_SIZE/4
#define VALUE_INDICATOR_HEIGHT 5*SQUARE_SIZE


short	Sdf, Vgt;

	/*
	 * Declare and initialize the rasters for each piece (and the empty
	 * square). Note that the raster representing each piece also
	 * shows the halftone for the shaded square on which it rests. This
	 * is necessary because in the VGTS, raster items overwrite each
	 * other, rather than being "or"d together.
	 * Note that if the constant SQUARE_SIZE is ever changed, then the
	 * rasters will have to be redesigned to still fit in a square.
	 */

#define raster_file_size SQUARE_SIZE*SQUARE_SIZE/16 + 1
	/* (Actually, this is a little conservative.) */

short	EMPTY_Raster[raster_file_size] = {
#include "rasters/empty.hex"
};

short	RC_Raster[raster_file_size] = {
#include "rasters/rc.hex"
};

short	BC_Raster[raster_file_size] = {
#include "rasters/bc.hex"
};

short	RK_Raster[raster_file_size] = {
#include "rasters/rk.hex"
};

short	BK_Raster[raster_file_size] = {
#include "rasters/bk.hex"
};

short	RHIGHLIGHT_Raster[raster_file_size] = {
#include "rasters/rhighlight.hex"
};


/*
 * The above rasters are stored in the array "SquareRaster", which
 * is indexed by the value of a piece.
 */

short *SquareRaster[5];


/*
 * FirstSquareItem is the VGTS item number of the item displayed in the
 * lower left square of the checker board.
 */

short	FirstSquareItem;

/*
 * ValueIndicator is the VGTS item number of the first item that is drawn
 * inside the perceived board value indicator after a move is made.
 */

short	ValueIndicator;

/*
 * Prompt_Item, Forced_Capture_Item, and Silly_Comment_Item are VGTS item
 * numbers for certain text prompts and comments that may appear by the board.
 */
short	Prompt_Item, Forced_Capture_Item, Silly_Comment_Item;

/*
 * Edit_Menu is the VGTS item number for the first item of the menu that is
 * displayed when "edit mode" is entered.
 */
short	Edit_Menu;

/* Edit_Menu_FirstSquare is the item number of the first selectable square
 * in the Edit mode menu.
 */
short	Edit_Menu_FirstSquare;


/* Define the positions and sizes of these items. */

#define TEXT_HEIGHT 16

#define PROMPT_X 129*SQUARE_SIZE/16
#define PROMPT_Y SQUARE_SIZE/2
#define PROMPT_WIDTH 120

#define FORCED_CAPTURE_X 4*SQUARE_SIZE
#define FORCED_CAPTURE_Y -7*SQUARE_SIZE/8
#define FORCED_CAPTURE_WIDTH 250

#define SILLY_COMMENT_X 2*SQUARE_SIZE
#define SILLY_COMMENT_Y 33*SQUARE_SIZE/4
#define SILLY_COMMENT_WIDTH 430



SetupVGT() /* Called by "main", to initialize the VGT. */
  {
    int		item, i, j;
    char	label[1];


    /* Create a SDF, in which we shall define the board. */
    Sdf = CreateSDF();

    /* Define the board. */
    DefineSymbol(Sdf, BOARD, 0);

    /*
     * Add the outline.
     * Note that we don't use SDF_OUTLINE, because then it gets partially
     * overwritten by the rasters for the squares near the edge (possibly
     * a bug in the VGTS).
     */

    item = BOARD + 1;

    AddItem(Sdf, item++, 0, 0, 0, 8*SQUARE_SIZE, 0,
	    SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++, 8*SQUARE_SIZE, 0, 0, 8*SQUARE_SIZE, 0,
	    SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++, 0, 8*SQUARE_SIZE, 0, 0, 0,
	    SDF_HORIZONTAL_LINE, 0);
    AddItem(Sdf, item++, 0, 8*SQUARE_SIZE, 8*SQUARE_SIZE, 0, 0,
	    SDF_HORIZONTAL_LINE, 0);

    /* Add vertical and horizontal lines (7 of each). */
    for (i = 1; i < 8; ++i)
      {
	AddItem(Sdf, item++, i*SQUARE_SIZE, 0, 0, 8*SQUARE_SIZE, 0,
		SDF_VERTICAL_LINE, 0);
      }
    for (i = 1; i < 8; ++i)
      {
	AddItem(Sdf, item++, 0, 8*SQUARE_SIZE, i*SQUARE_SIZE, 0, 0,
		SDF_HORIZONTAL_LINE, 0);
      }

    /* Add the letters at the bottom. */
    label[0] = 'A';
    for (i = 0; i<8; ++i)
      {
	AddItem(Sdf, item++,
		i*SQUARE_SIZE + SQUARE_SIZE/2 - 3, (i + 1)*SQUARE_SIZE,
		-SQUARE_SIZE/2, 0,
		0, SDF_SIMPLE_TEXT, label);
	++label[0];
      }

    /* Add the numbers at the left. */
    label[0] = '1';
    for (j = 0; j<8; ++j)
      {
	AddItem(Sdf, item++, -SQUARE_SIZE/2, 0,
		(j + 1)*SQUARE_SIZE-SQUARE_SIZE/2 - 8, (j + 2)*SQUARE_SIZE,
		0, SDF_SIMPLE_TEXT, label);
	++label[0];
      }

    /* Add the indicator of the perceived value of the board. */
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X + VALUE_INDICATOR_WIDTH,
	    VALUE_INDICATOR_Y, 0,
	    0, SDF_HORIZONTAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X + VALUE_INDICATOR_WIDTH,
	    VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT, 0,
	    0, SDF_HORIZONTAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, 0,
	    VALUE_INDICATOR_Y, VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT,
	    0, SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X+VALUE_INDICATOR_WIDTH, 0,
	    VALUE_INDICATOR_Y, VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT,
	    0, SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X + VALUE_INDICATOR_WIDTH,
	    VALUE_INDICATOR_Y + VALUE_INDICATOR_HEIGHT/2, 0,
	    0, SDF_HORIZONTAL_LINE, 0);
    ValueIndicator = item;

    /* The value indicator also contains 4 more items, to be drawn later. */

    Prompt_Item = (Forced_Capture_Item = (Silly_Comment_Item = item + 4)+1)+1;
    FirstSquareItem = Prompt_Item + 1;
    Edit_Menu = FirstSquareItem + 200; /* Way off on its own, I hope. */

    EndSymbol(Sdf, BOARD, 0);


    /*
     * We now set up the array "SquareRaster", each element of which is a
     * raster array (in .hex format).
     */

    SquareRaster[EMPTY] = EMPTY_Raster;
    SquareRaster[RC] = RC_Raster;
    SquareRaster[BC] = BC_Raster;
    SquareRaster[RK] = RK_Raster;
    SquareRaster[BK] = BK_Raster;


    /* Now create and display the VGT. */
    Vgt = CreateVGT(Sdf, GRAPHICS, BOARD, "Checker Board");
    DefaultView(Vgt, 11*SQUARE_SIZE, 10*SQUARE_SIZE,
		-SQUARE_SIZE, -SQUARE_SIZE, 0, 0, 0, 0);
  }


ToString(count, str)
int count; char str[];
    /*
     * Sets the 2-character string "str" to the string representing the
     * (decimal) value of "count". We assume 0<=count<100. In fact, in any
     * real game, "count" will be <= 12, but some bozo may decide to edit the
     * board so that it has a ridiculously large number of pieces on one side.
     */
  {
    if (count > 9)
      {
	str[0] = '0' + count/10;
	count %= 10;
      }
    else
	str[0] = ' ';
    str[1] = '0' + count;
  }


UpdateValueIndicator()
    /* Update the board value indicator. */
    /* (a Value < -512 or > 512 is "off the scale".) */
  {
    extern int  Value, Rcount, Bcount;

    int		shownValue;
    short	i;
    char	countString[2];

    /* Open the BOARD symbol for editing. */
    EditSymbol(Sdf, BOARD);

    /* Delete what's currently there. */
    for (i=0; i<4; ++i)
	DeleteItem(Sdf, ValueIndicator+i);

    if (Value < -512)
	shownValue = 0;
    else if (Value > 512)
	shownValue = VALUE_INDICATOR_HEIGHT;
    else
	shownValue = ((Value+512)*VALUE_INDICATOR_HEIGHT)>>10; /* ie. /1024 */

    /* Insert shaded rectangle representing the human's situation. */
    AddItem(Sdf, ValueIndicator,
	    VALUE_INDICATOR_X+1, VALUE_INDICATOR_X+VALUE_INDICATOR_WIDTH-1,
	    VALUE_INDICATOR_Y+1, VALUE_INDICATOR_Y+shownValue,
	    1, SDF_FILLED_RECTANGLE, 0);

    /* Insert shaded rectangle representing the computer's situation. */
    AddItem(Sdf, ValueIndicator+1,
	    VALUE_INDICATOR_X+1, VALUE_INDICATOR_X+VALUE_INDICATOR_WIDTH-1,
	    VALUE_INDICATOR_Y+shownValue,
	    VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT,
	    6, SDF_FILLED_RECTANGLE, 0);

    /* Add the numerals counting each side's inventory. */
    ToString(Rcount, countString);
    AddItem(Sdf, ValueIndicator+2,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X+20,
	    VALUE_INDICATOR_Y-32, VALUE_INDICATOR_Y+16,
	    0, SDF_SIMPLE_TEXT, countString);
    ToString(Bcount, countString);
    AddItem(Sdf, ValueIndicator+3,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X+20,
	    VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT+16,
	    VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT+32,
	    0, SDF_SIMPLE_TEXT, countString);

    EndSymbol(Sdf, BOARD, Vgt);
  }



FullDisplay( board, vflag)
    int *board; short vflag;

   /*
    * This subprogram displays the board "board" in Vgt.
    * At present it is somewhat gauche - it looks at each playable square in
    * turn, deleting the item (raster) that is currently displayed there,
    * and replacing it with the new raster that is to be displayed there.
    * Of course, Vgt itself is redrawn once only.
    * If "vflag" is set, then the value indicator is also displayed.
    */

  {
    extern int	Index[];

    register unsigned    i, j;
    short	boardIndex, itemNumber;

    /* Open the BOARD symbol for editing. */
    EditSymbol(Sdf, BOARD);

    /* Look at each playable square (i,j) */
    for (i = 0; i < 8; ++i)
	for (j = i&1; j < 8; j += 2)
	  {
	    /*
	     * The piece to be displayed is stored in board[k], where
	     * k = Index[8*i + (7-j)]. The VGTS item number for this board
	     * position is FirstSquareItem + k. We use the array "Index" to
	     * find "k", since the elements of the array "board" are not
	     * stored in row-wise order.
	     */
	    boardIndex = Index[(i<<3) + (7-j)];
	    itemNumber = FirstSquareItem + boardIndex;

	    /* Delete the item that is currently there. */
	    DeleteItem(Sdf, itemNumber);

	    /*
	     * Now add a copy of the raster for the new contents
	     * of the square.
	     */
	    AddItem(Sdf, itemNumber,
		    i*SQUARE_SIZE + 1, (i + 1)*SQUARE_SIZE - 1,
		    j*SQUARE_SIZE + 1, (j + 1)*SQUARE_SIZE - 1,
		    0, SDF_RASTER,
		    SquareRaster[board[boardIndex]]);
	  }

    /* Close the BOARD symbol, and display the new board. */
    EndSymbol(Sdf, BOARD, Vgt);

    if (vflag)
        UpdateValueIndicator();
  }





UpdateDisplayedSquare( boardIndex, highlightFlag)
    short boardIndex, highlightFlag;

    /*
     * This routine updates the item displayed for square "boardIndex" only.
     * (ie. it does not redisplay every square on the board, as does Display.)
     * (If "highlightFlag" is set, then the special 'selected square' raster
     * is used instead of the normal raster for the piece. )
     */

  {
    extern int	Board[];

    short	itemNumber;
    unsigned	xmin, xmax, ymin, ymax;
    register i, j;

    itemNumber = boardIndex + FirstSquareItem;

    /* Open the BOARD symbol for editing. */
    EditSymbol(Sdf, BOARD);

    /* Get the location of the square. */
    /* We would like to call InquireItem, as shown below, but the returned
       values of xmin, etc. are garbage. Instead we use a kludge.
    InquireItem(Sdf, itemNumber, &xmin, &xmax, &ymin, &ymax, 0, 0, 0);
    */
    j = boardIndex-1 - (boardIndex-7)/13;
    i = (j%6)*2;
    j = 8 - j/6;
    i = i + (j&1);
    xmin = i*SQUARE_SIZE + 1; xmax = (i+1)*SQUARE_SIZE - 1;
    ymin = j*SQUARE_SIZE + 1; ymax = (j+1)*SQUARE_SIZE - 1;
    /* End of kludge. */

    /* Delete the current item. */
    DeleteItem(Sdf, itemNumber);

    /* and add the new item (raster). */
    AddItem(Sdf, itemNumber,
	    xmin, xmax, ymin, ymax,
	    0, SDF_RASTER,
	    highlightFlag ? RHIGHLIGHT_Raster
			  : SquareRaster[Board[boardIndex]]);
    
    /* Close the BOARD symbol, and display the new board. */
    EndSymbol(Sdf, BOARD, Vgt);
  }


Prompt(str)
    char str[];
    /*
     * This routine prompts the human with "str", (or turns off the
     * prompt, if "str" is 0.
     */
  {
    EditSymbol(Sdf, BOARD);
    DeleteItem(Sdf, Prompt_Item);
    if (str)
	AddItem(Sdf, Prompt_Item,
		PROMPT_X, PROMPT_X+PROMPT_WIDTH,
		PROMPT_Y, PROMPT_Y+TEXT_HEIGHT,
		0, SDF_SIMPLE_TEXT, str);
    EndSymbol(Sdf, BOARD, Vgt);
  }


Silly_Comment(str)
    char str[];
    /*
     * This routine prints the "silly comment" "str" on the board.
     * (or turns it off, if "str" is 0.)
     */
  {
    EditSymbol(Sdf, BOARD);
    DeleteItem(Sdf, Silly_Comment_Item);
    if (str)
	AddItem(Sdf, Silly_Comment_Item,
		SILLY_COMMENT_X, SILLY_COMMENT_X+SILLY_COMMENT_WIDTH,
		SILLY_COMMENT_Y, SILLY_COMMENT_Y+TEXT_HEIGHT,
		0, SDF_SIMPLE_TEXT, str);
    EndSymbol(Sdf, BOARD, Vgt);
  }


#define FCMAX 30
static int fci;
static char fcstring[FCMAX];

Forced_Capture(str)
    char str[];
    /*
     * This routine adds the string "str" to the end of the forced captures
     * prompt (or deletes the prompt, if "str" is 0).
     */
  {
    register int j = 0;

    if (str)
	while ((fcstring[fci] = str[j++]) != 0)
	    ++fci;
    else
      {
	EditSymbol(Sdf, BOARD);
	DeleteItem(Sdf, Forced_Capture_Item);
	EndSymbol(Sdf, BOARD, Vgt);
      }
  }

begin_Forced_Capture()
  {
    fci = 0;
    Forced_Capture(0);
    Forced_Capture("Forced Captures:");
  }


end_Forced_Capture()
  {
    EditSymbol(Sdf, BOARD);
    AddItem(Sdf, Forced_Capture_Item,
	    FORCED_CAPTURE_X, FORCED_CAPTURE_X+FORCED_CAPTURE_WIDTH,
	    FORCED_CAPTURE_Y, FORCED_CAPTURE_Y+TEXT_HEIGHT,
	    0, SDF_SIMPLE_TEXT, fcstring);
    EndSymbol(Sdf, BOARD, Vgt);
  }




begin_Edit_Mode()
  /* Deletes the  value indicator, displaying the edit menu in its place. */
  {
    short item, i;
    int y;

    EditSymbol(Sdf, BOARD);

    for (item = ValueIndicator-5; item < ValueIndicator+4; ++item)
	DeleteItem(Sdf, item);

    /* Create the edit menu. */
    item = Edit_Menu;
    y = EDIT_MENU_Y;
    for (i = 0; i<8; ++i)
      {
        AddItem(Sdf, item++,
	        EDIT_MENU_X, EDIT_MENU_X + SQUARE_SIZE,
	        y, 0,
	        0, SDF_HORIZONTAL_LINE, 0);
	y += SQUARE_SIZE;
      }
    AddItem(Sdf, item++,
	    EDIT_MENU_X, 0,
	    EDIT_MENU_Y, EDIT_MENU_Y + 7*SQUARE_SIZE,
	    0, SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++,
	    EDIT_MENU_X + SQUARE_SIZE, 0,
	    EDIT_MENU_Y, EDIT_MENU_Y + 7*SQUARE_SIZE,
	    0, SDF_VERTICAL_LINE, 0);
    Edit_Menu_FirstSquare = item;
    y = EDIT_MENU_Y + SQUARE_SIZE;
    for (i = 0; i < 5; ++i)
      {
	AddItem(Sdf, item++,
	    EDIT_MENU_X+1, EDIT_MENU_X+SQUARE_SIZE-1,
	    y+1, y + SQUARE_SIZE-1,
	    0, SDF_RASTER,
	    SquareRaster[i]);
	y += SQUARE_SIZE;
      }
    AddItem(Sdf, item++,
	    EDIT_MENU_X + SQUARE_SIZE/8, EDIT_MENU_X + 3*SQUARE_SIZE/4,
	    EDIT_MENU_Y + 3*SQUARE_SIZE/8, EDIT_MENU_Y + 3*SQUARE_SIZE/4,
	    0, SDF_SIMPLE_TEXT, "Clear");
    AddItem(Sdf, item++,
	    EDIT_MENU_X + SQUARE_SIZE/4, EDIT_MENU_X + 3*SQUARE_SIZE/4,
	    EDIT_MENU_Y + 6*SQUARE_SIZE + 3*SQUARE_SIZE/8,
	    EDIT_MENU_Y + 6*SQUARE_SIZE + 3*SQUARE_SIZE/4,
	    0, SDF_SIMPLE_TEXT, "Done");

    EndSymbol(Sdf, BOARD, Vgt);

    Silly_Comment("You are now in Edit Mode - click \"Done\" to leave it.");
}


end_Edit_Mode()
  /* Deletes the edit menu, redisplaying the value indicator. */
  {
    short item, i;
    int y;

    EditSymbol(Sdf, BOARD);

    for (item = Edit_Menu; item < Edit_Menu+17; ++item)
	DeleteItem(Sdf, item);

    item = ValueIndicator - 5;

    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X + VALUE_INDICATOR_WIDTH,
	    VALUE_INDICATOR_Y, 0,
	    0, SDF_HORIZONTAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X + VALUE_INDICATOR_WIDTH,
	    VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT, 0,
	    0, SDF_HORIZONTAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, 0,
	    VALUE_INDICATOR_Y, VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT,
	    0, SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X+VALUE_INDICATOR_WIDTH, 0,
	    VALUE_INDICATOR_Y, VALUE_INDICATOR_Y+VALUE_INDICATOR_HEIGHT,
	    0, SDF_VERTICAL_LINE, 0);
    AddItem(Sdf, item++,
	    VALUE_INDICATOR_X, VALUE_INDICATOR_X + VALUE_INDICATOR_WIDTH,
	    VALUE_INDICATOR_Y + VALUE_INDICATOR_HEIGHT/2, 0,
	    0, SDF_HORIZONTAL_LINE, 0);

    EndSymbol(Sdf, BOARD, Vgt);

    Silly_Comment(0);

    UpdateValueIndicator();
}


Edit_Menu_Highlight(piece, on)
    short piece, on;
    /*
     * Highlights the Edit mode menu square for the piece 'piece', or turns
     * the highlighting off, depending upon whether or not the flag 'on'
     * is set.
     */

  {
    EditSymbol(Sdf, BOARD);

    DeleteItem(Sdf, Edit_Menu_FirstSquare + piece);

    if (on)
      { /* These braces are necessary to avoid "dangling else" confusion. */
	if (piece != EMPTY) /* Use the 'highlighted piece' raster. */
	    AddItem(Sdf, Edit_Menu_FirstSquare + piece,
	        EDIT_MENU_X+1, EDIT_MENU_X+SQUARE_SIZE-1,
	        EDIT_MENU_Y + SQUARE_SIZE + piece*SQUARE_SIZE + 1,
	        EDIT_MENU_Y + 2*SQUARE_SIZE + piece*SQUARE_SIZE - 1,
	        0, SDF_RASTER,
	        RHIGHLIGHT_Raster);
      }
    else /* Redisplay the proper piece for the square. */
	    AddItem(Sdf, Edit_Menu_FirstSquare + piece,
	        EDIT_MENU_X+1, EDIT_MENU_X+SQUARE_SIZE-1,
	        EDIT_MENU_Y + SQUARE_SIZE + piece*SQUARE_SIZE + 1,
	        EDIT_MENU_Y + 2*SQUARE_SIZE + piece*SQUARE_SIZE - 1,
	        0, SDF_RASTER,
	        SquareRaster[piece]);

    EndSymbol(Sdf, BOARD, Vgt);
  }
