/*  amaze : a distributed multi-person Vkernel game.
 *  Copyright (c) 1983 Eric J. Berglund  &  David R. Cheriton
 *
 *   This is the 1st of 8 files containing the data structures and routines
 *   which implement the game amaze.  The files are amaze.h, processes.c,
 *   init.c, inputhandler.c, communicate.c, state.c, draw.c, and test.c.
 *
 * General structs, constants, etc. for the program are defined here.
 *
 */

#include <framebuf.h>
#include <rasterops.h>
#include <lines.h>
#include <Venviron.h>

#define GAME_MANAGER	20	/* Logical process id of game manager. */
#define MAX_PLAYERS	5       /* Players 0 thru 4. */
#define GAME_CLICKS	2	/* Clicks per screen update. */
#define NULL_PROCESS	0	/* Pid for autopilot when it isn't running
				   and for managers of nonexistent players. */
#define FALSE	0
#define TRUE	1


/* Maze data structure values. */

#define NUM_CORNERS	65
#define NUM_CORRIDORS	76
#define NUM_VERT_WALLS	22
#define NUM_HORIZ_WALLS	26


/* Game manager request codes */

#define GAME_INPUT_CHAR		0x0050	/* Random value but consistent */
#define GAME_TIMER		0x0051	/* with Vkernel conventions */
#define REQUEST_MONSTER_STATE	0x2052
#define REPORT_MONSTER_STATE	0x0053
#define JOIN_GAME		0x2054
#define AUTO_PILOT_CHAR		0x0055
#define MISSILE_REPORT		0x2056


/* The only possible values of a monster living_status field.  Each 
 * living_status value is examined and possibly changed in state.c,
 * and then displayed graphically in draw.c.  Changing a monster's 
 * living_status gives an implicit command to the UpdateState and
 * DrawPicture routines in those two files.
 */

#define GLINT		0	/* Signal that conception should occur.	 */
#define CONCEIVED	1	/* Display the player as dormant, then	 */
				/* change to EMBRYO.			 */
#define EMBRYO		2	/* Wait for change to ALIVE.		 */
#define ALIVE		3	/* Move as commanded, erasing the former */
				/* display and drawing one at the new	 */
				/* ( x, y ) position.			 */
#define HIDDEN		4	/* Move as commanded, but do not allow	 */
				/* other managers to display the new	 */
				/* positions.				 */
				/* HIDDEN monsters can't shoot, though.	 */
#define DYING		5	/* Flash the monster DYING_COUNT times,	 */
				/* then change living_status to DEAD.	 */
#define SHOT		6	/* Flash the monster SHOT_COUNT times,	 */
				/* alternating it with a skull, then	 */
				/* change its status to STIFF.		 */
#define STIFF		7	/* Display a skull and crossbones for	 */
				/* STIFF_COUNT clicks, then change 	 */
				/* status to JUST_DIED.			 */
#define JUST_DIED	8	/* Erase the skull and change to DEAD.	 */
#define DEAD		9	/* Stay DEAD for DEAD_COUNT, then	 */
				/* change to CONCEIVED.			 */
#define CREMATED	10	/* Signal to the local manager that a	 */
				/* remote monster is so far gone that	 */
				/* he can't send a DEAD notice--this is	 */
				/* used when a remote manager doesn't	 */
				/* exist any longer.			 */


/* The possible values of a missile's state field.  Each of these, as 
 * well, is an implicit command to UpdateState and DrawPicture.
 */

#define JUST_LOADING	0	/* Clear away an explosion, then	 */
				/* switch state to LOADING.		 */
#define LOADING		1	/* Stay LOADING for LOAD_COUNT clicks */
				/* and then change to JUST_READY.	 */
#define JUST_READY	2	/* Display the missile away from the	 */
				/* maze, then switch to READY.		 */
#define READY		3	/* Wait to be fired.			 */
#define TRIGGER_PULLED	4	/* Change to JUST_FIRED.		 */
#define JUST_FIRED	5	/* Display the missile, and then change	 */
				/* to WHIZZING.				 */
#define WHIZZING	6	/* Advance by MISSILE_SPEED pixels 	 */
				/* until hitting a wall.		 */
#define HIT_WALL	7	/* Display an explosion for EXPLOSION_	 */
				/* COUNT clicks, then change to 	 */
				/* JUST_LOADING.			 */

/* COUNTs specify the number of calls to UpdateState before a monster or
 *  missile changes state.  VISIBLE_COUNT is the number of calls between
 *  emerging from hiding and being able to hide again.
 */

#define HIDDEN_COUNT		500
#define VISIBLE_COUNT		800
#define DYING_COUNT		100
#define SHOT_COUNT		100
#define STIFF_COUNT		50
#define DEAD_COUNT		50

#define LOAD_COUNT		300
#define EXPLOSION_COUNT		60


/* Directions of movement for monsters and missiles.  Despite the blow to
 * elegant programming, several optimizations have been added that depend
 * on these specific values for the direction constants.  I have tried to
 * flag these in the code, but be extremely wary about changing these
 * values.  --EJB
 */

#define NONE		-1
#define UP		0
#define DOWN		1
#define LEFT		2
#define RIGHT		3


/* Every corridor in the maze is represented by its two endpoints, and an
 * orientation--either HORIZONTAL or VERTICAL.  If a corridor is VERTICAL
 * its NEAR_CORNER endpoint is the higher of the two endpoints--its FAR_CORNER,
 * the lower.  If a corridor is HORIZONTAL, its NEAR_CORNER is the leftmost
 * endpoint--its FAR_CORNER, the rightmost.  Since the SUN workstation
 * designers chose the upper leftmost pixel of the screen as the origin,
 * the NEAR_CORNER endpoint of a corridor will always have a smaller coordinate
 * than the FAR_CORNER endpoint.
 */

#define NEAR_CORNER	0
#define FAR_CORNER	1
#define HORIZONTAL	0
#define VERTICAL	1

/* Each corner in the maze are represented by its x and y coordinates,
 * and by a list of the four corridors leading to it from the UP, DOWN,
 * LEFT, and RIGHT.  If there is no corridor coming from that direction,
 * the value in the list is WALLED.  Note that there is no corridor 0.
 */

#define WALLED		0

#define FIRST_CORRIDORINDEX	39
#define FIRST_CORNERINDEX	28
#define START_CORNER		29
#define LEFT_WRAP_AROUND_CORNER		0
#define RIGHT_WRAP_AROUND_CORNER	64


/* Velocities specify the number of pixels that each ALIVE or HIDDEN
 * monster moves every time UpdateState is called.
 * Missiles always move at the same velocity.
 */

#define VELOCITY1	1
#define VELOCITY2	2
#define VELOCITY3	3
#define VELOCITY4	4
#define VELOCITY5	5
#define MAX_VELOCITY	VELOCITY5
#define START_VELOCITY	VELOCITY3
#define MISSILE_SPEED	25


/* Graphics dimensions for rasters defined in draw.c. */

#define MONSTER_HEIGHT		40
#define MONSTER_WIDTH		40
#define MISSILE_HEIGHT		32
#define MISSILE_WIDTH		32
#define EXPLOSION_HEIGHT	40
#define EXPLOSION_WIDTH		40


/* Monsters which are not exactly at, but close to, a corner, may
 * still be able to shoot down another corridor at that corner, or
 * be shot by a missile passing through that corner, if they are...
 */

#define CLOSE_ENOUGH_TO_CORNER	20


/* Coordinates for EMBRYOs, new borns, and loaded missiles.
 */

#define MISSILE_REST_X		920
#define MONSTER_REST_Y		459
#define MONSTER_DISPLAY_X	840
#define FIRST_ALIVE_Y		368
#define FIRST_ALIVE_X		345


/* Flags used in communicate.c and state.c. */

#define NO_NOTICE_NEEDED	0	/* No significant change. */
#define NOTIFIABLE_CHANGE	1	/* Change in direction just took
					    place. */

#define RESPOND_WHEN_ASKED	1	/* Big news is pending--transmit
					   when asked. */
#define NO_ONE_WAITING		0	/* Nothing's happened and no one
					   cares. */

#define Abs( x )  ( (x) > 0  ?  (x) : (-(x))  )


/* The structures that define the corners and corridors in the maze--Node
 * for corners, Edge for corridors.
 */

typedef struct Nodetype
  {
     int x;			/* X-coordinate of corner.		*/
     int y;			/* Y-coordinate of corner.		*/
     int corridorindex[4];	/* The indices of the corridors above,	*/
				/* below, left of, and right of this	*/
				/* corner.				*/
     int corridorsadj;		/* Number of adjacent corridors.	*/
  } Node;

typedef struct Edgetype
  {
     int endptindex[2];		/* The indices of the corners which are	*/
				/* the endpoints of this corridor.  The	*/
				/* first is the NEAR_CORNER--the corner	*/
				/* that is UP or LEFT--and the second	*/
				/* is the FAR_CORNER.			*/
     int orientation;		/* HORIZONTAL or VERTICAL.		*/
  } Edge;
    


/* The state information maintained about each player's missile */

typedef struct
  {
    int state;			/* JUST_LOADING, LOADING, JUST_READY,	*/
				/* READY, TRIGGER_PULLED, JUST_FIRED,	*/
				/* WHIZZING, HIT_WALL?			*/
    int x;			/* x-coordinate.			*/
    int y;			/* y-coordinate.			*/
    int old_x;			/* last displayed x-coordinate.		*/
    int old_y;			/* last displayed y-coordinate.		*/
    int direction;		/* UP, DOWN, LEFT, or RIGHT?		*/
    int cornerindex;		/* Corner the missile's heading toward.	*/
    int count;			/* Time 'til JUST_LOADING or JUST_READY.*/
    struct mem_raster *raster;	/* Graphics display of this missile.	*/
  } MissileState;


/* The state information maintained about each player's monster. */

typedef struct
  {
    ProcessId gamemanager;	/* Pid of this monster's game manager.	*/
    ProcessId remoteasker;	/* Status inquirer created  by this 	*/
				/* player to monitor my monster.  Holds	*/
				/* RESPOND_WHEN_ASKED when my monster	*/
				/* has undergone a significant change	*/
				/* in state, and this monster hasn't	*/
				/* asked about me recently.		*/
				/* Holds NO_ONE_WAITING when he hasn't	*/
				/* asked and I wouldn't respond anyway.	*/
				/* Note the strategy here--unless my	*/
				/* monster does something unusual	*/
				/* (change direction, corridor, corner	*/
				/* velocity, or living_status) I don't	*/
				/* have to respond to a waiting		*/
				/* remoteasker; his manager can		*/
				/* extrapolate my position from		*/
				/* previous info.			*/
    ProcessId statusinquirer;	/* My status inquirer for this player.	*/
    int x;			/* X coordinate of monster.		*/
    int y;			/* Y coordinate of monster.		*/
    int old_x;			/* Old X coordinate of monster.		*/
    int old_y;			/* Old Y coordinate of monster.		*/
    int velocity;		/* Velocity of monster.			*/
    int current_dir;		/* Current direction of monster.	*/
    int next_dir;		/* Next direction of monster.		*/
    int living_status;		/* EMBRYO, ALIVE, DYING, DEAD, etc.	*/
    int old_living_status;	/* Status last time we drew it.		*/
    int cornerindex;		/* Corner the monster is approaching.	*/
    int corridorindex;		/* Corridor the monster is in.		*/
    int count;			/* Countdown for DYING, SHOT, DEAD,etc.	*/
    struct mem_raster *raster;	/* Pointer to monster's raster.		*/
    MissileState missile;
  } MonsterState;


/* The state maintained by the game manager process.
 * This is effectively its perception of the state of the game.
 */

typedef struct
  {
    ProcessId mymanager;	/* My manager's pid.			*/
    ProcessId mykeyboardreader;	/* My keyboard reader process.		*/
    ProcessId mytimer;		/* My timer process.			*/
    ProcessId myautopilot;	/* My auto pilot process, if any.	*/
    unsigned mymonsternumber;	/* My monster number.			*/
    MonsterState monsterstates[MAX_PLAYERS];	/* Monster states.	*/
  } GameState;



/* Format of request and reply messages for distributing monster state
 * information. Note that the reply message contains the monster state
 * which must be the same format as the request, which is then sent
 * to the respective game manager.  GameRequests and GameReplies are used
 * for GAME_INPUT_CHAR, AUTO_PILOT_CHAR, GAME_TIMER, and JOIN_GAME requests
 * and replies.  The first two send only a requestcode and a character--
 * the latter two, a requestcode only.  Replies consist of a replycode only
 * except for JOIN_GAME replies, which pass back the pids of the managers
 * currently playing the game.
 */

typedef struct
  {
    SystemCode requestcode;
    char inputchar;
  } GameRequest;

typedef struct
  {
    SystemCode replycode;
    ProcessId monstermanagers[MAX_PLAYERS]; /* Only used in JOIN_GAME */
  } GameReply;


/* Status inquirer requests and replies are used by local status inquirers
 * to first determine and then pass along information about a remote
 * monster to the local game manager.  The status inquirer first sends
 * only a requestcode, its monsternumber, and its local manager's pid,
 * but receives a reply consisting of all the information below.  It then
 * passes along that reply in the form of a request to its own manager,
 * which extracts the information, and sends back a mere replycode.
 * The fact that the request and reply are identically formatted saves
 * the status inquirer the work of converting the first reply to give
 * the second request.
 * Note that the managers do not require or supply all the state information
 * about their own players--the values not passed in messages can be 
 * determined by remote managers without assistance.
 */

typedef struct
  {
     SystemCode requestcode;
     unsigned monsternumber;
     ProcessId manager;
     short int xcoord;
     short int ycoord;
     short int velocity;
     short int direction;
     short int next_direction;
     short int living_status;
     short int cornerindex;
     short int corridorindex;
  }  StatusInquirerRequest;


typedef struct
  {
    SystemCode replycode;
    unsigned monsternumber;
    ProcessId manager;
    short int xcoord;
    short int ycoord;
    short int velocity;
    short int direction;
    short int next_direction;
    short int living_status;
    short int cornerindex;
    short int corridorindex;
  } StatusInquirerReply;


/* When a missile is fired, a MISSILE_REPORT is sent to every remote
 * manager, consisting of the following information:
 */

typedef struct
  {
    SystemCode requestcode;
    unsigned monsternumber;
    int x;
    int y;
    int direction;
    int cornerindex;
  } MissileRequest;
