/*
 * fill.c - RasterFill SUN graphics primitive
 *
 *  Take a region on the frame buffer and update it as determined by the
 * current contents of the sixteen bit SOURCE and MASK registers, and under the
 * operation specified by the FUNCTION register in the framebuffer.
 *  This routine simply "touches" locations, carrying out the operation
 * pre-specified by the user (which should have already be set)
 *
 * Originally done by Dave Brown
 * Updated June 1982 by Bill Nowicki
 */

#include <rasterops.h>
#include <framebuf.h>

RasterFill(dst)
    struct fb_raster	*dst;
{
    register short	h = dst->height,		/* d7, d6 */
    			w = dst->width;


        if (h<=0 || w<=0) return;       /* return if raster has weird size */

/*--------------------------------------------------------------------------
============================================================================
 Perform W/16 iterations of a full-width (16 bit) Vertical fill of height h
============================================================================
--------------------------------------------------------------------------*/

	{ register 	i,j;				/* d5, d4 */
	  register short				/* a5, a4 */
	    *Y_Address =
		(short *)(GXBase|GXselectY|GXupdate + ((dst->y)<<1)),
	    *X_Address =
		(short *)(GXBase|GXselectX + ((dst->x)<<1));


        GXwidth = 16;

   /**********************************************************************
    * The outer loop fills width/16 word-wide columns.  Then if w mod 16 *
    * is non-zero, one iteration of a special narrow-width fill is done  *
    **********************************************************************/

Outer:	for (i=w>>4 ; i>0; i--){

            *X_Address = j;	/* establish the X coordinate in Framebuf */
                                /* by accessing the appropriate address   */
	    Y_Address += h;	/* adjust Y address for a decrement loop  */

   /************************************************************************
    * The following code linearizes the inner loop of the Fill routine     *
    * in order to optimize the speed with which the operation is performed *
    *                                                                      *
    * The optimization strategy is the use of a move multiple register     *
    * instruction, to access addresses in the frame buffer where the raster*
    * is to be "filled".                                                   *
    *   First, a loop is performed which fills height/16 blocks of 16 words*
    * Second, the remaining height mod 16 words are filled in, based on bit*
    * tests of the low order 5 bits of height.  That is, one, two, four,   *
    * eight and 16 words are filled if corresponding bits  0, 1, 2, 3, or  *
    * 4 of height are set.                                                 *
    ************************************************************************/

    /* First perform h/16 iterations of fill16 */
    /* if h/16 == 0 then skip the div16 loop */

#ifdef debug
	puts("about to do Inner");
	getchar();
#endif

Inner:      j = h>>4;			/* remember j is d4 */
            asm("	andw d4,d4");   /* if it's zero, do no loops of 16 */
            asm("	beq mod16");
	    j--;
            asm("fill16: movemw #/ffff,a5@-");
            asm("	dbf d4,fill16");

            /* now case on h%16 (low order 4 bits of h) */
            /* and fill the appropriate number of words */

            asm("mod16:");
            switch(h & 0xf){
                case 1:    asm("	movemw #/0001,a5@-"); break;
                case 2:    asm("	movemw #/0003,a5@-"); break;
                case 3:    asm("	movemw #/0007,a5@-"); break;
                case 4:    asm("	movemw #/000f,a5@-"); break;
                case 5:    asm("	movemw #/001f,a5@-"); break;
                case 6:    asm("	movemw #/003f,a5@-"); break;
                case 7:    asm("	movemw #/007f,a5@-"); break;
                case 8:    asm("	movemw #/00ff,a5@-"); break;
                case 9:    asm("	movemw #/01ff,a5@-"); break;
                case 10:   asm("	movemw #/03ff,a5@-"); break;
                case 11:   asm("	movemw #/07ff,a5@-"); break;
                case 12:   asm("	movemw #/0fff,a5@-"); break;
                case 13:   asm("	movemw #/1fff,a5@-"); break;
                case 14:   asm("	movemw #/3fff,a5@-"); break;
                case 15:   asm("	movemw #/7fff,a5@-");
            }

	    X_Address += 16;    /* update the Framebuffer X address */
	}

   /***********************************************************************
    *   Now, if there's any leftover width, perform one special iteration
    * with a narrow width setting.
    *
    *   The following bit of code is a real hack to save space (note the use
    * of the dreaded GOTO).  
    *
    *   It sets things up so that when we get back to this point again, 
    *  1) the condition will be false (i.e. low order 4 bits are zero)
    *  2) the value of w is modified such that exactly one iteration of
    *	Outer (above) is performed.
    *
    ***********************************************************************/

	if (w &= 0xf){		/* is there any leftover width ?? */
	    GXwidth = w;	/* if so this is the width to fill */
            /*
             * set up w for exactly one iteration when we get back to  Outer
             * (above).  Also clear the low order bits so that next time
             * we get here, the condition above will be false (hack, hack...)
             */
            w = 16;
            goto Outer;
        }

	}
}
