/*
 * memtest.c
 *
 * MC68000 memory test program
 *
 * Version 2 board - ROM monitor level 0.7 and later only
 *
 * Jeffrey Mogul 23 March 1981, 18 June 1981
 * Mark Grossman 3 Dec 81
 *
 * Bill Nowicki December 1981
 *	- attempted to retrofit the Design Module version
 *	  (define MACSBUG to get this version, but Untested!)
 *
 * Mike Nielsen 27 Jan 82 
 *	- removed DM stuff and added promts for base,size of memory to test.
 *
 * Mike Nielsen 17 July 82
 *  	- changed to std include files and added default base,top parsing.
 *
 * Benjy Levy 9 Nov 82
 *  	- added code to detect parity error indications generated by
 *  	  MultiBus memory cards. Interrupt level 4 and IO port 0x1f0201
 *
 * Tim Mann 9 Sep 83
 *	- ifdef'ed out reference to IO port 0x1f0201 since Chrislin memory
 *	  cards we are using do not have this port.  Also, stops with
 *	  interrupts disabled on Multibus parity errors to avoid going into
 *	  a loop.  Removed "actual a0" since it is garbage.  Fixed bus
 *	  error code to not assume PC is a short.  Added ability to write
 *	  error log to a file.  Added byte-mode test since this exercises
 *	  additional circuitry on Multibus memory cards.  Version 7.
 */

#include "vectors.h"
#include "reentrant.h"
#include "m68000.h"
#include "stdio.h"

#define CHRISLIN		/* Chrislin Multibus memory w/o I/O port */

#define PASSES 10000		/* number of passes */
#define DELAY 10		/* seconds to wait in delayed test */
#define IGNORE(RTN)	\
	asm("	addql	#2,sp");	/* pt to access addr */	\
	asm("	movl	sp@+,AccessAddr");	/* save it */	\
	asm("	addql	#4,sp");		/* pt to return PC */ \
	asm("	movl	sp@,BadPC");		/* save it */	\
	asm("	movl	#RTN,sp@");		/* patch w/ error rtn */  \
	asm("	subql	#2,sp")			/* set up for RTE */

#define VECREAD	BusErrorVector = (long)IgnoreRd	    /* Ignore Bus errors*/
#define VECWRITE BusErrorVector = (long)IgnoreWrt


nullfn()
{
	label(IgnoreRd);
	IGNORE(RdErr);
    	rte;

	label(IgnoreWrt);
	IGNORE(WrtErr);
    	rte;
}


int version = 7;
extern short _end;
int   AccessAddr;
int   BadPC;
short *base;
short *top;
long size;
short Writ;
long   BESave;		/* Address of the old exeception handler */
short testno;
short *tp;
int seed = 433;

FILE *errorLog;

int curerrs = 0;
int toterrs = 0;


/*
 * This is where we come when MultiBus memory generates an interrupt.
 */

#define MemIOPort ((char *)0x1f0200)
reentrant(MemoryInterrupt)
 {
#ifdef CHRISLIN
 printf("\nReceived a Multibus memory parity error at location %x\n", 
 	 (int)tp);
 asm("	trap #14");
#else
 printf("\nReceived a MultiBus memory parity error at location %x, bank %d\n",
    	 (int)tp,(~*MemIOPort)&7);
 *MemIOPort = 1;
#endif CHRISLIN
 }


nullErr()
{
	label(RdErr);
	printbuserr("read");
	BusErrorVector = BESave;
	exit(0);

	label(WrtErr);
	printbuserr("write");
	BusErrorVector = BESave;
	exit(0);
}

printbuserr(mode)
	char *mode;
{
	printf("\nBus %s error, access addr %x, PC %x\n",
		mode,AccessAddr,BadPC);
	printf("  while testing location %x",tp);
}

gethex()
{
    register value = 0;
    char buf[12];
    register char c,*s;
    register digit;

    s = (char*)gets(buf);
    if ((c = *s) == 0)
    	return(-1);
    do
    {
    	if ( (c >= 'a') && (c <= 'f') )
    	    digit = c - 'a' + 10;
    	else if ( (c >= 'A') && (c <= 'F') )
    	    digit = c - 'A' + 10;
    	else if ( (c >= '0') && (c <= '9') )
    	    digit = c - '0';
    	else 
    	    return(-1);
    	value = (value << 4) + digit;

    	c = *(++s);
    } while (c);
    return(value);
}

main()
{
	register int i;
	short testvector;
    	long location;
	char fileName[128];
    	
	printf("Sun Memory Test Program, Version %d\n",version);

    	IRQ4Vect = (int) MemoryInterrupt;
    	intlevel(0);

    	base = &_end;
        printf("\nBase of memory block <CR for default of 0x%x>? ",base);
    	if ((location = gethex()) != -1)
    	    base = (short*)(location & ~1); /* make sure it's a word address*/

    	top = (short*)emt_getmemsize();
        printf("Top of memory block <CR for default of 0x%x>? ",top);
    	if ((location = gethex()) != -1)
    	    top = (short*)(location & ~1); /* make sure it's a word address*/

	printf("Data for additional test(hex) <CR to skip test>? ");
	testvector = gethex();

getfilename:
	printf("Write error log to file <CR for none>: ");
	gets(fileName);
	if (fileName[0] != '\0')
	  {
	    errorLog = fopen(fileName, "w");
	    if (errorLog == NULL)
	      {
	        printf("Can't open %s for writing\n", fileName);
		goto getfilename;
	      }
	  }
	else
	  errorLog == NULL;

	BESave = BusErrorVector;
	printf("\nTesting memory between 0x%x and 0x%x\n",base,top);
	if (errorLog)
	  {
	    fprintf(errorLog, "\nTesting memory between 0x%x and 0x%x\n",
	        base,top);
	    fflush(errorLog);
	  }
	for(i=0; i<PASSES;i++) {
	    printf("Begin pass %d, test ",i);
	    if (errorLog)
	      {
		fprintf(errorLog, "Begin pass %d, test ",i);
		fflush(errorLog);
	      }
	    curerrs=0;
	    testno=0;
    	    if (testvector != -1)
	    	curerrs += testn(base,top,testvector);
	    curerrs += testn(base,top,0);
	    curerrs += testn(base,top,-1);
	    curerrs += testn(base,top,0xAAAA);
	    curerrs += testn(base,top,0xC11C);
	    curerrs += testn(base,top,0x5555);
	    curerrs += testrand(base,top);
	    curerrs += testadr(base,top);
	    curerrs += testdelay(base,top);
	    curerrs += testbyte(base,top);
	    toterrs+=curerrs;

	    printf("  End pass ");
	    if (errorLog)
	      {
	        fprintf(errorLog, "  End pass ");
		fflush(errorLog);
	      }
	    if (curerrs) {
		printf("  Errors this pass: %d",curerrs);
		if (errorLog)
		  {
		    fprintf(errorLog, "  Errors this pass: %d",curerrs);
		    fflush(errorLog);
		  }
	    }
	    if (toterrs) {
		printf("  Errors (total): %d",toterrs);
		if (errorLog)
		  {
		    fprintf(errorLog, "  Errors (total): %d",toterrs);
		    fflush(errorLog);
		  }
	    }
	    printf("\n");
	    if (errorLog)
	      {
		fprintf(errorLog, "\n");
		fflush(errorLog);
	      }
	}

	if (toterrs) {
	    printf("%d total errors\n",toterrs);
	    if (errorLog)
	      {
		fprintf(errorLog, "%d total errors\n",toterrs);
		fflush(errorLog);
	      }
	}
	else {
	    printf("No errors\n");
	    if (errorLog)
	      {
		fprintf(errorLog, "No errors\n");
		fflush(errorLog);
	      }
	}
	if (errorLog) fclose(errorLog);
	BusErrorVector = (long)BESave;

}



testn(a,b,n)	/* fill memory with n, read it back */
short *a;
short *b;
short n;
{
	int errs=0;

	printf("%d",++testno);
	if (errorLog)
	  {
	    fprintf(errorLog, "%d",testno);
	    fflush(errorLog);
	  }
	VECWRITE;
	for (tp=a; tp<b;*tp++=n);

	VECREAD;
	for (tp=a;tp<b;tp++) {
		if (*tp != n) {
			errs++;
			report("constant",n,*tp,tp);
			}
		}
	
	return(errs);
}

testadr(a,b)	/* fill each short with its address, read it back */
short *a;
short *b;
{
	/* use global *tp */
	int errs=0;
	register short *lp = &Writ;	/* local copy of contents */

	printf("%d",++testno);
	if (errorLog)
	  {
	    fprintf(errorLog, "%d",testno);
	    fflush(errorLog);
	  }
	VECWRITE;
	for (tp=a; tp<b;tp++) *tp = *lp = (unsigned short)(0xFFFF&(int)tp);

	VECREAD;	
	for (tp=a;tp<b;tp++) {
		if ((*lp = *tp) != (unsigned short)(0xFFFF&(int)tp)) {
			errs++;
			report("variable",(unsigned short)(0xFFFF&(int)tp),
				*lp,tp);
			}
		}
	
	return(errs);
}

testrand(a,b)	/* random data test */
  short *a;
  short *b;
{
	int errs = 0;
	int rand;
	register short *lp = &Writ;

	printf("%d",++testno);
	if (errorLog)
	  {
	    fprintf(errorLog, "%d",testno);
	    fflush(errorLog);
	  }
	rand = seed;
	VECWRITE;
	for (tp=a; tp<b;tp++)
	   {
		rand = rand<<2 + rand + 17623;
		*tp = *lp = (unsigned short)(0xFFFF&rand);
	   }
	rand = seed;
	VECREAD;
	for (tp=a;tp<b;tp++)
	   {
		rand = rand<<2 + rand + 17623;
		if ((*lp = *tp) != (unsigned short)(0xFFFF&rand))
		   {
			errs++;
			report("random",(unsigned short)(0xFFFF&rand),*lp,tp);
		   }
	   }
	seed = rand - 25;
	return(errs);
}

testdelay(a,b)	/* fill each short with its address, delay, read it back */
  short *a;
  short *b;
{
	int timer;
	int errs=0;

	printf("%d",++testno);
	if (errorLog)
	  {
	    fprintf(errorLog, "%d",testno);
	    fflush(errorLog);
	  }
	VECWRITE;
	for (tp=a; tp<b;tp++) *tp = (unsigned short)(0xFFFF&(int)tp);

	for (timer=100000*DELAY;timer >0; timer--) ; /* sleep */
	
	VECREAD;
	for (tp=a;tp<b;tp++) {
		if (*tp != (unsigned short)(0xFFFF&(int)tp)) {
			errs++;
			report("delayed",(unsigned short)0xFFFF&(int)tp,
				*tp,tp);
			}
		}
	
	return(errs);
}

report(type,expect,got,adr)
char *type;
unsigned short expect;
unsigned short got;
unsigned short *adr;
{
	printf("\nError: %s data test; expected %x, got %x at address %x\n",
		type, expect, got, adr);

	if (errorLog)	
	  {
	    fprintf(errorLog, 
	        "\nError: %s data test; expected %x, got %x at address %x\n",
		type, expect, got, adr);
	    fflush(errorLog);
	  }
}

testbyte(a,b)	/* random byte data test */
  char *a;
  char *b;
{
	int errs = 0;
	int rand;
	register char *lp = (char *) &Writ;

	printf("%d",++testno);
	if (errorLog)
	  {
	    fprintf(errorLog, "%d",testno);
	    fflush(errorLog);
	  }
	rand = seed;
	VECWRITE;
	for (tp=(short *)a; (char *)tp<b; (char *)tp++)
	   {
		rand = rand<<2 + rand + 17623;
		*(char *)tp = *lp = (char)(0xFF&rand);
	   }
	rand = seed;
	VECREAD;
	for (tp=(short *)a; (char *)tp<b; (char *)tp++)
	   {
		rand = rand<<2 + rand + 17623;
		if ((*lp = *(char *)tp) != (char)(0xFF&rand))
		   {
			errs++;
			report("random",(char)(0xFF&rand),*lp,tp);
		   }
	   }
	seed = rand - 25;
	return(errs);
}

