/*
 * pupprint.c - print out a PUP packet
 *
 * Bill Nowicki September 1982
 *
 * with some stuff from Jeff Mogul's pupwatch program
 *
 * Modified:
 * 22 June 1983	  Jeffrey Mogul
 *	- complains if Pup checksums are bad
 */

# include <stdio.h>
# include <puplib.h>
# include <pupconstants.h>
# include <puppacket.h>
# include <leaf.h>


PupPrint( buf, out, count)
  struct PupPacket *buf;
  FILE *out;		/* file to print on */
  int count;		/* size of paccket in short words */
  {
    /*
     * decode a PUP packet onto given file
     */
    struct Port src, dst;
    int length;
    char *PupTypeName();
    ushort checksum();
    int checksumwrd;
    ushort pkcksum;
    
    PortUnpack( &buf->PupSrc, &src);
    PortUnpack( &buf->PupDst, &dst);
    length = buf->PupLength - PUPACKOVER;

    if (count*2 < buf->PupLength + (buf->PupLength & 1))
      {
        fprintf( out, "Bad packet size=%d, PUP Length=%d\n",
		count, buf->PupLength);
      }

    fprintf( out, "PUP Type %s Length %d (%d Data)\n", 
    	PupTypeName(buf->PupType, dst.socket, src.socket),
	buf->PupLength, length);
    
    fprintf( out, "From "); PortPrint( &src, out );
    fprintf( out, " to ");  PortPrint( &dst, out );
    fprintf( out, " ID 0%o", buf->PupID );

    /* check checksum */
    checksumwrd = roundup(length);
    pkcksum = *((ushort *)&(buf->PupData[checksumwrd]));
    if ((pkcksum != NOCKSUM) && (pkcksum !=
    		checksum(buf, checksumwrd + PUPACKOVER - 2))) {
	fprintf(out, " *BAD CHECKSUM*");
    }

    fprintf( out, "\n" );

    switch (buf->PupType)
     {
       case BSP_ACK:
          fprintf(out, "Bytes/Pup: %d, Pups: %d, Total bytes: %d\n",
	      *(short *)(buf->PupData), *(short *)(buf->PupData+2),
	      *(short *)(buf->PupData+4)	);
       	  break;

       case RFC:
	   fprintf(out,"Connection ");
       case NAMERESP:
       case ADDREQ:
           PortUnpack( buf->PupData, &dst );
	   fprintf(out,"Port: ");
           PortPrint( &dst, out);
           fprintf( out, "\n" );
	   break;
	   
       case ERRORPUP:
           PrintErrorPUP(out, buf->PupData, length );
	   break;

       case SEQUIN:
       	  PrintSequin(out, buf->PupID, length, buf->PupData);
	  break;

       case GWINFREP:
           PrintGatewayInfo(out, buf->PupData, length );
	   break;

       default:
          if (length>0)
            {
	        int count;
		extern int PrintData;

	        fprintf( out, "Data: ");
		for (count=0; count<length && (count<64 || PrintData); count++)
	  	    PrettyPutchar( buf->PupData[count], out);
		if (count<length) fprintf(out,"...");
        	fprintf( out, "\n");
	   }
      }
  }


PortPrint(prt, out)
  struct Port *prt;
  FILE *out;
{
	fprintf( out, "%o#%o#",prt->net,prt->host);
	
	switch(prt->socket) {

	case TELNET:
		fprintf( out, "Telnet");
		break;
	case GATEWAYINFO:
		fprintf( out, "GatewayInfo");
		break;
	case FTP:
		fprintf( out, "FTP");
		break;
	case MISCSERVICES:
		fprintf( out, "MiscServices");
		break;
	case ECHOSERVER:
		fprintf( out, "EchoServer");
		break;
	case MAIL:
		fprintf( out, "Mail");
		break;
	case EFTPSERVER:
		fprintf( out, "EftpServer");
		break;
	case EARSSTATUS:
		fprintf( out, "EarsStatus");
		break;
	case LEAF:
		fprintf( out, "Leaf");
		break;
	
	default:
		fprintf( out, "%o",prt->socket);
		break;
	}
}


char *PupTypeName(pt,dsock,ssock)
  unsigned char pt;
  unsigned long dsock;
  unsigned long ssock;
 {
    /*
     * Given a Pup type and a pair of sockets, PupTypeName
     * tries to come up with a human-sensible name for the pup
     * type.
     */
  static char numtype[100];

  if (pt < 0100)
    {	/* Registered pup type */
	
	switch (pt)
	  {
	    case ECHOME:	return("Echome");
	    case IMANECHO:	return("ImAnEcho");
	    case IMABADECHO:	return("ImABadEcho");
	    case ERRORPUP: 	return("Error");
	    case RFC:		return("RFC");
	    case RTP_ABORT: 	return("RTP Abort");
	    case RTP_END:	return("RTP End");
	    case RTP_ENDR: 	return("RTP EndReply");
	    case BSP_DATA: 	return("BSP Data");
	    case BSP_ADATA:	return("BSP AData");
	    case BSP_ACK:	return("BSP Ack");
	    case BSP_MARK:	return("BSP Mark");
	    case BSP_INTR:	return("BSP Interrupt");
	    case BSP_INTRR:	return("BSP InterruptReply");
	    case BSP_AMARK:	return("BSP AMark");

	    case EFTPDATA:	return("Eftp Data");
	    case EFTPACK:	return("Eftp Ack");
	    case EFTPEND:	return("Eftp End");
	    case EFTPABORT:	return("Eftp Abort");
	    
	    default:
		sprintf(numtype,"[strange registered type %o]",pt);
		return(numtype);
	    }
	}


	if ((ssock == MISCSERVICES) ||(dsock == MISCSERVICES)) 
	{
		/* MiscServices pup type */

	 switch (pt) 
	   {
	    case STIMEREQ:    	return("string time request");
	    case STIMERESP:    	return("string time reply");
	    case ATIMEREQ:    	return("Alto time request");
	    case ATIMERESP:    	return("Alto time response");
	    case NAMEREQ:	return("Name lookup request");
	    case NAMERESP:	return("Name lookup response");
	    case ADDREQ:	return("Address lookup request");
	    case ADDRESP:	return("Address lookup response");
	    case LOOKUPERR:	return("Lookup Error");
	    case MMAILREQ:    	return("Msg-style Mail Check Request");
	    case LMAILREQ:    	return("Laurel-style Mail Check Request");
	    case NEWMAIL:	return("Mail Check - New Mail Exists");
	    case NONEWMAIL:	return("Mail Check - No New Mail");
	    case NOSUCHMBOX:	return("Mail Check - No Such Mailbox");
	    case WHRUSREQ:    	return("Where is User request");
	    case WHRUSRESP:    	return("Where is User response");
	    case WHRUSERR:    	return("Where is User Error");
	    case NETDIRVERSION:	return("Net Directory Version");
	    case SENDNETDIR:	return("Send Net Directory");
	    case BOOTFILEREQ:  	return("Boot File Request");
	    case KISSOFDEATH:  	return("Kiss of Death");
	    case AUTHREQ:    	return("User authentication request");
	    case AUTHPOSRESP:  	return("User authentication positive response");
	    case AUTHNEGRESP:  	return("User authentication negative response");
	    case BOOTSTRQ:    	return("Boot Stats request");
	    case BOOTSTRP:    	return("Boot Stats reply");
	    case BOOTDREQ:    	return("Boot directory request");
	    case BOOTDREP:    	return("Boot directory reply");

	    case SENDUMSG:	return("Send User Message");
	    case SENDUACK:	return("Send User Message Ack");
	    case SENDUERR:    	return("Send User Message Error");
	    case SUNBOOTREQ:   	return("Sun Boot request");
	    case SUNBOOTDREQ:  	return("Sun Boot directory request");
	    case SUNBOOTDREP:  	return("Sun Boot directory reply");
	    
	    default:
	    	sprintf(numtype,"[unknown MiscServices type %o]",pt);
		return(numtype);
	    }
	}

	if ((ssock == GATEWAYINFO) ||(dsock == GATEWAYINFO)) 
	 {
	  switch (pt) 
	   {
	    case GWINFREQ:    	return("GateWay Information Request");
	    case GWINFREP:    	return("GateWay Information Reply");

	    default:
	    	sprintf(numtype,"[unknown GatewayInfo type %o]",pt);
		return(numtype);
	    }
	}
	
	if ((ssock == EARSSTATUS) ||(dsock == EARSSTATUS)) {
	
	 switch (pt) 
	  {

	    case EARSTATREQ:	return("Ears Status request");
	    case EARSTATREP:	return("Ears Status reply");
	    default:	    	sprintf(numtype,"[unknown Ears-Status type %o]",pt);
				return(numtype);
	  }
	}

	if ((ssock == LEAF) ||(dsock == LEAF)) 
	 {
	  switch (pt) 
	   {
	    case SEQUIN:	return("Sequin");
	
	    default:	sprintf(numtype,"[unknown Leaf/Sequin type %o]",pt);
			return(numtype);
	    }
	 }
	
	sprintf(numtype,"[unknown unregistered type %o]",pt);
	return(numtype);

}

PrintSequin(f,id,length,buf)
    FILE *f;
    char *buf;
    union SequinID id;
  {
  	/*
	 * print out a sequin packet
	 */
     register struct LeafPacket *lp = (struct LeafPacket *)buf;
     
     fprintf(f,"Sendseq=%d, Recvseq=%d, Allocate=%d, SequinOP=",
       id.codes.Sendseq, id.codes.Recvseq, id.codes.Allocate );
     switch (id.codes.Control)
       {
	  case SEQUINDATA:	fprintf(f,"Data");	break;
	  case SEQUINACK:	fprintf(f,"Ack");	break;
	  case SEQUINNOP:	fprintf(f,"Nop");	break;
	  case SEQUINRESTART:	fprintf(f,"Restart");	break;
	  case SEQUINCHECK:	fprintf(f,"Check");	break;
	  case SEQUINOPEN:	fprintf(f,"Open");	break;
	  case SEQUINBREAK:	fprintf(f,"Break");	break;
	  case SEQUINCLOSE:	fprintf(f,"Close");	break;
	  case SEQUINCLOSED:	fprintf(f,"Closed");	break;
	  case SEQUINDESTROY:	fprintf(f,"Destroy");	break;
	  case SEQUINDALLYING:	fprintf(f,"Dallying");	break;
	  case SEQUINQUIT:	fprintf(f,"Quit");	break;
	  case SEQUINBROKEN:	fprintf(f,"Broken");	break;
	  case SEQUINDEAD:	fprintf(f,"Dead");	break;
	  default:		fprintf(f,"Strange op %d",id.codes.Control);
       }
     fprintf(f,"\n");

     while (length>0)
       {
       lp = (struct LeafPacket *)buf; 
       switch (lp->LeafOp&076000)
         {
	  case LEAFERROR:	fprintf(f,"Leaf Error: ");
	  			PrintLeafError(f,lp->LeafData.Error.Subcode);
	  							break;
	  case LEAFOPEN:	fprintf(f,"LeafOpen");		break;
	  case LEAFCLOSE:	fprintf(f,"LeafClose");		break;
	  case LEAFDELETE:	fprintf(f,"LeafDelete");	break;
	  case LEAFLENGTH:	fprintf(f,"LeafLength");	break;
	  case LEAFTRUNCATE:	fprintf(f,"LeafTruncate");	break;
	  case LEAFREAD:	fprintf(f,"LeafRead");		break;
	  case LEAFWRITE:	fprintf(f,"LeafWrite");		break;
	  case LEAFRESET:	fprintf(f,"LeafReset");		break;
	  case LEAFNOOP:	fprintf(f,"LeafNoop");		break;
	  case LEAFPARAMS:	fprintf(f,"LeafParams");	break;

	  case LEAFERRORANS:	fprintf(f,"Leaf Error Answer: ");
				PrintLeafError(f,lp->LeafData.Error.Subcode);
	  							break;
	  case LEAFOPENANS:	fprintf(f,"LeafOpenAnswer");	break;
	  case LEAFCLOSEANS:	fprintf(f,"LeafCloseAnswer");	break;
	  case LEAFDELETEANS:	fprintf(f,"LeafDeleteAnswer");	break;
	  case LEAFLENGTHANS:	fprintf(f,"LeafLengthAnswer");	break;
	  case LEAFTRUNCATEANS:	fprintf(f,"LeafTruncateAnswer");break;
	  case LEAFREADANS:	fprintf(f,"LeafReadAnswer");	break;
	  case LEAFWRITEANS:	fprintf(f,"LeafWriteAnswer");	break;
	  case LEAFRESETANS:	fprintf(f,"LeafResetAnswer");	break;
	  case LEAFNOOPANS:	fprintf(f,"LeafNoopAnswer");	break;
	  case LEAFPARAMSANS:	fprintf(f,"LeafParamsAnswer");	break;
         }
	buf += (lp->LeafOp & 01777);
	length -= (lp->LeafOp & 01777);
	fprintf(f," of %d bytes ",(lp->LeafOp & 01777) );
	if ( (lp->LeafOp & 01777)==0) break;
     }
    fprintf(f,"\n");
  }

PrintLeafError(f,code)
    FILE *f;
    short code;
  {
    switch(code)
      {
    case IllegalLookupControl:	fprintf(f,"Illegal Lookup Control "); break;
    case Namemalformed:		fprintf(f,"Name Malformed "); break;
    case IllegalChar:		fprintf(f,"Illegal Char "); break;
    case IllegalStar:		fprintf(f,"Illegal Star "); break;
    case IllegalVersion:	fprintf(f,"Illegal Version "); break;
    case NameTooLong:		fprintf(f,"Name Too Long "); break;
    case IllegalDIFAccess:	fprintf(f,"IllegalDIFAccess "); break;
    case FileNotFound:		fprintf(f,"FileNotFound "); break;
    case AccessDenied:		fprintf(f,"AccessDenied "); break;
    case FileBusy:		fprintf(f,"FileBusy "); break;
    case DirNotFound:		fprintf(f,"DirNotFound "); break;
    case AllocExceeded:		fprintf(f,"AllocExceeded "); break;
    case FileSystemFull:	fprintf(f,"FileSystemFull "); break;
    case CreateStreamFailed:	fprintf(f,"CreateStreamFailed "); break;
    case FileAlreadyExists:	fprintf(f,"FileAlreadyExists "); break;
    case FileUndeletable:	fprintf(f,"FileUndeletable "); break;
    case Username:		fprintf(f,"Username "); break;
    case Userpassword:		fprintf(f,"Userpassword "); break;
    case FilesOnly:		fprintf(f,"FilesOnly "); break;
    case ConnectName:		fprintf(f,"ConnectName "); break;
    case ConnectPassword:	fprintf(f,"ConnectPassword "); break;
    case BrokenLeaf:		fprintf(f,"BrokenLeaf "); break;
    case BuddingLeaf:		fprintf(f,"BuddingLeaf "); break;
    case BadHandle:		fprintf(f,"BadHandle "); break;
    case LeafFileTooLong:	fprintf(f,"LeafFileTooLong "); break;
    case IllegalLeafTruncate:	fprintf(f,"IllegalLeafTruncate "); break;
    case AllocLeafVMem:		fprintf(f,"AllocLeafVMem "); break;
    case IllegalLeafRead:	fprintf(f,"IllegalLeafRead "); break;
    case IllegalLeafWrite:	fprintf(f,"IllegalLeafWrite "); break;
    }
  }

PrintErrorPUP(out, buf, length )
    FILE *out;
    struct PupPacket *buf;
    int length;
  {
  	/*
	 * decode an error PUP
	 */
    struct Port src, dst;
    int badLength, count;
    char *PupTypeName();
    
    PortUnpack( &buf->PupSrc, &src);
    PortUnpack( &buf->PupDst, &dst);
    badLength = buf->PupLength - PUPACKOVER;

    fprintf( out, "Erroneous PUP was of Type %s Length %d (%d Data)\n", 
    	PupTypeName(buf->PupType, dst.socket, src.socket),
	buf->PupLength, badLength);
    
    fprintf( out, "From "); PortPrint( &src, out );
    fprintf( out, " to ");  PortPrint( &dst, out );
    fprintf( out, " ID 0%o", buf->PupID );

    length -= PUPACKOVER - 2;
    fprintf( out, " Error codes %o,%o\n", 
    		*(short *)buf->PupData, *(short *)(buf->PupData+2) );
    fprintf( out, "Error string: ");
    for (count=4; count<length && count<64; count++)
	  	    PrettyPutchar( buf->PupData[count], out);
    fprintf( out, "\n" );
  }


PrintGatewayInfo(out, buf, length )
    FILE *out;
    unsigned char *buf;
  {
  	/*
	 * decode a routing packet
	 */
    while (length>0)
      {
        fprintf(out,"Target net: 0%o ", buf[0]);
	if (buf[1]==0 && buf[2]==0 && buf[3]==0)
	  fprintf(out,"Directly connected\n");
	else
          fprintf(out,"Gateway net: 0%o, Host: 0%o, Hops: %d\n",
	    buf[1], buf[2], buf[3]);
        length -= 4;
	buf += 4;
      }
  }