/*
 * ipprint.c - decode Internet Datagrams
 *
 * Bill Nowicki September 1982
 *
 * Copyright (c) 1982 Stanford University
 *
 * This source code can be copied ONLY if all changes and improvements
 * are promptly sent to the Author.
 */

# include <stdio.h>
# include "ip.h"

IPPrint( buf, out )
  IpHdrPtr buf;
  FILE *out;
  {
    /*
     * Decode an IP packet, and print information on the
     * given file about it.
     */
    int headerLen, version, fragment = 0;
    
    headerLen = buf->VersionIhl & HDR_LENGTH;
    version = buf->VersionIhl>>4;

    fprintf( out, "IP from "); IPaddrPrint(buf->source, out);
    fprintf( out, " to "); IPaddrPrint(buf->dest, out);
    if (version != CurrentInternetVersion)
        fprintf( out, ", version %d", version );
    if (headerLen != NormalHeaderLength )
        fprintf( out, ", header length %d", headerLen);
    fprintf( out, "\n");
    if (buf->SvcType)
      {
        fprintf( out, "Service: ");
	if (buf->SvcType & 16) fprintf( out, "Low Delay ");
	if (buf->SvcType &  8) fprintf( out, "Throughput ");
	if (buf->SvcType &  4) fprintf( out, "Reliable ");
	switch ( (buf->SvcType>>5)&7)
	  {
	    case 1: fprintf( out, "Priority ");		break;
	    case 2: fprintf( out, "Immediate ");	break;
	    case 3: fprintf( out, "Flash ");		break;
	    case 4: fprintf( out, "Flash Override ");	break;
	    case 5: fprintf( out, "Critic/ECP ");	break;
	    case 6: fprintf( out, "Internet Control ");	break;
	    case 7: fprintf( out, "Network Control ");	break;
	  }
	fprintf( out, "\n");
      }
     fprintf( out, "Length: %d, ", buf->length);
     if (buf->id) fprintf( out, "id: %d, ",  buf->id);
     if (buf->FlagsFragOffset&FragOffset)
	{
       	  fprintf( out, "Fragment offset: %d, ", 
       		buf->FlagsFragOffset&FragOffset);
	  fragment++;
	}
     if ( (buf->FlagsFragOffset>>13)&1)
        fprintf( out, "More Fragments ");
     if ( (buf->FlagsFragOffset>>14)&1)
        fprintf( out, "Don't Fragment");
     fprintf( out, "Time to live: %d\n", buf->TimeToLive & 0xFF );
     if (headerLen>5)
       {
         fprintf( out, "IP Options: ");
	 PrintIPoptions( ( (long *) buf) + 5, 
	 	(headerLen-5)*4, out );
       }
       
      if (fragment)
        {
	  fprintf(out,"Data: %d bytes\n", buf->length - headerLen*4);
	  return;
	}

      switch ( buf->protocol & 255 )
        {
	  case 1:
	      fprintf( out, "ICMP\n");
	      break;

	  case 6:
	      TCPprint( ( (long *)buf ) + headerLen, 
	      		buf->length - headerLen*4, out );
	      break;

	  case 12:
	      PupPrint(( (long *)buf ) + headerLen, out, 
	      		buf->length/2 - headerLen*2 );
	      break;

	  case 17:
	      UDPprint( ( (long *)buf ) + headerLen,  
	      		buf->length - headerLen*4, out );
	      break;


	  case 77:
	      NDprint( ( (long *)buf ) + headerLen,  
	      		buf->length - headerLen*4, out );
	      break;

	  default:
	      fprintf( out, "Protocol %d\n", buf->protocol & 255);
	      break;
	}
  }


IPaddrPrint( addr, out )
  register addr;
  FILE *out;
  {
    /*
     * print the given IP address on the given file
     */
    int part;
    fprintf( out, "[");
    for (part=0; part<4; part++)
      {
        fprintf( out, "%d", (addr>>24) & 0xFF );
		addr = addr<<8;
	if (part<3) fprintf( out, ".");
      }
    fprintf( out, "]");
  }


PrintIPoptions( buf, count, out )
  register char *buf;
  int count;
  FILE *out;
  {
    /*
     * decode some IP options
     */
    int length;
     
    while (count-->0)
      switch ( (*buf++) & 255)
        {
	  case 0:		/* End of options */
	    fprintf( out, "\n");
	    return;

	  case 1:		/* NO Operation */
	     break;
	     
	  case 130:
	     fprintf( out, "Security ");
	     length = *buf++;
	     buf += length-2;
	     break;
	    	      	     
	  case 131:
	     fprintf( out, "Loose Source and Record Route\n");
	     length = *buf++;
	     buf += length-2;
	     break;
	    	      	    	      	     
	  case 137:
	     fprintf( out, "Strict Source and Record Route\n");
	     length = *buf++;
	     buf += length-2;
	     break;
	    	      
	  case 7:
	     fprintf( out, "Record route ");
	     length = *buf++;
	     buf += length-2;
	     break;
	    	      
       	  case 136:
	     fprintf( out, "Stream ID ");
	     length = *buf++;
	     buf += length-2;
	     break;
	    	      
	  case 68:
	     fprintf( out, "Timestamp ");
	     length = *buf++;
	     buf += length-2;
	     break;
	    	      
	  default:
	     fprintf( out, "<0%o>", buf[-1] );
	     break;
	}
  }


TCPprint( buf, count, out)
  short *buf;
  int count;
  FILE *out;
  {
    int ack, urgent, headerLen, flags, window, checksum;

    fprintf( out, "TCP from port "); IPprintPort( *buf++, out);
    fprintf( out, " to port "); IPprintPort( *buf++, out);
    fprintf( out, " Sequence # %u ", *(long *)buf );
    buf += 2;
    ack = *(long *)buf;
    buf += 2;
    flags = *buf++;
    headerLen = (flags>>12) & 15;
    window = *buf++;
    checksum = *buf++;
    urgent = *buf++;
    fprintf( out, "Window %u, ", window );
    if (flags &  1) fprintf( out, "Fin ");
    if (flags &  2) fprintf( out, "Syn ");
    if (flags &  4) fprintf( out, "Reset ");
    if (flags &  8) fprintf( out, "Push ");
    if (flags & 16) fprintf( out, "Ack # %u", ack);
    if (flags & 32) fprintf( out, "Urgent %d", urgent);
    fprintf( out, "\n");
    count -= headerLen*4;
    headerLen -= 5;
    if (headerLen>0)
      {
        TCPoptions( buf, headerLen*4, out );
	buf += headerLen *2;
      }
    if (count>0)
      {
        int i;
        char *p = (char *)buf;
	extern int PrintData;
	
	fprintf( out, "Data: ");
	for (i=0; i<count && (i<64 || PrintData); i++)
	  {
	    if (*p == IAC) 
	      {
	        if (i++ == count) break;
		p++;
	        switch (*p & 255)
		  {
# define Option(s) p++; i++; \
  PrintTelnetOption( out, s, *p++); i++; continue;
		    case WILL: Option("<Will>");	
		    case WONT: Option("<Won't>");
		    case DO:   Option("<Do>");
		    case DONT: Option("<Don't>");
		    default:   fprintf( out, "<IAC>");
		  }
	      }
	    PrettyPutchar( *p++, out );
	  }
	if (count>64) fprintf( out, "..." );
        fprintf( out, "\n");
      }
  }


PrintTelnetOption( out, s, c )
    FILE *out;
    char *s;
    unsigned char c;
  {
  	/*
	 * decode the telnet options
	 */

    fprintf( out, s);
    switch (c)
      {
	case OPT_BINARY:  fprintf( out, "<Transmit binary>"); return;
	case OPT_ECHO:    fprintf( out, "<Echo>"); return;
	case OPT_RECONNECT:fprintf( out, "<Reconnect>"); return;
	case OPT_SUPPRESS_GA:
			  fprintf( out, "<Suppress go ahead>"); return;
	case OPT_RECSIZE: fprintf( out, "<Approximate message size>"); return;
	case OPT_STATUS:  fprintf( out, "<Status>"); return;
	case OPT_TMARK:   fprintf( out, "<Timing mark>"); return;
	case OPT_RCTE:    fprintf( out, "<RCTE>"); return;
	case OPT_LINEWIDTH:fprintf( out, "<Output line width>"); return;
	case OPT_PAGESIZE:fprintf( out, "<Output size>"); return;
	case OPT_CR_DISP: fprintf( out, "<Output <cr> disposition>"); return;
	case OPT_HT_SET:  fprintf( out, "<Horizontal tab stops>"); return;
	case OPT_HT_DISP: fprintf( out, "<Horizontal tap disposition>"); 
				return;
	case OPT_FF_DISP: fprintf( out, "<Output Formfeed Disposition>"); 
				return;
	case OPT_VT_SET:  fprintf( out, "<Nego. about Vertical Tabstops>"); 
				return;
	case OPT_VT_DISP: fprintf( out, "<Nego. about Output V. T. Disp.>"); 
				return;
	case OPT_LF_DISP: fprintf( out, "<Nego. Output L.f. Disp.>"); return;
	case OPT_XASCII:  fprintf( out, "<Extended ascii option>"); return;
	case OPT_VTERM:   fprintf( out, "<Special VTSS option>"); return;
      }
    fprintf( out, "<Option: %d>", c );
  }

IPprintPort( port, out)
    short port;
    FILE *out;
  {
    switch (port)
      {
        case  1:    fprintf( out, "Old Telnet");	break;
        case  3:    fprintf( out, "Old FTP");		break;
        case  5:    fprintf( out, "RJE");		break;
        case  7:    fprintf( out, "Echo");		break;
        case  9:    fprintf( out, "Black Hole");	break;
        case 11:    fprintf( out, "Systat");		break;
        case 17:    fprintf( out, "Message");		break;
        case 19:    fprintf( out, "TTY Test");		break;
        case 20:    fprintf( out, "FTP Data");		break;
        case 21:    fprintf( out, "FTP");		break;
        case 23:    fprintf( out, "Telnet");		break;
        case 25:    fprintf( out, "SMTP (Mail)");	break;
	case 35:    fprintf( out, "Printer Server");	break;
	case 37:    fprintf( out, "Time Server");	break;
	case 41:    fprintf( out, "Graphics");		break;
	case 42:    fprintf( out, "Name	Server");	break;
        case 57:    fprintf( out, "MTP (Old Mail)");	break;
        case 513:   fprintf( out, "Remote Who");	break;
        default:
	    fprintf( out, "%d", port & 0xFFFF);
	    break;
      }
  }


TCPoptions( buf, count, out )
  register char *buf;
  int count;
  FILE *out;
  {
    /*
     * decode some TCP options
     */
    int length;
     
    fprintf( out, "Options: ");
    while (count-->0)
      switch ( (*buf++) & 255)
        {
	  case 0:		/* End of options */
	    fprintf( out, "\n");
	    return;

	  case 1:		/* NO Operation */
	     break;
	     
	  case 2:
	     length = *buf++;
	     fprintf( out, "Max segment size %d ",
	          ( ( *buf & 255)<<8 ) + ( buf[1] & 255 ) );  
	     buf += length-2;
	     count -= length-2;
	     break;

	  default:
	     fprintf( out, "<0%o>", buf[-1] & 255 );
	     break;
	}
    fprintf( out, "\n");
  }



UDPprint( buf, count, out)
  short *buf;
  int count;
  FILE *out;
  {
    char *p;
    int len,chk, t;
    short destPort;

    if (count < 8) 
      {
	fprintf(out,"Ill formated UDP:\n");
      }
    else 
      {
	fprintf( out, "UDP from port "); IPprintPort( *buf++, out);
	destPort = *buf++;
	fprintf( out, " to port "); IPprintPort( destPort, out);
	len = (unsigned short) *buf++;
	chk = (unsigned short) *buf++;
	fprintf( out, ", Length = %d, Checksum = %d\n",len,chk);
	count -= 8;
      }
    p = (char *) buf;
    switch (destPort)
      {
        case 513:
	  buf += 4; count-= 8;
	  fprintf(out,"Host: ");
	  PrettyString(out,buf,20);
	  buf += 16; count -= 32;
	  fprintf(out,"Loads: %.1f,", GetInt(buf));
	  buf += 2; count -= 4;
	  fprintf(out," %.1f,", GetInt(buf));
	  buf += 2; count -= 4;
	  fprintf(out," %.1f", GetInt(buf));
	  buf += 2; count -= 4;
	  t = GetInt(buf);
	  fprintf(out," Boot time: %15.15s\n", ctime(&t)+4 );
	  buf += 2; count -= 4;
	  while (count > 0)
 	    {
	     fprintf(out,"Terminal: ");
	     PrettyString(out,buf,8);
	     buf += 4;
	     fprintf(out," User: ");
	     PrettyString(out,buf,8);
	     buf += 4;
	     t = GetInt(buf);
	     fprintf(out," Time on: %8.8s", ctime(&t)+11 );
	     buf += 2;
	     t = GetInt(buf);
	     buf += 2;
	     t /= 60;	     
	     fprintf(out," Idle: %2d:%02d", t/60, t % 60 );
	     count -= 24;
	     fprintf(out,"\n");
	    }
	  break;

        default:
    	  while (count > 0) 
      	    {
		PrettyPutchar( *p++, out );
		count--;    
            }
      }
    fprintf( out, "\n");
}


NDprint( buf, count, out)
  short *buf;
  int count;
  FILE *out;
  {
	/*
	 * Print out the weird SMI network disk protocol
	 */
    char *p;
    int op, minor, err, ver;

    if (count < 28) 
      {
	fprintf(out,"ND packet too short!:\n");
      }
    else 
      {
	fprintf( out, "ND ");
	op = *buf >> 8;
	switch (op&7)
	  {
	    case 1: fprintf(out,"Read ");	break;
	    case 2: fprintf(out,"Write ");	break;
	    case 3: fprintf(out,"Error ");	break;
	  }
	if (op&010) fprintf(out,"Wait ");
	if (op&020) fprintf(out,"Done ");
	minor = *buf++ & 0xFF;
	err = *buf >> 8;
	ver = *buf++ & 0xFF;
	fprintf(out,"minor 0%o err=0%o ver=%d\n", minor, err, ver );
	
	fprintf(out,"sequence number %d, n", GetInt(buf));
	buf += 2;
	fprintf(out,"block number %d, ", GetInt(buf));
	buf += 2;
	fprintf(out,"byte count %d\n", GetInt(buf));
	buf += 2;
	fprintf(out,"Residual %d ", GetInt(buf));
	buf += 2;
	fprintf(out,"offset %d ", GetInt(buf));
	buf += 2;
	fprintf(out,"in this packet %d\n", GetInt(buf));
	buf += 2;
      }
    fprintf(out,"%d bytes of data\n", count - 28 );
  }

GetInt(buf)
    register unsigned char *buf;
  {
    register int i;
    
    i = *buf++;
    i <<= 8;
    i |= *buf++;
    i <<= 8;
    i |= *buf++;
    i <<= 8;
    i |= *buf++;
    return(i);
  }

PrettyString(out,s,count)
    FILE *out;
    char *s;
    int count;
  {
  	/*
	 * Print up to count characters
	 */
    for (;count > 0 && *s;count--)
         PrettyPutchar(*s++,out);
   
    while (count-- > 0)
      PrettyPutchar(' ',out);
  }
