/*
 * edp.c - Ethernet diagnostic program (for the 68000)
 *
 * Copyright (c) 1982 by William I. Nowicki
 *
 * NOTE: Permission to copy this program is given ONLY
 *	 if all improvements and bug fixes are promptly returned
 *	 to the Author.
 *
 * Written by Bill Nowicki, March 1981
 *
 * October 1981		Bill Nowicki
 *	- Fixed for PC processor board
 *
 * December 1981	Bill Nowicki
 *	- Added a minimal command interpreter
 *	- Allow for multiple interfaces (gateways)
 *	- Added loopback command
 *
 * July 1982		Bill Nowicki
 *	- Added cooperating host capabilities (68000 or Alto running edp) 
 *	  (can also be echo server)
 *	- Options to disable printing
 *	- check the time and calculate baud rates
 *	- Incorporated "Bud" Spurgeon's May 1982 modifications
 *	- incorporated Bill Yeager's May modifications
 *
 * May 1983		Bill Nowicki
 *	- Added PUP mode, both user and server
 *
 * November 1983	Philip Almquist
 *	- Make echo server generate correct echo replies
 *	- Add "refect" mode, a variant of the echo server useful for
 *	  different (and obscure) things.
 *	- Add an option to make the echo server compute checksums for
 *	  PUP echo replies.  The option also controls whether checksums
 *	  are computed for outgoing packets in reflect mode.
 *
 * Wish list:
 *	- Check for Bus errors (i.e. interface does not respond)
 *	- Check for timeouts on interrupts
 *	- Test loopback mode and collision test
 *	- Check back-back packets
 * 	- Add more error-prone data sequences & randomize
 */

# include "m68000.h"
# include "reentrant.h"
# include "vectors.h"
# include "chars.h"
# include "m68enet.h"

# include "puplib.h"
# include "pupstatus.h"
# include "puppacket.h"
# include "pupconstants.h"

# define Disable IntEnable0
# define Enable  IntEnable1

# define MAXLENGTH 512

/*
 * Some echo stuff for local net echoing.
 * These are Ethernet Types.
 */

# define EchoMe 	0700
# define IamAnEcho 	0701

short dump;

unsigned short
	InputBuffer[MAXLENGTH*2],	/* buffer for incoming packet */
	OutputBuffer[MAXLENGTH+10],	/* buffer for outgoing packet */
					/* these are saved at interrupt level: */
	RxStatus,			/* status of last received packet */
	TxStatus,			/* status of last transmitted packet */
	Lookahead;			/* lookahead word in input FIFO */

   /*
    * Global flags and counters.
    *   Note: Global variables have first letter capitalized!
    */

int
	PupMode = 0,		/* true if doing PUP echos instead of EDP */
	Checksumming = 0;	/* true if computing checksums for outgoing */
				/* PUP echo reply and reflected packets */
	OtherHost = 0,		/* another host we are cooperating with */
				/* zero means self */
	Interrupt = 0,		/* we got an interrupt since sending packet */
	AwaitingInterrupt = 0,	/* we are waiting for a transmit interrupt */
	AwaitingFrame = 0,	/* we are waiting for an incoming packet */
	SentFrame = 0,		/* we got the transmit interrupt */
	ReceivedFrame = 0,	/* we got a packet since the last one sent */
	LoopbackFlag = Loopback1,/* state of the loopback bit */
	SpurriousInterrupt = 0,	/* Interrupt when we were not expecting */
    	LineCount = 0,		/* counts bangs on the line */
	BadCount = 0,		/* size of received wrong */
	FramesSent = 0,		/* number of frames sent */
    	Timeouts = 0,		/* counts timeouts */
    	RxBadStatus = 0,	/* counts bad Rx Statuses */
    	TxBadStatus = 0,	/* same for Tx Status */
    	MisMatches = 0,		/* data compare error */
    	GoodEchos = 0,		/* good packet counter */
    	GoodBits = 0,		/* good bits counter */
	TicksAfterGood = 250,	/* ms after a good packet to do another */
	TicksToWait = 2000,	/* ms after sending to declare a timeout */
	TestStarted = 0,	/* time when test started */
	FixedLength = 0,	/* fixed length packets */
	PrintDebug = 0,		/* print debugging information */
	PrintPerPacket = 1,	/* print per packet */
	PrintErrors = 1,	/* print error information */
	LoopbackCount = 0,	/* counts time to flip loopback bit */
	MaxLoop = 10,		/* how often to flip it */
	LoopbackAlternate = 1,	/* Turn Lookpback on and off */
	CurrentLength = 16,	/* length of our output packet */
	InputLength = 16,	/* length of our input packet */
	MaxLength =  280,	/* maximum size of packet sent */
	MinLength =  16;	/* minimum size of packet sent */


int nCRCd;
int Overflow, Collision;

char Name[256];			/* PUP name of echosending host */

struct Port RemotePort;		/* Remote PUP port to echo to */
struct Port LocalPort;		/* Local PUP port */

extern char OurNetNumber;

short	BadWord, TooBig, TooSmall;

short	 eiUnit = 1,		/* Ethernet Unit number; 
				   following point to its registers */
 	*eiData0, 		/* write: output data FIFO */
				/* read: input data FIFO */
	*eiData1,		/* Don't know what this is for */
	*eiStatus, 		/* write: control bits */
				/* read: some status bits */
	*eiAddress;		/* write: set/clear bit in reception vector */
				/* read: host address switch settings */


char MyAddress = 0;		/* The switch setting on the ethernet board */


InitEthernetInterface(IntHandler)
 int IntHandler; 	/* really a procedure pointer */
   {
       /*
        * initialize the ethernet interface. Set up the interrupt vector,
	* read the address, and set up the status register
	*/
     register short address;
     int count;

	intlevel(2);

	printf( "  \nTesting Ethernet interface unit number %d\n", eiUnit);
	printf( ". means packet was sent, ! means received\n");

	eiData0   = (short*)(IOLOWADR+eiUnit*0x100);
	eiData1   = eiData0 + 1;
	eiStatus  = eiData1 + 1;
	eiAddress = eiStatus + 1;
	IRQ2Vect  = IntHandler;

	*eiStatus = Init1+Disable+FilterData0+Loopback0;
	printf( "Found Ethernet board number %d\n", eiUnit );

	for (address=0; address<256; address++)	/* Clear Address Filter */
		*eiAddress = address;

	MyAddress = address = *eiAddress;	/* Activate Switch Address */
 	*eiStatus = Init1+Disable+FilterData1+Loopback0;
	*eiAddress = address;	
	*eiStatus = Init0+Enable+LoopbackFlag+IntLevel2;

	printf( "My Ethernet address is %d. (0%o)\n", address, address);
	if ( address == 0 || (address & 0xff) == 0xff )
	    printf("Bad address value read from switches\n%s",
	           "(zero and FF are special ethernet addresses!)\n"  );

	for ( count=0; count<=81292; count++)
	   address = *eiData0;
	if ( !(address&QueueEmptyBit) )
	  printf( " input queue did not drain properly!\n" );
	intlevel(0);

      FramesSent = 0;
      nCRCd = 0;
      Overflow = 0;
      Collision = 0;
      Timeouts = 0;
      GoodEchos = 0;
      GoodBits = 0;
      MisMatches = 0;
      BadCount = 0;
      BadWord = 0; 
      TooBig = 0; 
      TooSmall = 0;

   }


reentrant(EthernetInterrupt)
 {
  register unsigned short count, status;
  short i;

 /*
  * The ethernet interrupt service routine
  */
	Interrupt = 1;

	Lookahead = status = *eiData0;
	if ( status & QueueEmptyBit ) status = *eiData0;	

	while ( !(status & QueueEmptyBit))
	    {
  	        RxStatus = status;
		count = status & CountMask;
		for (i=0; i<count; i++)
	  	   {
		 	InputBuffer[i] = *eiData0;
		   }
		if (count != CurrentLength+1) 
		  {
		    ++BadCount;
		    if (count < CurrentLength) ++TooSmall;
		    if (count > CurrentLength+1) ++TooBig;
		    BadWord = InputBuffer[count];
		  }
		InputLength = count - 1;
		ReceivedFrame = 1;
		status = *eiData0;
	    }

	TxStatus = *eiStatus;
	if (TxStatus & TSent)
	   {
		status = *eiAddress;	/* Clear Tx Interrupt */ 
		SentFrame = 1;
		AwaitingInterrupt = 0;
	   }
	/* *eiStatus = Init0+Disable+LoopbackFlag+IntLevel2; */
}


TransmitFrame(count, buffer)
 int count; 
 unsigned short *buffer;
 {  
 	/*
	 * send the indicated frame on the Ethernet, and set the appropriate
	 * counters and flags.
	 */
     register int i;

     AwaitingInterrupt = 1;
     AwaitingFrame = 1;
     ReceivedFrame = 0;
     ++FramesSent;
     *eiStatus = Init0+Enable+LoopbackFlag+FilterData0+IntLevel2;
     *eiData0 = count;
     for (i=0; i<count; i++) *eiData0 = buffer[i];
  }


DisplayMessage()
 {
  register int i;

   printf("Received message: \t");
   for (i=1; i<CurrentLength; i++) printf("%o\t",InputBuffer[i],"\t");  
   putchar( '\n' );
 }


DecodeTx( value )
 {
   /*
    * print a human readable version of the transmitter status word
    */
  if ( value == 0 )
   {
     printf( "Zero\n" );
     return;
   }

  if ( LoopbackFlag == Loopback1 )
       printf( "Loopback ");
  else printf( "Ether ");
  
  if ( value & TSent         ) printf(  "Sent ");
  if ( value & TimeoutBit    ) printf(  "Timeout ");
  if ( value & EIDone	     ) printf(  "Done ");
  if ( value & RxRcvd        ) printf(  "Packet Received ");
  putchar( '\n' );
 }


DisplayTransmitStatus()
 {
    /*
     * display the transmit status if anything is amiss
     */
   if ( ! (TxStatus & TimeoutBit) ) return;
   TxBadStatus++;
   if (PrintErrors)
     {
       printf("Transmitted Message Status:" ); 
       DecodeTx( TxStatus );
     }
 }


DecodeRx( value )
 {
  /*
   * print out a human readable form of the Receiver status register
   */

  if ( value == 0 )
   {
     printf( "Zero\n" );
     return;
   }
  
# define RxErrors (OverflowBit | CollisionBit | CRCerrorBit)

  if ( value & QueueEmptyBit ) printf(  "Empty ");
  if ( value & OverflowBit   ) printf(  "Overflow ");
  if ( value & CollisionBit  ) printf(  "Collision  ");
  if ( value & CRCerrorBit   ) printf(  "CRC Error ");
  if ( value & CountMask     ) 
    printf( "Count=%d ", value & CountMask );
  putchar( '\n' );
 }

int newCRC, newCOL, eerror;	/* some more Yeager hacks */


DisplayStatus()
 {
    /*
     * display the receiver status if anything interesting is going on
     * The length of packets read includes the CRC, so add one.
     *
     * Returns 0 on errors, 1 if everything went OK.
     * 
     */
   if ( (RxStatus & RxErrors)  || (RxStatus & CountMask) != CurrentLength+1 ) 
    {
      RxBadStatus++;
      if (PrintErrors)
        {
          printf(" Current Length %d, RxStatus: ", CurrentLength);  
          DecodeRx(RxStatus);
	}
      return(0);
   }
  return(1);
 }



BuffersMatch()
 {
  /*
   * returns true if the message received matches the message 
   * that we sent. Also prints out up to ten differences.
   * Starts at two because Ethernet Type is first word, 
   * Ethernet address is zeroth word.
   * Also skip the port if this is a PUP echo.
   */
  register int i, bad = 0;
  int first, last;
  
  if (PupMode)
    {
      if (InputBuffer[1] != PUP) 
        {
	  printf("Strange packet type: 0%o\n", InputBuffer[1] & 0xFFFF);
	  return(0);
	}
      if ( (InputBuffer[3] & 255) != IMANECHO) 
        {
	  printf("Strange PUP type: 0%o, expected Echo response: 0%o\n", 
	  	InputBuffer[2] & 255, IMANECHO);
	  return(0);
	}
      first = PUPACKOVER/2 + 1;
      last  = CurrentLength - 1;
    }
  else
    {
      first = 2;
      last = CurrentLength;
    }

  for (i = first; i < last; i++)
     if ( InputBuffer[i] != OutputBuffer[i] )
        {
	  if ( bad == 0 ) 
	     printf( "\nComparison error on a packet of length %d\n", 
	                     CurrentLength );
	  if ( bad++ < 10 ) 
	   {
	    register unsigned In= 0, Out= 0;
	    In = InputBuffer[i]; Out = OutputBuffer[i];
 	    printf( "Word %d should be %x, is %x (difference %x, xor %x)\n", 
	       i, Out,  In, (Out - In) & 0x0FFFF, (Out ^ In) & 0x0FFFF);
	   }
	 }

  if ( bad > 10 )
    printf( "Total of %d words were bad in that packet\n", bad );

  if ( bad )
    {
     MisMatches++;
     return( 0 );
    }
  else
    {
      GoodEchos++;
      GoodBits += CurrentLength*16;
      return( 1 );
    }
 }
 

CheckTransmit()
 {
 	/*
	 * This routine is called periodically to check the
	 * transmitter status and print a dot when done.
	 */
   if (SentFrame) 
     {
      DisplayTransmitStatus();
      if (PrintPerPacket)
	{
          if ( LineCount++ > 32 ) 
	   {
	      putchar( '\n' );
	      LineCount = 0;
	   }
          putchar('.');
	} 
      SentFrame = 0;
     }
 }


char *CRCs = {"CRC ERROR"};
char *COLs = {" COLLISION ERROR"};
char *OVFs = {"OVERFLOW ERROR"};
char *GOODY= {"GOOD PACKET"};
short lastOK;

char *errType() 
  {
    if (lastOK) return(GOODY);
    if (newCRC) return(CRCs);
    if (newCOL) return(COLs);
    return(OVFs);
  }

	/*
	 * All of this terribly unreadable code is Bill Yeager's.
	 * complain to him about it.
	 */

dumppacket()
  {
    register short i, j; 
    unsigned short Adds[3];
    
    j = CurrentLength+1;
    printf ("\nDumping packet, length= %d %s\n", CurrentLength, 
	     errType());
    printf ("  Sent    Rcvd    XOR\n");

    for (i=0; i<j; ++i) 
      {
        register unsigned In, Out;

	Out = OutputBuffer[i] & 0x0FFFF; In = InputBuffer[i] & 0x0FFFF;
	/* DON'T print Output CRC */
	if (i != CurrentLength) printf ("%06O  ", Out);
	else
	    {printf ("        "); Out = In;}
 	printf ("%06O  ", In);
 	printf ("%06O", In ^ Out);
	if (i == CurrentLength) printf ("  <== IN CRC ONLY");
	printf ("\n");
      }
  }

short thewait;
char *helpline =
       {"\n   Type P for error message printout\n   Type H to hold display"},
     *dumpline =
       {"   Type D to dump next ERRORED packet, G next GOOD one.\n"},
     *anyline = {"(ANY char continues)\n"};


SetupPacket()
  {
	/*
	 * Prepare the packet to be sent.
	 *	The zeroth word is the source and destination address,
	 *	and the first is the Ethernet type.
	 * If this is a PUP echo, then we have more work to do.
	 */
    register int i;
    int destination = OtherHost;
    
    if (OtherHost==0) destination = MyAddress;
    
    if (destination!=MyAddress)
      {
        LoopbackAlternate = 0;
	LoopbackFlag = Loopback0;
      }

    OutputBuffer[0] = ( MyAddress & 0x0ff ) | ( (destination & 0xff) << 8 );
    i = 2;
    if (PupMode)
      {
        OutputBuffer[1] = PUP;				/* Ether Type */
	OutputBuffer[i++] = CurrentLength*2 - 4;	/* PUP Length */
	OutputBuffer[i++] = ECHOME;			/* PUP Type */
	OutputBuffer[i++] = 0;				/* PUP ID */
	OutputBuffer[i++] = 0;
	PortPack( &RemotePort, OutputBuffer+i );
	i += sizeof(struct packedPort)/2;
        LocalPort.host = MyAddress;
        LocalPort.net = OurNetNumber;			
        LocalPort.socket = ECHOSERVER;
	PortPack( &LocalPort, OutputBuffer+i );
	i += sizeof(struct packedPort)/2;
      }
    else
      {
        OutputBuffer[1] = EchoMe;
      }
    
    for (; i<MAXLENGTH; i++) OutputBuffer[i] = i;
    if (PupMode)
      {
      		/*
		 * set the last word of the packet to the PUP checksum
		 */
	OutputBuffer[CurrentLength - 1] = 
		checksum( OutputBuffer+2, CurrentLength*2 - 6 );
      }
  }


PrintSummary()
  {
    /*
     * Print out some information on the counts so far
     */

   if (FramesSent) 	printf( " Frames sent: %d, ", FramesSent );
   if (GoodEchos)	printf( "Good: %d, ", GoodEchos );
   if (MisMatches) 	printf( "Mismatches: %d, ", MisMatches );
   if (Timeouts) 	printf( "Timeouts: %d, ", Timeouts );
   if (RxBadStatus) 	printf( "RxBadStatus: %d, ", RxBadStatus );
   if (TxBadStatus) 	printf( "TxBadStatus: %d, ", TxBadStatus );
   if (GoodEchos)	printf( "Packets/sec: %.1f, ", 
  		(1000.0 * GoodEchos)/(emt_ticks()-TestStarted) );
   if (GoodEchos)	printf( "K Bits/sec: %d ", 
  		GoodBits/(emt_ticks()-TestStarted) );
  }


HitKey()
  {
	/*
	 * return true if a key was hit, handling
	 * both SMI suns and serial line interfaces
	 */
    int c = emt_mayget();

    if (c != -1) return(1);
    if (linereadyrx(0)==0) return(0);
    getchar();
    return(1);
  }


Test()
 {
   /*
    * main test loop here
    */
  int timeout;		/* time to transmit another packet */
  int start;
  
  InitEthernetInterface( EthernetInterrupt );
  SetupPacket();

  TransmitFrame( CurrentLength, OutputBuffer);
  timeout = emt_ticks() + TicksToWait;
  TestStarted = emt_ticks();
  start = PupMode ? (PUPACKOVER/2) : 2;

  while ( !HitKey(0) )
    {
      CheckTransmit();
      if (ReceivedFrame) 
        {
          AwaitingFrame = 0;
          ReceivedFrame = 0;
	  if ( DisplayStatus() && BuffersMatch() ) 
	    {
		timeout = emt_ticks() + TicksAfterGood;
	        if (PrintPerPacket) putchar( '!' );
	    }

	  if ( LoopbackCount++ > MaxLoop )
	    {
	     /*
	      *  Turn loopback on & off periodically, if the
	      * LoopbackAlternate flag is set.
	      */
	      if (LoopbackAlternate) LoopbackFlag ^= Loopback1;
	      LoopbackCount = 0;
	    }

          if ( !FixedLength )
	    {
	     if (CurrentLength++ > MaxLength ) 
	      {
	         /*
		  * after the length hits MAX, restore it to the minimum
		  * and invert the data.
		  */
		register int i;

	        CurrentLength = MinLength;

		for ( i=start; i<MAXLENGTH; i++)
		    OutputBuffer[i] = ~ OutputBuffer[i];
	      }
	    SetupPacket();
         }
	  continue;

	} /* end of if ReceivedFrame */

      if (timeout < emt_ticks()) 
	{
         if ( AwaitingInterrupt  )
	     {
	        printf("Interrupt timeout\n");
		Timeouts++;
	     }
	  else if (AwaitingFrame) 
	    {
	      Timeouts++;
	      printf("Echo Timeout, length=%d words\n", CurrentLength);
	    }
          if ( !FixedLength )
	    {
	     if (CurrentLength++ > MaxLength ) 
	      {
	         /*
		  * after the length hits MAX, restore it to the minimum
		  * and invert the data.
		  */
		register int i;

	        CurrentLength = MinLength;

		for ( i=start; i<MAXLENGTH; i++)
		    OutputBuffer[i] = ~ OutputBuffer[i];
	      }
	    SetupPacket();
         }
 	  TransmitFrame( CurrentLength, OutputBuffer);
	  timeout = emt_ticks() + TicksToWait;
	}
    } /* end of while loop */

  putchar( '\n' );
  PrintSummary();

# ifdef Yeager

      switch (getchar() & 0177) 
	{
	  case 'd': case 'D': /* dump next errored packet */
	      dump = 1; continue;

	  case 'g': case 'G': /* dump next good packet */
	      gooddump = 1; continue;

	  case 'p': case 'P': /* Print error status */
	      CheckTransmit();
              DisplayStatus(); continue;

	  case 'H': case 'h': /* hold display until conQ */
	      getchar();      /* ANY char. continues. */
	      continue;

	  case '?': 
  	      printf("%s%s%s", helpline, anyline, dumpline);
  	      printf("(ANY char continues)\n");
	      continue;
	  default: break;
	}
      break;

# endif Yeager
  } /* end of Test procedure */


EchoServer()
  {
  	/*
	 * In this mode we just wait for packets to come in
	 * and, and send then back whence they came.
	 */

    LoopbackFlag = Loopback0;
    InitEthernetInterface( EthernetInterrupt );
    TestStarted = 0;

    while ( !HitKey(0) )
      {
        CheckTransmit();
	if (ReceivedFrame)
	  {
	    if (TestStarted==0)  TestStarted = emt_ticks();  
	    ReceivedFrame = 0;
	    if (RxStatus & RxErrors)
	      {
	       DecodeRx( RxStatus );
	      }
	     else GoodEchos++;
      	    GoodBits += InputLength*16;
	    InputBuffer[0] = ( MyAddress & 0x0ff ) | ( (InputBuffer[0] & 0xff) << 8 );
	    if (InputBuffer[1] == PUP)
	      {
	        if ( (InputBuffer[3] & 255) != ECHOME)
		  {
		    printf( "Strange Pup Type: 0%o, expected 0%o\n", 
		    	InputBuffer[3] & 255, ECHOME);
		    continue;
		  }
	        InputBuffer[3] = IMANECHO;
		PortUnpack( InputBuffer+9, &RemotePort );
		PortPack( &RemotePort, InputBuffer+6 );
        	LocalPort.host = MyAddress;
        	LocalPort.net = OurNetNumber;			
        	LocalPort.socket = ECHOSERVER;
		PortPack( &LocalPort, InputBuffer+9 );
		if (Checksumming)
		    InputBuffer[InputLength-1] =
				checksum( InputBuffer+2, InputLength*2 - 6);
		else
		    InputBuffer[InputLength-1] = NOCKSUM;
	      }
	    else
	      InputBuffer[1] = IamAnEcho;
	    TransmitFrame( InputLength, InputBuffer );
	    if (PrintPerPacket) putchar( '!' );
	  }
      }
    putchar( '\n' );
    PrintSummary();
  }


ReflectServer()
  {
  	/*
	 * In this mode we receive PUP packets and send them back to
	 * the host that they came from.  That sounds like an echo
	 * server?  Well, not quite, because:
	 *
	 *  - any PUP packet (not just echo requests) are accepted
	 *  - the PUP type of the packet is not changed to echo reply
	 *    (in fact, it is not changed).
	 *  - In the reflected packet the source and destination nets
	 *    and hosts are swapped, but the source and destination
	 *    sockets are not.
	 *
	 * But why?  Because (for debugging reasons) I needed to make
	 * a SUN act like it understood a protocol that I didn't have
	 * the patience to make it really understand.  Sorry you asked?
	 *
	 * Non-PUP packets are discarded.  Some features that might be
	 * useful in the future are address filters of some sort.
	 */

    Socket TempSocket;

    LoopbackFlag = Loopback0;
    InitEthernetInterface( EthernetInterrupt );
    TestStarted = 0;

    while ( !HitKey(0) )
      {
        CheckTransmit();
	if (ReceivedFrame)
	  {
	    if (TestStarted==0)  TestStarted = emt_ticks();  
	    ReceivedFrame = 0;
	    if (RxStatus & RxErrors)
	      {
	       DecodeRx( RxStatus );
	      }
	     else GoodEchos++;
      	    GoodBits += InputLength*16;
	    InputBuffer[0] = ( MyAddress & 0x0ff ) | ( (InputBuffer[0] & 0xff) << 8 );
	    if (InputBuffer[1] == PUP)
	      {
		PortUnpack( InputBuffer+9, &RemotePort );
		PortUnpack( InputBuffer+6, &LocalPort );
		TempSocket = LocalPort.socket;
		LocalPort.socket = RemotePort.socket;
		RemotePort.socket = TempSocket;
		PortPack( &RemotePort, InputBuffer+6 );
		PortPack( &LocalPort, InputBuffer+9 );
		if (Checksumming)
		    InputBuffer[InputLength-1] =
				checksum( InputBuffer+2, InputLength*2 - 6);
		else
		    InputBuffer[InputLength-1] = NOCKSUM;
		TransmitFrame( InputLength, InputBuffer );
		if (PrintPerPacket) putchar( '!' );
	      }
	  }
      }
    putchar( '\n' );
    PrintSummary();
  }


CheckBusErrors()
 {
  	/*
	 * I guess this routine should do something with Bus Errors?
	 */
 }


main()
  {
    sp = (int *)emt_getmemsize()-4;
    linedisarmtx(0);
    linedisarmrx(0);
    putchar( '\f' );
    printf( "Ethernet Diagnostic Program version 3.1 \n");
    LoopbackFlag = Loopback0;

    while (1) 
     {
        putchar( '\n' );
        CheckBusErrors();
	printf( "0-4 - select Ethernet interface number\n" );
	
	printf( "a - specify address for echo sending\n");

	printf( "c - Turn %s checksumming of outgoing PUP packets\n",
	   Checksumming ? "off" : "on" );

	printf( "e - echo to another host           ");
	if (OtherHost) printf( "(current = 0%o)", OtherHost );
	putchar('\n');

	printf( "f - fixed length packets           %s\n",
		!FixedLength ? "" : "(current setting)" );

	printf( "i - interval between packets       (current = %d millisecs)\n",
	    TicksAfterGood );

	printf( "l - loopback options               (currently %s)\n",
	   LoopbackAlternate ? "alternating" :
	   ( LoopbackFlag == Loopback1) ? "local" : "ether" );

	printf( "m - Message printout control       (current = %s good, %s errors)\n",
	   PrintPerPacket ? "print" : "ignore",
	   PrintErrors    ? "print" : "ignore" );

	printf( "p - PUP mode                       (currently %s)\n",
	   PupMode ? "ON" : "raw Ether" );

	printf( "r - reflect server mode\n");

	printf( "s - echo server mode\n");

	printf( "t - time out interval              (current = %d millisecs)\n",
	    TicksToWait );

	printf( "v - variable length packets        %s\n",
		FixedLength ? "" : "(current setting)" );

	printf( "w - set words per packet           (current = %d)\n" ,
	 CurrentLength );

	printf( "q - quit the program\n" );

	printf( "Return or space performs test\n" );
        printf( "Select command:" );
	switch ( getchar() & 0177)
	  {
	   case '0': eiUnit = 0; printf( "\n" ); break;
	   case '1': eiUnit = 1; printf( "\n" ); break;
	   case '2': eiUnit = 2; printf( "\n" ); break;
	   case '3': eiUnit = 3; printf( "\n" ); break;
	   case '4': eiUnit = 4; printf( "\n" ); break;
	   
	   case 'l':
	   case 'L':
	       if (LoopbackAlternate) 
	             printf( "\nLoopback is alternating\n" );
		else printf( "\nMode is %s\n", LoopbackFlag==Loopback1 ?
			 "always loopback" : "transmit" );
		printf( "Options are Alternate, Loopback, Transmit, or Don't change:" );
		switch (getchar())
		  {
		    case 'l':
		    case 'L':
		        printf( "\nWill always loopback\n");
			LoopbackAlternate = 0;
			LoopbackFlag = Loopback1;
			break;

		    case 't':
		    case 'T':
		        printf( "\nWill always transmit\n");
			LoopbackAlternate = 0;
			LoopbackFlag = Loopback0;
			break;

		    case 'a':
		    case 'A':
		        printf( "\nWill alternate loopback and transmitting\n");
			LoopbackAlternate = 1;
			LoopbackFlag = Loopback1;
			break;

		
		   default:
		        printf( "\nMode unchanged\n");
			break;
		  }
		 continue;

	   
	   case 'm':
	   case 'M':
	       if (PrintPerPacket) 
	             printf( "\nPrinting good packets\n" );
		else printf( "\nNot printing good packets\n" );
	       if (PrintErrors) 
	             printf( "\nPrinting errors\n" );
		else printf( "\nNot printing errors\n" );
		printf( "Options are Good, Bad, Silent, Normal, or Don't change:" );
		switch (getchar())
		  {
		    case 'g':
		    case 'G':
			printf( "\nWill print only good packets\n");
			PrintPerPacket = 1;
			PrintErrors = 0;
			break;
		
		    case 'b':
		    case 'B':
			printf( "\nWill print only bad packets\n");
			PrintPerPacket = 0;
			PrintErrors = 1;
			break;
		
		    case 's':
		    case 'S':
			printf( "\nWill not print anything\n");
			PrintPerPacket = 0;
			PrintErrors = 0;
			break;
		
		    case 'n':
		    case 'N':
			printf( "\nWill print good packets and errors\n");
			PrintPerPacket = 1;
			PrintErrors = 1;
			break;
		
		   default:
		        printf( "\nPrintout mode unchanged\n");
			break;
		  }
		 continue;

	    case 'D':
	    case 'd': 
	        PrintDebug = 1; 
		break;

	    case 'e':
	    case 'E': 
		printf("\nEcho to (octal) LOCAL host # (zero for none)");
		if ( scanf("%o", &OtherHost ) != 1 ) break;
		if (OtherHost==0)
		    printf( "Will Echo to myself!\n");
		break;

	    case 's':
	    case 'S': 
		printf( "\nEcho server started...\n");
		EchoServer();
		break;

	    case 'r':
	    case 'R': 
		printf( "\nReflect server started...\n");
		ReflectServer();
		break;

	    case 't':
	    case 'T':
		printf("\nEnter MILLI-seconds before timeout:");
		scanf( "%u", &TicksToWait );
		if (TicksToWait < 1) TicksToWait = 1; 
		break;

	    case 'i':
	    case 'I':
		printf("\nEnter MILLI-seconds between output packets:");
		scanf( "%u", &TicksAfterGood );
		if (TicksAfterGood < 1) TicksAfterGood = 1;
		break;

	    case 'p':
	    case 'P':
		if (PupMode)
		    printf("\nTurning off PUP mode, back to raw Ether\n");
		else
		    printf("\nTurning on PUP mode\n");
		PupMode = !PupMode;
		break;

	    case 'c':
	    case 'C':
		if (Checksumming)
		    printf("\nWon't checksum outgoing PUP packets\n");
		else
		    printf("\nWill checksum outgoing PUP packets\n");
		Checksumming = !Checksumming;
		break;

	    case 'a':
	    case 'A':
	        printf( "\nEnter address for PUP echos: ");
		gets(Name);
		if ( mlookup(Name, &RemotePort) == OK)
		  {
		    unsigned char dest = 0;

		    if (puproute(&RemotePort, &dest, NULL))
		      {
		        printf("Cannot route to that host\n");
		      }
		    else OtherHost = dest;
		    printf("Using host %s (%o#%o#%o) via host 0%o\n", Name,
			RemotePort.net, RemotePort.host, 
			RemotePort.socket, dest);
		    if (RemotePort.socket == 0) 
		    	RemotePort.socket = ECHOSERVER;
		  }
		else
		  {
		    printf("Name not found\n");
		  }
		break;

	    case 'w':
	    case 'W':
	    	printf( "\nEnter size in words to transmit:" );
		scanf( "%u", &CurrentLength );
	        if (CurrentLength < 4) CurrentLength = 4;
		if (CurrentLength > MAXLENGTH) 
		  {
		    CurrentLength = MAXLENGTH;
		    printf ("Using %d words\n", MAXLENGTH);
		  }
		break;

	    case 'f':
	    case 'F':
		printf( "\nFixed length packets of size %d\n", CurrentLength);
		printf( "Use 'w' to change the size.\n");
		FixedLength = 1;
		break;

	    case 'v':
	    case 'V':
		printf( "\nVariable length packets set\n");
		FixedLength = 0;
		break;

	    case 'q':
	    case 'Q':
	    case '\003':
	        printf( "\nExiting to monitor.\n" );
	        return;
		
	    case ' ':
	    case CR:
	        printf( "\nType any character to stop the test\n" );
		Test();
	        continue;
		
	    default:
	       printf( "\n No such command!!!\n" );
	   }
      }

  }
