/*
 * pcmap.h
 *
 * Macros and definitions for Sun-1 pc board version memory mapping
 * PLEASE DO NOT EDIT THIS FILE WITHOUT CHECKING FIRST WITH ME!
 *
 * Jeffrey Mogul	20 March 1981
 *
 * 4 November 1981 (jcm) -- now maps highest page as invalid.
 * 8 July 1981 (jcm) -- changed Multibus Memory map to cover ca. 1mb
 * 7 May 1981 (jcm) -- revised for PC board version
 * 6 May 1981 (jcm) -- added mapping for Multibus Memory space
 *
 * stolen from code written by Luis Trabb-Pardo
 */

/*
 * address space allocations
 */
#define	ADRSPC_RAM	0x0		/* Read/Write, segmented memory */
#define ADRSPC_PROM0	0x200000	/* ROM #0, boot space */
#define ADRSPC_PROM1	0x400000	/* ROM #1 */
#define	ADRSPC_UART	0x600000	/* onboard UART */
#define ADRSPC_TIMER	0x800000	/* onboard Timer */
#define ADRSPC_PAGEMAP	0xA00000	/* base of page table */
#define ADRSPC_SEGMAP	0xC00000	/* base of segment table */
#define ADRSPC_CONTEXT	0xE00000	/* context register */

#define ADRSPC_SIZE	0x200000	/* size of each address space */


/*
 * Memory is divided into 16 contexts; the current context is
 * defined by the context register.
 */

/*
 * Configuration register: read the "context register" to get
 * the contents of the configuration register.
 */
#define	GETCONFIG	(*(unsigned short *)ADRSPC_CONTEXT)

/*
 * Context Register -- only 4 high order bits (of 16) count
 */
#define CONTEXTREG (*(unsigned short *)ADRSPC_CONTEXT)

#define SETCONTEXT(x) CONTEXTREG = ((x)<<12)

/*
 * Surprise! to read the context register, look at the
 * high 4 bits of any segment map entry. (?)
 */
#define	GETCONTEXT	((*(unsigned short *)ADRSPC_SEGMAP) >> 12)

#define NUMCONTEXTS	16

/*

   A memory address is 

	struct {
		unsigned reserved:3;		-- not currently used, MBZ 
		unsigned SegInContext:6;	-- segment within contex 
		unsigned PageInSeg:4;		-- page within segment 
		unsigned ByteInPage:11;		-- byte within page 
		}	MemAddress;

   The segment number is computed as <context><SegInContext>, yielding
   a 10-bit segment table index.
   
   A segment table entry is
   
   	struct {
		unsigned CurContext:4;		-- Current context (Read only)
		unsigned SegProt_User:1;	-- allows User access
		unsigned SegProt_Write:1;	-- allows Write access
		unsigned SegProt_Read:1;	-- allows Read access
		unsigned SegProt_Exec:1;	-- allows Execute access
		unsigned reserved:2;		-- not currently used, MBZ
		unsigned PageNumHi:6;		-- high bits of Page number
		}	SegTabEntry;
		
   The virtual page number is computed as <PageNumHi><PageInSeg>,
   yielding a 10-bit page table index.
   
   A page table entry is
   
   	struct {
		unsigned PageUsed:1;		-- "Used" flag
		unsigned PageDirty:1;		-- "Dirty" bit
		unsigned PageSpace:2;		-- physical page space
		unsigned PageFrameNumber:12;	-- physical page index
		}	PageTabEntry;
   
   The physical memory address is computed as <PageFrameNumber><ByteInPage>,
   yielding a 23-bit byte address.

*/

/*
 * Segment map constants/macros
 */
#define	SEGMAPBASE	ADRSPC_SEGMAP	/* base of segment map */
#define CONTEXTSIZE	0x40		/* 64. segments per context */
#define SEGMAPSIZE	0x400		/* 16*64 segments */

/* return address of segment map entry #<seg> */
#define SEGMAPADR(seg) ((short *)(SEGMAPBASE+((seg)<<15)))

/* set segment table entry #<seg> to <entry> */
#define SETSEGMAP(seg,entry) *SEGMAPADR(seg) = (entry)

/* get segment table entry #<seg> */
#define GETSEGMAP(seg) (*SEGMAPADR(seg))
	
/* segment protection bits -- can be set independently */
#define SEGPRO_EXEC	0x100
#define SEGPRO_READ	0x200
#define SEGPRO_WRITE	0x400
#define SEGPRO_USER	0x800
#define SEGPRO_ALL	(SEGPRO_EXEC|SEGPRO_READ|SEGPRO_WRITE|SEGPRO_USER)

/*
 * Page map constants/macros
 */
#define	PAGEMAPBASE	ADRSPC_PAGEMAP	/* base of page map */
#define PAGEMAPSIZE	0x400		/* 1024. pages total */
#define PAGESIZE	0x800		/* 2048. bytes per page */

/* return address of page map entry #<page> */
#define PAGEMAPADR(page) ((short *)(PAGEMAPBASE+((page)<<11)))

/* set page table entry #<page> to <entry> */
#define SETPAGEMAP(page,entry) *PAGEMAPADR(page) = (entry)

/* get page table entry #<page> */
#define GETPAGEMAP(page) (*PAGEMAPADR(page))
	
/* page "address space modes" -- mutually exclusive */
#define PGSPC_MEM	0x0000		/* on-board memory */
#define PGSPC_NXM	0x1000		/* invalid page */
#define PGSPC_MBMEM	0x2000		/* multibus memory */
#define PGSPC_MBIO	0x3000		/* multibus I/O */

/*
 * Initial mapping of memory:
 *
 * We need an initial mapping of memory that will suffice
 * to get the monitor started and allow memory to be sized;
 * we make virtual memory map directly onto physical memory
 * for all 16 contexts.  We leave things in context 0.
 */
#define INITMAP\
{\
	register i;\
	register j;\
	for (j = 0; j < NUMCONTEXTS; j++) {\
	    SETCONTEXT(j);\
	    for (i = 0; i < CONTEXTSIZE; i++) {\
		SETSEGMAP(i, (i|SEGPRO_ALL));\
	    }\
	}\
	SETCONTEXT(0);\
	for (i = 0; i < PAGEMAPSIZE; i++) {\
		SETPAGEMAP(i,(i|PGSPC_MEM));\
	}\
}	

/*
   Boot State:

In boot state (the state of the system after reset) the address
	0XXXXX 		reads and executes from address 2XXXXX
			writes onto address 0XXXXX,
in this way it is possible to initialize RAM just after reset.
Also, all interrupts including normally non-maskable ones are
disabled.

Note that CLR instructions should not be used on RAM space in
boot state; this is because the CLR instructions do a read cycle
before writing.

To exit boot state, perform a write operation into the
PROM0 address space.

*/

#define EXITBOOT *(unsigned short *)ADRSPC_PROM0 = 1

/*
 * MULTIBUS MEMORY AND I/O MAPPING
 */

/*
 * Multibus I/O mapping
 *
 * By convention, MultiBus I/O is mapped at the top of the
 * mappable address space.
 */

#define	IOLOWPAGE	0x3E0		/* first MultiBus page */
#define	IOLOWADR	0x1F0000	/* IOLOWPAGE << 11 */
#define	NUMIOPAGES	(0x020 - 1)	/* number of MultiBus pages */

/* We don't map top page because of the following problem: the 68000
 * sets A4-A23 to all 1's during vector acquisition, thus causing a
 * spurious multibus operation.  This apparently screws up Multibus
 * DMA transactions.
 */

/*
 * MultiBus(x) returns the mapped address for MultiBus address x
 */
#define MultiBus(x)	(IOLOWADR+(x))

/*
 * initIOMAP initializes the top pages of the map to conform to
 * the convention described above
 */
#define initIOMAP  \
{\
	register i;\
	for (i=0; i<NUMIOPAGES; i++)\
		SETPAGEMAP((i+IOLOWPAGE),(i|PGSPC_MBIO));\
}

/*
 * Multibus memory mapping
 *
 * By convention, MultiBus memory is mapped just below 
 * Multibus I/O in the mappable address space.
 */

#define	MBMEMLOWPAGE	0x200	/* first MultiBus memory page */
#define	MBMEMLOWADR	0x100000	/* MBMEMLOWPAGE << 11 */
#define	NUMMBMEMPAGES	0x1e0	/* number of MultiBus memory pages */

/*
 * MultiBusMem(x) returns the mapped address for MultiBus memory address x
 */
#define MultiBusMem(x)	(MBMEMLOWADR+(x))

/*
 * initMBMEMMAP initializes the top pages of the map to conform to
 * the convention described above
 */
#define initMBMEMMAP  \
{\
	register i;\
	for (i=0; i<NUMMBMEMPAGES; i++)\
		SETPAGEMAP((i+MBMEMLOWPAGE),(i|PGSPC_MBMEM));\
}

#if NUMIOPAGES == NUMMBMPAGES
/*
 * InitMultiBusMap initializes the maps for both Multibus memory and
 * I/O; the loop-jamming of initMBMEMMAP and initIOMAP saves
 * code in the ROM monitor.  This works because NUMIOPAGES ==
 * NUMMBMEMPAGES.
 */
#define InitMultiBusMap  \
{\
	register i;\
	for (i=0; i<NUMIOPAGES; i++) {\
		SETPAGEMAP((i+MBMEMLOWPAGE),(i|PGSPC_MBMEM));\
		SETPAGEMAP((i+IOLOWPAGE),(i|PGSPC_MBIO));\
	}\
}

#else
/* do each initialization; cannot jam loops */
#define InitMultiBusMap \
	initMBMEMMAP;\
	initIOMAP;
#endif
