/*
 * map.h
 *
 * Macros and definitions for Sun-1 memory mapping
 *
 * Vaughan Pratt	12 July 1981
 *
 */

#define NUMCONTEXTS 16
#define NUMPHYSSEGS 64

/* 		     MEMORY ADDRESS STRUCTURE
*/

struct {
	unsigned domain:3;		/* 8 domains, see list below */
	unsigned segment:6;		/* segment within contex  */
	unsigned page:4;		/* page within segment  */
	unsigned byte:11;		/* byte within page  */
}	memaddress;

/* 			ADDRESS SPACE DOMAINS
*/

#define MAPPED	((char *)0x000000)	/* mapped virtual memory */
#define PROM0	((char *)0x200000)	/* ROM #0, boot space */
#define PROM1	((char *)0x400000)	/* ROM #1 */
#define	UART	((char *)0x600000)	/* onboard UART */
#define TIMER	((char *)0x800000)	/* onboard Timer */
#define PAGEMAP	((char *)0xA00000)	/* base of page table */
#define SEGMAP	((char *)0xC00000)	/* base of segment table */
#define CONTEXT	((char *)0xE00000)	/* context register */

/* 				PAGES
*/

#define pfloor(p) ((page *)((int)(p) & -sizeof(page)))
#define pceil(p) ((page *)((int)(p)+sizeof(page)-1 & -sizeof(page)))

/* Physical pages */

typedef int physpage;

/*  Page Table Entries
*/ 

typedef	union {
	    long longno[512];			/* define sizeof(page) */
	    short entry;			/* whole entry as one short */
	    struct {
		unsigned used:1;		/* "Used" bit */
		unsigned dirty:1;		/* "Dirty" bit */
		unsigned physspace:2;		/* physical page space */
		unsigned physpage:12;		/* physical page index */
	    } field;
	} page;

#define pagemapentry(used,dirty,physspace,physpage)			\
	(((((((used)<<1) + (dirty)) << 1) + (physspace)) << 12) + (physpage))

#define pmap (0xA00000/sizeof(page))
/* p[pmap].entry accesses the map entry for page *p 
 * p[pmap].field.physpage accesses the physpage field for page *s 
 */

/* page.field.used */
#define UNUSED 0
#define USED 1

/* page.field.dirty */
#define CLEAN 0
#define DIRTY 1

/* page.field.physspace */
#define ONBOARD 0
#define INVALIDPAGE 1
#define OFFBOARD 2
#define IO 3

/* 				SEGMENTS
*/

#define sfloor(s) ((segment *)((int)(s) & -sizeof(segment)))
#define sceil(s) ((segment *)((int)(s)+sizeof(segment)-1 & -sizeof(segment)))

/* Physical Segments */
/* These reside in the page map.  Only the physical segment allocator needs to
know about the representation of a physical segment. */

typedef int physseg;


/* Segment Map Entries
*/

typedef union {
	    page pageno[16];		/* define sizeof(segment) */
	    short entry;		/* whole entry as one short */
	    struct {
		unsigned contextfield:4;/* readonly copy of current context */
		unsigned access:4;	/* access code */
		unsigned physseg:8;	/* High 8 bits of Page number */
	    } field;
	} segment;

#define segmapentry(access,physseg) (((access)<<8) + (physseg))

#define smap (0xC00000/sizeof(segment))
/* s[smap].entry accesses the map entry for segment *s 
 * s[smap].field.physseg accesses the physseg field for segment *s 
 */

/* Access codes */

/* System-only */
#define ______	0	
#define __x___	1
#define r_____	2
#define r_x___	3
#define rw____	4
#define rwx___	5

/* General data */
#define r__r__	6		/* Read-only */
#define rw_r__	7		/* User read, system write */
#define r__rw_	8		/* System read, user write */
#define rw_rw_	9		/* No-holds-barred data */

/* "Unsafe" code */
#define rw_r_x	10		/* source of code not trusted */
#define rw_rwx	11		/* user not trusted */

/* "Safe" code */
#define r_xr_x	12		/* Read-only */
#define rwxr_x	13		/* System writable */

/* Proprietary code */
#define rwx__x	14

/* Unprotected */
#define rwxrwx	15


/* Suggestive names for some access codes */
#define INVALIDSEG ______
#define UNPROTECTED RWXRWX

/* Context Register */

/*
The context register holds values of type 'contextval', which act like
short integers shifted left 12 (= CXOFF).  The loop 
		cv = 0; do ... while (cv += cval(1));
will enumerate all contextvals.  sizeof(contextval) == 2.

The context register must be called 'curcontext' when its rvalue is intended
(when being fetched from), and 'newcontext' when its lvalue is intended (when
being assigned to).  

cval(i) converts integer i to a contextval (by shifting i left 12);
uncval(c) is the inverse.  It is the intent that most operations
involving the context register be performed with contextvals rather than
ints, to avoid unnecessary shifting.
*/


#define CXOFF		12		/* context register offset */

typedef segment context[NUMPHYSSEGS];	/* determines sizeof(context) */
typedef unsigned short contextval;

	/* The hardware reads the context register from the segment map: */
#define curcontext ((contextval)(*(unsigned short *)SEGMAP & 0xF000))

	/* but writes it via the context register */
#define newcontext   *(contextval *)CONTEXT

#define cval(c) ((contextval)(c<<CXOFF))	/* converts int to contextval*/
#define uncval(c) ((int)c>>CXOFF)		/* converts contextval to int*/


/* Perform operation s in remote context c.  Must be executed in space mapped
 * to the same text in current and remote contexts. 
 */
#define remotecontext(c,s) 						\
{	register contextval oldc=curcontext;				\
	newcontext = c;							\
	s;								\
	newcontext = oldc;						\
}


/* Regions
	A region is a sequence of pages from lo to hi, and an access code
*/

typedef struct {page *lo; page *hi; char access;} region;

