/* inliner.c */
/*
 * A proprietary product of Human Computing Resources Corporation
 *			    10 St. Mary Street
 *			    Toronto, Ontario
 *			    Canada  M4Y 1P9
 *			    (416) 922-1937
 * Copyright (C) 1984, HCR
 *
 * Not to be reproduced, transmitted, or disclosed in any way without written
 * permission.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: inliner.c,v 5.5 89/05/12 12:51:01 pcc Rel-3_0 $";
/* static char ID[] = "@(#)inliner.c	15.7	of 86/12/08"; */
#endif

#ifdef	INLINER
/* Inliner: substitute function bodies for calls
 *
 * Pass 1: analysis and gathering
 *
 * - outputs nothing
 * - builds table of fns, noting:
 *   + info needed to decide on inlining:
 *     * number of suitable calls
 *     * "size"
 *     * naughtiness (setjmp, asm, ...)
 *     * info to see if closed version ever needed
 *	 o external (so could be called from outside)
 *	 o address taken
 *	 o calls that will not be replaced (eg. recursion)
 * - finds largest label number used
 * - saves the local symbol table for each body (these are,
 *   in some sense, formals for the substitution process).
 * - builds the global symbol table, so no body will be copied
 *   out of the scope of a global it uses.  Example:
 *
 *	void x();
 *	void y(){ x(); }   <----+
 *	void z() { ; }		|  x will move before the decl of z
 *	void x() { z(); }  >>>--+
 * - inlining decision is only based on the structure of the callee,
 *   not on how or where it is used.
 *
 * Pass 2: perform substitutions, then optimize
 *
 * - substitution involves:
 *   + instantiating local variables of callee:
 *     * fusing activation records of callee/caller
 *     * declaring new instances
 *     * modifying nodes that refer to locals
 *     * generating code to assign actual to formal
 *   + carrying result back, if any
 */

#include <assert.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
extern int atoi();

#include <blocks.h>
#include <bool.h>
#include <cmanifest.h>
#include <dagsymbol.h>
#include <erroro.h>
#include <flow.h>
#include <identifier.h>
#include <instruct.h>
#include <opt.h>
#include <option.h>
#include <pcc.h>
#include <readero.h>
#include <storage.h>
#include <target.h>
#include <temp.h>
#include <tree.h>

#include <inliner.h>

/*
 *	Globals
 */

int inlining = False;	/* shall we inline at all? */

/*
 *	Private Data
 */

static int
	callExpansionLimit = 100,	/* maximum size of expanded call */
	bodyExpansionLimit = 1000,	/* maximum size of expanded function */
	callRoom,		/* space left for this call */
	bodyRoom;		/* space left for this body */

static Boolean ShowExcuse = False;	/* explain whenever fn not inlined */

typedef struct SI {
	struct SI *next;
	Operator op;
	CONSZ cookie;
	TWORD type;
#ifdef	FLEXNAMES
	char *name;
#else
	char name[NCHNAM];
#endif	/* FLEXNAMES */
	Identifier actual;
#ifdef	FORT
	Boolean may_not_promote;
	Boolean addressed;
#endif
} staticInfo;

/* Function Table */

typedef struct FI {
	struct FI *next;
	Operator op;
	CONSZ cookie;
	Boolean isParam;
	TWORD type;
#ifdef	FLEXNAMES
	char *name;
#else
	char name[NCHNAM];
#endif	/* FLEXNAMES */
	Identifier actual;
	TreeNode assign_actual;
} formalInfo;

typedef struct FT {
	struct FT	*next;
	char		*identifier;
	formalInfo	*formals;
	staticInfo	*statics;
	int		size;
	InstrList	body;
	int		requested;	/* -1: don't!; 1: do! */
	char		*naughty;
	Boolean		active;
} iFunc;

static iFunc *iFuncTab = NULL;		/* table of functions to inline */

/*
 * Forward Declarations
 */

static void InlinerAnalyze(/*Instruction*/);
static Boolean LocalsBad();	/* test for existence of locals */
static formalInfo *grabFormals(); /* form list of formals (parms and locals) */
static staticInfo *grabStatics(); /* form list of statnames */
static Boolean paramMatch();	/* see if number and types of formals match actuals */
static TreeNode instantiateFormals();	/* create actuals for formal list */
static void enterStatics();		/* enter STATNAMES into symbol table */
static void substituteFormals();	/* within tree t, replace all refs to formals in f */
static void subStatics();	/* within tree t, replace all refs to formals in st */
static iFunc *iFLookUp(/*id*/);		/* lookup function to be inlined */
static void doSubst(/*body, f, topLevel*/);


void
InlinerPass()
{
	Init();
	LocalInit();		/* machine dep. global init */
	ReadICode(&header, &ParamLoads, &function, &trailer, InlinerAnalyze);
}

static void
InlinerAnalyze(s)
	Instruction s;
{
	Boolean keepBody = False;	/* keep this body for inlining? */

	if (pdebug > 1) {
		printf("Header\n");
		PrintIList(header);
		printf("Parameter Loads\n");
		PrintIList(ParamLoads);
		printf("Function code\n");
		PrintIList(function);
		printf("Function trailer\n");
		PrintIList(trailer);
	}

	if (AsmStmt)
		;	/* WarnNoOpt("used asm statement", (char *) NULL); */
	else
	if (BadFcn >= 0 && BadFcnNames[BadFcn] != NULL)
		;	/* WarnNoOpt("called %s()", BadFcnNames[BadFcn]); */
	else
	if (AssignedGoto) {
		/*
		 *	For f77, an assigned goto makes the flow graph very difficult
		 *	to build, so we simply don't do it, and do no transformations
		 *	on the function.
		 *	For C, this is a constant False.
		 *
		 *	This test must be before any tests that examine the
		 *	basic blocks or the flow graph.
		 */

		;	/* WarnNoOpt("used assigned GOTO statement", (char *) NULL); */
	} else {		/* We can do it */
		register iFunc *f = iFLookUp(ftnname);
		int size = 0;
		Instruction ip = function.first;

		assert(f != NULL);	/* ftnname must be a legitimate public name */
		assert(ip==NULL || function.last->next==NULL);
		while (ip != NULL) {
			size++;
			ip = ip->next;
		}
		f->size = size;
		if (LocalsBad())
			f->naughty =
	"callee has local struct or array, or takes address of a local";
		else if (f->requested < 0)
			f->naughty = "callee forbidden";
		else if (f->requested==0 && size>callExpansionLimit)
			f->naughty = "callee too big";
		else {
			/* Seems good enough to consider inlining. */
			f->naughty = NULL;
			keepBody = True;
			f->formals = grabFormals();
			f->statics = grabStatics();
			f->body = function;
		}
	}
	FreeIList(&header);
	if (!keepBody)
		FreeIList(&function);
	FreeIList(&trailer);
	FreeIList(&ParamLoads);
	ResetProg();
}

static iFunc *
iFLookUp(id)		/* lookup function to be inlined */
	char *id;
{
	register iFunc *f;

	if (id == NULL)
		return NULL;	/* not a public name */
	for (f=iFuncTab; f!=NULL; f=f->next)
		if (*id==*f->identifier && strcmp(id, f->identifier)==0)
/**/			return f;

	/* not found -- make a new entry */
	f = GetStorage(s_NoStats, iFunc);
	assert(f!=NULL);
	f->next = iFuncTab;
	f->identifier = SaveIdText(id);
	f->formals = NULL;
	f->statics = NULL;
	f->size = -1;
	InitList(&f->body);
	f->requested = 0;
	f->naughty = "callee not in file";
	f->active = False;
	iFuncTab = f;
	return f;
}

void
InlinerRequest(f)	/* inline this function, please */
	char *f;
{
	int req = 1;

	for (;;) {
		switch (*f) {
		case '\0':
			break;

		case '!':
			inlining = True;
			break;

		case '?':
			ShowExcuse = True;
			break;

		case ',':
			break;

		case '-':
			req = -1;
			break;

		case '+':
			req = 1;
			break;

		case '|':
		case '#':
			{
				char	which = *f;
				int	n = 0;

				for (;;) {
					f++;
					if (!isdigit(*f))
						break;
					n = n*10 + (*f-'0');
				}
				if (which == '|')
					bodyExpansionLimit = n;
				else
					callExpansionLimit = n;
			}
			f--;
			break;

		case '@':
			{
				/* take the rest as a filename */
				FILE	*fp = fopen(f+1, "r");

				if (fp == NULL)
					UserError("can\'t open inclusion/exclusion file");
				else {
					for (;;) {
						char	name[3+sizeof ftnname];
						char	*s;
						iFunc	*fe;

						s = fgets(name, sizeof name, fp);
						if (s == NULL) {
							if (ferror(fp))
		UserError("I/O error reading inclusion/exclusion file");
							break;
						}
						s += strlen(s);
						if (s==name || s[-1]!='\n') {
		UserError("line too long in inclusion/exclusion file");
							break;
						}
						s[-1] = '\0';
						fe = iFLookUp(name);
						assert(fe != NULL);
						fe->requested = req;
					}
				}
			}
			f += strlen(f);
			break;

		default:
			{
				char b[sizeof ftnname];
				char *bp;
				Boolean bad = !isalpha(*f);

				bp=b;
				for (; *f!='\0' && *f!=','; f++) {
					if ((isalnum(*f) || *f=='_')
					    && bp!=&b[(sizeof b)-1])
						*bp++ = *f;
					else
						bad = True;
				}
				*bp = '\0';
				if (bad)
					UserError("invalid inliner flag");
				else {
					iFunc	*fe = iFLookUp(b);

					assert(fe != NULL);
					fe->requested = req;
				}
			}
			break;
		}
		if (*f == '\0')
			break;
		f++;
	}
}


static TreeNode
doExprSubst(apre, expr, stmt)	/* try to inline calls made by an expression */
	InstrList	*apre;	/* prefix for expression being transformed:
				 * where inlined instructions are appended.
				 */
	TreeNode	expr;	/* Address of root of expression to be
				 * transformed.
				 */
	Instruction	stmt;	/* statement containing expr
				 * (for line number and filename)
				 */
{
	iFunc *f;
	char *excuse = NULL;
    	char strbuf[BUFSIZ];
	char namework[sizeof ftnname];

	assert(expr != NULL);
#ifdef	FORT
	/* in fortran, only inline subroutines: no need to process children */
	switch (expr->in.op) {
	case UNARY CALL:
	case CALL:
	case FORTCALL:
	case UNARY FORTCALL:
		break;	/* go process this node! */
	default:
		return expr;
	}
#else
	/* process children */
	switch (optype(expr->in.op)) {
	case LTYPE:
		return expr;
	case BITYPE:
		switch (expr->in.op) {
		case ANDAND:
		case OROR:
		case COMOP:
		case QUEST:
			break;
		default:
			expr->in.right = doExprSubst(apre, expr->in.right, stmt);
		}
		/* fall through ... */
	case UTYPE:
		expr->in.left = doExprSubst(apre, expr->in.left, stmt);
		switch (expr->in.op) {
		case UNARY CALL:
		case CALL:
		case FORTCALL:
		case UNARY FORTCALL:
			break;	/* go process this node! */
		default:
			return expr;
		}
		break;
	default:
		InternalFault("doExprSubst: unknown optype");
	}
#endif 	/* FORT */
#ifndef MPX
	if (   expr->in.left->tn.op != CADDR
	    && expr->in.left->tn.op != ADDR)
#else
	if (expr->in.left->tn.op != ADDR)
#endif
		excuse = "callee name not manifest";
	else if (!(CanBeLNAME(DECREF(DECREF(expr->in.left->tn.type)))
		   || (DECREF(expr->in.left->tn.type) == TVOID
		       && expr == stmt->u.ex.root)))
		excuse = "this result type can\'t be an LNAME";
	else if ((f=iFLookUp(PublicName(
			expr->in.left->in.identifier, namework)))==NULL)
		excuse = "callee isn\'t a user\'s function";
	else if (f->naughty != NULL)
		excuse = f->naughty;
	else if (f->active)
		excuse = "recursive call";
	else if (f->size>callRoom && f->requested<=0)
		excuse = "cumulative size would exceed limit";
	else if (!paramMatch(f->formals, expr))
		excuse = "actuals don\'t match formals";
	else {
		/* copy body of fn */
		register Instruction fip;	/* formal instruction ptr */
		register Instruction iip;	/* instantiated instruction ptr */
		int firstLab = LAST_P2_LABEL;
		int lastLab = 0;
		int retLab = 0;
		int retCnt = 0;
		InstrList cpy;
		TreeNode resVar;	/* variable in which to store result */

		callRoom -= f->size;
		bodyRoom -= f->size;
		InitList(&cpy);
		iip = AddInstruction(Passed, &cpy);
		sprintf(strbuf, "%c! Entering %s", ASM_CCHAR, f->identifier);
		iip->u.tx.line = StoreText(strbuf);
		resVar = instantiateFormals(f->formals, &cpy, expr, stmt);
		enterStatics(f->statics);
		for (fip=f->body.first; fip!=NULL; fip=fip->next) {
			switch (fip->tag) {
			case FcnStart:
			case FcnEnd:
			case ParamHere:
			case ParamFetch:
			case ParamSave:
			case Block_Start:
			case Block_End:
#ifdef	FORT
			case CopyOut:
#endif	/* FORT */
				/* discard */
				break;
			case Return:
				if (retCnt++ == 0)
					retLab = ++MaxP1ILabel;
				AddInstruction(UBranch, &cpy)->u.lb.lab_num =
					retLab;
				break;
			case Expr:
				{
				TreeNode t = CopyTree(fip->u.ex.root);

				iip = CreateInstruction(fip->tag);
				*iip = *fip;
				iip->next = NULL;
				iip->u.ex.root = t;
				if (t->in.op==FORCE && fip->next->tag==Return) {
					/* turn a Force-before-Return into
					 * an assignment to the result var.
					 */
					assert(resVar!=NULL);
					t->in.op = ASSIGN;
					t->in.right = t->in.left;
					t->in.left = CopyTree(resVar);
				}
				subStatics(t, f->statics);
				substituteFormals(t, f->formals, resVar);
				AppendInstruction(iip, &cpy);
				}
				break;
			case Switch:
				{
				register unsigned i = fip->u.sw.ncases;

				iip = CreateInstruction(fip->tag);
				*iip = *fip;
				iip->next = NULL;
				iip->u.sw.swtable = (struct sw *)
					calloc(i, sizeof(struct sw));
				while (i != 0) {
					i--;
					iip->u.sw.swtable[i] =
						fip->u.sw.swtable[i];
				}
				AppendInstruction(iip, &cpy);
				}
				break;
			case Passed:
				if (strcmp(fip->u.tx.line, "T.fartext 2") == 0){
				    /* Don't copy constant pool declarations */
				    /* Drop this line and the next */
				    fip = fip->next;
				}else{
				    iip = CreateInstruction(fip->tag);
				    *iip = *fip;
				    iip->next = NULL;
				    iip->u.tx.line = StoreText(fip->u.tx.line);
				    AppendInstruction(iip, &cpy);
				}
				break;
			case Label:
				/* note range of labels */
				if (fip->u.lb.lab_num < firstLab)
					firstLab = fip->u.lb.lab_num;
				if (fip->u.lb.lab_num > lastLab)
					lastLab = fip->u.lb.lab_num;
				/* fall through ... */
			default:
				/* blindly copy */
				iip = CreateInstruction(fip->tag);
				*iip = *fip;
				iip->next = NULL;
				AppendInstruction(iip, &cpy);
				break;
			}
		}
		if (retCnt != 0) {
			iip = cpy.last;
			assert(iip != NULL);
			if (iip->tag==UBranch && iip->u.lb.lab_num==retLab) {
				/* delete useless final branch */
				DelInstruction(iip, &cpy);
				retCnt--;
			}
			if (retCnt != 0)
				AddInstruction(Label, &cpy)->u.lb.lab_num = retLab;
		}
		if (firstLab != LAST_P2_LABEL) {
			/* renumber all labels (except retlab) */
			register int labDelta = MaxP1ILabel-firstLab + 1;

#			define labAdjust(l) { \
				if ((l) != retLab) { \
					assert(firstLab<=(l) && (l)<=lastLab); \
					(l) += labDelta;  \
				} \
			}

			assert(0<firstLab && firstLab<=lastLab && lastLab<=MaxP1ILabel);
			MaxP1ILabel = lastLab + labDelta;
			assert(lastLab<LAST_P2_LABEL);
			for (iip=cpy.first; iip!=NULL; iip=iip->next) {
				switch (iip->tag) {
				case UBranch:
				case Label:
					labAdjust(iip->u.lb.lab_num);
					break;
				case Expr:
					if (IsCBranch(iip)) {
						register TreeNode t =
							iip->u.ex.root->in.right;

						assert(t!=NULL && t->in.op==STLABEL);
						assert(t->tn.identifier==NULL || *t->tn.identifier=='\0');
						labAdjust(t->tn.lval);
					}
					break;
				case Switch:
					{
					register int i;

					for (i=0; i!=iip->u.sw.ncases; i++)
						labAdjust(iip->u.sw.swtable[i].slab);
					labAdjust(iip->u.sw.defaultlabel);
					}
					break;
#ifdef	FORT
				case ArithIf:
					{
					register int i;

					for (i=0; i!=ArithIfExits; i++)
						labAdjust(iip->u.aif.labs[i]);
					}
					break;
#endif	/* FORT */
				}
			}
		}
		doSubst(&cpy, f, False);
		iip = AddInstruction(Passed, &cpy);
		sprintf(strbuf, "%c! Exiting %s", ASM_CCHAR, f->identifier);
		iip->u.tx.line = StoreText(strbuf);
		ConcatInstructions(apre, &cpy);

		/*
		 * Turn expr into a reference to resVar (this is
		 * suppressed if expr is the root of the statement's
		 * expression).
		 */

		if (expr == stmt->u.ex.root) {
			walkf(expr, TreeFree);
			expr = NULL;
		} else {
			assert(resVar!=NULL);
			resVar = CopyTree(resVar);
			resVar->tn.type = expr->in.type;
			walkf(expr, TreeFree);
			expr = FixType(resVar);
		}
	}
	if (ShowExcuse && excuse!=NULL) {
		sprintf(strbuf, "%c! Inliner excuse: %s", ASM_CCHAR, excuse);
		AddInstruction(Passed, apre)->u.tx.line = StoreText(strbuf);
	}
	return expr;
}

void
InlineSubstitute(body, name)
	InstrList *body;
	char *name;
{
	bodyRoom = bodyExpansionLimit;
	doSubst(body, iFLookUp(name), True);
}
static void
doSubst(body, f, topLevel)		/* do inline substitution */
	InstrList *body;	/* starting instruction of code */
	iFunc *f;		/* current function */
	Boolean topLevel;	/* top level fn body? */
{
	/*register*/ Instruction ip, newip, ipred;

	assert(f != NULL);
	assert(body->first==NULL? body->last==NULL : body->last!=NULL && body->last->next==NULL);
    	f->active = True;
	for (ipred=NULL, ip=body->first; ip!=NULL; ipred=ip, ip=ip->next) {
		if (ip->tag == Expr) {
			InstrList pre;	/* prefix for expr: inlined code */

			InitList(&pre);
			if (topLevel) {
				callRoom = callExpansionLimit;
				if (callRoom > bodyRoom)
					callRoom = bodyRoom;
			}
			ip->u.ex.root = doExprSubst(&pre, ip->u.ex.root, ip);
			assert(pre.first==NULL? pre.last==NULL : pre.last!=NULL && pre.last->next==NULL);
			newip = pre.first;
			if (newip != NULL) {
				/* splice new instructions before old one */
				if (ipred == NULL)
					body->first = newip;
				else
					ipred->next = newip;
				ipred = pre.last;
				assert(ipred->next == NULL);
				ipred->next = ip;
			}
			assert(ipred==NULL || ipred->next==ip);
			if (ip->u.ex.root == NULL) {
				/* Inlining has eliminated the expression:
				 * get rid of the instruction.
				 */
				DelInstruction(ip, body);
				/* So next ip will be successor of the
				 * deleted instruction, we back ip up.
				 * ipred will be instantaneously wrong.
				 */
				ip = ipred;
			}
		}
	}
	assert(ipred == body->last);
    	f->active = False;
}

static Boolean
LocalsBad()			/* test for existence of bad locals */
{
	Identifier id;

	for (id = LastRegId+1; id <= MaxIdentifier; ++id) {
		Operator op = IdOp(id);

		if (op!=NAME
		    && op!=STATNAME
		    && (!CanBeLNAME(SymType(id)) || WasAddressed(id)
		        || op==PADDR || op==LADDR))
/**/			return True;
	}
	return False;
}

static formalInfo *
grabFormals()	/* form list of formals (parms and locals) */
{
	Identifier id;
	formalInfo *f = NULL;

	for (id = LastRegId+1; id <= MaxIdentifier; ++id) {
		Operator op = IdOp(id);

		if (op != NAME && op != STATNAME) {
			formalInfo *new = GetStorage(s_NoStats, formalInfo);

			assert(new!=NULL);
			new->next = f;
			f = new;
			new->op = op;
			new->cookie = IdCookie(id);
#ifdef	FORT
			new->isParam = op==PNAME;
#else
			new->isParam = id<=LastParam;
#endif	/* FORT */
			new->type = SymType(id);
			assert(CanBeLNAME(new->type));
#ifdef	FLEXNAMES
			new->name = SaveIdText(IdText(id));
#else
			strncpy(new->name, IdText(id), NCHNAM);
#endif	/* FLEXNAMES */
		}
	}
	return f;
}

static staticInfo *
grabStatics()	/* form list of statnames */
{
	Identifier id;
	staticInfo *st = NULL;

	for (id = LastRegId+1; id <= MaxIdentifier; ++id) {
		Operator op = IdOp(id);

		if (op == STATNAME) {
			staticInfo *new = GetStorage(s_NoStats, staticInfo);

			assert(new!=NULL);
			new->next = st;
			st = new;
			new->op = op;
			new->cookie = IdCookie(id);
			new->type = SymType(id);
#ifdef	FLEXNAMES
			new->name = SaveIdText(IdText(id));
#else
			strncpy(new->name, IdText(id), NCHNAM);
#endif	/* FLEXNAMES */
#ifdef	FORT
			new->may_not_promote = MayNotPromote(id);
			new->addressed = WasAddressed(id);
#endif 	/* FORT */
		}
	}
	return st;
}


static TWORD
paramType(t)	/* find canonical type of parameter */
	TWORD t;
{
	if ((t&~BTMASK) == 0 ) switch (t) {
	case CHAR:
	case SHORT:
	case INT:
	case UCHAR:
	case USHORT:
	case UNSIGNED:
	case ENUMTY:
#if SZINT!=SZLONG
		t = INT;
		break;
#endif
		/* fall through? */
	case ULONG:
	case LONG:
		t = LONG;
		break;
	case FLOAT:
	case DOUBLE:
		t = DOUBLE;
		break;
	}
	return t;
}

static Boolean
paramMatch(f, c)	/* see if formals and actuals match */
	formalInfo *f;	/* (backward) list of formals */
	TreeNode c;	/* call tree */
{
	TreeNode al;

	if (c->in.op == UNARY CALL || c->in.op == UNARY FORTCALL)
		al = NULL;
	else {
		assert(c->in.op == CALL || c->in.op == FORTCALL);
		al = c->in.right;
	}
	for (;;) {
		if (f == NULL)
/**/			return al==NULL;	/* same number of actuals and formals? */
		else if (f->isParam) {
			TreeNode a;

			if (al == NULL)
/**/				return False;	/* more formals than actuals */
			if (al->in.op == CM) {
				a = al->in.right;
				al = al->in.left;
			} else {
				a = al;
				al = NULL;
			}
			/* Here is the matching.  It need to be more inclusive. */
			if (   f->type==STRTY || f->type==UNIONTY
			    || paramType(f->type)!=paramType(a->in.type))
/**/				return False;
		}
		f = f->next;
	}
}

static TreeNode
instantiateFormals(f, body, call, stmt)	/* create actuals for formal list */
	formalInfo *f;		/* (backward) list of formals */
	InstrList *body;	/* body of replacement (where to hang instrs) */
	TreeNode call;		/* call tree (from which to grab actuals) */
	Instruction stmt;	/* statement containing call */
{
	TreeNode al;

	/* This could be done on demand so unused ones would not be allocated. */
	assert(call!=NULL && stmt!=NULL);
	switch (call->in.op) {
	case CALL:
	case FORTCALL:
		al = call->in.right;
		break;
	case UNARY CALL:
	case UNARY FORTCALL:
		al = NULL;
		break;
	default:
		InternalFault("instantiateFormals: not a CALL");
	}
	for (; f!=NULL; f=f->next) {
		TreeNode t;

		f->actual = GetNamedTemp(LNAME, f->type, f->name);
		t = IdTree(f->actual);
		if (f->isParam) {
			Instruction i = AddInstruction(Expr, body);
			TreeNode r = TreeAllocate();
			TreeNode r_right;

			i->u.ex.lineno = stmt->u.ex.lineno;
			i->u.ex.filename = stmt->u.ex.filename;
			i->u.ex.root = r;
			r->in.op = ASSIGN;
			r->in.type = f->type;
			r->in.identifier = NULL;
			r->in.left = CopyTree(t);
			assert(al!=NULL);
			if (al->in.op == CM) {
				t = al->in.right;
				al = al->in.left;
			} else {
				t = al;
				al = NULL;
			}

			r_right = r->in.right = CopyTree(t);;
			switch (r_right->in.op) {
			case ADDR:
			case STADDR:
			case LADDR:
			case PADDR:
#ifndef MPX
			case CADDR:
#endif
				f->assign_actual = r_right;
				break;
			default:
				f->assign_actual = NULL;
				break;
			}
		}
	}
	{
		/* create variable to hold the result of the function */
		TWORD ty = call->in.left->tn.type;
		assert(ISPTR(ty) && ISFTN(DECREF(ty)));
		if (DECREF(ty) == TVOID)	/* TVOID is void()! */
			return NULL;
		else {
			TreeNode t;
			t = IdTree(GetTemp(LNAME, DECREF(DECREF(ty)), 'r'));
			return t;
		}
	}
}

static void
enterStatics(st)
	staticInfo *st;
{
	for(; st!=NULL; st=st->next) {
		Identifier id;
		id = st->actual = EnterSymbol(STATNAME,st->cookie,st->type,st->name,False);
#ifdef FORT		
		SetNotPromotable(id, st->may_not_promote);
		if(st->addressed)
			SetAddressed(id);
#else
		SetNotPromotable(id, True);
		SetAddressed(id);
#endif /* FORT */
	}
}


static void
substituteFormals(t, f, resVar)	/* within tree t, replace all refs to formals in f */
	TreeNode t;
	formalInfo *f;
	TreeNode resVar;	/* result variable of inlining */
{
	formalInfo *p;
	CONSZ cookie;

	for (;;) {
		assert(t!=NULL);
		switch (optype(t->in.op)) {
		case LTYPE:
			switch (t->in.op) {
			case LNAME:
				if (resVar != NULL &&
				    t->tn.rval == resVar->tn.rval)
					break;
				/* fall through ... */
			case REG:
			case PNAME:
				cookie = t->tn.op==REG? t->tn.lval : t->tn.rval;
				p = f;
				for (;;) {
					assert(p!=NULL);
					if (p->op==t->tn.op && p->cookie==cookie)
						break;
					p = p->next;
				}
				if (p->isParam
				    && p->assign_actual != NULL)
					*t = *p->assign_actual;
				else
					*t = *IdTree(p->actual);	/* overwrite! */
				break;
			case LADDR:
			case PADDR:
				InternalFault("LADDR or PADDR as inliner formal");
			}
			break;	/* and out of loop... */
		case BITYPE:
			substituteFormals(t->in.right, f, resVar);
			/* fall through ... */
		case UTYPE:
			t = t->in.left;/* tail recursion */
	/**/		continue;	/* so around again... */
		default:
			InternalFault("impossible optype: %d", optype(t->in.op));
		}
		break;
	}
}

static void
subStatics(t, st)	/* within tree t, replace all refs to formals in st */
	TreeNode t;
	staticInfo *st;
{
	staticInfo *p;
	CONSZ cookie;

	for (;;) {
		assert(t!=NULL);
		switch (optype(t->in.op)) {
		case LTYPE:
			switch (t->in.op) {
			case STATNAME:
			case STADDR:
				cookie = t->tn.rval;
				p = st;
				for (;;) {
					assert(p!=NULL && p->op == STATNAME);
					if (p->cookie==cookie)
						break;
					p = p->next;
				}
				t->tn.identifier = IdText(p->actual);	/* overwrite! */
				break;
			}
			break;	/* and out of loop... */
		case BITYPE:
			subStatics(t->in.right, st);
			/* fall through ... */
		case UTYPE:
			t = t->in.left;/* tail recursion */
	/**/		continue;	/* so around again... */
		default:
			InternalFault("impossible optype: %d", optype(t->in.op));
		}
		break;
	}
}

/*
 * For irreducible function, most PCO processing is suppressed.  But
 * inlining is still done.  This routine is called to ensure that any
 * inliner-generated temps (which will be LNAMES) are allocated.
 */

static void
IrrNodeAllocate(node)
	NODE *node;
{
	if (node->in.op==LNAME && node->tn.lval==NOOFFSET) {
		Identifier id = IdLookUp(node->in.op, (CONSZ) node->tn.rval, ANYTYPE);
		TreeNode sn = IdTree(id);

		if (sn==NULL || !HasAllocation(id)) {
			Boolean success;

			success = AllocResource(id, StackResource);
			assert(success);
			PccAdvanceAllocation();
			sn = IdTree(id);
		}
		assert(sn!=NULL && sn->in.op==LNAME && sn->tn.lval!=NOOFFSET);
		*node = *sn;
	}
	return;
}

void
IrrAllocate()	/* for irreducible fn, allocate any inliner temps */
{
	BasicBlock b;
	Instruction ip;

	for (b = FirstBlock; b != NULL; b = b->next)
		if (b->reachable)
			for (ip = b->code.first; ip!=NULL; ip=ip->next)
				if(ip->tag == Expr)
					(void) fwalk(ip->u.ex.root, IrrNodeAllocate, 0/*dummy*/);
}

#endif	/* INLINER */
