/* findloop.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */


#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: findloop.c,v 5.5 89/05/12 12:50:34 pcc Rel-3_0 $";
/* static char ID[] = "@(#)findloop.c	15.4	of 86/10/25"; */
#endif

/*
 *	Loop Finding
 *
 * See: Aho and Ullman, Algorithm 13.1
 *
 * NOTE: This routine produces loops in nested order - from the inside out.
 * The loop invariant code and storage allocation algorithms make use of this.
 *
 * NOTE 2: The present version of the loop invariant algorithm uses the fact
 * that the loops will be produced in the same order when FindLoops() is
 * called after adding a pre-header.  The world will end if this assumption is
 * wrong.  Once we have an expandable flow graph, this problem will go away.
 */

/*
 * Import
 */

#include <loop.h>
#include <assert.h>
#include <option.h>
#include <flow.h>
#include <longset.h>
#include <storage.h>
#include <target.h>
#include <erroro.h>

/*
 * Export
 */

int nloops = 0;			/* number of loops in program */
LTabEntry *LoopPtrs = NULL;	/* an array of pointers into LoopTable */

/*
 * Private
 */


static int LoopSize;		/* current size of LoopTable */
LTabEntry LoopTable = NULL;	/* a table of loop information */
LTabEntry LoopRoot;		/* Root of loop tree */
static int natural_loops = 0;	/* number of natural loops a la Aho and
				 * Ullman */

static void	InsLoopNode();
static void	GenLoop();
static void	LoopInitialize();
static void	LoopTreeWalk();
static void	MergeLoops();
static void	FindWeights();
static void	FindExits();

void
FindLoops(SameLoops)
	Boolean SameLoops;	/* Will we be finding the same loops as before? */
{
	FlowIndex f;
	int n, i;

	LoopInitialize(SameLoops);

	LoopRoot = NULL;
	natural_loops = 0;

	for (n=0; n < NumReachableNodes; ++n) {
	    f = DFN[n];
	    for (i=0; i < NumExits(f); ++i) {
		if (Successor(f, i) != NullFlow) {
		    /* f --> exits[i] is an edge; tail==f, head == exits[i] */
		    if (BackEdge(f, Successor(f, i))) {
			GenLoop(f, Successor(f, i));
		    }
		}
	    }
	}

	if( ldebug )
	{
		if( ldebug > 1 )
			DumpLoopTree();
		printf("Loops in nested order:\n");
	}

	nloops = 0;
	LoopTreeWalk(LoopRoot);
	FindExits();
	FindWeights();
}

static void
LoopInitialize(SameLoops)
	Boolean SameLoops;	/* Will we be finding the same loops as before? */
{
	int i, max_init;

	if( LoopPtrs != NULL )
	{
		for( i = 0; i < natural_loops; i++ )
		{
			DestroySet(LoopTable[i].loop);
			if( LoopTable[i].exits != NULL )
				DestroySet(LoopTable[i].exits);

			if( !SameLoops )
				FreeLTempList(LoopTable[i].ltemps);
		}

		if( !SameLoops && LoopSize < NumFlowNodes )
		{
			free(LoopTable);
			DecreaseSpace(s_LoopTable, LoopSize * sizeof(LTE_type));
			free(LoopPtrs);
			DecreaseSpace(s_LoopTable, LoopSize * sizeof(LTabEntry));
			LoopSize = 0;
			LoopPtrs = NULL;
		}
	}

	if( LoopPtrs == NULL )
	{
		LoopSize = NumFlowNodes;
		LoopTable = GetArray(s_LoopTable, LoopSize, LTE_type);
		CheckStorage(LoopTable, "storage for loop table (%d)", LoopSize);
		LoopPtrs = GetArray(s_LoopTable, LoopSize, LTabEntry);
		CheckStorage(LoopPtrs, "storage for loop pointers (%d)", LoopSize);
	}

	max_init = SameLoops ? natural_loops : LoopSize;
	for( i = 0; i < max_init; ++i )
	{
		if( !SameLoops )
		{
			LoopTable[i].ltemps  = NULL;
			LoopTable[i].prehead = NULL;
			LoopTable[i].loop    = NULL;
			LoopTable[i].exits   = NULL;
		}
		LoopTable[i].child   = NULL;
		LoopTable[i].sibling = NULL;
	}
}

static void
GenLoop(a, b)				/* Compute loop including a -> b */
	FlowIndex a, b;
{
	LongSet Loop;

	if (ldebug)
		printf("Find loop for %d --> %d ", a, b);
	Loop = CreateSet(NumFlowNodes);
	Insert( (int) b, Loop);
	InsLoopNode(a, Loop);
	if (ldebug) {
		DumpSet(Loop);
		putchar('\n');
	}

	LoopTable[natural_loops].loop = Loop;
	LoopTable[natural_loops].head = b;
	AddLoopToTree(&LoopTable[natural_loops], &LoopRoot);
	natural_loops++;
}

static void
InsLoopNode(n, L)			/* Insert n into loop L */
	FlowIndex n;
	LongSet L;
{
	PredNode p;

	if (!IsElement( (int) n, L)) {
		Insert( (int) n, L);
		/* Put each predecessor in the loop */
		for (p=FlowGraph[n].preds; p != NULL; p=p->next)
		    if( FlowGraph[p->p].block->reachable )
			InsLoopNode(p->p, L);
	}
}

/*
 * Add loop "newloop" to tree whose root is *adroot
 */
AddLoopToTree(newloop, adroot)
	LTabEntry	newloop;
	LTabEntry	*adroot;
{
	LTabEntry	root = *adroot;
	LTabEntry	p, pnext;

	if( root == NULL )
	{	/* Adding to an empty tree - just put newloop place */

		*adroot = newloop;
	}
	else
	if( Disjoint(newloop->loop, root->loop ) )
	{	/* root and newloop have nothing in common.  Make newloop
		 * a sibling
		 */
		AddLoopToTree(newloop, &root->sibling);
	}
	else
	if( SetEq(newloop->loop, root->loop ) )
	{
		/* New loop and root are equal.  Treat as if
		 * neither was a subset of the other.
		 * Merge them, and re-insert the resulting loop
		 * into the tree.
		 */
		MergeLoops(root, newloop);
		*adroot = root->sibling;
		root->sibling = NULL;
		AddLoopToTree(root, adroot);
	}
	else
	if( Subset(newloop->loop, root->loop ) )
	{	/* New loop is a subset of the loop at the root of the
		 * current tree.  Make the new loop a child.
		 */

		AddLoopToTree(newloop, &root->child);
	}
	else
	if( Subset(root->loop, newloop->loop) )
	{	/* The loop at the root of the current tree is a subset of
		 * the new loop.  Replace the root by the new loop.
		 * Then add the old root and each of its siblings to the
		 * new tree.
		 */

		*adroot = newloop;
		for( p = root; p != NULL; p = pnext )
		{
			pnext = p->sibling;
			p->sibling = NULL;
			AddLoopToTree(p, adroot);
		}
	}
	else
	{
		/* New loop and root are not disjoint, and neither is
		 * a subset of the other.  Merge them, and re-insert
		 * the resulting loop into the tree.
		 */
		MergeLoops(root, newloop);
		*adroot = root->sibling;
		root->sibling = NULL;
		AddLoopToTree(root, adroot);
	}
}

/*
 * Merge loops "root" and "newloop" 
 */
static void
MergeLoops(root, newloop)
	LTabEntry root;
	LTabEntry newloop;
{
	LTabEntry p, pnext;

	assert(root->head == newloop->head);
	Union(root->loop, root->loop, newloop->loop);
	for( p = newloop->child; p != NULL; p = pnext )
	{
		pnext = p->sibling;
		p->sibling = NULL;
		AddLoopToTree(p, &root->child);
	}
}

/* Run through loop tree and produce an array of loop pointers in order from
 * the inside out.
 */

static void
LoopTreeWalk(root)
	LTabEntry root;
{
	for( ; root != NULL; root = root->sibling )
	{
		LoopTreeWalk(root->child);
		LoopPtrs[nloops++] = root;
		if( ldebug )
			DumpLTabEntry(root);
	}
}

/*
 *	Determine correct nesting depth and weight for each block in each loop.
 */

static void
FindWeights()
{
	int n;
	int i;
	LongSet loopset;

	for( i = 0; i < NumReachableNodes; i++ )
	    FlowGraph[DFN[i]].nesting = 0;	/* Initialize nesting depth */

	/* now compute nesting depth for each block */
	for( n = 0; n < nloops; n++ )
	{
		loopset = LoopPtrs[n]->loop;
		for(i = FirstElement(loopset); i != NoElement; i = NextElement(i,loopset) )
			FlowGraph[i].nesting++;
	}

	for( i = 0; i < NumReachableNodes; i++ )
	{
		FlowGraph[DFN[i]].loopweight = LoopWeight(FlowGraph[DFN[i]].nesting);
	}
}

/*
 *	Find all blocks through which control might leave each loop
 */

static void
FindExits()
{
	int n, i, j;
	FlowIndex f;
	LTabEntry l;

	for( n = 0; n < nloops; n++ )
	{
		l = LoopPtrs[n];
		if( l->exits == NULL )
			l->exits = CreateSet(NumFlowNodes);
		NullSet(l->exits);

		for( i = FirstElement(l->loop); i != NoElement;
		     i = NextElement(i, l->loop))
		{
			f = (FlowIndex) i;
			for( j = 0; j < NumExits(f); j++ )
			{
				if(!IsElement( (int)Successor(f,j), l->loop))
				{
					Insert(i, l->exits);
			/**/		break;
				}
			}
		}
	}
}

void
DumpLTabEntry(l)
	LTabEntry l;
{
	printf("Loop %d: ", l - LoopTable);
	printf(" Header: %d PreHeader:", l->head);
	if( l->prehead != NULL )
		printf(" %d ", l->prehead->FGindex);
	else
		printf(" *** ");
	if( l->loop != NULL )
		DumpSet(l->loop);
	if( l->exits != NULL )
	{
		printf("\n	Exits:");
		DumpSet(l->exits);
	}
	printf("\n");

	if( ldebug > 1 )
	{
		printf("First sibling:");
		if( l->sibling == NULL )
			printf(" ****");
		else
			printf(" %d", l->sibling - LoopTable);

		printf(" First child:");
		if( l->child == NULL )
			printf("****\n");
		else
			printf(" %d\n", l->child - LoopTable);
	}
}

void
DumpLoopTree()
{
	int n;

	printf("Original Natural Loops\n");
	printf("Tree root: %d\n", LoopRoot - LoopTable);
	for( n = 0; n < natural_loops; n++ )
		DumpLTabEntry(&LoopTable[n]);
}
