/* constant.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */
/*
 *	Constant Propagation
 */

/*
 *	See: Aho and Ullman, Principles of Compiler Design, Section 14.1
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: constant.c,v 5.5 89/05/12 12:49:45 pcc Rel-3_0 $";
/* static char ID[] = "@(#)constant.c	15.2	of 86/09/25"; */
#endif

/*	Import
 */

# include <constant.h>
# include <udchain.h>
# include <assert.h>
# include <blocks.h>
# include <dag.h>
# include <dagsymbol.h>
# include <longset.h>
# include <identifier.h>
# include <ops.h>
# include <option.h>
# include <prune.h>
# include <erroro.h>
# include <arith.h>


/*	Export
 */

int cfdebug;
int NoConstantProp = 1;	/* turn off constant propagation */

/*	Private
 */

#define IsConversion(d)	((d)->op == OCONVLEAF || (d)->op == OCONVTREE || \
			 (d)->op == SCONV )
#define IsConst(d)	((d)->op == ICON || \
			 (d)->op == FCON || \
			 (IsConversion(d) && ((d)->u.in.left->op == ICON || \
			 		      (d)->u.in.left->op == FCON)))

#define IsAddrOrConst(d)(Literal((d)->op) || \
			 (IsConversion(d) && Literal((d)->u.in.left->op)))

static Boolean	CWalk();
static TreeNode	TreeForConstant();
static void CondConst();
static void FoldDAG();

void
ConstantProp()
{
	BasicBlock b;
	Boolean changed;
	DAG_Node d;

	if( NoConstantProp )
		return;

	/* To save space at the expense of time, define SAVE_SPACE
	 */
#ifndef SAVE_SPACE
	/* Walk the DAG for each block and attach IN sets where needed
	 */
	for( b = FirstBlock; b != NULL; b = b->next )
	{
		if( b->reachable )
			UDSecondWalk(b,True);
	}
#endif

	do {
		changed = False;

		for( b = FirstBlock; b != NULL; b = b->next )
		    if( b->reachable )
		    {
#ifdef SAVE_SPACE
			UDSecondWalk(b,True);
#endif

			changed |= CWalk(b);	/* really mean ||= here but we
						 * want to force evaluation */

#ifdef SAVE_SPACE
			for( d = b->Dag; d != NULL; d = d->next)
			if( d->In != NULL )
			{
				DestroySet(d->In);
				d->In = NULL;
			}
#endif
		    }
	} while( changed );

#ifndef SAVE_SPACE
	for( b = FirstBlock; b != NULL; b = b->next )
	{
		if( b->reachable )
		{
			for( d = b->Dag; d != NULL; d = d->next)
			if( d->In != NULL )
			{
				DestroySet(d->In);
				d->In = NULL;
			}
		}
	}
#endif
}

static Boolean
CWalk(b)
	BasicBlock b;
{
	DAG_Node d, dconst;
	DAG_Node left_dag, right_dag;
	LongSet tmpset, sdefs;
	AttachedID aid;
	int i;
	TreeNode left_tree, right_tree, t;
	Boolean changed;

	tmpset = CreateSet(UDSetSize);
	changed = False;

	for( d = b->Dag; d != NULL; d = d->next )
		switch( optype(d->op) )
		{
		case LTYPE:
			/* Fold this leaf only if
			 * 1) is is a NAME, LNAME, PNAME or STATNAME
			 * 2) there is exactly one def of this leaf reaching
			 *    this node, and that def is not in a conditional
			 * 3) the def is a literal
			 *
			 * Could also try to fold REGs here, but it usually
			 * does more harm than good.  Should only fold if we
			 * know that a parent can be folded also
			 * if( d->op != REG && d->op != PNAME && .....
			 */
			if( d->op != PNAME && d->op != LNAME && d->op != STATNAME &&
			    (!GoodGlobals || d->op != NAME) )
		/**/		break;
			sdefs = GetDefs(d->leaf_id);
			assert(sdefs != NULL && d->In != NULL );
			Intersection(tmpset, sdefs, d->In);
			if( Cardinality(tmpset) != 1 )
		/**/		break;
			i = FirstElement(tmpset);
			assert(i >= 0);
			dconst = UDDagPtrs[i];
			if( !IsAddrOrConst(dconst) || dconst->in_cond != NotConditional )
		/**/		break;

			if( cfdebug > 0 )
				fprintf(stderr,"constant node %s(%x) replaced by %s(%x)\n",
					opst[d->op], d,
					opst[dconst->op], dconst);

			/* Transform this node into the appropriate constant.
			 * First, if it has a delayed store, undo it.  Setting
			 * aid->leaf_ref as we do is a lie, but is better than
			 * setting it to NULL.  (At the moment, only delay.c
			 * uses leaf_ref, and it has been called before we
			 * are).
			 */

			if( d->delayed != NULL )
			{
				aid = AttachID(d->leaf_id, d->delayed);
				aid->leaf_ref = d;
				aid->UDIndex = d->UDIndex;
				--(d->delayed->delay_count);
				d->delayed = NULL;
				d->special_delay = False;
			}

			t = TreeForConstant(dconst);
			assert(t != NULL && d->type == dconst->type);
			FoldDAG(d, t);
			TreeFree(t);
			changed = True;
			break;

		case UTYPE:
			/*
			 * Only fold ICONs.  We cannot fold addresses in DAG
			 * nodes.
			 *
			 * In addition, we do not fold conversions.  This is
			 * not quite right - we should fold conversions that
			 * will cause code to be generated - but it will do
			 * as a first approximation.
			 */

			if( IsConversion(d) )
		/**/		break;

			left_dag = d->u.in.left;
			if( !IsConst(left_dag) )
		/**/		break;

			/* Build a tree representing the left ICON */

			left_tree = TreeForConstant(left_dag);
			if( left_tree == NULL )
		/**/		break;
			if( conval(left_tree, d->op, d->type, NIL) )
			{
				if( cfdebug > 0 )
					fprintf(stderr, "Op %s(%x) with constant operand %s(%x) folded\n",
						opst[d->op], d,
						opst[left_dag->op], left_dag);

				left_dag->in_degree--;
				FoldDAG(d, left_tree);
				changed = True;
			}
			TreeFree(left_tree);
			break;
		case BITYPE:
			/*
			 * Only fold ICONs.  We cannot fold addresses in DAG
			 * nodes.
			 */

			left_dag = d->u.in.left;
			right_dag = d->u.in.right;
			if( !IsConst(left_dag) )
		/**/		break;
			if( !IsConst(right_dag) &&
			    d->op != ANDAND && d->op != OROR )
		/**/		break;

			/* Build trees for the left and right ICONs */

			left_tree = TreeForConstant(left_dag);
			if( left_tree == NULL )
		/**/		break;
			if( !IsConst(right_dag) )
			{
				if( (d->op == ANDAND && left_tree->tn.lval) ||
				    (d->op == OROR && !left_tree->tn.lval) )
				{
					/* Right hand side will be evaluated.
					 * No point in folding anything.
					 */
					TreeFree(left_tree);
		/**/			break;
				}
				else
				{
					/* rhs not evaluated.  Fake an ICON
					 * for conval
					 */
					right_tree = TreeForConstant(left_dag);
				}
			}
			else
				right_tree = TreeForConstant(right_dag);

			if( right_tree == NULL )
			{
				TreeFree(left_tree);
		/**/		break;
			}
	

			if( conval(left_tree, d->op, d->type, right_tree) )
			{
				if( cfdebug )
					fprintf(stderr, "Op %s(%x) with constant operands %s(%x) %s(%x) folded\n",
						opst[d->op], d,
						opst[left_dag->op], left_dag,
						opst[right_dag->op], right_dag);
				left_dag->in_degree--;
				right_dag->in_degree--;
				if( d->op == ANDAND || d->op == OROR )
					CondConst(d);
	
				FoldDAG(d, left_tree);
				changed = True;
			}
			TreeFree(right_tree);
			TreeFree(left_tree);
			break;
		}

	DestroySet(tmpset);
	return changed;
}

/*
 * d is a conditional node (&& or ||) that we have just constant folded.
 * (the indegrees of its children have been decremented, but the
 * pointers are still available.)  Clean up the left hand side.
 */
static void
CondConst(d)
	DAG_Node d;
{
	TreeNode left_tree;
	DagCount right_cond;
	DAG_Node dp;
	AttachedID aid;

	left_tree = TreeForConstant(d->u.in.left);
	right_cond = d->u.in.right->in_cond;

	if( ( left_tree->tn.lval && d->op == OROR) ||
	    (!left_tree->tn.lval && d->op == ANDAND) )
	{
		/* RHS does not happen.  Make it vanish.
		 */
		for(dp = d->u.in.left; dp != d; dp = dp->next )
		    if( dp->in_cond >= right_cond )
		    {
			assert(dp->delay_count == 0);

			while( dp->attached != NULL )
				DeleteID(dp->attached->id, dp, True);

			if( dp->delayed != NULL )
			{
				aid = AttachID(dp->leaf_id, dp->delayed);
				aid->leaf_ref = dp;
				aid->UDIndex = dp->UDIndex;
				--(dp->delayed->delay_count);
				dp->delayed = NULL;
				dp->special_delay = False;
			}

			if( dp->in_degree == 0 )
				PruneTree(dp, right_cond, RecursivePrune);
		    }
	}
	else
	if( ( left_tree->tn.lval && d->op == ANDAND) ||
	    (!left_tree->tn.lval && d->op == OROR) )
	{
		/* RHS happens.  Make all nodes happen with the same
		 * "conditionalness" as d
		 */
		for(dp = d->u.in.left; dp != d; dp = dp->next )
		    if( dp->in_cond == right_cond )
		    {
			dp->in_cond = d->in_cond;

			if( cfdebug > 1 )
				fprintf(stderr, "Op %s(%x) given condit level %d\n",
					opst[dp->op], dp, dp->in_cond);
		    }
	}
	else
		InternalFault("Conditional neither true nor false: node #%d",
				d->order);

	TreeFree(left_tree);
}

static TreeNode
TreeForConstant(d)
	DAG_Node d;
{
	TreeNode t;

	t = TreeAllocate();
	if( optype(d->op) != LTYPE )
	{
		assert(IsConversion(d));
		t->in.op = d->op;
		t->in.type = d->type;
		t->tn.rval = d->u.tn.rval;
		t->in.left = TreeForConstant(d->u.in.left);
		if( t->in.left == NULL )
		{
			TreeFree(t);
			return NULL;
		}
		t = FinalOpt(t);
		if( optype(t->in.op) != LTYPE )
		{
			walkf(t, TreeFree);
			return NULL;
		}
	}
	else
	{
		assert(Literal(d->op));
		if( d->op == FCON ) {
			t->in.op   = d->op;
			t->in.type = d->type;
			t->fpn.dval = d->u.fpn.dval;
		} else {
			*t = *(IdTree(d->leaf_id));
			t->tn.type = d->type;
		}
	}
	return t;
}

/*
 *	Change DAG node d into a constant.  Get the value of the constant from
 *	tree node t.
 */

static void
FoldDAG(d, t)
	DAG_Node d;
	TreeNode t;
{
	d->op = t->tn.op;
	d->type = t->tn.type;
	d->carrier_required = False;
	if (d->op == FCON)
	{	/* Special case */
		d->u.fpn.dval = t->fpn.dval;
	}
	else
	{
		switch(d->op)
		{
		case ICON:
		case LABCON:
			d->leaf_id = EnterSymbol(d->op, t->tn.lval,
					t->tn.type, (char *)NULL, True);
			break;

		case ADDR:
		case LADDR:
		case PADDR:
		case STADDR:
			d->leaf_id = EnterSymbol(d->op, (CONSZ) t->tn.rval,
					t->tn.type, (char *)NULL, True);
			break;

		default:
			InternalFault("FoldDAG called for op %d", d->op);
		}
		assert(d->leaf_id != NoId);
		SetNodeID(d->leaf_id,d);
		if( IdTree(d->leaf_id) == NULL )
			SetIdTree(d->leaf_id,t);
		d->u.tn.lval = 0;
		d->u.tn.rval = 0;
	}
}
