/* benefit.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 *	Compute the benefit of various allocation options.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: benefit.c,v 5.5 89/05/12 12:49:21 pcc Rel-3_0 $";
/* static char ID[] = "@(#)benefit.c	15.2	of 86/10/20"; */
#endif

/*
 *	Import
 */

#include <assert.h>
#include <blocks.h>
#include <cost.h>
#include <dag.h>
#include <dagsymbol.h>
#include <flow.h>
#include <identifier.h>
#include <target.h>
#ifdef FILLSPILL
#include <option.h>
#endif /* FILLSPILL */

/*
 *	Private
 */

#define max(a, b)	((a) > (b) ? (a) : (b))
#define min(a, b)	((a) < (b) ? (a) : (b))

/*
 *	This computes the benefit gained (in COST units) from allocating an
 *	object (variable or temporary) to a resource.  Each DAG is scanned
 *	and the benefit for each mentioned identifier is updated.
 */

/*
 *	Is it really a reference?
 */

#define IsRef(op)	(optype(op) == LTYPE && (op) != LEAFNOP)

/*
 *	This computes the benefit gained by one reference to id
 *	if it is in r.  Eval_cost is the cost of re-evaluating
 *	the node instead of referring to the id.
 */

static int
RefBenefit(id, eval_cost, r)
	Identifier id;
	CostType eval_cost;
	ResrcIndex r;
{
	CostType this_cost;		/* cost in this resource */
	CostType next_cost;		/* cost in next avail resource */
	TWORD type;
	ResrcIndex next_r;

	/*
	 *	Compute the benefit of a reference to the id in this
	 *	resource.  The cost of this reference is compared to
	 *	the cost of re-evaluating the node and the cost
	 *	of referencing it in the next resource for which
	 *	it is eligible.
	 */

	assert(id != NoId);
	type = SymType(id);
	this_cost = ResRefCost(r, type);

	/*
	 *	Find next eligible resource
	 */

	next_r = r + 1;
	while (next_r < NumResources && !CanAllocate(id, next_r))
		++next_r;
	/*
	 *	next_r == NumResources => no place left to allocate.
	 *				  choices are this resource or
	 *				  re-evaluate.
	 *		OR
	 *	next_r < NumResouces &&
	 *	CanAllocate(id, next_r) => choices are this r, next_r,
	 *				  or re-evaluate.
	 */

	if (next_r == NumResources)
/**/		return eval_cost - this_cost;
	else {
		/*
		 *	The benefit is the advantage of this resource
		 *	over the cheaper of the other two alternatives.
		 */

		next_cost = ResRefCost(next_r, type);
/**/		return min(next_cost, eval_cost) - this_cost;
	}
}

/*
 *	Handle a reference to n.  The weight represents
 *	the importance of this reference.
 */

static void
UpdateRefBenefit(n, weight, r)
	DAG_Node n;
	int weight;
	ResrcIndex r;
{
	Identifier id;

	/*
	 * A special case: if the node is an OCONVLEAF without
	 * a carrier, then the benefit accrues to the child.
	 */

	if (n->op == OCONVLEAF && n->carrier == NoId) /* skip to child */
		n = n->u.in.left;

	id = n->carrier;
	if (id == NoId && optype(n->op) == LTYPE && IsRef(n->op))
		id = n->leaf_id;

	if (id != NoId && CanAllocate(id, r))
		ChangeBenefit(id, weight * RefBenefit(id, n->cost, r));
}

static void
DagBenefit(d, weight, r)	/* Compute benefits of DAG d at weight */
	DAG_Node d;
	int weight;
	ResrcIndex r;
{
	DAG_Node n;
	Identifier id, sid;
	int b;		/* benefit for this node */
	AttachedID aid;
	TWORD t;

	/*
	 *  For each PNAME, LNAME or TNAME in the DAG, we compute the savings
	 *  gained by putting it in the current resource.  This is the sum over
	 *  all references of the reference benefit + the sum over all stores
	 *  of the store benefit.
	 *  A leaf node counts as one reference to its leaf-id, if there are
	 *  attached identifiers (to account for the load).
	 *  Each attached identifier contributes one store to the benefit.
	 *  Each reference to a node with a carrier contributes to the
	 *  benefit for that carrier.
	 *  For FORTRAN, STATNAMES are also eligible
	 */

	for (n=d; n != NULL; n=n->next) {

		/*
		 * Estimate references to carrier.  If no carrier, and
		 * node is a real leaf reference, use the leaf id.
		 */


		UpdateRefBenefit(n, (int) (n->refs * weight), r);

		/* Evaluate the node */

		if (n->attached != NULL && optype(n->op) == LTYPE
		    && n->leaf_id != NoId)
		{
			id = n->leaf_id;
			if (CanAllocate(id, r) && IsRef(n->op)) {
				t = SymType(id);
				b = IdRefCost(id) - ResRefCost(r ,t);
				ChangeBenefit(id, b*weight);
			}
		}

		/*  Account for stores to attached ids */

		for (aid=n->attached; aid != NULL; aid=aid->next) {
			if (CanAllocate(aid->id, r)) {
				id = aid->id;
				t = SymType(id);
				b=IdStoreCost(id)-ResStoreCost(r,t);
				ChangeBenefit(id, b*weight);
			}
		}

		/* Handle possible delayed store */

		if (n->delayed != NULL) {

			/* We have to fetch n->delayed, and store to n
			 * We know n is a leaf
			 */

			id = n->leaf_id;
			if (id != NoId && CanAllocate(id, r)) {
				b=IdStoreCost(id)-ResStoreCost(r,t);
				ChangeBenefit(id, b*weight);
			}

			sid = n->delayed->carrier;
			if (sid != NoId && CanAllocate(sid, r)) {
				b = RefBenefit(sid, n->delayed->cost, r);
				ChangeBenefit(sid, b*weight);
			}
		}

# ifdef FILLSPILL
		if( !NoFillSpill && callop(n->op) )
			HandleFortCalls( n, weight );
# endif /* FILLSPILL */

	}
}

void
CompBenefits(r)		/* Compute benefits for this function */
	ResrcIndex r;
{
	Identifier id;
	FlowIndex f;
	BasicBlock b;
	int ben;

	/*
	 *	Initially, the benefit depends on the resource.
	 *	For most registers, it is slightly < 0 to reflect
	 *	additional overhead in saving the register.
	 *	For parameters, the parameter must be moved from
	 *	where it is passed to the resource r.
	 *
	 */

	ben = InitialBenefit(r);
	for (id = FirstId; id <= MaxIdentifier; ++id) {
		if (IdOp(id) == PNAME)
			SetBenefit(id, (ben - (int) CostParamLoad(id, r)));
		else
			SetBenefit(id, ben);
	}

	ComputeCosts();

	/* Compute the benefit for each node in the flow graph */

	for (f=0; f < NumFlowNodes; ++f) {
		b = FlowGraph[f].block;
		if (b->reachable)
		{
			DagBenefit(b->Dag, FlowGraph[f].loopweight, r);
		}
	}
}

