/* allocate.c */
/*
 * HCR Confidential
 *
 * These computer programs are the confidential, proprietary property
 * of HCR (Human Computing Resources Corporation, 10 St. Mary Street,
 * Toronto, Ontario, Canada), and may not be disclosed except with the
 * prior written agreement of HCR.
 *
 * Copyright (c) 1984, 1985, 1986 Human Computing Resources Corporation
 * All Rights Reserved
 */

/*
 *	Variable and temporary allocation
 *
 *	This handles the allocation of variables to registers, and
 *	temporaries to registers or stack storage.  This is discussed
 *	in more detail in the documentation.
 */

#ifndef lint
static char *rcsid = "@(#) (Gould) $Header: allocate.c,v 5.5 89/05/12 12:49:02 pcc Rel-3_0 $";
/* static char ID[] = "@(#)allocate.c	15.8	of 86/11/29"; */
#endif

/*
 *	Import
 */

# include <activity.h>
# include <allocate.h>
# include <allocvect.h>
# include <assert.h>
# include <bool.h>
# include <benefit.h>
# include <config.h>
# include <dag.h>
# include <storage.h>
# include <instruct.h>
# include <temp.h>
# include <identifier.h>
# include <dagsymbol.h>
# include <flow.h>
# include <cost.h>
# include <tree.h>
# include <loop.h>
# include <longset.h>
# include <pcc.h>
# include <target.h>
# include <prune.h>
# include <refcount.h>
# include <erroro.h>

int AllocDebug = 0;
int NoShareDebug = 0;

/*
 *	Private variables
 */

static int NumForcedTemps = 0;		/* temps allocated that have
					 * no benefit!
					 */

/*
 *	This global is used to pass the current resource from
 *	BlockVector to the routines called by activity count
 *	processing. 
 */

static ResrcIndex activity_r;

/*
 * The set of user variables yet to be localized to a lump
 * (i.e., to a basic block, loop, or the whole function)
 */

LongSet UnresolvedVars;

/*
 *	Forward
 */

static void TempDone();
static void UselessTemp();
static AllocationVector DoEnclosedBlocks();
static void PrintBenefits();	/* debug */

/*******************************************************
 *	Allocation Vector Construction		       *
 *******************************************************/

/*
 *	This constructs the allocation vector for the temporaries
 *	in a DAG.  A temporary is "allocated" when it is first stored
 *	into.  It will be allocated to the first vacant position in
 *	the vector.  That position will be made vacant when there
 *	are no more outstanding references to the temporary, as
 *	determined by UpdateActivity.  The benefit of a vector is
 *	thus the sum of the benefits of all the temporaries that
 *	were allocated to that spot.
 */

typedef struct AS {
	AllocationVector v;	/* Allocation entry for this "slot" */
	Identifier occupant;	/* Current occupant of this "slot" */
} *StatusAlloc;

static StatusAlloc status;	/* array[slot] of status entries */
static int next_slot;		/* First slot never used */

static void
HandleDelay(n)			/* Handle a Delayed store */
	DAG_Node n;
{
	assert(n->delayed->activity != 0);
	--(n->delayed->activity);
	if (n->delayed->activity == 0)
		TempDone(n->delayed);
}

static void
TempDone(n)		/* Global activity_r is an implied parameter */
	DAG_Node n;
{
	Identifier id;
	int slot;

	/*
	 *	We have to check IsAllocated and CanAllocate again,
	 *	since there might be temps that aren't in the list.
	 */

	id = n->carrier;
	if (id != NoId && !IsAllocated(id) &&
	    IsTemp(id) && CanAllocate(id, activity_r))
	{
		slot = 0;
		while (slot < next_slot && status[slot].occupant != id)
			++slot;
		assert(slot < next_slot);	/* Not Allocated!! */
		status[slot].occupant = NoId;
	}

	/*
	 *	Handle ordinary delayed store
	 */

	if (n->delayed != NULL && !(n->special_delay))
		HandleDelay(n);
}

static void
TempAlloc(id)
	Identifier id;
{
	int slot;
	int ben;		/* Benefit */

	if (NoShareDebug)
		slot = next_slot;
	else {
		slot = 0;
		while (slot < next_slot && status[slot].occupant != NoId)
			++slot;
	}

	/* slot = next_slot or slot is empty */

	if (slot == next_slot) {	/* Create a new entry */
		assert(status[slot].v == NULL);
		status[slot].v = CreateVector(0, id);
		++next_slot;
	}
	status[slot].occupant = id;
	ben = Benefit(id);
	status[slot].v->benefit += ben;
	Insert((int) id, status[slot].v->s);
}

/*
 *	Called by ActivityDag when a node is evaluated.  Puts the
 *	temp in a slot if necessary.
 */

static Boolean
TempEval(n)		/* activity_r is an implicit parameter */
	DAG_Node n;
{
	Identifier id = n->carrier;

	if (id != NoId && !IsAllocated(id) &&
	    IsTemp(id) && CanAllocate(id, activity_r))
	{
		/* Allocate a temporary */
		TempAlloc(id);
	}
	UpdateActivity(n, TempDone);
	if (n->special_delay) {
		assert(n->delayed != NULL);
		HandleDelay(n);
	}
	return False;
}

static AllocationVector
BlockVector(b, r)		/* Compute allocation vector for b */
	BasicBlock b;
	ResrcIndex r;
{
	AllocationVector v;	/* result */
	int slot;

	/* Initialize the allocation status */

	assert(status != NULL);		/* Should be allocated already */
	next_slot = 0;
	activity_r = r;			/* set implied parameter */

	(void) ActivityDag(b->Dag, TempEval);

	v = NULL;

	/*
	 * Each vector entry in status reflects one group of temporaries
	 * that can share a storage location.  The appropriate benefit
	 * sum is also recorded.  The result vector is constructed by
	 * repeated merging - an insertion sort.  We go backwards
	 * down the list for speed - the list is already likely partially
	 * ordered.
	 */

	for (slot = next_slot - 1; slot >= 0; --slot) {
		/*  assert(status[slot].occupant == NoId); /* dangling temp */
		status[slot].occupant = NoId;
		VectorMerge(&v, &status[slot].v);
	}

	/* Put any Unresolved variable that is live only in this block
	 * into the allocation vector.  We conservatively assume that
	 * it is live throughout this block.
	 */

	{
		int i;

		for (i = FirstElement(UnresolvedVars); i !=NoElement;
		     i=NextElement(i, UnresolvedVars))
		{
			Identifier id = (Identifier) i;
			int f = FirstNeed(id);	/* FlowIndex, or NoElement */

			if (f==b->FGindex &&
			    NextElement(f, Needs(id))==NoElement)
			{
				AllocationVector t = CreateVector(Benefit(id), id);

				VectorMerge(&v, &t);
			    	DelMember((int) id, UnresolvedVars);
			}
		}
	}

	if (AllocDebug > 2) {
		printf("Temp Vector for block #%d is\n\n", b->blocknum);
		PrintVector(v);
	}
	return v;

}

/*
 *	End of Temporary Allocation Vector Construction
 */

/*
 * Local Variable Allocation Vector Construction
 */

typedef struct UL {
	AllocationVector v;	/* for these user variables.
				 * Note that the vector is of length 1.
				 */
	LongSet life;		/* set(FlowNode): lifetime of these variables */
	struct UL *next;
} *UserList;

static UserList ul_free = NULL;	/* free cells */
static UserList ul_head = NULL;	/* head of current User Variable List */

static void
UserAlloc(id)
	Identifier id;
{
	UserList u;

	for (u=ul_head; ; u=u->next) {
		if (u == NULL) {
			/* We have fallen off the end of the current list.
			 * We must create a new entry.  We will add it at
			 * the front (this may not be optimal).
			 */
			if (ul_free == NULL) {
				u = GetStorage(s_NoStats, struct UL);
				CheckStorage(u, "storage for user variable allocation data", 0);
			} else {
				u = ul_free;
				ul_free = u->next;
			}
			u->next = ul_head;
			ul_head = u;
			u->v = CreateVector(Benefit(id), id);
			u->life = CreateSet((int) NumFlowNodes);
			CopySet(u->life, Needs(id));
	/**/		break;
		} else if (Disjoint(u->life, Needs(id))) {
			/*
			 * We have found an entry that can accomodate us:
			 * use it.
			 */
			AllocationVector t = CreateVector(Benefit(id), id);

			VectorSum(&u->v, &t);
			Union(u->life, u->life, Needs(id));
	/**/		break;
		}
	}
    	DelMember((int) id, UnresolvedVars);
}

static AllocationVector
UserDone(vec)
	AllocationVector vec;
{
	while (ul_head != NULL) {
		UserList u = ul_head;

		ul_head = u->next;
		VectorMerge(&vec, &u->v);
		DestroySet(u->life);
		u->life = NULL;
		u->next = ul_free;
		ul_free = u;
	}
	return vec;
}

/*
 *	Loop Temporary Allocation Vector Construction
 */

static AllocationVector
LoopVector(l, r)	/* Compute the allocation vector for l's variables */
	int l;		/* loop index */
	ResrcIndex r;
{
	AllocationVector v, lv;
	LTempList ltp;
	int ben;		/* benefit temporary */

	v = NULL;
	ltp = LoopPtrs[l]->ltemps;

	while (ltp != NULL) {
		assert(ltp->id != NoId);
		ben = Benefit(ltp->id);
		if (!IsAllocated(ltp->id) && CanAllocate(ltp->id, r)) {
			lv = CreateVector(ben, ltp->id);
			VectorMerge(&v, &lv);	/* Loop Temps compete */
		}
		ltp = ltp->next;
	}

	if (AllocDebug > 2) {
		printf("Loop Temp Vector for loop #%d is\n\n", l);
		PrintVector(v);
	}
	return v;
}

static void
DoLoops(r)
	ResrcIndex r;
{
	AllocationVector v, sv;		/* for building a.v. */
	int loopx;			/* current loop */
	BasicBlock b;			/* current block */
	LongSet blocks;			/* the blocks in the loop + p.h. */


	/*
	 *	We now compute the allocation vector for the loops.
	 *	We do the loops from the inside out.
	 */

	blocks = CreateSet(NumFlowNodes);

	for (loopx = 0; loopx < nloops; ++loopx) {
		CopySet(blocks, LoopPtrs[loopx]->loop);
		b = LoopPtrs[loopx]->prehead;
		if (b != NULL)
			Insert((int) b->FGindex, blocks);

		v = DoEnclosedBlocks(blocks, r);  /* Blocks inside the loop */
		if (AllocDebug > 2) {
			printf("vector for blocks enclosed in %d\n\n", loopx);
			PrintVector(v);
		}

		sv = LoopVector(loopx, r);	/* the loop temporaries */
		VectorMerge(&v, &sv);		/* They compete */

		{
			/* Put any variable that is live only in this loop
			 * into the allocation vector
			 */

			int FirstInBlocks;	/* FlowIndex, or NoElement */
			int i;

			assert(ul_head == NULL);
			FirstInBlocks = FirstElement(blocks);
			assert(FirstInBlocks != NoElement);
			for (i = FirstElement(UnresolvedVars); i !=NoElement;
			     i=NextElement(i, UnresolvedVars))
			{
				Identifier id = (Identifier) i;

				if (FirstInBlocks<=FirstNeed(id) &&
				    Subset(Needs(id),blocks))
					UserAlloc(id);
			}
		}
		FlowGraph[LoopPtrs[loopx]->head].v = UserDone(v);
		if (AllocDebug > 2) {
			printf("vector for loop in %d\n\n", loopx);
			PrintVector(v);
		}
	}

	DestroySet(blocks);
}

/*
 *	Construct allocation vector for local variables
 */

static AllocationVector
LocalVector()		/* Compute allocation vector for locals and params */
{
	AllocationVector v;
	int i;

	/*
	 *	This routine is never called for resource NumResources - 1,
	 *	since it is assumed that stack space has already
	 *	been allocated by pass 1.  Furthermore, we don't
	 *	include variables with non-positive benefit in the
	 *	vector.
	 */

	assert(ul_head == NULL);
	for (i = FirstElement(UnresolvedVars); i !=NoElement;
	     i=NextElement(i, UnresolvedVars))
		UserAlloc((Identifier) i);
	v = UserDone((AllocationVector) NULL);
	if (AllocDebug > 2) {
		printf("Local Variables Vector is\n\n");
		PrintVector(v);
	}

	return v;
}

/*
 *	Compute the composite allocation vector for those blocks that
 *	are contained in a loop.
 */

static AllocationVector
DoEnclosedBlocks(s, r)		/* Do the blocks making up a loop */
	LongSet s;
	ResrcIndex r;
{
	register FlowIndex f;		/* Current flownode ( =block) */
	register int iif;		/* Value from set functions */
	AllocationVector bv, v;

	v = NULL;
	iif = FirstElement(s);
	while (iif != NoElement) {
		f = (FlowIndex) iif;	/* Conversion now safe */

		assert(FlowGraph[f].block->reachable);
		if (!FlowGraph[f].visited) {
			/* This node is local only to this loop */
			bv = BlockVector(FlowGraph[f].block, r);
			VectorSum(&v, &bv);
			FlowGraph[f].visited = True;
		} else
		if (FlowGraph[f].v != NULL) {
			/* This is the header of an enclosed loop */
			VectorSum(&v, &FlowGraph[f].v);
			assert(FlowGraph[f].v == NULL);
		}
		/* Other blocks are already done! */
		iif = NextElement(iif, s);
	}
	return v;
}

/*
 *	Allocate variables and temporaries.
 */

/*
 *	Having determined that a particular temporary isn't needed,
 *	this removes it from the DAG.
 */

static void
UselessTemp(id)
	Identifier id;
{
	DAG_Node n;
	TempDefList dl, dl_next;

	dl = GetTempDefs(id);
	assert(dl != NULL);	/* at least one! */
	while (dl != NULL) {
		n = dl->def;
		assert(n != NULL);
		if (n->op != LEAFNOP) {		/* if so, already zapped */
			assert(n->attached != NULL && n->attached->id == id);
			assert(IsTransparent(id));
			DeleteID(id, n, True);		/* Remove it from the list */
			if( n->carrier == id )
				n->carrier = NoId;
			PruneTree(n, NotConditional, RecursivePrune);
		}
		dl_next = dl->next;
		DeleteTempDef(id, dl);
		dl = dl_next;
	}

	if (AllocDebug > 1)
		printf("Temp %d useless; zapped\n", id);
}

/*
 *	If we've just allocated an LNAME to the stack, we might
 *	have taken its address.  In this case, we have to go
 *	fix up the offset in the symbol table tree for the ADDR.
 */

static void
AllocLADDR(id)
	Identifier id;
{
	Operator op = IdOp(id);
	Identifier addr_id;
	TreeNode t, addr_t;

	assert(op == LNAME);

	t = IdTree(id);
	assert(t != NULL);

	addr_id = IdLookUp(ToADDR(op), (CONSZ) t->tn.rval, ANYTYPE);
	if (addr_id != NoId) {
		if (AllocDebug > 1)
			printf("Allocating &id: %d\n", id);
		addr_t = IdTree(addr_id);
		assert(addr_t != NULL);
		addr_t->tn.lval = t->tn.lval;
	}
}

static void
AllocId(params, id, r)
	InstrList *params;
	Identifier id;
	ResrcIndex r;
{
	TreeNode t;
	CONSZ old_lval;		/* usually, the offset */
	int ns;			/* name space of object */
	Boolean allocated;	/* If allocation successful */

	t = IdTree(id);
	if (t != NULL) {
		ns = IdOp(id);
		old_lval = t->tn.lval;		/* in case PNAME */
		if (IdBenefit(id) > 0) {
			allocated = AllocResource(id, r);
			if (allocated) {
				SetAllocated(id, True);
				UndoTransparent(id);
				if (ns == PNAME)
					LoadParam(r, params, t, old_lval);
				else
				if ((ns == LNAME) && (r == NumResources-1))
					AllocLADDR(id);
			}
		} else {
			/*
			 * 	There was no benefit in allocating it to r.
			 *	If the was the last resource, see if we
			 *	must allocate it.
			 */

			if (r == NumResources - 1) {
				if (!IsTransparent(id)) {
					++NumForcedTemps;
					allocated = AllocResource(id, r);
					assert(allocated);
					if (ns == LNAME)
						AllocLADDR(id);
				} else {
					UselessTemp(id);
				}
			}
		}
	}
}

static void
DoAllocation(params, v, r)
	InstrList *params;		/* Param Loads */
	AllocationVector v;
	ResrcIndex r;
{
	AllocationVector sv;	/* Scan Vector */
	int iid;		/* id (as int) returned from set op */
	Identifier id;		/* The actual id in question */

	sv = v;			/* Scan the allocation vector */
	while (sv != NULL) {

		/*
 		 *	Attempt to allocate objects in this
		 *	slot to resource r.
		 */

		iid = FirstElement(sv->s);
		while (iid != NoElement) {
			id = (Identifier) iid;      /* Conversion now safe */
			assert(CanAllocate(id, r));
			AllocId(params, id, r);
			iid = NextElement(iid, sv->s);
		}
		PccAdvanceAllocation();
		sv = sv->next;
	}
}

static void
Allocate(params, r)
	InstrList *params;
	ResrcIndex r;
{
	FlowIndex f;
	AllocationVector v;		/* developing answer */
	AllocationVector sv;		/* scratch vector */
	LongSet blocks;			/* Loop set + pre-header */
	Identifier id;			/* for scan of symbol table */

	/*
	 *	Initialize temporary storage
	 */

	blocks = CreateSet(NumFlowNodes);
	status = (StatusAlloc) calloc((unsigned) NextTempId, sizeof(struct AS));
	CheckStorage(status, "storage for %d allocation status entries", NextTempId);

	CompBenefits(r);

	for (f=0; f < NumFlowNodes; ++f) {
		if (FlowGraph[f].block->reachable) {
			FlowGraph[f].visited = False;
			FlowGraph[f].v = NULL;	       /* allocation vector */
		}
	}

	/*
	 * Calculate which user variables we wish to try to promote
	 * If this is the last resource, we only include LNAMEs that
	 * that don't already have stack allocations.  These may
	 * be included with negative benefit, because some may have
	 * to be allocated.
	 *
	 * Note: STATNAMEs occur only in f77.  UsedFirst determines
	 * if the variable is used before it is defined.  In f77
	 * this can happen for variables initialized in DATA
	 * statements, or when the variable is assumed to be
	 * static.
	 *
	 * The conditions for inclusion in the vector are:
	 *
	 *	0) Re-allocation (promotion) valid (i.e., !MayNotPromote)
	 *	   (This condition checked in CalcNeeds())
	 * 	1) LNAME, PNAME, or (STATNAME and not UsedFirst)
	 *	   (This condition checked in CalcNeeds())
	 * 	2) not already allocated by us
	 * 	3) can be allocated to this resource
	 * 	4) benefit > 0
	 */

	NullSet(UnresolvedVars);
	for (id = FirstId; id <= MaxIdentifier; ++id) {
		if (Needs(id)!=NULL && CanAllocate(id, r) && !IsAllocated(id)
		    &&
		    ((r != NumResources-1 && Benefit(id) > 0) || 
		     (IdTree(id) != NULL && !HasAllocation(id)))
		   )
			Insert((int) id, UnresolvedVars);
	}


	DoLoops(r);

	/*	We've computed the vectors for all loops.  To get the
	 *	vector for the entire function, we consider the function
	 *	as a loop (executed once!), with its locals and parameters
	 *	the loop temporaries.
	 */

	NullSet(blocks);
	for (f = 0; f < NumFlowNodes; ++f) {
		/*  Since we scan anyway, only look at those that matter */
		if ((!FlowGraph[f].visited || FlowGraph[f].v != NULL) &&
		    FlowGraph[f].block->reachable )
			Insert((int) f, blocks);
	}

	v = DoEnclosedBlocks(blocks, r);

	sv = LocalVector();
	VectorMerge(&v, &sv);		/* v is the final answer */

	if (AllocDebug > 1) {
		printf("Final allocation vector\n\n");
		PrintVector(v);
	}

	if (AllocDebug > 3) {
		printf("Final benefit values\n\n");
		PrintBenefits();
	}

	/*
	 *	We now allocate the temps and promote the variables
	 */

	DoAllocation(params, v, r);

	/* Allocation Complete */

	free(status);
	VectorFree(v);
	DestroySet(blocks);
}

static void
CalcNeeds()	/* compute NeedsSpace sets for each variable */
{
	Identifier id;
	FlowIndex f;
	LongSet bn = CreateSet((int) (MaxIdentifier+1));

	/* empty all NeedsSpace fields */
	for (id = FirstId; id <= MaxIdentifier; ++id) {
		Operator ns = IdOp(id);

		if (!MayNotPromote(id) && 
		    (ns == LNAME || ns == PNAME ||
		    (ns == STATNAME && !UsedFirst(id))))
		{
			assert(!IsAllocated(id));
		    	EmptyNeeds(id);
		}
	}

	/* fill in each NeedsSpace from each block's info */
	for (f=0; f<NumFlowNodes; f++) {
		BasicBlock b = FlowGraph[f].block;
		int i;

		if (b->reachable) {
			/* bn := b->ld.In | b->ld.Out | b->ld.PDef */
			Union(bn, b->ld.In, b->ld.Out);
			Union(bn, bn, b->ld.PDef);
			for (i=FirstElement(bn); i!=NoElement; i=NextElement(i, bn))
				AddNeeds(f, (Identifier) i);
		}
	}
	DestroySet(bn);

	/* fill in FirstNeed field */
	for (id = FirstId; id <= MaxIdentifier; ++id)
		SetFirstNeed(id);
}

/*
 *	Run the allocation algorithm for each available resource
 */

void
AllocVars(params)
	InstrList *params;
{
	ResrcIndex r;
	Identifier id;

	InitRefs();
	CalcNeeds();
	UnresolvedVars = CreateSet((int)(MaxIdentifier+1));
	for (r = 0; r < NumResources; ++r) {
		if (AllocDebug > 1)
			printf("\nAllocating Resource # %d\n", r);
		Allocate(params, r);
	}

	for (id = FirstId; id <= MaxIdentifier; id++ )
		if (Needs(id)!=NULL)
			FreeNeeds(id);

	DestroySet(UnresolvedVars);
}

static void
PrintBenefits()
{
	Identifier id;
	char *p;

	for( id = FirstId; id <= MaxIdentifier; id++ ) {
		p = IdText(id);
		printf("%s(%d): benefit = %d\n",
			(p != NULL) ? p : "..no name..",
			id, Benefit(id));
	}
}
