/************************************************************************
*									*
*		uccio.c - io package for compiler to avoid stdio	*
*									*
************************************************************************/

#include "uccio.h"
#include "varargs.h"

#define reg register
#define uns unsigned
#ifndef VOID
#define VOID
#endif
#ifndef CHARPTR
#define CHARPTR char *
#endif

#define VBUFSIZ	8192
#define NFILES 10

FILE	_ucciob[] = {
	0,0,0,0,0,R_MODE,               /* standard input               */
	0,0,0,0,1,W_MODE,               /* standard output              */
	0,0,0,0,2,W_MODE,               /* standard error               */
	0,0,0,0,-1,0,
	0,0,0,0,-1,0,
	0,0,0,0,-1,0,
	0,0,0,0,-1,0,
	0,0,0,0,-1,0,
	0,0,0,0,-1,0,
	0,0,0,0,-1,0
};

#define         EOL             '\n'
#define ufopen(fp) ((fp)->_ucmd & (R_MODE|W_MODE))

static char	*idbase = 0;
#ifdef linux
static char	dbuf[16];
#else
static char	dbuf[16] = 0;
#endif
static char	hextab[] = "0123456789abcdef";
static char	fillch = 0;
static short	fieldlen = 0;
static short	left = 0;
static short	vopen = 0;
static char	*vtop = 0,		/* max memory actually written */
		*vmax = 0;		/* max memory actually acquired */
char		*sbrk();
long		lseek();

/* This io package is tailored for the Unidot compiler system.
   It has several attributes:  1) it uses only the read/write interface
   of the system, therefore should be easier to port to different
   operating systems.  2) It does not call malloc, depending rather on
   specific calls to setbuf if the user wants buffered IO.  3) It does
   not implement the full set of printf routines, but instead just enough
   to make the compiler happy (hence is much smaller).
*/

/*      _ucget() -- standard i/o fill-buffer function.          */
VOID
_ucget( fp ) reg FILE *fp;{
	static char	inbuf[2];	/* short term usage only	*/
	reg		i;
	long		l;
	int             x;


/* If the stream lacks a buffer,  read a single byte and return it	*/

	if( !ufopen(fp) ) return EOF;
	fp->_ucct = 0;
	if( fp->_ucmd & W_LAST || !(fp->_ucmd & R_MODE) ) return EOF;
	fp->_ucmd |= R_LAST;
	if( fp->_ucba == NULL ){
		i = (int)fp->_ucpt;		/* check unget char */
		fp->_ucpt = 0;
		if( i ) return i & 0xff;
		if( read( fp->_ucfi, inbuf, 1 ) != 1 ) return EOF;
		return inbuf[0] & 0xff;
	}
	if( fp->_ucmd & V_MODE ){		/* virtual file		*/
		l = (long)vtop - (long)fp->_ucpt;
		if( l <= 0 ) return EOF;
		fp->_ucct = l > VBUFSIZ ? VBUFSIZ : l;
	} else {
	
		fp->_ucpt = fp->_ucba;
		i = read( fp->_ucfi, fp->_ucba, fp->_ucsz);
		if( i < 1 ) return EOF;		/* read failed		*/
		fp->_ucct = i;
	}
	fp->_ucct--;
	return *fp->_ucpt++ & 0xff;
}
/*	_ucput() -- standard i/o flush-buffer function.			*/

VOID
_ucput( ch, fp ) reg ch; reg FILE *fp;{

	reg char	*p;
	reg		i;
	static short	rqsize = VBUFSIZ;	/* size of VM getter	*/
	static char	outbuf[2];
	long		l;
	int             x;

	fp->_ucct = 0;
	if( !ufopen(fp) || !(fp->_ucmd & W_MODE) || (fp->_ucmd & R_LAST) )
		return EOF;
	fp->_ucmd |= W_LAST;
	if( fp->_ucba == NULL ){
		outbuf[0] = ch;
		if( write( fp->_ucfi, outbuf, 1 ) == EOF ) return EOF;
		return ch & 0xff;
	}
	if( fp->_ucmd & V_MODE ){		/* virtual file */
		vtop = fp->_ucpt;
		l = (long)vmax - (long)vtop;
		if( l <= 0 ){
			while( sbrk(rqsize) == (char *)-1 ){
				if( rqsize <= 128 )
					MSG( "6vm file overflow at size %d",
							fp->_ucpt-fp->_ucba);
				rqsize >>= 1;	/* divide by two and try again*/
			}
			l += rqsize;
			vmax += rqsize;
		}
		if( l > VBUFSIZ ) l = VBUFSIZ;
		fp->_ucct = l;
	} else if( fp->_ucmd & L_MODE ){        /* line buffered */
		fp->_ucct = 0;
		i = fp->_ucpt - fp->_ucba;
		if (i >= fp->_ucsz) {
			fp->_ucpt = fp->_ucba;
			if( i > 0 && write( fp->_ucfi, fp->_ucba, i ) != i)
				return EOF;
			i = 0;
		}
		p = fp->_ucpt;
		*p = ch;
		fp->_ucpt = ++p;
		if (ch == EOL) {
			fp->_ucpt = fp->_ucba;
			i++;
			if (write( fp->_ucfi, fp->_ucba, i ) != i)
				return EOF;
		}
		return ch & 0xff;
	} else {
		i = fp->_ucpt - fp->_ucba;
		fp->_ucct = fp->_ucsz;
		fp->_ucpt = fp->_ucba;
		if( i != 0 && write( fp->_ucfi, fp->_ucba, i ) != i) return EOF;
	}
	p = fp->_ucpt;
	*p = ch;
	fp->_ucpt = ++p;
	fp->_ucct--;
	return ch & 0xff;
}

/* "_cleanup()" -- close all standard i/o files
   int _cleanup();

   "_cleanup()" closes all open standard i/o files, flushing all
   files in  write mode.  "_cleanup()" is called  automatically
   by "exit()".	     */

/* dv: the Gould has got things mixed up so there is some redefining
   needed.  To do this #define exit(a) uexit(a) which calls ucleanup and
   the exit */

#ifndef exit
#define _ucleanup _cleanup
#endif

VOID
_ucleanup(){
	reg FILE *fp;
	for( fp = &_ucciob[0]; fp < &_ucciob[NFILES]; fp++ ) fclose( fp );
}

/*	fclose() -- close a buffered i/o file			*/
VOID
fclose( fp ) reg FILE *fp;{
	if( !ufopen(fp) ) return;
	if( fp->_ucmd & V_MODE ){
		if( fp->_ucpt != fp->_ucba ) vtop = fp->_ucpt;
	} else {
		if( fp->_ucmd & W_MODE )fflush( fp );
		close( fp->_ucfi );
	}
	fp->_ucfi = -1;
	fp->_ucpt = fp->_ucba;
	fp->_ucct = 0;
	fp->_ucmd = 0;
}

/*	ftrunc() -- truncate file to 0 length */
VOID
ftrunc( fname, fp ) char *fname; reg FILE *fp; {
	/* both the name and the file descriptor are required because
	   the action is different depending on the way the file was
	   opened	*/

	if( !ufopen(fp) ) return;
	if( fp->_ucmd & V_MODE ){
		vtop = fp->_ucba;
	} else {
		lseek( fp->_ucfi, 0L, 0 );
		close( creat( fname, 0666 ) );
	}
	fp->_ucmd &= ~(W_LAST|R_LAST);
	fp->_ucpt = fp->_ucba;
	fp->_ucct = 0;
}

/*	fflush() -- flush a buffered i/o file			*/
int
fflush( fp ) reg FILE *fp;{
	int i;
	int x;

	if( !ufopen(fp) ) return;
	if( !(fp->_ucmd & V_MODE) &&
	    fp->_ucba && (i = fp->_ucpt - fp->_ucba) &&
	    (fp->_ucmd & W_MODE) && !(fp->_ucmd & R_LAST)) {
		write( fp->_ucfi, fp->_ucba, i );
	}
	fp->_ucpt = fp->_ucba;
	fp->_ucmd &= ~(R_LAST|W_LAST);
	fp->_ucct = 0;
	return 0;			/* return success	     */
}

/*	fopen and freopen - uccio buffered open functions		*/
FILE *
freopen(fname, fmode, fp ) char *fname, *fmode; reg FILE *fp; {
	int nn, x;

	/* freopen connects a named file in a specifed mode to a
	   specific iob, usually the stdin, stdout, or stderr iob's
	*/

	if( ufopen(fp) ) fclose( fp );
	fp->_ucmd = 0;
	switch( *fmode ){

case 'v':	if( vopen ) return NULL;
		vtop = vmax = fp->_ucba = fp->_ucpt = sbrk(0);
		vopen++;
		fp->_ucmd |= V_MODE|R_MODE|W_MODE;
		return fp;

case 'r':	fp->_ucfi = open( fname, 0 );	/* open for reading	*/
		fp->_ucmd |= R_MODE;
		break;

case 'w':	close( creat( fname, 0666 ) );
		fp->_ucfi = open( fname, 1 );	/* open for write	 */
		fp->_ucmd |= W_MODE;
		break;

case 'u':       fp->_ucfi = open( fname, 2 );   /* open for update       */
			/* NOTE: must be 2 for vms */
		if( (fp->_ucfi & 0xff) != 0xff ){
			fp->_ucmd |= W_MODE;
			break;
		}

case 't':		/* open for read and write (temp use) */
		fp->_ucfi = creat( fname, 0666 );
		close( fp->_ucfi );
		fp->_ucfi = open( fname, 2 );
		fp->_ucmd |= R_MODE|W_MODE;
		break;
		
default:	return NULL;			/* bad modeletter */
	}
	if( (fp->_ucfi & 0xff) == 0xff ) return NULL;
	fp->_ucct = 0;
	fp->_ucpt = fp->_ucba;
	return fp;
}

FILE *
fopen(fname, fmode) char *fname, *fmode; {
    /* Fopen opens a named file in a specified mode, and attaches same to a
     * free iop.  Endopen is called to do the actual work.  Return value is
     * the fopened iop if no errors arose; else NULL.		 */
	reg		i;

	for( i=0; i<NFILES; i++ ) if( !ufopen(&_ucciob[i]) ) break;
	if( i >= NFILES ) return NULL;
	return freopen(fname, fmode, &_ucciob[i] );
}

VOID
ungetc( c, fp ) reg FILE *fp; {
	if( !(fp->_ucmd & R_LAST) ) return EOF;
	if( fp->_ucba == NULL )
		fp->_ucpt = (char *)(0x100 | c);
	else {
		fp->_ucct++;
		*--(fp->_ucpt) = c;
	}
}

VOID
setbuf( fp, buf ) reg FILE *fp; char *buf; {
	if( fp->_ucmd & (R_LAST|W_LAST|V_MODE) ) return;
	fp->_ucba = fp->_ucpt = buf;
	fp->_ucct = 0;
	fp->_ucsz = BUFSIZ;
}

VOID
setvbuf( fp, buf, type, sz ) reg FILE *fp; char *buf; {

	if( fp->_ucmd & (R_LAST|W_LAST|V_MODE) ) return;
	if( type == _IONBF ){
		fp->_ucba = NULL;
	} else {
		fp->_ucba = fp->_ucpt = buf;
		fp->_ucsz = sz;
		if( type == _IOLBF ){
			fp->_ucmd |= L_MODE;
		}
	}
	fp->_ucct = 0;
}

VOID
setstr( idp ) char *idp; {

	idbase = idp;
}

VOID
fseek( fp, p, flag ) reg FILE *fp; long p; {

	reg char	*xp;
	reg		k;
	long		l;

	if( !ufopen(fp) ) return;
	if( fp->_ucmd & V_MODE ){
		if( fp->_ucpt > vtop ) vtop = fp->_ucpt;
		switch( flag ){
		case 0:	xp = fp->_ucba;	break;
		case 1:	xp = fp->_ucpt;  break;
		case 2:	xp = vtop;
		}
		xp += p;
		if( xp < fp->_ucba || xp > vmax ) MSG("7bad vm seek");
		fp->_ucpt = xp;
		fp->_ucct = 0;			/* show buffer empty	*/
		fp->_ucmd &= ~(W_LAST|R_LAST);
		return;
	}
	if( (!(fp->_ucmd & W_MODE) || (fp->_ucmd & R_LAST)) &&
	    fp->_ucba && (p || flag) ){

		/* to get here we must either be reading the file or
		   have an unwriteable file and the file must be
		   buffered, and we must be going to someplace other
		   than the start of the file

		   the idea here is to maximize the opportunities for
		   doing raw reads from the disk.  Hence we try to
		   force block aligns, and further we try to avoid
		   re-reads if the required location is already in
		   the buffer.
		*/

		l = lseek( fp->_ucfi, 0L, 1 ); /* get where we really are */
		switch( flag ){
	case 1:		p += l - fp->_ucct;			break;
	case 2:		p += lseek( fp->_ucfi, 0L, 2 );
		}

		/* p now set to where we really want to go */

		k = fp->_ucct + (fp->_ucpt - fp->_ucba);

		/* k has the size of the last read */

		if( p < l && p >= l - k ){

			/* the seek position is in buffer */

			fp->_ucpt = fp->_ucba + (p - l + k);
			fp->_ucct = l - p;
		} else {	

			/* must really do a seek and a read  */

			k = p & (fp->_ucsz-1);  /* offset in block      */
			p -= k;			/* start of block	*/
			lseek( fp->_ucfi, p, 0 );
			fp->_ucct = read(fp->_ucfi,fp->_ucba,fp->_ucsz);
			if( k > fp->_ucct ) k = fp->_ucct;
			if( k < 0 ) fp->_ucct = k = 0;	/* error	*/
			fp->_ucpt = fp->_ucba + k;	/* set pointer	*/
			fp->_ucct -= k;			/* adjust count	*/
		}
		fp->_ucmd |= R_LAST;
	} else {

		/* all other cases	*/

		fflush( fp );			/* start with a flush	*/
		lseek( fp->_ucfi, p, flag );	/* use raw seek		*/
		fp->_ucmd &= ~(R_LAST|W_LAST);	/* no last operation	*/
	}
}

long
ftell(fp) reg FILE *fp; { /* where are we */

	long	l;
	if( !ufopen(fp) ) return -1L;
	l = lseek( fp->_ucfi, 0L, 1 );
	if( fp->_ucba == NULL ){
		if( fp->_ucpt ) l--;
		return l;
	}
	if( fp->_ucmd & R_LAST ) return l - fp->_ucct;
	if( fp->_ucmd & W_LAST ) return l + (fp->_ucpt - fp->_ucba);
	return l;
}

/*	here starts the printf simulator		*/
VOID
_doprf(fmt,args,fp) char *fmt; va_list *args; FILE *fp; {

	reg va_list	argp;
	reg char	*cp;
	reg int		c;
	reg int		i;
	reg long	l;
	reg char	*push;
	reg int		prec;

	argp = *args;
	cp = fmt;
	fillch = ' ';		/* default fill character	*/
	fieldlen = -1;		/* default field length		*/
	left = 0;
	push = 0;
	prec = -1;
	for(;;){
		c = *cp++;
		if( c == 0 ){
			if( push ){
				cp = push;
				push = 0;
				continue;
			}
			return;
		}
		if( c != '%' ){
pc:			putc(c,fp);
			continue;
		}
		c = *cp++;
		if( c == '-' ){
			left++;
			c = *cp++;
		}
		if( c == '0' ){
			fillch = '0';
			c = *cp++;
		}
		if( c >= '0' && c <= '9' ){
			fieldlen = 0;
			while( c >= '0' && c <= '9' ){
				fieldlen = fieldlen*10 + c - '0';
				c = *cp++;
			}
		} else
		if( c == '*' ){
			fieldlen = va_arg(argp, unsigned );
			c = *cp++;
		}
		if( fieldlen < 0 ) left = 0;
		if( c == '.' && cp[0] == '*' && cp[1] == 's' ){
			/* from dv: %.*s get string len from variable */
			c = 's';
			cp += 2;
			prec = va_arg(argp, unsigned );
		}
		if( c == 'l' ){		/* handle a long	*/
			l = va_arg(argp, long );
			c = *cp++;
			switch( c ){
	case 'o':		putu(fp,l,3);		continue;
	case 'x':		putu(fp,l,4);		continue;
	case 'd':		putl(fp,l);		continue;
			}
			goto pc;
		}
		switch( c ){

	case 'o':	putu(fp, (long)va_arg(argp,unsigned), 3); continue;
	case 'x':	putu(fp, (long)va_arg(argp,unsigned), 4); continue;
	case 'd':	putl(fp, (long)va_arg(argp,int));	    continue;
	case 'r':	push = cp; cp = va_arg(argp,CHARPTR);     continue;
	case 's':	puts(fp,va_arg(argp,CHARPTR));		   continue;
	case 'n':	i = va_arg(argp,unsigned);
			puts(fp,i?idbase+i:NULL);		   continue;
	case 'c':	c = va_arg(argp,unsigned) & 0xff;
			if( c >= ' ' && c <= 0176 )		   goto pc;
			if( c == '\t' )				   goto pc;
			if( c == '\n')				   goto pc;
			if( c == 014 )				   goto pc;
			putc( '<', fp );
			putu( fp, (long)c, 3 );
			c = '>';				   goto pc;
		}
		goto pc;
	}
}

static
_putdb( fp, n ) reg FILE *fp;reg n;{

	reg	i;

	i = fieldlen - n;
	if( left == 0 ) while( --i >= 0 ) putc( fillch, fp );
	while( --n >= 0 ) putc( dbuf[ n ], fp );
	while( --i >= 0 ) putc( fillch, fp );
	fillch = ' ';
	fieldlen = -1;
	left = 0;
}

VOID
putu( fp, n, b ) FILE *fp;long	n; {		/* put out unsigned stuff */

	reg	i,
		m;

	m = (1 << b) - 1;	/* form a mask */
	i = 0;
	do dbuf[ i++ ] = hextab[n & m]; while( (n >>= b) && i < 8 );
	_putdb( fp, i );
}

VOID
putl( fp, n ) FILE *fp; long n;{		/* put out decimal stuff */

	reg	i;
	reg	sign;

	sign = 0;
	if( n < 0 ){
		sign++;
		if( (n = -n) < 0 ){
			puts( fp, "-2147483648" );
			return;
		}
	}
	i = 0;
	do dbuf[i++] = (n % 10) + '0'; while( (n /= 10) != 0 );
	if( sign ) dbuf[i++] = '-';
	_putdb( fp, i );
}


VOID
puts( fp, s ) FILE *fp; reg char *s; {

	reg	c;
	reg	n;
	char	*ssave;

	if( s == NULL ) s = "<nil>";
	n = 0;
	if( left == 0 ){
		ssave = s;
		while( *s != 0 ) s++;
		c = s - ssave;
		s = ssave;
		if( fieldlen < 0 ) fieldlen = c;
		c = fieldlen - c;
		for(; n<c; n++ ) putc( ' ', fp );
	}
	for( ; n<fieldlen && (c = *s++); n++ ) putc( c, fp );
	while( ++n <= fieldlen ) putc( ' ', fp );
	fieldlen = -1;
	left = 0;
	fillch = ' ';
}

VOID
fprf(fp,fmt,va_alist) FILE *fp;  char *fmt; va_dcl {
	va_list	pvar; va_start(pvar); _doprf(fmt,&pvar,fp); }

						/*VARARGS1*/
VOID
dprt(fmt,va_alist) char *fmt; va_dcl {
	va_list	pvar; va_start(pvar); _doprf(fmt,&pvar,stderr); }

						/*VARARGS1*/
VOID
prt(fmt,va_alist) char *fmt; va_dcl {
	va_list	pvar; va_start(pvar); _doprf(fmt,&pvar,stdout); }

						/*VARARGS2*/
VOID
sprt(buf,fmt,va_alist) char *buf,*fmt; va_dcl {
	va_list	pvar;
	static FILE	fake;

	va_start(pvar);
	fake._ucpt = buf;
	fake._ucct = 4095;
	fake._ucfi = 2;				/* if we spill */
	_doprf(fmt,&pvar,&fake);
	*fake._ucpt = 0;			/* terminate the string */
}
